/*
   File: leftcorner.c
   Defines a leftcorner recursive backup parser generator
*/

/* global includes */
#include <stdio.h>

/* libeag includes */
#include <export.h>
#include <error.h>
#include <memalloc.h>
#include <warshall.h>

/* local includes */
#include <tree.h>
#include <main.h>
#include <common.h>
#include <typecheck.h>
#include <numbering.h>
#include <recursion.h>
#include <codemeta.h>
#include <eag_unparser.h>
#include <editor.h>
#include <gentemplates.h>
#include <topdown.h>
#include <lookahead.h>
#include <leftcorner.h>

/*
   Allocate space and calculate the leftcorner relation
   Note that:
   member_leftcorner[i * nr_of_hyper_rules + j] = 1 <=>
   member i is a leftcorner of rule j <=>
   member i <LC rule j
*/
private char *member_leftcorner;
private char *member_leftcorner_closure;
private void allocate_space_for_relations ()
	{ int i,j;
	  member_leftcorner = (char *) ckcalloc
			(nr_of_members * nr_of_hyper_rules, sizeof (char));
	  member_leftcorner_closure = (char *) ckcalloc
			(nr_of_members * nr_of_hyper_rules, sizeof (char));
	  for (i=0; i < nr_of_members; i++)
	     for (j=0; j < nr_of_hyper_rules; j++)
		{ member_leftcorner [i*nr_of_hyper_rules + j] = 0;
		  member_leftcorner_closure [i*nr_of_hyper_rules + j] = 0;
		};
	};

private void compute_member_relation_in_mem (hyper_rule rule, member m)
	{ int mem_nr = m -> number;
	  int rule_nr = rule -> number;
	  switch (m -> tag)
	     { case tag_call:
		  { hyper_rule callee = m -> u.cl -> def;
		    if (callee -> kind & (h_predicate | h_semipredicate))
		       return;
		  }; break;
	       case tag_terminal:
		  if (strlen (m -> u.terminal) == 0) return;
		  break;
	       case tag_cut: return;
	       default: break;
	     };
	  member_leftcorner [mem_nr * nr_of_hyper_rules + rule_nr] = 1;
	};

private void compute_member_relation_in_mems (hyper_rule rule, member_list mems)
	{ int i;
	  if (mems == member_list_nil) return;
	  for (i = 0; i < mems -> nrofms; i++)
	     { compute_member_relation_in_mem (rule, mems -> ms[i]);
	       if (mems -> ms[i] -> empty == h_neverproducesempty) return;
	     };
	};

private void compute_member_relation (hyper_rule rule)
	{ int i;
	  alt_list alts = rule -> alts;
	  if (rule -> ext) return;
	  if (rule -> kind & (h_predicate | h_semipredicate)) return;
	  for (i=0; i < alts -> nrofas; i++)
	     compute_member_relation_in_mems (rule, alts -> as[i] -> members);
	};

private void compute_member_relations ()
	{ int i;
	  for (i = 0; i < nr_of_hyper_rules; i++)
	     compute_member_relation (all_hyper_rules[i]);
	};

private void dump_leftcorner_relation ()
	{ int i,j;
	  wlog ("dumping member left corner relation");
	  for (i = 0; i < nr_of_members; i++)
	     for (j = 0; j < nr_of_hyper_rules; j++)
		if (member_leftcorner [i*nr_of_hyper_rules + j])
		   { unparse_member (all_members[i]);
		     eprint_log ("  < %s\n", all_hyper_rules[j] -> nonterminal);
		   };
	};

/*
   refl_leftc_relation [j * nr_of_hyper_rules + i] = 1 <=>
   j is called by i in zero or more steps as a (possibly) leftmost member <=>
   rule j <LC* rule i <=>
   rule j <LC+ rule i V j = i
*/
private char *refl_leftc_relation;
private void build_reflexive_leftcorner_relation ()
	{ int i,j;
	  refl_leftc_relation = (char *) ckcalloc
			(nr_of_hyper_rules * nr_of_hyper_rules, sizeof (char));
	  for (i=0; i < nr_of_hyper_rules; i++)
	     for (j=0; j < nr_of_hyper_rules; j++)
		refl_leftc_relation [i * nr_of_hyper_rules + j] =
		   leftc_relation [i * nr_of_hyper_rules + j];
	  /* add reflexivety */
	  for (i=0; i < nr_of_hyper_rules; i++)
	     refl_leftc_relation [i * nr_of_hyper_rules + i] = 1;
	};

/*
   Compute the reflexive closure of the member_leftcorner relation
   Note that:
   member_leftcorner_closure[i * nr_of_hyper_rules + j] = 1 <=>
   member i <LC* rule j
*/
private void compute_closure_of_member_relation ()
	{ int i,j,k;
	  for (i = 0; i < nr_of_members; i++)
	     for (j = 0; j < nr_of_hyper_rules; j++)
		if (member_leftcorner[i*nr_of_hyper_rules + j])
		   for (k=0; k < nr_of_hyper_rules; k++)
		      if (refl_leftc_relation[j * nr_of_hyper_rules + k])
			 member_leftcorner_closure[i*nr_of_hyper_rules + k] = 1;
	};

private void dump_leftcorner_closure_relation ()
	{ int i,j;
	  wlog ("dumping closure of left corner relation");
	  for (i = 0; i < nr_of_members; i++)
	     for (j = 0; j < nr_of_hyper_rules; j++)
		if (member_leftcorner_closure [i*nr_of_hyper_rules + j])
		   { unparse_member (all_members[i]);
		     eprint_log ("  <* %s\n",all_hyper_rules[j] -> nonterminal);
		   };
	};

private void compute_leftcorner_relations ()
	{ warning ("computing left corner relations...");
	  allocate_space_for_relations ();
	  compute_member_relations ();
	  if (full_verbose) dump_leftcorner_relation ();
	  build_reflexive_leftcorner_relation ();
	  compute_closure_of_member_relation ();
	  if (full_verbose) dump_leftcorner_closure_relation ();
	};

/*
   Code goals
*/
private int goalnr;
private void generate_goal_for_rule (hyper_rule rule)
	{ if (rule -> ext) return;
	  if (!rule -> reachable) return;
	  if (rule -> kind & (h_predicate | h_semipredicate)) return;
	  rule -> goal = goalnr;
	  goalnr++;
	};

private void generate_goal_for_rules ()
	{ int i;
	  warning ("generating goals for hyper rules");
	  goalnr = 0;
	  for (i = 0; i < nr_of_hyper_rules; i++)
	     generate_goal_for_rule (all_hyper_rules[i]);
	};

private char *goal_relation;
private void allocate_goal_relation ()
	{ int i,j;
	  goal_relation = ckcalloc (goalnr * goalnr, sizeof (char));
	  for (i = 0; i < goalnr; i++)
	     for (j = 0; j < goalnr; j++)
		goal_relation [i * goalnr + j] = 0;
	};

private void fill_goal_relation_for_rules (int i, int j)
	{ hyper_rule rule_i = all_hyper_rules[i];
	  hyper_rule rule_j = all_hyper_rules[j];
	  int goal_i = rule_i -> goal;
	  int goal_j = rule_j -> goal;

	  if (rule_i -> ext) return;
	  if (!rule_i -> reachable) return;
	  if (rule_i -> kind & (h_predicate | h_semipredicate)) return;
	  if (rule_j -> ext) return;
	  if (!rule_j -> reachable) return;
	  if (rule_j -> kind & (h_predicate | h_semipredicate)) return;
	  goal_relation [goal_i * goalnr + goal_j] = 
	     refl_leftc_relation [i * nr_of_hyper_rules + j];
	};

private void generate_goal_relation ()
	{ int i,j;
	  warning ("generating needed subset of leftcorner relation");
	  allocate_goal_relation ();
	  for (i = 0; i < nr_of_hyper_rules; i++)
	     for (j = 0; j < nr_of_hyper_rules; j++)
		fill_goal_relation_for_rules (i,j);
	};

/*
   Forward declare parser routines for every rule
*/
private void generate_declarations_for_rule (hyper_rule rule)
	{ if (!rule -> reachable) return;
	  if (rule -> kind & (h_predicate | h_semipredicate))
	     fprintf (out, "%s void %s_%s ();\n",
		rule -> ext?"import":"private",
		rule_qualifier (rule), rule -> nonterminal);
	  else
	     { fprintf (out, "private void rule_%s ();\n", rule -> nonterminal);
	       fprintf (out, "private void get_%s ();\n", rule -> nonterminal);
	       fprintf (out, "private void red_%s ();\n", rule -> nonterminal);
	       if (rule -> empty == h_mayproduceempty)
		  { fprintf (out, "private void empv_%s ();\n",
			     rule -> nonterminal);
		    fprintf (out, "private void emp_%s ();\n",
			     rule -> nonterminal);
	          };
	     };
	};

private void generate_rule_declarations ()
	{ int i;
	  for (i = 0; i < nr_of_hyper_rules; i++)
	     generate_declarations_for_rule (all_hyper_rules[i]);
	  fprintf (out, "\n");
	};

/*
   Code the goals
*/
private void code_goal_for_rule (hyper_rule rule)
	{ if (rule -> ext) return;
	  if (!rule -> reachable) return;
	  if (rule -> kind & (h_predicate | h_semipredicate)) return;
	  fprintf (out, "#define goal_%s %d\n",
		   rule -> nonterminal, rule -> goal);
	};

private void code_goal_for_rules ()
	{ int i;
	  fprintf (out, "/* code goals for non predicate syntax rules */\n");
	  for (i = 0; i < nr_of_hyper_rules; i++)
	     code_goal_for_rule (all_hyper_rules[i]);
	  fprintf (out, "\n");
	};

private void code_goal_relation ()
	{ int i,j;
	  int eltnr = 0;
	  int mincol = (goalnr < 20)?goalnr:20;
	  fprintf (out,
		"#define on_spine_to_goal(x) lc_rel[x * %d + topi()]\n",
		goalnr);
	  fprintf (out, "private char lc_rel[] =\n\t{ ");
	  for (i = 0; i < goalnr; i++)
	     for (j = 0; j < goalnr; j++)
		{ fprintf (out, "%d, ", goal_relation[i * goalnr + j]);
		  eltnr++;
		  if (eltnr == mincol) 
		     { fprintf (out, "\n\t  ");
		       eltnr = 0;
		     };
	        };
	  fprintf (out, "%s\t};\n\n", (eltnr)?"\n":"");
	};

/*
   Generate parser routines for a leftcorner parser
*/
private void generate_rule_routine (hyper_rule rule)
	{ if (!rule -> reachable) return;
	  fprintf (out, "private void rule_%s ()\n", rule -> nonterminal);
	  fprintf (out, "\t{\n");
	  if (traceflag)
	     fprintf (out, "\t  trace_enter (\"rule_%s\");\n",
		     rule -> nonterminal);
	  may_generate_lookahead_check (rule, rule -> first);
	  fprintf (out, "\t     { pushi (goal_%s);\n", rule -> nonterminal);
	  fprintf (out, "\t       pushq (get_%s);\n", rule -> nonterminal);
	  fprintf (out, "\t       callq ();\n");
	  fprintf (out, "\t       pop(2);\n");
	  fprintf (out, "\t     };\n");
	  if (rule -> empty == h_mayproduceempty)
	     { may_generate_lookahead_check (rule, rule -> follow);
	       fprintf (out, "\t     { pushq (empv_%s);\n",
			rule -> nonterminal);
	       fprintf (out, "\t       callq ();\n");
	       fprintf (out, "\t       pop(1);\n");
	       fprintf (out, "\t     };\n");
	     };
	  fprintf (out, "\t  pushq (rule_%s);\n", rule -> nonterminal);
	  if (traceflag)
	     fprintf (out, "\t  trace_leave (\"rule_%s\");\n",
		     rule -> nonterminal);
	  fprintf (out, "\t}\n\n");
	};

private void generate_emp_call (call c, int *sonnr, int *nrpushes, int vstrict)
	{ if (layoutflag && (c -> def == layout_rule))
	     { fprintf (out, "\t  pushq (emp_%s);\n", c -> nonterminal);
	       *nrpushes += 1;
	       return;
	     };
	  fprintf (out, "\t  pushi (%d);\n", *sonnr);
	  fprintf (out, "\t  pushq (link_son);\n");
	  generate_display (c -> display, nrpushes);
	  fprintf (out, "\t  pushq (make_affix_link);\n");
	  if (c -> def -> kind & h_predicate)
	     fprintf (out, "\t  pushq (pred_%s);\n", c -> nonterminal);
	  else if (c -> def -> kind & h_semipredicate)
	     fprintf (out, "\t  pushq (semipred_%s);\n", c -> nonterminal);
	  else
	     fprintf (out, "\t  pushq (emp%s_%s);\n",
			   (vstrict)?"v":"", c -> nonterminal);
	  *nrpushes += 4;
	  *sonnr -= 1;
	};

private void generate_emp_semi (semiterminal sm, int *sonnr,
			        int *nrpushes, int vstrict)
	{ fprintf (out, "\t  pushi (%d);\n", *sonnr);
	  fprintf (out, "\t  pushq (link_son);\n");
	  generate_display (sm -> display, nrpushes);
	  fprintf (out, "\t  pushq (make_affix_link);\n");
	  fprintf (out, "\t  pushq (make_leafnode);\n");
	  fprintf (out, "\t  pushq (make_empty_strstore);\n");
	  if ((sm -> s -> kind & star) && (sm -> s -> kind & strict) && vstrict)
	     { fprintf (out, "\t  pushs (");
	       output_string (out, sm -> s -> string);
	       fprintf (out, ");\n");
	       if (sm -> s -> kind & non)
		  fprintf (out, "\t  pushq (fail_if_iptr_not_at_set);\n");
	       else fprintf (out, "\t  pushq (fail_if_iptr_at_set);\n");
	       *nrpushes += 2;
	     };
	  *nrpushes += 5;
	  *sonnr -= 1;
	};

private void generate_emp_member (member m, int *sonnr,
				  int *nrpushes, int vstrict)
	{ switch (m -> tag)
	     { case tag_call:
		  generate_emp_call (m -> u.cl, sonnr, nrpushes, vstrict);
	       case tag_terminal: break;
	       case tag_semiterminal:
		  generate_emp_semi (m -> u.semi, sonnr, nrpushes, vstrict);
	       case tag_cut:
	       default: break;
	     };
	};

private void generate_emp_members_before (member_list mems, int i, int *sonnr,
					  int *nrpushes, int vstrict)
	{ int j;
	  if (mems == member_list_nil) return;
	  for (j = i - 1; 0 <= j; j--)
	     generate_emp_member (mems -> ms[j], sonnr, nrpushes, vstrict);
	};

private void generate_emp_rhs (member_list mems, int nrsons,
			       int *nrpushes, int vstrict)
	{ int i;
	  int sonnr = nrsons;
	  if (mems == member_list_nil) return;
	  for (i=0; i < mems -> nrofms; i++)
	     generate_emp_member (mems -> ms[i], &sonnr, nrpushes, vstrict);
	};

private void generate_leftcorner_lhs (alt a, int *nrpushes)
	{ generate_display (a -> display, nrpushes);
	  fprintf (out, "\t  pushi (%d);\n", a -> nrsons);
	  fprintf (out, "\t  pushi (%d);\n", a -> nodenr);
	  fprintf (out, "\t  pushq (make_normalnode);\n");
	  *nrpushes += 3;
	};

/*
   Do not use look ahead inside the emp_routines.
   The input pointer may already point further in the text
*/
private void generate_emp_alt (hyper_rule rule, alt a, int i, int vstrict)
	{ int nrpushes = 0;
	  int code_build = !layoutflag || (rule != layout_rule);
	  char *emps = (vstrict)?"empv":"emp";
	  if (a -> empty != h_mayproduceempty) return;
	  fprintf (out, "\t  {\n");
	  if (code_build) generate_affix_decls (a);
	  if (traceflag)
	     fprintf (out, "\t  trace_alternative (\"%s_%s\", %d);\n",
			   emps, rule -> nonterminal, i);
	  if (code_build)
	     generate_emp_rhs (a -> members, a -> nrsons, &nrpushes, vstrict);
	  if (code_build) generate_leftcorner_lhs (a, &nrpushes);
	  fprintf (out, "\t  callq ();\n");
	  fprintf (out, "\t  pop (%d);\n", nrpushes);
	  if (code_build) generate_affix_undecls (a);
	  fprintf (out, "\t  };\n");
	};

private void generate_emp_routine (hyper_rule rule, int vstrict)
	{ int i;
	  char *emps = (vstrict)?"empv":"emp";
	  if (!rule -> reachable) return;
	  if (rule -> empty == h_mayproduceempty)
	     { fprintf (out, "private void %s_%s ()\n",
			emps, rule -> nonterminal);
	       fprintf (out, "\t{\n");
	       if (traceflag)
		  fprintf (out, "\t  trace_enter (\"%s_%s\");\n",
			   emps, rule -> nonterminal);
	       for (i=0; i < rule -> alts -> nrofas; i++)
		  generate_emp_alt (rule, rule -> alts -> as[i], i, vstrict);
	       if (traceflag)
		  fprintf (out, "\t  trace_leave (\"%s_%s\");\n",
			   emps, rule -> nonterminal);
	       fprintf (out, "\t  pushq (%s_%s);\n", emps, rule -> nonterminal);
	       fprintf (out, "\t}\n\n");
	     };
	};

private void generate_rule_call (call c, int *sonnr, int *nrpushes)
	{ if (layoutflag && (c -> def == layout_rule))
	     { fprintf (out, "\t  pushq (rule_%s);\n", c -> nonterminal);
	       *nrpushes += 1;
	       return;
	     };
	  fprintf (out, "\t  pushi (%d);\n", *sonnr);
	  fprintf (out, "\t  pushq (link_son);\n");
	  generate_display (c -> display, nrpushes);
	  fprintf (out, "\t  pushq (make_affix_link);\n");
	  fprintf (out, "\t  pushq (%s_%s);\n", rule_qualifier (c -> def),
			c -> nonterminal);
	  *nrpushes += 4;
	  *sonnr -= 1;
	};

private void generate_rule_member (member m, int *sonnr,
				   int *nrpushes, int code_build)
	{ switch (m -> tag)
	     { case tag_call:
		  generate_rule_call (m -> u.cl, sonnr, nrpushes); break;
	       case tag_terminal:
		  generate_terminal (m -> u.terminal, nrpushes); break;
	       case tag_semiterminal:
		  generate_semiterminal (m -> u.semi, sonnr,
			nrpushes, code_build);
	       default: break;
	     };
	};

private void generate_rule_members_from (member_list mems, int i, int *sonnr,
					 int *nrpushes, int code_build)
	{ int j;
	  for (j = mems -> nrofms - 1; i < j; j--)
	     generate_rule_member (mems -> ms[j], sonnr, nrpushes, code_build);
	};

/*
   The following routines generate code to parse placeholders,
*/
private void generate_untyped_placeholder_code (hyper_rule rule)
	{ int nrpushes = 5;
	  generate_placeholder_alt_header (rule, 1, untyped_symbol[0]);
	  fprintf (out, "\t  pushq (red_%s);\n", rule -> nonterminal);
	  generate_buildplaceholdernode (rule, 1, &nrpushes);
	  fprintf (out, "\t  pushq (rule_layout);\n");
	  fprintf (out, "\t  pushs (\"%s\");\n", untyped_symbol);
	  fprintf (out, "\t  pushq (parse_terminal);\n");
	  fprintf (out, "\t  pushq (rule_layout);\n");
	  generate_placeholder_alt_trailer (rule, 1, nrpushes);
	};

private void generate_typed_placeholder_code (hyper_rule rule)
	{ int nrpushes = 9;
	  generate_placeholder_alt_header (rule, 0, typed_open_symbol[0]);
	  fprintf (out, "\t  pushq (red_%s);\n", rule -> nonterminal);
	  generate_buildplaceholdernode (rule, 0, &nrpushes);
	  fprintf (out, "\t  pushq (rule_layout);\n");
	  fprintf (out, "\t  pushs (\"%s\");\n", typed_close_symbol);
	  fprintf (out, "\t  pushq (parse_terminal);\n");
	  fprintf (out, "\t  pushs (\"%s\");\n", rule -> nonterminal);
	  fprintf (out, "\t  pushq (parse_terminal);\n");
	  fprintf (out, "\t  pushs (\"%s\");\n", typed_open_symbol);
	  fprintf (out, "\t  pushq (parse_terminal);\n");
	  fprintf (out, "\t  pushq (rule_layout);\n");
	  generate_placeholder_alt_trailer (rule, 0, nrpushes);
	};

private void generate_placeholder_code (hyper_rule rule)
	{ if (!rule -> placeholder) return;
	  generate_untyped_placeholder_code (rule);
	  generate_typed_placeholder_code (rule);
	};

private void generate_lcit_alternative_for_terminal (hyper_rule rule,
						int i, alt a, int j, member m)
	{ int nrpushes = 1;
	  int sonnr = a -> nrsons;
	  int code_build = !layoutflag || (rule != layout_rule);
	  if (!rule -> reachable) return;
	  may_generate_lookahead_check (rule, m -> first);
	  generate_alt_header (rule, i, a, code_build, 0);
	  fprintf (out, "\t  pushq (red_%s);\n", rule -> nonterminal);
	  generate_rule_members_from (a -> members, j, &sonnr,
			&nrpushes, code_build);
	  if (code_build) generate_emp_members_before (a -> members, j,
			&sonnr, &nrpushes, 0);
	  if (code_build) generate_leftcorner_lhs (a, &nrpushes);
	  generate_terminal (m -> u.terminal, &nrpushes);
	  generate_alt_trailer (rule, i, a, nrpushes, code_build, 0);
	};

private void generate_leftcorner_semi (semiterminal sm, int sonnr,
				       int *nrpushes, int code_build)
	{ set s = sm -> s;
	  if (code_build)
	     { fprintf (out, "\t  pushi (%d);\n", sonnr);
	       fprintf (out, "\t  pushq (link_son);\n");
	       generate_display (sm -> display, nrpushes);
	       fprintf (out, "\t  pushq (make_affix_link);\n");
	       fprintf (out, "\t  pushq (make_leafnode);\n");
	       *nrpushes += 4;
	     };
	  fprintf (out, "\t  pushs (");
	  output_string (out, s -> string);
	  fprintf (out, ");\n");
	  fprintf (out, "\t  pushq (parse");
	  if (s -> kind & non) fprintf (out, "_non");
	  fprintf (out, "_set");
	  if (s -> kind & star) fprintf (out, "_plus");	/* leftc <> empty */
	  if (s -> kind & plus) fprintf (out, "_plus");
	  if (s -> kind & strict) fprintf (out, "_strict");
	  fprintf (out, ");\n");
	  *nrpushes += 2;
	};

private void generate_lcit_alternative_for_semiterminal (hyper_rule rule,
						int i, alt a, int j, member m)
	{ int nrpushes = 1;
	  int sonnr = a -> nrsons;
	  int code_build = !layoutflag || (rule != layout_rule);
	  int sonnr2;
	  if (!rule -> reachable) return;
	  may_generate_lookahead_check (rule, m -> first);
	  generate_alt_header (rule, i, a, code_build, 0);
	  fprintf (out, "\t  pushq (red_%s);\n", rule -> nonterminal);
	  generate_rule_members_from (a -> members, j, &sonnr, &nrpushes,
				code_build);
	  sonnr2 = sonnr;
	  sonnr -= 1;
	  if (code_build) generate_emp_members_before (a -> members, j,
				&sonnr, &nrpushes, 0);
	  generate_leftcorner_semi (m -> u.semi, sonnr2, &nrpushes, code_build);
	  if (code_build) generate_leftcorner_lhs (a, &nrpushes);
	  generate_alt_trailer (rule, i, a, nrpushes, code_build, 0);
	};

private void try_generate_lcit_alternative_for_member (hyper_rule rule,
						int i, alt a, int j, member m)
	{ switch (m -> tag)
	     { case tag_call: break;
	       case tag_terminal:
		  if (strlen (m -> u.terminal) > 0)
		     generate_lcit_alternative_for_terminal (rule, i, a, j, m);
		  break;
	       case tag_semiterminal:
		  generate_lcit_alternative_for_semiterminal (rule, i, a, j, m);
	       default: break;
	     };
	};

private void try_generate_lcit_alternative (hyper_rule rule, int i, alt a)
	{ member_list mems = a -> members;
	  int j;
	  if (mems == member_list_nil) return;
	  for (j=0; j < mems -> nrofms; j++)
	     { try_generate_lcit_alternative_for_member
		   (rule, i, a, j, mems -> ms[j]);
	       if (mems -> ms[j] -> empty == h_neverproducesempty) return;
	     };
	};

private void try_generate_lcit_alternatives (hyper_rule rule)
	{ int i;
	  if (rule -> ext) return;
	  if (rule -> kind & (h_predicate | h_semipredicate)) return;
	  for (i=0; i < rule -> alts -> nrofas; i++)
	     try_generate_lcit_alternative (rule, i, rule -> alts -> as[i]);
	};

private void generate_get_routine (hyper_rule rule)
	{ int i;
	  if (!rule -> reachable) return;
	  fprintf (out, "private void get_%s ()\n", rule -> nonterminal);
	  fprintf (out, "\t{\n");
	  if (traceflag)
	     fprintf (out, "\t  trace_enter (\"get_%s\");\n",
		     rule -> nonterminal);
	  for (i=0; i < nr_of_hyper_rules; i++)
	     if (refl_leftc_relation [i * nr_of_hyper_rules + rule -> number])
		{ if (placeholderflag)
		     generate_placeholder_code (all_hyper_rules[i]);
		  try_generate_lcit_alternatives (all_hyper_rules [i]);
		};
	  if (traceflag)
	     fprintf (out, "\t  trace_leave (\"get_%s\");\n",
		     rule -> nonterminal);
	  fprintf (out, "\t  pushq (get_%s);\n", rule -> nonterminal);
	  fprintf (out, "\t}\n\n");
	};

private void generate_check_goal (hyper_rule rule)
	{ fprintf (out, "\t  if (topi () == goal_%s)\n", rule -> nonterminal);
	  fprintf (out, "\t     { pop (1);\n");
	  fprintf (out, "\t       callq ();\n");
	  fprintf (out, "\t       pushi (goal_%s);\n", rule -> nonterminal);
	  fprintf (out, "\t     };\n");
	};

private void try_generate_lcin_alt_for_member (hyper_rule rule, int i, alt a,
					       int j, member m,
					       hyper_rule rule_N)
	{ call c;
	  int nrpushes = 1;
	  int sonnr = a -> nrsons;
	  int code_build = !layoutflag || (rule != layout_rule);
	  int sonnr2;
	  if (!rule -> reachable) return;
	  if (m -> tag != tag_call) return;
	  c = m -> u.cl;
	  if (c -> def != rule_N) return;
	
	  fprintf (out, "\t  if (on_spine_to_goal (goal_%s)){\n",
			rule -> nonterminal);
	  if (lookahead)
	     { char *fset = gather_lcin_director_set (rule, a -> members, j);
	       may_generate_lookahead_check (rule, fset);
	     };
	  generate_alt_header (rule, i, a, code_build, 0);
	  fprintf (out, "\t  pushq (red_%s);\n", rule -> nonterminal);
	  generate_rule_members_from (a -> members, j, &sonnr,
			&nrpushes, code_build);
	  sonnr2 = sonnr;
	  sonnr -= 1;
	  generate_emp_members_before (a -> members, j, &sonnr, &nrpushes, 0);
	  if (!layoutflag || (rule_N != layout_rule))
	     { fprintf (out, "\t  pushi (%d);\n", sonnr2);
	       fprintf (out, "\t  pushq (link_son);\n");
	       generate_display (c -> display, &nrpushes);
	       fprintf (out, "\t  pushq (make_affix_link);\n");
	       fprintf (out, "\t  pushq (exchange_top);\n");
	       nrpushes += 4;
	     };
	  if (code_build) generate_leftcorner_lhs (a, &nrpushes);
	  generate_alt_trailer (rule, i, a, nrpushes, code_build, 0);
	  fprintf (out, "\t  };\n");
	};

private void try_generate_lcin_alternative (hyper_rule rule, int i, alt a,
					    hyper_rule rule_N)
	{ int j;
	  member_list mems = a -> members;
	  if (mems == member_list_nil) return;
	  for (j=0; j < mems -> nrofms; j++)
	     { try_generate_lcin_alt_for_member
		  (rule, i, a, j, mems -> ms[j], rule_N);
	       if (mems -> ms[j] -> empty == h_neverproducesempty)
		  return;
	     };
	};

private void try_generate_lcin_alternatives (hyper_rule rule, hyper_rule rule_N)
	{ int i;
	  alt_list alts = rule -> alts;
	  for (i=0; i < alts -> nrofas; i++)
	     try_generate_lcin_alternative (rule, i, alts -> as[i], rule_N);
	};

private void generate_red_routine (hyper_rule rule)
	{ int i;
	  if (!rule -> reachable) return;
	  fprintf (out, "private void red_%s ()\n", rule -> nonterminal);
	  fprintf (out, "\t{\n");
	  if (traceflag)
	     fprintf (out, "\t  trace_enter (\"red_%s\");\n",
		     rule -> nonterminal);
	  generate_check_goal (rule);
	  for (i=0; i < nr_of_hyper_rules; i++)
	     try_generate_lcin_alternatives (all_hyper_rules[i], rule);
	  if (traceflag)
	     fprintf (out, "\t  trace_leave (\"red_%s\");\n",
		     rule -> nonterminal);
	  fprintf (out, "\t  pushq (red_%s);\n", rule -> nonterminal);
	  fprintf (out, "\t}\n\n");
	};

private void generate_parse_routines_for_rule (hyper_rule rule)
	{ if (rule -> ext) return;
	  if (!rule -> reachable) return;
	  if (rule -> kind & h_predicate) generate_predicate_topdown (rule);
	  else if (rule -> kind & h_semipredicate)
		generate_semipredicate_topdown (rule);
	  else 
	     { generate_rule_routine (rule);
	       generate_emp_routine (rule, 1);
	       generate_emp_routine (rule, 0);
	       generate_get_routine (rule);
	       generate_red_routine (rule);
	     };
	};

private void generate_parser_routines ()
	{ int i;
	  for (i = 0; i < nr_of_hyper_rules; i++)
	     generate_parse_routines_for_rule (all_hyper_rules[i]);
	};

public void generate_leftcorner_parser (char *fname,
					char **predefs, int nrpreds)
	{ compute_leftcorner_relations ();
	  generate_goal_for_rules ();
	  generate_goal_relation ();
	  warning ("generating left corner parser...");
	  open_output_file (fname, "leftcorner", "c");
	  generate_std_includes (predefs, nrpreds);
	  generate_meta_rules ();
	  generate_rule_declarations ();
	  code_nodenrs (fname);
	  if (editor) generate_enter_templates ();
	  warning ("coding syntax rules...");
	  code_goal_for_rules ();
	  code_goal_relation ();
	  generate_parser_routines ();
	  warning ("coding postamble and main...");
	  generate_start_rule ();
	  generate_module_interface (fname, predefs, nrpreds);
	  generate_main (fname, predefs, nrpreds);
	  close_output_file ();
	};
