/*
   File: layout.c
   Analyzes where layout may be inserted
*/

/* global includes */
#include <stdio.h>

/* libeag includes */
#include <export.h>
#include <error.h>

/* local includes */
#include <tree.h>
#include <numbering.h>
#include <main.h>
#include <layout.h>
#include <common.h>

/*
   Analyze the grammar for layout and invisible rules
*/
public int is_an_invisible_member (member m)
	{ switch (m -> tag)
	     { case tag_call:
		  { hyper_rule def = m -> u.cl -> def;
		    if (def == layout_rule) return (1);
		    if (def -> ext) return (1);
		    if (def -> kind & (h_predicate | h_semipredicate))
			return (1);
		    return (0);
		  };
	       case tag_terminal: return (strcmp (m -> u.terminal, "") == 0);
	       case tag_cut: return (1);
	       default: return (0);
	     };
	};

private void initial_check_if_alt_starts_with_layout (hyper_rule rule,
						      member_list mems)
	{ int j;
	  for (j=0; j < mems -> nrofms; j++)
	     { member m = mems -> ms[j];
	       if ((m -> tag == tag_call) && (m -> u.cl -> def == layout_rule))
		  { rule -> startswithlayout = 1;
		    return;
		  };
	       if (m -> empty == h_neverproducesempty) return;
	     };
	};

private void initial_check_if_alt_ends_in_layout (hyper_rule rule,
						  member_list mems)
	{ int j;
	  for (j=mems -> nrofms - 1; 0 <= j; j--)
	     { member m = mems -> ms[j];
	       if ((m -> tag == tag_call) && (m -> u.cl -> def == layout_rule))
		  { rule -> endsinlayout = 1;
		    return;
		  };
	       if (m -> empty == h_neverproducesempty) return;
	     };
	};

private void initial_analyze_layout_in_alt (hyper_rule rule, alt a)
	{ member_list mems = a -> members;
	  if (mems == member_list_nil) return;
	  initial_check_if_alt_starts_with_layout (rule, mems);
	  initial_check_if_alt_ends_in_layout (rule, mems);
	};

private void initial_analyze_layout_in_rule (hyper_rule rule)
	{ int i;
	  if (rule -> ext) return;
	  if (rule -> kind & (h_predicate | h_semipredicate)) return;
	  if (rule == layout_rule) return;
	  for (i=0; i < rule -> alts -> nrofas; i++)
	     initial_analyze_layout_in_alt (rule, rule -> alts -> as[i]);
	};

private void initial_analyze_layout_in_rules ()
	{ int i;
	  for (i = 0; i < nr_of_hyper_rules; i++)
	     initial_analyze_layout_in_rule (all_hyper_rules[i]);
	};

/*
   Incrementally find which members are shielded from
   members by layout
*/
private int change;
private void check_if_alt_starts_with_layout (hyper_rule rule, member_list mems)
	{ int j;
	  if (rule -> startswithlayout) return;
	  for (j=0; j < mems -> nrofms; j++)
	     { member m = mems -> ms[j];
	       if ((m -> tag == tag_call) &&
		   (m -> u.cl -> def -> startswithlayout))
		  { rule -> startswithlayout = 1;
		    change = 1;
		    return;
		  };
	       if (m -> empty == h_neverproducesempty) return;
	     };
	};

private void check_if_alt_ends_in_layout (hyper_rule rule, member_list mems)
	{ int j;
	  if (rule -> endsinlayout) return;
	  for (j=mems -> nrofms - 1; 0 <= j; j--)
	     { member m = mems -> ms[j];
	       if ((m -> tag == tag_call) &&
		   (m -> u.cl -> def -> endsinlayout))
		  { rule -> endsinlayout = 1;
		    change = 1;
		    return;
		  };
	       if (m -> empty == h_neverproducesempty) return;
	     };
	};

private void analyze_layout_in_alt (hyper_rule rule, alt a)
	{ member_list mems = a -> members;
	  if (mems == member_list_nil) return;
	  check_if_alt_starts_with_layout (rule, mems);
	  check_if_alt_ends_in_layout (rule, mems);
	};

private void analyze_layout_for_rule (hyper_rule rule)
	{ int i;
	  if (rule -> ext) return;
	  if (rule -> kind & (h_predicate | h_semipredicate)) return;
	  if (rule == layout_rule) return;
	  for (i=0; i < rule -> alts -> nrofas; i++)
	     analyze_layout_in_alt (rule, rule -> alts -> as[i]);
	};

private void analyze_layout_for_rules ()
	{ int i;
	  for (i = 0; i < nr_of_hyper_rules; i++)
	     analyze_layout_for_rule (all_hyper_rules[i]);
	};

private void report_on_layout_for_rule (hyper_rule rule)
	{ if (rule -> startswithlayout)
	     wlog ("rule %s may start with layout", rule -> nonterminal);
	  if (rule -> endsinlayout)
	     wlog ("rule %s may end with layout", rule -> nonterminal);
	};

private void report_on_layout ()
	{ int i;
	  for (i = 0; i < nr_of_hyper_rules; i++)
	     report_on_layout_for_rule (all_hyper_rules[i]);
	};

private void analyze_layout_in_grammar ()
	{ int nrpasses = 0;
	  warning ("analyzing layout...");
	  initial_analyze_layout_in_rules ();
	  do
	     { change = 0;
	       nrpasses++;
	       analyze_layout_for_rules ();
	     }
	  while (change);
	  hint ("needed %d pass%s for layout analysis", nrpasses,
		(nrpasses == 1)?"":"es");
	  if (full_verbose)
	     { wlog ("dumping layout analysis");
	       report_on_layout ();
	     };
	};

/*
   Determine layout insertion points
*/
private void mark_member_if_followed (member m, member_list mems, int from)
	{ int k;
	  for (k = from; k < mems -> nrofms; k++)
	     { if (!is_an_invisible_member (mems -> ms[k]))
		  { m -> followlayout = 1;
		    return;
		  };
	     };
	};

private void determine_layout_insertion_point (member_list mems, int i)
	{ int j;
	  member m = mems -> ms[i];
	  if (is_an_invisible_member (m)) return;
	  if ((m -> tag == tag_call) && (m -> u.cl -> def -> endsinlayout))
	     { mark_member_if_followed (m, mems, i+1);
	       return;
	     };
	  for (j = i + 1; j < mems -> nrofms; j++)
	     { member m2 = mems -> ms[j];
	       if (is_layout_member (m2))
		  { mark_member_if_followed (m, mems, j+1);
		    return;
		  };
	       if ((m2 -> tag == tag_call) &&
		   (m2 -> u.cl -> def -> startswithlayout))
		  { m -> followlayout = 1; return; };
	       if (!is_an_invisible_member (m2)) return;
	     };
	};

private void determine_layout_insertion_points_in_alt (alt a)
	{ int i;
	  member_list mems = a -> members;
	  if (mems == member_list_nil) return;
	  for (i=0; i < mems -> nrofms - 1; i++)
	     determine_layout_insertion_point (mems, i);
	};

private void determine_layout_insertion_points_in_rule (hyper_rule rule)
	{ int i;
	  if (rule -> ext) return;
	  if (rule -> kind & (h_predicate | h_semipredicate)) return;
	  for (i=0; i < rule -> alts -> nrofas; i++)
	     determine_layout_insertion_points_in_alt (rule -> alts -> as[i]);
	};

private void determine_layout_insertion_points_in_rules ()
	{ int i;
	  for (i = 0; i < nr_of_hyper_rules; i++)
	     determine_layout_insertion_points_in_rule (all_hyper_rules[i]);
	};

/*
   Dump insertion points
*/
private void dump_member (member m)
	{ if (is_an_invisible_member (m)) return;
	  switch (m -> tag)
	     { case tag_call: wlog (m -> u.cl -> nonterminal); break;
	       case tag_terminal: output_string (stderr, m -> u.terminal);
				  break;
	       case tag_semiterminal: wlog ("{}");
	       default: break;
	     };
	};

private void dump_insertion_points_of_alt (member_list mems)
	{ int j;
	  if (mems == member_list_nil) return;
	  for (j=0; j < mems -> nrofms; j++)
	     { member m = mems -> ms[j];
	       dump_member (m);
	       if (m -> followlayout) eprint_log (".");
	     };
	};

private void dump_insertion_points_of_rule (hyper_rule rule)
	{ int i;
	  if (rule -> ext) return;
	  if (rule -> kind & (h_predicate | h_semipredicate)) return;
	  if (rule == layout_rule) return;
	  for (i=0; i < rule -> alts -> nrofas; i++)
	     { eprint_log ("%s: ", rule -> nonterminal);
	       dump_insertion_points_of_alt (rule -> alts -> as[i] -> members);
	       eprint_log ("\n");
	     };
	};

private void dump_insertion_points_of_rules ()
	{ int i;
	  for (i = 0; i < nr_of_hyper_rules; i++)
	     dump_insertion_points_of_rule (all_hyper_rules[i]);
	};

private void determine_layout_insertion_points ()
	{ warning ("determining layout insertion points...");
	  determine_layout_insertion_points_in_rules ();
	  if (full_verbose)
	     { wlog ("dumping layout insertion points");
	       dump_insertion_points_of_rules ();
	     };
	};

public void analyze_layout ()
	{ analyze_layout_in_grammar ();
	  determine_layout_insertion_points ();
	};
