/*
   File: gentree.c
   Defines the actions necessary for building the syntax tree during parse
*/

/* global includes */
#include <stdio.h>

/* libeag includes */
#include <export.h>
#include <memalloc.h>

/* local includes */
#include <tree.h>
#include <typecheck.h>
#include <gentree.h>

/* Allocate a set */
public set new_set (int kind, char *string)
	{ set new = (set) ckmalloc (sizeof (struct set_rec));
	  new -> kind = kind;
	  new -> string = string;
	  return (new);
	};

/* Allocate an affix variable */
public affix_variable new_affix_variable (char *name)
	{ affix_variable new = (affix_variable) ckmalloc
				(sizeof (struct affix_variable_rec));
	  new -> variable = name;
	  new -> def = meta_rule_nil;
	  return (new);
	};

/* Allocate an affix */
public affix new_affix_nonterminal (affix_variable var)
	{ affix new = (affix) ckmalloc (sizeof (struct affix_rec));
	  new -> tag = tag_affix_nonterminal;
	  new -> name = string_nil;
	  new -> next = affix_nil;
	  new -> type = undefined_type;
	  new -> u.var = var;
	  return (new);
	};

public affix new_affix_terminal (char *s)
	{ affix new = (affix) ckmalloc (sizeof (struct affix_rec));
	  new -> tag = tag_affix_terminal;
	  new -> name = string_nil;
	  new -> next = affix_nil;
	  new -> type = undefined_type;
	  new -> u.string = s;
	  return (new);
	};

public affix new_affix_number (int val)
	{ affix new = (affix) ckmalloc (sizeof (struct affix_rec));
	  new -> tag = tag_affix_number;
	  new -> name = string_nil;
	  new -> next = affix_nil;
	  new -> type = undefined_type;
	  new -> u.number = val;
	  return (new);
	};

public affix new_affix_set (set s)
	{ affix new = (affix) ckmalloc (sizeof (struct affix_rec));
	  new -> tag = tag_affix_set;
	  new -> name = string_nil;
	  new -> next = affix_nil;
	  new -> type = undefined_type;
	  new -> u.aset = s;
	  return (new);
	};

/* Recursively duplicate an affix */
public affix rdup_affix (affix a)
	{ affix new = (affix) ckmalloc (sizeof (struct affix_rec));
	  new -> tag = a -> tag;
	  new -> name = string_nil;
	  new -> next = affix_nil;
	  new -> type = undefined_type;
	  switch (a -> tag)
	     { case tag_affix_nonterminal:
		  new -> u.var = rdup_affix_variable (a -> u.var); break;
	       case tag_affix_terminal: new -> u.string = a -> u.string; break;
	       case tag_affix_number: new -> u.number = a -> u.number; break;
	       case tag_affix_set:
		  new -> u.aset = rdup_set (a -> u.aset); break;
	     };
	  return (new);
	};

/* Announce to use 'room' affixes in an affix_list */
private void room_affix_list (affix_list al, int room)
	{ if (room <= al -> room) return;
	  al -> as = (affix *) ckrecalloc (al -> as, room, sizeof (affix));
	  al -> room = room;
	};

/* Allocate a new affix_list */
public affix_list new_affix_list ()
	{ affix_list new =
			(affix_list) ckmalloc (sizeof (struct affix_list_rec));
	  new -> nrofas = 0;
	  new -> room = 2;
	  new -> as = (affix *) ckcalloc (2, sizeof (affix));
	  return (new);
	};

/* Append affix to affix_list */
public void app_affix_list (affix_list al, affix a)
	{ if (al -> nrofas == al -> room)
	     room_affix_list (al, al -> nrofas << 1);
	  al -> as [al -> nrofas] = a;
	  al -> nrofas++;
	};

/* Recursively duplicate an affix_list */
public affix_list rdup_affix_list (affix_list old)
	{ int ix;
	  affix_list new;
	  if (old == affix_list_nil) return (affix_list_nil);
	  new = (affix_list) ckmalloc (sizeof (struct affix_list_rec));
	  new -> nrofas = old -> nrofas;
	  new -> room = old -> room;
	  new -> as = (affix *) ckcalloc (old -> nrofas, sizeof (affix));
	  for (ix = 0; ix < old -> nrofas; ix++)
	     new -> as[ix] = rdup_affix (old -> as[ix]);
	  return (new);
	};

/* Allocate an affix expression */
public expr new_expr_concat (affix_list al)
	{ expr new = (expr) ckmalloc (sizeof (struct expr_rec));
	  new -> tag = tag_concat;
	  new -> u.concat = al;
	  return (new);
	};

public expr new_expr_compos (affix_list al)
	{ expr new = (expr) ckmalloc (sizeof (struct expr_rec));
	  new -> tag = tag_compos;
	  new -> u.compos = al;
	  return (new);
	};

public expr new_expr_single (affix a)
	{ expr new = (expr) ckmalloc (sizeof (struct expr_rec));
	  new -> tag = tag_single;
	  new -> u.single = a;
	  return (new);
	};

/* Recursively duplicate an affix expression */
public expr rdup_expr (expr e)
	{ expr new = (expr) ckmalloc (sizeof (struct expr_rec));
	  new -> tag = e -> tag;
	  switch (e -> tag)
	     { case tag_single:
		  new -> u.single = rdup_affix (e -> u.single); break;
	       case tag_compos:
		  new -> u.compos = rdup_affix_list (e -> u.compos); break;
	       case tag_concat:
		  new -> u.concat = rdup_affix_list (e -> u.concat); break;
	     };
	  return (new);
	};

/* Allocate a position */
public pos new_pos (int kind, expr ex)
	{ pos new = (pos) ckmalloc (sizeof (struct pos_rec));
	  new -> kind = kind;
	  new -> ex = ex;
	  new -> type = undefined_type;
	  return (new);
	};

/* Recursively duplicate a position */
public pos rdup_pos (pos p)
	{ pos new = (pos) ckmalloc (sizeof (struct pos_rec));
	  new -> kind = p -> kind;
	  new -> ex = rdup_expr (p -> ex);
	  new -> type = undefined_type;
	  return (new);
	};

/* Allocate an empty position */
public pos new_empty_pos ()
	{ pos new = (pos) ckmalloc (sizeof (struct pos_rec));
	  new -> kind = undefinedflow;
	  new -> ex = expr_nil;
	  new -> type = undefined_type;
	  return (new);
	};

/* Announce to use 'room' positions in a pos_list */
private void room_pos_list (pos_list pl, int room)
	{ if (room <= pl -> room) return;
	  pl -> ps = (pos *) ckrecalloc (pl -> ps, room, sizeof (pos));
	  pl -> room = room;
	};

/* Allocate a new pos_list */
public pos_list new_pos_list ()
	{ pos_list new = (pos_list) ckmalloc (sizeof (struct pos_list_rec));
	  new -> nrofps = 0;
	  new -> room = 2;
	  new -> ps = (pos *) ckcalloc (2, sizeof (pos));
	  return (new);
	};

/* Append position to pos_list */
public void app_pos_list (pos_list pl, pos p)
	{ if (pl -> nrofps == pl -> room)
	     room_pos_list (pl, pl -> nrofps << 1);
	  pl -> ps [pl -> nrofps] = p;
	  pl -> nrofps++;
	};

/* Recursively duplicate an pos_list */
public pos_list rdup_pos_list (pos_list old)
	{ int ix;
	  pos_list new;
	  if (old == pos_list_nil) return (pos_list_nil);
	  new = (pos_list) ckmalloc (sizeof (struct pos_list_rec));
	  new -> nrofps = old -> nrofps;
	  new -> room = old -> room;
	  new -> ps = (pos *) ckcalloc (old -> nrofps, sizeof (pos));
	  for (ix = 0; ix < old -> nrofps; ix++)
	     new -> ps[ix] = rdup_pos (old -> ps[ix]);
	  return (new);
	};

/* Allocate a list of empty positions */
public pos_list new_empty_pos_list (int nrofps)
	{ int ix;
	  pos_list new = (pos_list) ckmalloc
				(sizeof (struct pos_list_rec));
	  new -> nrofps = nrofps;
	  new -> room = nrofps;
	  new -> ps = (pos *) ckcalloc (nrofps, sizeof (pos));
	  for (ix = 0; ix < nrofps; ix++)
	     new -> ps[ix] = new_empty_pos ();
	  return (new);
	};
 
/* Allocate a call */
public call new_call (char *nont, pos_list display)
	{ call new = (call) ckmalloc (sizeof (struct call_rec));
	  new -> nonterminal = nont;
	  new -> display = display;
	  new -> def = hyper_rule_nil;
	  return (new);
	};

/* Allocate a semiterminal */
public semiterminal new_semiterminal (set s, pos_list display)
	{ semiterminal m = (semiterminal) ckmalloc
				(sizeof (struct semiterminal_rec));
	  m -> s = s;
	  m -> display = display;
	  return (m);
	};

/* Allocate a member */
public member new_member_call (call c)
	{ member m = (member) ckmalloc (sizeof (struct member_rec));
	  m -> empty = h_undefined;
	  m -> first = "";
	  m -> followlayout = 0;
	  m -> number = 0;
	  m -> sonnr = 0;
	  m -> tag = tag_call;
	  m -> u.cl = c;
	  return (m);
	};

public member new_member_terminal (char *s)
	{ member m = (member) ckmalloc (sizeof (struct member_rec));
	  m -> empty = h_undefined;
	  m -> first = "";
	  m -> followlayout = 0;
	  m -> number = 0;
	  m -> tag = tag_terminal;
	  m -> u.terminal = s;
	  return (m);
	};

public member new_member_semiterminal (semiterminal semi)
	{ member m = (member) ckmalloc (sizeof (struct member_rec));
	  m -> empty = h_undefined;
	  m -> first = "";
	  m -> followlayout = 0;
	  m -> number = 0;
	  m -> tag = tag_semiterminal;
	  m -> u.semi = semi;
	  return (m);
	};

public member new_member_cut ()
	{ member m = (member) ckmalloc (sizeof (struct member_rec));
	  m -> empty = h_undefined;
	  m -> first = "";
	  m -> followlayout = 0;
	  m -> number = 0;
	  m -> tag = tag_cut;
	  return (m);
	};

/* Announce to use 'room' members in an member_list */
private void room_member_list (member_list ml, int room)
	{ if (room <= ml -> room) return;
	  ml -> ms = (member *) ckrecalloc (ml -> ms, room, sizeof (member));
	  ml -> room = room;
	};

/* Allocate a new member_list */
public member_list new_member_list ()
	{ member_list new =
		(member_list) ckmalloc (sizeof (struct member_list_rec));
	  new -> nrofms = 0;
	  new -> room = 2;
	  new -> ms = (member *) ckcalloc (2, sizeof (member));
	  return (new);
	};

/* Append member to member_list */
public void app_member_list (member_list ml, member m)
	{ if (ml -> nrofms == ml -> room)
	     room_member_list (ml, ml -> nrofms << 1);
	  ml -> ms [ml -> nrofms] = m;
	  ml -> nrofms++;
	};

/* Allocate a meta alternative */
public meta_alt new_meta_alt (expr e)
	{ meta_alt new = (meta_alt) ckmalloc (sizeof (struct meta_alt_rec));
	  new -> e = e;
	  new -> number = 0;
	  return (new);
	};

/* Announce to use 'room' meta_alts in an meta_alt_list */
private void room_meta_alt_list (meta_alt_list mal, int room)
	{ if (room <= mal -> room) return;
	  mal -> as = (meta_alt *)
		ckrecalloc (mal -> as, room, sizeof (meta_alt));
	  mal -> room = room;
	};

/* Allocate a new meta_alt_list */
public meta_alt_list new_meta_alt_list ()
	{ meta_alt_list new =
		(meta_alt_list) ckmalloc (sizeof (struct meta_alt_list_rec));
	  new -> nrofas = 0;
	  new -> room = 2;
	  new -> as = (meta_alt *) ckcalloc (2, sizeof (meta_alt));
	  return (new);
	};

/* Append meta_alt to meta_alt_list */
public void app_meta_alt_list (meta_alt_list mal, meta_alt ma)
	{ if (mal -> nrofas == mal -> room)
	     room_meta_alt_list (mal, mal -> nrofas << 1);
	  mal -> as [mal -> nrofas] = ma;
	  mal -> nrofas++;
	};

/* Allocate a meta rule */
public meta_rule new_meta_rule (char *nont, meta_alt_list alts)
	{ meta_rule new = (meta_rule) ckmalloc (sizeof (struct meta_rule_rec));
	  new -> meta_nonterminal = nont;
	  new -> meta_alts = alts;
	  new -> type = undefined_type;
	  new -> kind = undefined_meta_value;
	  new -> ext = 0;
	  new -> empty = 0;
	  new -> number = 0;
	  new -> mvalue = value_nil;
	  new -> reachable = 0;
	  return (new);
	};

public meta_rule new_external_meta_rule (char *nont, int type,
					 int kind, int empty)
	{ meta_rule new = (meta_rule) ckmalloc (sizeof (struct meta_rule_rec));
	  new -> meta_nonterminal = nont;
	  new -> meta_alts = meta_alt_list_nil;
	  new -> type = type;
	  new -> ext = 1;
	  new -> empty = empty;
	  new -> kind = kind;
	  new -> number = 0;
	  new -> mvalue = value_nil;
	  new -> reachable = 0;
	  return (new);
	};

/* Allocate an alternative */
public alt new_alt (pos_list pl, member_list ml)
	{ alt new = (alt) ckmalloc (sizeof (struct alt_rec));
	  new -> display = pl;
	  new -> members = ml;
	  new -> nodenr = 0;
	  new -> nrsons = 0;
	  new -> locals = affix_nil;
	  new -> empty = h_undefined;
	  new -> number = 0;
	  new -> first = "";
	  new -> director = "";
	  new -> rule_nr = 0;
	  return (new);
	};

/* Announce to use 'room' alts in an alt_list */
private void room_alt_list (alt_list al, int room)
	{ if (room <= al -> room) return;
	  al -> as = (alt *) ckrecalloc (al -> as, room, sizeof (alt));
	  al -> room = room;
	};

/* Allocate a new alt_list */
public alt_list new_alt_list ()
	{ alt_list new = (alt_list) ckmalloc (sizeof (struct alt_list_rec));
	  new -> nrofas = 0;
	  new -> room = 2;
	  new -> as = (alt *) ckcalloc (2, sizeof (alt));
	  return (new);
	};

/* Append alt to alt_list */
public void app_alt_list (alt_list al, alt a)
	{ if (al -> nrofas == al -> room)
	     room_alt_list (al, al -> nrofas << 1);
	  al -> as [al -> nrofas] = a;
	  al -> nrofas++;
	};

/* Allocate a hyper rule */
public hyper_rule new_hyper_rule (char *nont, alt_list alts)
	{ hyper_rule new = (hyper_rule) ckmalloc
				(sizeof (struct hyper_rule_rec));
	  new -> nonterminal = nont;
	  new -> alts = alts;
	  new -> ext = 0;
	  new -> kind = h_undefined;
	  new -> empty = h_undefined;
	  new -> number = 0;
	  new -> placeholder = 0;
	  new -> startswithlayout = 0;
	  new -> endsinlayout = 0;
	  new -> first = "";
	  new -> follow = "";
	  new -> proto_display = pos_list_nil;
	  new -> reachable = 0;
	  new -> has_sons = 0;
	  new -> goal = -1;
	  return (new);
	};

public hyper_rule new_external_hyper_rule (char *nont, alt_list alts, int kind)
	{ hyper_rule new = (hyper_rule) ckmalloc
				(sizeof (struct hyper_rule_rec));
	  new -> nonterminal = nont;
	  new -> alts = alts;
	  new -> ext = 1;
	  new -> kind = kind;
	  new -> empty = h_undefined;
	  new -> number = 0;
	  new -> placeholder = 0;
	  new -> startswithlayout = 0;
	  new -> endsinlayout = 0;
	  new -> first = "";
	  new -> follow = "";
	  new -> proto_display = pos_list_nil;
	  new -> reachable = 0;
	  new -> has_sons = 0;
	  new -> goal = -1;
	  return (new);
	};

