/*
   File: flow.c
   Defines a flow check of the eag
*/

/* global includes */
#include <stdio.h>
#include <stdlib.h>
#include <stdarg.h>

/* libeag includes */
#include <export.h>
#include <error.h>

/* local includes */
#include <sizes.h>
#include <tree.h>
#include <numbering.h>
#include <main.h>
#include <typecheck.h>
#include <common.h>
#include <flow.h>

/*
   check indicated flow in rules
*/
private int flow_errors;
private void init_flow ()
	{ flow_errors = 0;
	};

private void flow_error (char *format, ...)
	{ char buf[MAXSTRLEN];
	  va_list arg_ptr;
	  va_start (arg_ptr, format);
	  vsprintf (buf, format, arg_ptr);
	  va_end (arg_ptr);

	  flow_errors++;
	  error ("flow error: %s", buf);
	};

private void merge_flow_in_position (hyper_rule rule, int i, pos protops, pos p)
	{ if (protops -> kind == undefinedflow)
	     protops -> kind = p -> kind;
	  else if (protops -> kind != p -> kind) flow_error (
		   "Flow mismatch between alternatives in rule %s, alt %d",
		   rule -> nonterminal, i+1);
	};

private void merge_flow_in_displays (hyper_rule rule, int i, pos_list proto, 
				     pos_list dpy)
	{ int j;
	  if (dpy == pos_list_nil) return;
	  for (j=0; j < proto -> nrofps; j++)
	     merge_flow_in_position (rule, i, proto -> ps[j], dpy -> ps[j]);
	};

private void deduce_initial_flow_in_alt (hyper_rule rule, int i, alt a)
	{ merge_flow_in_displays (rule, i, rule -> proto_display, a -> display);
	};

private void deduce_initial_flow_in_alts (hyper_rule rule, alt_list alts)
	{ int i;
	  for (i=0; i < alts -> nrofas; i++)
	     deduce_initial_flow_in_alt (rule, i, alts -> as[i]);
	};

private void deduce_initial_flow_in_rule (hyper_rule rule)
	{ deduce_initial_flow_in_alts (rule, rule -> alts);
	};

private void deduce_initial_flow_in_rules ()
	{ int i;
	  for (i = 0; i < nr_of_hyper_rules; i++)
	     deduce_initial_flow_in_rule (all_hyper_rules[i]);
	};

/*
   Determine which predicates are actually semipredicates
   by noting if they have no inherited positions at all
*/
private int no_inherited_positions (pos_list proto)
	{ int i;
	  for (i=0; i < proto -> nrofps; i++)
	     if (proto -> ps[i] -> kind == inherited)
		return (0);
	  return (1);
	};

private void check_if_semipredicate (hyper_rule rule)
	{ if (rule -> ext) return;
	  if (!(rule -> kind & h_predicate)) return;
	  if (no_inherited_positions (rule -> proto_display))
	     { rule -> kind &= ~h_predicate;
	       rule -> kind |= h_semipredicate;
	     };
	};

private void check_for_semipredicates ()
	{ int i;
	  for (i = 0; i < nr_of_hyper_rules; i++)
	     check_if_semipredicate (all_hyper_rules[i]);
	};

public void flow_check ()
	{ warning ("checking flow...");
	  init_flow ();
	  deduce_initial_flow_in_rules ();
	  if (flow_errors)
	     panic ("%d flow error%s w%s found", flow_errors,
		    (flow_errors == 1)?"":"s", (flow_errors == 1)?"as":"ere");
	  check_for_semipredicates ();
	};
