/*
   File: unparser.c
   Unparses the internal datastructures.
*/

/* global includes */
#include <stdio.h>

/* libeag includes */
#include <export.h>
#include <error.h>

/* local includes */
#include <tree.h>
#include <numbering.h>
#include <main.h>
#include <typecheck.h>
#include <eag_unparser.h>

private void unparse_string_or_set (char *s)
	{ char *ptr;
	  for (ptr = s; *ptr; ptr++)
	     { char c = *ptr;
	       if (c == '\n') eprint_log ("\\n");
	       else if (c == '\t') eprint_log ("\\t");
	       else if (c == '"')  eprint_log ("\\\"");
	       else if (c == '}')  eprint_log ("\\}");
	       else if (c == '\\') eprint_log ("\\\\");
	       else eprint_log ("%c", c);
	     };
	};

private void unparse_set (set s)
	{ if (s -> kind & non) eprint_log ("^");
	  eprint_log ("{");
	  unparse_string_or_set (s -> string);
	  eprint_log ("}");
	  if (s -> kind & star) eprint_log ("*");
	  if (s -> kind & plus) eprint_log ("+");
	  if (s -> kind & strict) eprint_log ("!");
	};

private void unparse_string (char *s)
	{ eprint_log ("%c", '"');
	  unparse_string_or_set (s);
	  eprint_log ("%c", '"');
	};

private void unparse_number (int n)
	{ eprint_log ("%d", n);
	};

private void unparse_affix_variable (affix_variable v)
	{ eprint_log ("%s", v -> variable);
	};

private void unparse_affix (affix a)
	{ switch (a -> tag)
	     { case tag_affix_nonterminal:
		  unparse_affix_variable (a -> u.var); break;
	       case tag_affix_terminal:
	          unparse_string (a -> u.string); break;
	       case tag_affix_number:
	          unparse_number (a -> u.number); break;
	       case tag_affix_set:
	          unparse_set (a -> u.aset); break;
	     };
	};

private void unparse_affix_list (affix_list affl, char sep)
	{ int i;
	  for (i = 0; i < affl -> nrofas; i++)
	     { unparse_affix (affl -> as[i]);
	       if (i != affl -> nrofas - 1) eprint_log (" %c ", sep);
	     };
	};

private void unparse_expr (expr e)
	{ if (e == expr_nil) return;
	  switch (e -> tag)
	     { case tag_single: unparse_affix (e -> u.single); break;
	       case tag_compos: unparse_affix_list (e -> u.compos, '*'); break;
	       case tag_concat: unparse_affix_list (e -> u.concat, '+'); break;
	     };
	};

private void unparse_meta_alts (meta_alt_list al)
	{ int i;
	  if (al == meta_alt_list_nil) return;
	  for (i = 0; i < al -> nrofas; i++)
	     { eprint_log ("\n   ");
	       unparse_expr (al -> as[i] -> e);
	     };
	};

private char *string_from_meta_kind (int k)
	{ switch (k)
	     { case single_meta_value: return ("single");
	       case multiple_meta_value: return ("finite");
	       case recognizer_meta_value: return ("recognizer");
	     };
	  return ("unknown");
	};

private char *string_from_hyper_kind (int k)
	{ if (k & h_nonpredicate) return ("normal");
	  if (k & h_predicate) return ("predicate");
	  if (k & h_semipredicate) return ("semipredicate");
	  return ("error kind");
	};

private void unparse_external_meta_rule (meta_rule rule)
	{ eprint_log ("\n$ %s :: ", rule -> meta_nonterminal);
	  eprint_log ("%s, %s.\n", string_from_type (rule -> type),
			string_from_meta_kind (rule -> kind));
	};

private void unparse_meta_rule (meta_rule rule)
	{ if (rule -> ext) unparse_external_meta_rule (rule);
	  else
	     { eprint_log ("\n%s ::", rule -> meta_nonterminal);
	       unparse_meta_alts (rule -> meta_alts);
	       eprint_log (".\n");
	     };
	};
private void unparse_meta_rules ()
	{ int i;
	  for (i = 0; i < nr_of_meta_rules; i++)
	     unparse_meta_rule (all_meta_rules[i]);
	};

private void unparse_position (pos p)
	{ if (p -> kind == inherited) eprint_log (">");
	  unparse_expr (p -> ex);
	  if (p -> kind == derived) eprint_log (">");
	};

private void unparse_display (pos_list pl)
	{ int i;
	  if (pl == pos_list_nil) return;
	  eprint_log (" (");
	  for (i=0; i < pl -> nrofps; i++)
	     { unparse_position (pl -> ps[i]);
	       if (i != pl -> nrofps - 1) eprint_log (", ");
	     };
	  eprint_log (")");
	};

private void unparse_call (call c)
	{ eprint_log (c -> nonterminal);
	  unparse_display (c -> display);
	};

private void unparse_semiterminal (semiterminal s)
	{ unparse_set (s -> s);
	  unparse_display (s -> display);
	};

public void unparse_member (member m)
	{ switch (m -> tag)
	     { case tag_call: unparse_call (m -> u.cl); break;
	       case tag_terminal: unparse_string (m -> u.terminal); break;
	       case tag_semiterminal: unparse_semiterminal (m -> u.semi); break;
	       case tag_cut: eprint_log ("->");
	       default: break;
	     };
	};

private void unparse_member_list (member_list ml)
	{ int i;
	  eprint_log ("   ");
	  if (ml == member_list_nil) return;
	  for (i = 0; i < ml -> nrofms; i++)
	     { unparse_member (ml -> ms[i]);
	       if (i != ml -> nrofms - 1)
		  { eprint_log (", ");
		    if (i%3 == 2) eprint_log ("\n      ");
		  };
	     };
	};

private void unparse_alt (hyper_rule rule, alt a)
	{ eprint_log ("%s", rule -> nonterminal);
	  unparse_display (a -> display);
	  eprint_log (":\n");
	  unparse_member_list (a -> members);
	};

private void unparse_alts (hyper_rule rule, alt_list alts)
	{ int i;
	  eprint_log ("\n");
	  for (i=0; i < alts -> nrofas; i++)
	     { unparse_alt (rule, alts -> as[i]);
	       if (i != alts -> nrofas - 1) eprint_log (";\n");
	     };
	  eprint_log (".\n");
	};

private void unparse_external_hyper_rule (hyper_rule rule)
	{ eprint_log ("\n$ %s", rule -> nonterminal);
	  unparse_display (rule -> alts -> as[0] -> display);
	  eprint_log (": ");
	  eprint_log ("%s.\n", string_from_hyper_kind (rule -> kind));
	};

private void unparse_hyper_rule (hyper_rule rule)
	{ if (rule -> ext) unparse_external_hyper_rule (rule);
	  else unparse_alts (rule, rule -> alts);
	};

private void unparse_hyper_rules ()
	{ int i;
	  for (i = 0; i < nr_of_hyper_rules; i++)
	     unparse_hyper_rule (all_hyper_rules[i]);
	};

private void unparse_start_rule ()
	{ unparse_call (start_rule);
	  eprint_log (".\n");
	};

void unparse_eag ()
	{ if (!full_verbose) return;
	  warning ("unparsing eag...");
	  unparse_start_rule ();
	  unparse_hyper_rules ();
	  unparse_meta_rules ();
	};
