/*
   File: common.c
   Defines some general support routines for output generation
*/

/* General includes */
#include <stdio.h>
#include <sys/types.h>
#include <sys/time.h>

/* libeag includes */
#include <export.h>
#include <error.h>
#include <textparsing.h>

/* libedt includes */
#include <editor.h>

/* local includes */
#include <sizes.h>
#include <tree.h>
#include <numbering.h>
#include <common.h>
#include <main.h>

/*
   Do initial output for generated parsers
*/
public FILE *out;
public void open_output_file (char *basename, char *infix, char *suffix)
	{ char totalname[MAXFNAME];
	  time_t thetime;
	  char *atime;
	  sprintf (totalname, "%s_%s.%s", basename, infix, suffix);
	  if (!(out = fopen (totalname, "w")))
	     panic ("can't open outputfile %s", totalname);
	  time (&thetime);
	  atime = ctime (&thetime);
	  fprintf (out, "/*\n");
	  fprintf (out, "   File: %s\n", totalname);
	  fprintf (out, "   Generated on %s", atime);
	  fprintf (out, "*/\n\n");
	};

public void close_output_file ()
	{ fclose (out);
	};

public void generate_std_includes (char **predefs, int nrpreds)
	{ int i;
	  warning ("coding preamble...");
	  fprintf (out, "/* Global includes */\n");
	  fprintf (out, "#include <stdio.h>\n\n");
	  fprintf (out, "/* Libeag includes */\n");
	  fprintf (out, "#include <export.h>\n");
	  fprintf (out, "#include <error.h>\n");
	  fprintf (out, "#include <ds.h>\n");
	  fprintf (out, "#include <textparsing.h>\n");
	  fprintf (out, "#include <buildtree.h>\n");
	  fprintf (out, "#include <buildaffixgraph.h>\n");
	  fprintf (out, "#include <propagate.h>\n");
	  fprintf (out, "#include <trace.h>\n");
	  fprintf (out, "#include <nodenrs.h>\n");
	  fprintf (out, "#include <init.h>\n");
	  if (interface_to_c) fprintf (out, "#include <c_interface.h>\n");
	  fprintf (out, "\n");
	  fprintf (out, "/* Predefines includes */\n");
	  for (i = 0; i < nrpreds; i++)
	     fprintf (out, "#include <%s.h>\n", predefs [i]);
	  fprintf (out, "\n");
	  if (editor)
	     { fprintf (out, "/* libedt includes */\n");
	       fprintf (out, "#include <templates.h>\n");
	       fprintf (out, "#include <cpmerge.h>\n");
	       fprintf (out, "#include <unparser.h>\n");
	       fprintf (out, "#include <editmain.h>\n");
	       fprintf (out, "#include <initedit.h>\n");
	       fprintf (out, "#include <editorparsing.h>\n\n");
	     };
	};

public char *rule_qualifier (hyper_rule rule)
	{ if (rule -> kind & h_predicate) return ("pred");
	  if (rule -> kind & h_semipredicate) return ("semipred");
	  return ("rule");
	};

/*
   Assign nodenrs to every reachable alternative
   Nodenr 0 is reserved for the root node
   Nodenr 1 is reserved for leaf nodes
*/
private int nodenr;
private void code_nodenr (hyper_rule rule)
	{ fprintf (out, "\t       case %d: return (\"%s_%s\");\n",
			nodenr, rule_qualifier (rule), rule -> nonterminal);
	  nodenr++;
	};

private void code_nodenr_for_alt (hyper_rule rule, alt a)
	{ fprintf (out, "\t       case %d: return (\"%s_%s\");\n",
		   a -> nodenr, rule_qualifier (rule), rule -> nonterminal);
	  nodenr++;
	};

private void code_nodenrs_in_alts (hyper_rule rule, alt_list alts)
	{ int i;
	  for (i=0; i < alts -> nrofas; i++)
	     code_nodenr_for_alt (rule, alts -> as[i]);
	};

private void code_nodenrs_in_rule (hyper_rule rule)
	{ if (rule -> ext) return;
	  if (!rule -> reachable) return;
	  if (layoutflag && (rule == layout_rule)) return;
	  if (rule -> kind & h_predicate) code_nodenr (rule);
	  else if (placeholderflag && (rule -> placeholder)) code_nodenr (rule);
	  code_nodenrs_in_alts (rule, rule -> alts);
	};

public void code_nodenrs (char *fname)
	{ int i;
	  nodenr = 2;
	  fprintf (out, "public char *%s_name_from_nodenr (int nodenr)\n",
				fname);
	  fprintf (out, "\t{ switch (nodenr)\n");
	  fprintf (out, "\t     { case 0: return (\"rootnode\");\n");
	  fprintf (out, "\t       case 1: return (\"leafnode\");\n");
	  for (i = 0; i < nr_of_hyper_rules; i++)
	     code_nodenrs_in_rule (all_hyper_rules[i]);
	  fprintf (out, "\t       default: panic (\"strange node ");
	  fprintf (out, "%%d in %s\", nodenr);\n", fname);
	  fprintf (out, "\t     };\n");
	  fprintf (out, "\t  return (NULL);\n");
	  fprintf (out, "\t};\n\n");
	};

/*
   If the string is large break it up in parts
*/
public void output_char (FILE *file, char c)
	{ if (c == '\n') fprintf (file, "\\n");
	  else if (c == '\t') fprintf (file, "\\t");
	  else if (c == EOFCHAR) fprintf (file, "\\%3o", c & 0377);
	  else if (c < ' ') fprintf (file, "\\%o", c);
	  else if (c == '"') fprintf (file, "\\\"");
	  else if (c == '\'') fprintf (file, "\\'");
	  else if (c == '\\') fprintf (file, "\\\\");
	  else fputc (c, file);
	};

#define MAXCOUNT 50
public void output_unquoted_string (FILE *file, char *string)
	{ char *ptr;
	  int count = 0;
	  for (ptr = string; *ptr; ptr++)
	     {
#ifndef Debug
	       if (count == MAXCOUNT)
		  { fprintf (file, "\\\n");
		    count = 0;
		  };
#endif
	       output_char (file, *ptr);
#ifndef Debug
	       count++;
#endif
	     };
	};

public void output_string (FILE *file, char *string)
	{ fputc ('"', file);
	  output_unquoted_string (file, string);
	  fputc ('"', file);
	};

/*
   Code lookahead sets
*/
private void generate_lookahead_condition (char *s)
	{ if (!s || (strlen (s) == 0)) fprintf (out, "0");
	  else if (strlen (s) == 1)
	     { fprintf (out, "iptr_at ('");
	       output_char (out, *s);
	       fprintf (out, "')");
	     }
	  else
	     { fprintf (out, "in_lookahead (");
	       output_string (out, s);
	       fprintf (out, ")");
	     };
	};

public void may_generate_lookahead_check (hyper_rule rule, char *s)
	{ if (!lookahead) return;
	  fprintf (out, "\t  if (");
	  if (lookahead_error_messages) fputc ('!', out);
	  generate_lookahead_condition (s);
	  if (lookahead_error_messages)
	     { fprintf (out, ") lookahead_failure (\"%s\");\n",
			rule -> nonterminal);
	       fprintf (out, "\t  else\n");
	     }
	  else fprintf (out, ")\n");
	};

public void generate_terminal (char *s, int *nr_pushes)
	{ fprintf (out, "\t  pushs (");
	  output_string (out, s);
	  fprintf (out, ");\n");
	  fprintf (out, "\t  pushq (parse_terminal);\n");
	  *nr_pushes += 2;
	};

public void generate_semiterminal (semiterminal sm, int *sonnr,
				   int *nrpushes, int code_build)
	{ set s = sm -> s;
	  if (code_build)
	     { fprintf (out, "\t  pushi (%d);\n", *sonnr);
	       fprintf (out, "\t  pushq (link_son);\n");
	       generate_display (sm -> display, nrpushes);
	       fprintf (out, "\t  pushq (make_affix_link);\n");
	       fprintf (out, "\t  pushq (make_leafnode);\n");
	       *nrpushes += 4;
	     };
	  fprintf (out, "\t  pushs(");
	  output_string (out, s -> string);
	  fprintf (out, ");\n");
	  fprintf (out, "\t  pushq(parse");
	  if (s -> kind & non) fprintf (out, "_non");
	  fprintf (out, "_set");
	  if (s -> kind & star) fprintf (out, "_star");
	  if (s -> kind & plus) fprintf (out, "_plus");
	  if (s -> kind & strict) fprintf (out, "_strict");
	  fprintf (out, ");\n");
	  *nrpushes += 2;
	  *sonnr -= 1;
	};

public void generate_cut (int *nrpushes)
	{ fprintf (out, "\t  puship (&cut_set);\n");
	  fprintf (out, "\t  pushq (cut);\n");
	  *nrpushes += 2;
	};

public void generate_affix_output (pos_list dpy, int *nrpushes)
	{ int i;
	  if (dpy == pos_list_nil) return;
	  for (i=dpy -> nrofps - 1; 0 <= i; i--)
	     { pos p = dpy -> ps[i];
	       if (p -> kind != inherited)
		  { affix a = p -> ex -> u.single;
		    fprintf (out, "\t  pusha (%s);\n", a -> name);
		    fprintf (out, "\t  pushq (output_affix);\n");
		    *nrpushes += 2;
		  };
	     };
	};

private void generate_collect_affix_output (pos_list dpy, int *nrpushes)
	{ int i;
	  if (dpy == pos_list_nil) return;
	  for (i=dpy -> nrofps - 1; 0 <= i; i--)
	     { pos p = dpy -> ps[i];
	       if (p -> kind != inherited)
		  { affix a = p -> ex -> u.single;
		    fprintf (out, "\t  pusha (%s);\n", a -> name);
		    fprintf (out, "\t  pushi (%d);\n", i);
		    fprintf (out, "\t  pushq (collect_output_affix);\n");
		    *nrpushes += 3;
		  };
	     };
	};

private void generate_assign_affix_input (pos_list dpy, int *nrpushes)
	{ int i;
	  if (dpy == pos_list_nil) return;
	  for (i=dpy -> nrofps - 1; 0 <= i; i--)
	     { pos p = dpy -> ps[i];
	       if (p -> kind == inherited)
		  { affix a = p -> ex -> u.single;
		    fprintf (out, "\t  pusha (%s);\n", a -> name);
		    fprintf (out, "\t  pushi (%d);\n", i);
		    fprintf (out, "\t  pushq (assign_input_affix);\n");
		    *nrpushes += 3;
		  };
	     };
	};

private void generate_reserve_collection_space (pos_list dpy, int *nrpushes)
	{ int nr = (dpy == pos_list_nil)?0:dpy -> nrofps;
	  fprintf (out, "\t  pushi (%d);\n", nr);
	  fprintf (out, "\t  pushq (reserve_collection_space);\n");
	  *nrpushes += 2;
	};

public void generate_affix_decls (alt a)
	{ affix ptr;
	  for (ptr = a -> locals; ptr != affix_nil; ptr = ptr -> next)
	     switch (ptr -> tag)
		{ case tag_affix_nonterminal:
		     { meta_rule def = ptr -> u.var -> def;
		       fprintf (out, "\t  affixnode %s = ", ptr -> name);
		       if (def != meta_rule_nil)
			  { fprintf (out, "make_%s_value ();\n",
					def -> meta_nonterminal);
			  }
		       else fprintf (out, "new_affixnode (\"%s\");\n",
					ptr -> name);
		     };
		  case tag_affix_set:
		     /* can't occur any more */
		     break;
		  case tag_affix_terminal:
		     { fprintf (out,
				"\t  affixnode %s = string_to_affix (\"%s\", ",
				ptr -> name, ptr -> name);
		       output_string (out, ptr -> u.string);
		       fprintf (out, ");\n");
		     };
		     break;
		  case tag_affix_number:
		     { fprintf (out,
			"\t  affixnode %s = number_to_affix (\"%s\", %d);\n",
				ptr -> name, ptr -> name, ptr -> u.number);
		     };
		     break;
		};
	};

public void generate_affix_undecls (alt a)
	{ affix ptr;
	  for (ptr = a -> locals; ptr != affix_nil; ptr = ptr -> next)
	     { fprintf (out, "\t  detach_valuenode (%s -> value);\n",
			ptr -> name);
	       fprintf (out, "\t  free_affixnode (%s);\n", ptr -> name);
	     };
	};

private void generate_affix (affix a, int *nr_pushes)
	{ fprintf (out, "\t  pusha (%s);\n", a -> name);
	  *nr_pushes += 1;
	};

private void generate_affix_list (affix_list affl, int *nr_pushes)
	{ int i;
	  for (i = affl -> nrofas - 1; 0 <= i; i--)
	     generate_affix (affl -> as[i], nr_pushes);
	  fprintf (out, "\t  pushi (%d);\n", affl -> nrofas);
	  *nr_pushes += 1;
	};

private void generate_expression (expr e, int *nr_pushes)
	{ switch (e -> tag)
	     { case tag_single:
		  { generate_affix (e -> u.single, nr_pushes);
		    fprintf (out, "\t  pushi (singleaffix);\n");
		  }; break;
	       case tag_compos:
		  { generate_affix_list (e -> u.compos, nr_pushes);
		    fprintf (out, "\t  pushi (composaffix);\n");
		  }; break;
	       case tag_concat:
		  { generate_affix_list (e -> u.concat, nr_pushes);
		    fprintf (out, "\t  pushi (concataffix);\n");
		  }; break;
	     };
	  *nr_pushes += 1;
	};

public void generate_display (pos_list pl, int *nr_pushes)
	{ if (pl == pos_list_nil)
	     { fprintf (out, "\t  pushi (0);\n");	/* no affixes */
	     }
	  else
	     { int i;
	       for (i=pl -> nrofps - 1; 0 <= i; i--)
		  generate_expression (pl -> ps[i] -> ex, nr_pushes);
	       fprintf (out, "\t  pushi (%d);\n", pl -> nrofps);
	     };
	  *nr_pushes += 1;
	};

private void generate_startcall (call c, int *nr_pushes)
	{ fprintf (out, "\t  pushi(1);\n");
	  fprintf (out, "\t  pushq(link_son);\n");
	  *nr_pushes += 2;
	  generate_display (c -> display, nr_pushes);
	  fprintf (out, "\t  pushq(make_affix_link);\n");
	  fprintf (out, "\t  pushq(%s_%s);\n", rule_qualifier (c -> def),
		c -> nonterminal);
	  *nr_pushes += 2;
	};

/*
   The following routines generate code to enter alternatives
*/
public void generate_alt_header (hyper_rule rule, int i, alt a,
				 int code_build, int cut)
	{ fprintf (out, "\t  {\n");
	  if (cut) fprintf (out, "\t  int cut_set = 0;\n");
	  if (qstackflag)
	     fprintf (out, "\t  cont* lqptr = qptr;\n");
	  if (code_build) generate_affix_decls (a);
	  if (traceflag)
	     fprintf (out, "\t  trace_alternative (\"%s_%s\", %d);\n",
		rule_qualifier (rule), rule -> nonterminal, i);
	};

private void generate_qstack_check (hyper_rule rule, int i)
	{ fprintf (out, "\t  if (lqptr != qptr)\n");
	  fprintf (out, "\t     panic (\"qstack inconsistency\
 detected in rule %s, alt %d\");\n", rule -> nonterminal, i);
	};

public void generate_alt_trailer (hyper_rule rule, int i, alt a,
				  int nrpushes, int code_build, int cut)
	{ fprintf (out, "\t  callq ();\n");
	  fprintf (out, "\t  pop (%d);\n", nrpushes);
	  if (code_build) generate_affix_undecls (a);
	  if (qstackflag) generate_qstack_check (rule, i);
	  if (cut) fprintf (out, "\t  if (cut_set) goto leave_%s;\n",
		rule -> nonterminal);
	  fprintf (out, "\t  };\n");
	};

/*
   The following routines generate code to parse placeholders,
*/
public void generate_placeholder_alt_header (hyper_rule rule,
					     int untyped, char look)
	{ int nraffs = rule -> proto_display -> nrofps;
	  int i;
	  char fset[2];
	  fset[0] = look;
	  fset[1] = '\0';
	  may_generate_lookahead_check (rule, fset);
	  fprintf (out, "\t {\n");
	  if (qstackflag) fprintf (out, "\t  cont *lqptr = qptr;\n");
	  for (i=0; i < nraffs; i++)
	     fprintf (out,
		"\t  affixnode uaf%d = make_undefined_affix ();\n", i);
	  if (traceflag)
	     fprintf (out, "\t  trace_alternative (\"%s_%s\", %d);\n",
			rule_qualifier (rule), rule -> nonterminal, 
			(untyped)?-2:-1);
	};

public void generate_placeholder_alt_trailer (hyper_rule rule,
					      int untyped, int nrpushes)
	{ int nraffs = rule -> proto_display -> nrofps;
	  int i;
	  fprintf (out, "\t  callq ();\n");
	  fprintf (out, "\t  pop (%d);\n", nrpushes);
	  for (i=0; i < nraffs; i++)
	     { fprintf (out, "\t  detach_valuenode (uaf%d -> value);\n", i);
	       fprintf (out, "\t  free_affixnode (uaf%d);\n", i);
	     };
	  if (qstackflag) generate_qstack_check (rule, (untyped)?-2:-1);
	  fprintf (out, "\t };\n");
	};

public void generate_buildplaceholdernode (hyper_rule rule,
					   int untyped, int *nrpushes)
	{ int nraffs = rule -> proto_display -> nrofps;
	  int i;
	  for (i = nraffs - 1; 0 <= i; i--)
	     { fprintf (out, "\t  pusha (uaf%d);\n", i);
	       fprintf (out, "\t  pushi (singleaffix);\n");
	       *nrpushes += 2;
	     };
	  fprintf (out, "\t  pushi (%d);\n", nraffs);
	  fprintf (out, "\t  pushi (%d);\n",
		rule -> alts -> as[0] -> nodenr - 1);
	  fprintf (out, "\t  pushq (make_%stypedplaceholdernode);\n",
		(untyped)?"un":"");
	  *nrpushes += 3;
	};

public void generate_start_rule ()
	{ int nrpushes = 0;
	  fprintf (out, "public void transduce ()\n");
	  fprintf (out, "\t{\n");
	  generate_affix_decls (start_alt);
	  fprintf (out, "\t  pushq (dummy_continuation);\n");
	  fprintf (out, "\t  pushq (increment_nrparses);\n");
	  if (editor)
	     { fprintf (out, "\t  pushq (copy_tree);\n");
	       nrpushes += 1;
	     }
	  else if (interface_to_c)
	     generate_collect_affix_output (start_rule -> display, &nrpushes);
	  else generate_affix_output (start_rule -> display, &nrpushes);
	  if (dumpflag)
	     { fprintf (out, "\t  pushq (dump_parse_tree);\n");
	       nrpushes += 1;
	     };
	  fprintf (out, "\t  pushq (endofsentence);\n");
	  nrpushes += 2;
	  generate_startcall (start_rule, &nrpushes);
	  generate_display (start_rule -> display, &nrpushes);
	  fprintf (out, "\t  pushi (1);\n");
	  fprintf (out, "\t  pushi (0);\n");
	  fprintf (out, "\t  pushq (make_normalnode);\n");
	  nrpushes += 3;
	  if (interface_to_c)
	     { generate_assign_affix_input (start_rule -> display, &nrpushes);
	       generate_reserve_collection_space
					(start_rule -> display, &nrpushes);
	     };
	  fprintf (out, "\t  callq ();\n");
	  fprintf (out, "\t  pop (%d);\n", nrpushes);
	  generate_affix_undecls (start_alt);
	  fprintf (out, "\t};\n\n");
	};

public void generate_module_interface (char *fname, char **predefs, int nrpreds)
	{ int i;
	  fprintf (out, "public char *module_name_from_nodenr (int nodenr)\n");
	  fprintf (out, "\t{ int mnr = modnr_from_nodenr (nodenr);\n");
	  fprintf (out, "\t  switch (mnr)\n");
	  fprintf (out, "\t     { case 0: return (\"%s\");\n", fname);
	  for (i=0; i < nrpreds; i++)
	     fprintf (out, "\t       case %d: return (\"%s\");\n",
		i+1, predefs [i]);
	  fprintf (out, "\t       default: panic (\"strange module");
	  fprintf (out, " nr %%d\", mnr);\n");
	  fprintf (out, "\t     };\n");
	  fprintf (out, "\t  return (NULL);\n");
	  fprintf (out, "\t};\n\n");
	  fprintf (out, "public char *name_from_nodenr (int nodenr)\n");
	  fprintf (out, "\t{ int mnr = modnr_from_nodenr (nodenr);\n");
	  fprintf (out, "\t  switch (mnr)\n");
	  fprintf (out,
	"\t     { case 0: return (%s_name_from_nodenr (nodenr));\n", fname);
	  for (i=0; i < nrpreds; i++)
	     fprintf (out,
	"\t       case %d: return (%s_name_from_nodenr (nodenr));\n",
		i+1, predefs [i]);
	  fprintf (out, "\t       default: panic (\"strange module");
	  fprintf (out, " nr %%d\", mnr);\n");
	  fprintf (out, "\t     };\n");
	  fprintf (out, "\t  return (NULL);\n");
	  fprintf (out, "\t};\n\n");
	};

/*
   This will fail if 'fname' contains quotes (")
*/
public void generate_main (char *fname, char **predefs, int nrpreds)
	{ int i;
	  if (interface_to_c) return;
	  fprintf (out, "public void main (int argc, char **argv)\n");	
	  fprintf (out, "\t{ int status;\n");
	  fprintf (out, "\t  char *fname;\n");
	  fprintf (out, "\t  init_error ();\n");
	  if (traceflag) fprintf (out, "\t  init_trace ();\n");
	  if (editor) fprintf (out,
		"\t  status = init_editor (\"%s\", &argc, argv);\n", fname);
	  else
	     fprintf (out,
		"\t  status = init_transducer (argc, argv, 0, &fname);\n");
	  for (i=0; i < nrpreds; i++)
	     fprintf (out, "\t  init_%s (%d);\n", predefs[i], i+1);
	  if (editor)
	     { fprintf (out, "\t  enter_templates ();\n");
	       fprintf (out, "\t  initial_transduce_and_unparse (status);\n");
	       fprintf (out, "\t  start_editor ();\n");
	     }
	  else
	     { fprintf (out, "\t  transduce ();\n");
	       fprintf (out, "\t  complain_on_found_parses ();\n");
	       if (matchflag) fprintf (out, "\t  dump_matches ();\n");
	     };
	  fprintf (out, "\t  exit (0);\n");
	  fprintf (out, "\t};\n\n");
	};
