/*
   File: codeskel.c
   Codes a skeleton library out of a specification
*/

/* global includes */
#include <stdio.h>
#include <sys/types.h>
#include <sys/time.h>

/* libeag includes */
#include <export.h>
#include <error.h>

/* local includes */
#include <sizes.h>
#include <tree.h>
#include <numbering.h>
#include <main.h>
#include <codeskel.h>

/*
   Transput stuff
*/
private FILE *out;
private void open_output_file (char *basename, char *suffix)
	{ char totalname[MAXFNAME];
	  time_t thetime;
	  char *atime;
	  sprintf (totalname, "%s.%s", basename, suffix);
	  if (!(out = fopen (totalname, "w")))
	     panic ("can't open outputfile %s", totalname);
	  time (&thetime);
	  atime = ctime (&thetime);
	  fprintf (out, "/*\n");
	  fprintf (out, "   File: %s\n", totalname);
	  fprintf (out, "   Generated on %s", atime);
	  fprintf (out, "   Implements the predefines library '%s'\n",
			basename);

	  fprintf (out, "*/\n");
	};
#define close_output_file() fclose (out)

/*
   Code skeleton include file
*/
private char *rule_qualifier (hyper_rule rule)
	{ if (rule -> kind & h_semipredicate) return ("semipred");
	  if (rule -> kind & h_predicate) return ("pred");
	  return ("rule");	/* should not occur actually */
	};

private void open_conditional (char *name)
	{ fprintf (out, "#ifndef Inc_%s\n", name);
	  fprintf (out, "#define Inc_%s\n\n", name);
	};

private void close_conditional (char *name)
	{ fprintf (out, "#endif /* Inc_%s */\n", name);
	};

private void code_include_rule (hyper_rule rule)
	{ if (!rule -> ext) return;
	  fprintf (out, "export void %s_%s ();\n", rule_qualifier (rule),
			rule -> nonterminal);
	};

private void code_include_rules ()
	{ int i;
	  for (i = 0; i < nr_of_hyper_rules; i++)
	     code_include_rule (all_hyper_rules[i]);
	};

private void code_include_metarule (meta_rule rule)
	{ if (!rule -> ext) return;
	  fprintf (out, "export affixnode make_%s_value ();\n",
			rule -> meta_nonterminal);
	  fprintf (out, "export void rec_%s_value ();\n",
			rule -> meta_nonterminal);
	};

private void code_include_metarules ()
	{ int i;
	  for (i = 0; i < nr_of_meta_rules; i++)
	     code_include_metarule (all_meta_rules[i]);
	};

private void code_interface_spec (char *name)
	{ fprintf (out, "\nexport char *%s_name_from_nodenr (int nodenr);\n",
					name);
	  fprintf (out, "export void init_%s ();\n", name);
	};

private void generate_include_file (char *name)
	{ open_output_file (name, "h");
	  open_conditional (name);
	  code_include_rules ();
	  code_include_metarules ();
	  code_interface_spec (name);
	  close_conditional (name);
	  close_output_file();
	};

/*
   Code skeleton file
*/
private char *module_name; 
private int cnodenr;
private void code_source_header (char *name)
	{ fprintf (out, "/* Global includes */\n");
	  fprintf (out, "#include <stdio.h>\n\n");
	  fprintf (out, "/* Libeag includes */\n");
	  fprintf (out, "#include <memalloc.h>\n");
	  fprintf (out, "#include <textstorage.h>\n");
	  fprintf (out, "#include <ds.h>\n");
	  fprintf (out, "#include <buildtree.h>\n");
	  fprintf (out, "#include <buildaffixgraph.h>\n");
	  fprintf (out, "#include <propagate.h>\n");
	  fprintf (out, "#include <nodenrs.h>\n\n");
	  fprintf (out, "/* Local includes */\n");
	  fprintf (out, "#include <%s.h>\n\n", name);
	  fprintf (out, "/* our module number */\n");
	  fprintf (out, "private int %s_modnr = 2;\n\n", name);
	  module_name = name;
	  cnodenr = 0;
	};

/*
   Code the skeletons of hyper rules
*/
private void code_calculate_inherited_values (pos_list dpy)
	{ int i;
	  int first = 1;
	  for (i=0; i < dpy -> nrofps; i++)
	     if (dpy -> ps[i] -> kind == inherited)
		{ fprintf (out,
		"\t%svaluenode v%d = calc_affix_value (ps%d, lower_side);\n",
			(first)?"{ ":"  ", i+1, i+1);
		  first = 0;
		};
	};

private int code_calculate_derived_values (pos_list dpy)
	{ int i;
	  int first = 1;
	  int der = 0;
	  fprintf (out, "\t  if (/* insert code to test args */ 1)\n");
	  for (i=0; i < dpy -> nrofps; i++)
	     if (dpy -> ps[i] -> kind != inherited)
	        { fprintf (out, "\t     %svaluenode v%d;\n",
			(first)?"{ ":"  ", i+1);
		  first = 0;
		  der = 1;
	        };
	  if (derived) fprintf (out,
		"\t       /* insert code to calculate derived values */\n");
	  else fprintf (out,
		"\t     /* insert code to use inherited values */;\n");
	  return (der);
	};

private void code_propagate_derived_values (pos_list dpy)
	{ int i;
	  int nrpushes = 0;
	  for (i=0; i < dpy -> nrofps; i++)
	     if (dpy -> ps[i] -> kind != inherited)
		{ fprintf (out, "\t       pushp (ps%d);\n", i+1);
		  fprintf (out, "\t       pushv (v%d);\n", i+1);
		  fprintf (out,
			"\t       pushq (propagate_predicate_value);\n");
		  nrpushes += 3;
		};
	  fprintf (out, "\t       callq ();\n");
	  fprintf (out, "\t       pop (%d);\n", nrpushes);
	};
 
private void code_detach_derived_values (pos_list dpy)
	{ int i;
	  for (i=0; i < dpy -> nrofps; i++)
	     if (dpy -> ps[i] -> kind != inherited)
		fprintf (out, "\t       detach_valuenode (v%d);\n", i+1);
	  fprintf (out, "\t     };\n");
	};

private void code_detach_inherited_values (pos_list dpy)
	{ int i;
	  for (i=0; i < dpy -> nrofps; i++)
	     if (dpy -> ps[i] -> kind == inherited)
		fprintf (out, "\t  detach_valuenode (v%d);\n", i+1);
	};

private void code_actual_predicate (hyper_rule rule, pos_list dpy)
	{ int i;
	  int der;
	  fprintf (out, "private void act_%s (", rule -> nonterminal);
	  for (i=0; i < dpy -> nrofps; i++)
	     fprintf (out, "%sposnode ps%d", (i==0)?"":", ", i+1);
	  fprintf (out, ")\n");
	  code_calculate_inherited_values (dpy);
	  der = code_calculate_derived_values (dpy);
	  if (der)
	     { code_propagate_derived_values (dpy);
	       code_detach_derived_values (dpy);
	     };
	  code_detach_inherited_values (dpy);
	  fprintf (out, "\t};\n\n");
	};

private void generate_check_inherited_affixes (char *cond,
					       pos_list dpy, char *check)
	{ int first = 1;
	  int i;
	  fprintf (out, "\t  %s", cond);
	  for (i=0; i < dpy -> nrofps; i++)
	     { if (dpy -> ps[i] -> kind == inherited)
		 { fprintf (out, "%s!%s (ps%d)", (first)?"":" || ", check, i+1);
		   first = 0;
		 };
	     };
	  fprintf (out, ")\n\t     callq();\n");
	};

private void code_delayed_predicate (hyper_rule rule, pos_list dpy)
	{ int i;
	  fprintf (out, "private void delayed_%s (posnode *ps)\n",
			rule -> nonterminal);
	  for (i=0; i < dpy -> nrofps; i++)
	     fprintf (out, "\t%sposnode ps%d = ps[%d];\n",
		(i==0)?"{ ":"  ", i+1, i);
	  generate_check_inherited_affixes ("if (", dpy, "position_is_defined");
	  generate_check_inherited_affixes ("else if (",
		dpy, "position_has_value");
	  fprintf (out, "\t  else\n");
	  for (i=0; i < dpy -> nrofps; i++)
	     fprintf (out, "\t   %sps%d -> delayed = 0;\n",
		(i==0)?"{ ":"  ", i+1);
	  fprintf (out, "\t     act_%s (", rule -> nonterminal);
	  for (i=0; i < dpy -> nrofps; i++)
	     fprintf (out, "%sps%d", (i==0)?"":", ", i+1);
	  fprintf (out, ");\n");
	  for (i=0; i < dpy -> nrofps; i++)
	     fprintf (out, "\t     ps%d -> delayed = 1;\n", i+1);
	  fprintf (out, "\t   };\n");
	  fprintf (out, "\t};\n\n");
	};

private void code_proper_predicate (hyper_rule rule, pos_list dpy)
	{ int i;
	  fprintf (out, "public void pred_%s ()\n", rule -> nonterminal);
	  for (i=0; i < dpy -> nrofps; i++)
	     fprintf (out,
		"\t%saffixnode af%d = new_affixnode (\"%s_af%d\");\n",
		(i==0)?"{ ":"  ", i+1, rule -> nonterminal, i+1);
	  fprintf (out, "\t  pushq (delayed_%s);\n", rule -> nonterminal);
	  fprintf (out, "\t  pushq (make_node_delayed);\n");
	  for (i=dpy -> nrofps; 1 <= i; i--)
	     { fprintf (out, "\t  pusha (af%d);\n", i);
	       fprintf (out, "\t  pushi (singleaffix);\n");
	     };
	  fprintf (out, "\t  pushi (%d);\n", dpy -> nrofps);
	  fprintf (out, "\t  pushi (0);\n");
	  fprintf (out, "\t  pushi (mk_nodenr (%s_modnr, %d));\n",
		module_name, cnodenr);
	  cnodenr++;
	  fprintf (out, "\t  pushq (make_predicatenode);\n");
	  fprintf (out, "\t  callq ();\n");
	  fprintf (out, "\t  pop (%d);\n", 2 * dpy -> nrofps + 6);
	  for (i=0; i < dpy -> nrofps; i++)
	     { fprintf (out, "\t  detach_valuenode (af%d -> value);\n", i+1);
	       fprintf (out, "\t  free_affixnode (af%d);\n", i+1);
	     };
	  fprintf (out, "\t  pushq (pred_%s);\n", rule -> nonterminal);
	  fprintf (out, "\t};\n\n");
	};

private void code_predicate (hyper_rule rule)
	{ alt a = rule -> alts -> as[0];
	  pos_list dpy = a -> display;
	  if (dpy == pos_list_nil)
	     panic ("predicate %s has 0 positions\n", rule -> nonterminal);
	  code_actual_predicate (rule, dpy);
	  code_delayed_predicate (rule, dpy);
	  code_proper_predicate (rule, dpy);
	};

private void code_semipredicate (hyper_rule rule)
	{ alt a = rule -> alts -> as[0];
	  pos_list dpy = a -> display;
	  int nrps = (dpy == pos_list_nil)?0:dpy -> nrofps;
	  int i; 

	  fprintf (out, "public void semipred_%s ()\n", rule -> nonterminal);
	  fprintf (out, "\t{\n");
	  for (i=0; i < nrps; i++) fprintf (out,
		"\t  affixnode af%d = new_affixnode (\"%s_af%d\");\n",
		i+1, rule -> nonterminal, i+1);
	  for (i=nrps; 1 <= i; i--)
	     { fprintf (out, "\t  pusha (af%d);\n", i);
	       fprintf (out, "\t  pushi (single_affix);\n");
	     };
	  fprintf (out, "\t  pushi (%d);\n", nrps);
	  fprintf (out, "\t  pushi (0);\n");
	  fprintf (out, "\t  pushi (mk_nodenr (%s_modnr, %d));\n",
		module_name, cnodenr);
	  cnodenr++;
	  fprintf (out, "\t  pushq (make_semipredicatenode);\n");
	  fprintf (out, "\t  callq ();\n");
	  fprintf (out, "\t  pop (%d);\n", 2 * nrps + 4);
	  for (i=0; i < nrps; i++)
	     { fprintf (out, "\t  detach_valuenode (af%d -> value);\n", i+1);
	       fprintf (out, "\t  free_affixnode (af%d);\n", i+1);
	     };
	  fprintf (out, "\t  pushq (semipred_%s);\n", rule -> nonterminal);
	  fprintf (out, "\t};\n\n");
	};

private void code_hyper_rule (hyper_rule rule)
	{ if (!rule -> ext) return;
	  if (rule -> kind & h_predicate) code_predicate (rule);
	  else code_semipredicate (rule);
	};

private void code_hyper_rules ()
	{ int i;
	  for (i = 0; i < nr_of_hyper_rules; i++)
	     code_hyper_rule (all_hyper_rules[i]);
	};

/*
   Code the skeletons of meta rules
*/
private void code_recognize_rule (meta_rule rule)
	{ fprintf (out, "public void rec_%s_value ()\n",
			rule -> meta_nonterminal);
	  fprintf (out, "\t{ valuenode v = popv ();\n");
	  fprintf (out, "\t  /* insert code to recognize v */\n");
	  fprintf (out, "\t  /* end of insertion */\n");
	  fprintf (out, "\t  pushv (v);\n");
	  fprintf (out, "\t  pushq (rec_%s_value);\n",
			rule -> meta_nonterminal);
	  fprintf (out, "\t};\n\n");
	};

private void code_make_rule (meta_rule rule)
	{ fprintf (out, "public affixnode make_%s_value ()\n",
			rule -> meta_nonterminal);
	  fprintf (out, "\t{ affixnode new = new_affixnode (\"predef_%s\");\n",
			rule -> meta_nonterminal);
	  fprintf (out, "\t  new -> defined = 1;\n");
	  fprintf (out, "\t  new -> mfunc = rec_%s_value;\n",
			rule -> meta_nonterminal);
	  fprintf (out, "\t  return (new);\n");
	  fprintf (out, "\t};\n\n");
	};

private void code_meta_rule (meta_rule rule)
	{ code_recognize_rule (rule);
	  code_make_rule (rule);
	};

private void code_meta_rules ()
	{ int i;
	  for (i = 0; i < nr_of_meta_rules; i++)
	     code_meta_rule (all_meta_rules[i]);
	};

private void code_hyper_rule_nodenr (hyper_rule rule)
	{ if (!rule -> ext) return;
	  fprintf (out, "\t     %scase %d: return (\"%s_%s\");\n",
			(cnodenr)?"  ":"{ ",
			cnodenr, rule_qualifier (rule), rule -> nonterminal);
	  cnodenr++;
	};

private void code_hyper_rule_nodenrs ()
	{ int i;
	  for (i = 0; i < nr_of_hyper_rules; i++)
	     code_hyper_rule_nodenr (all_hyper_rules[i]);
	};

private void code_source_trailer (char *name)
	{ cnodenr = 0;
	  fprintf (out, "public char *%s_name_from_nodenr (int nodenr)\n",
			name);
	  fprintf (out, "\t{ int lnr = lnodenr_from_nodenr (nodenr);\n");
	  fprintf (out, "\t  switch (lnr)\n");
	  code_hyper_rule_nodenrs ();
	  fprintf (out, "\t       default: fprintf (stderr, ");
	  fprintf (out, "\"strange nodenr %%d in %s\\n\", lnr);\n", name);
	  fprintf (out, "\t     };\n");
	  fprintf (out, "\t  exit (4);\n");
	  fprintf (out, "\t};\n\n");
	  fprintf (out, "void init_%s (modnr)\n", name);
	  fprintf (out, " int modnr;\n");
	  fprintf (out, "\t{ %s_modnr = modnr;\n", name);
	  fprintf (out, "\t};\n");
	};

private void generate_source_file (char *name)
	{ open_output_file (name, "c");
	  code_source_header (name);
	  code_hyper_rules ();
	  code_meta_rules ();
	  code_source_trailer (name);
	  close_output_file();
	};

public void generate_skeleton (char *name)
	{ generate_include_file (name);
	  generate_source_file (name);
	};
