/*
   File: codemeta.c
   Codes the meta rules
*/

/* global includes */
#include <stdio.h>

/* libeag includes */
#include <export.h>
#include <error.h>

/* local includes */
#include <tree.h>
#include <numbering.h>
#include <common.h>
#include <main.h>
#include <typecheck.h>
#include <codemeta.h>

/*
   Forward declare all meta rule creator and parsing routines
*/
private void generate_meta_rule_declaration (meta_rule rule)
	{ if (!rule -> reachable) return;
	  if (rule -> ext)
	     { fprintf (out, "import affixnode make_%s_value ();\n",
			rule -> meta_nonterminal);
	       fprintf (out, "import void rec_%s_value ();\n",
			rule -> meta_nonterminal);
	       return;
	     };
          fprintf (out, "private affixnode make_%s_value ();\n",
			rule -> meta_nonterminal);
	  fprintf (out, "private void rec_%s_value ();\n",
			rule -> meta_nonterminal);
	  if ((rule -> type == tuple_type) ||
	      (rule -> type == any_type)) return;
	  fprintf (out, "private void recbody_%s_value ();\n",
			rule -> meta_nonterminal);
	};

private void generate_meta_rule_declarations ()
	{ int i;
	  for (i = 0; i < nr_of_meta_rules; i++)
	     generate_meta_rule_declaration (all_meta_rules[i]);
	  fprintf (out, "\n");
	};

/*
   Generate creator routines
*/
private void tuple_creator_indenter (int i)
	{ int j;
	  fputs ("\t  ", out);
	  for (j=0; j < 5 * i; j++) fputc (' ', out);
	};

private void generate_tuple_creator (char *dest, value v, int indent)
	{ int j;
	  tuple_creator_indenter (indent);
	  fprintf (out, "   { valuenode t%d = new_tuple_value (%d);\n",
		indent, v -> v.tuple -> nrofvs);
	  for (j=0; j < v -> v.tuple -> nrofvs; j++)
	     { value sv = v -> v.tuple -> vals[j];
	       char TBuffer[40];
	       sprintf (TBuffer, "t%d -> v.co.vals[%d]", indent, j);
	       switch (sv -> type)
		  { case string_type:
		       { tuple_creator_indenter (indent+1);
			 fprintf (out, "%s = string_to_value (", TBuffer);
		         output_string (out, sv -> v.string);
			 fputs (");\n", out);
		       };
		       break;
		    case integer_type:
		       { tuple_creator_indenter (indent+1);
			 fprintf (out, "%s = number_to_value (%d);\n",
				TBuffer, sv -> v.number);
		       };
		       break;
		    case tuple_type:
		       generate_tuple_creator (TBuffer, sv, indent+1);
		       break;
		  };
	     };
	  tuple_creator_indenter (indent+1);
	  fprintf (out, "%s = t%d;\n", dest, indent);
	  tuple_creator_indenter (indent);
	  fputs ("   };\n", out);
	};

private void generate_creator_body (meta_rule rule)
	{ if (rule -> kind == single_meta_value)
	     { value val = rule -> mvalue;
	       switch (rule -> type)
		  { case string_type:
		       { fprintf (out, "string_to_affix (\"meta_%s\", ",
			 	rule -> meta_nonterminal);
		         output_string (out, val -> v.string);
		         fprintf (out, ");\n");
		         return;
		       };
		    case integer_type:
		       { fprintf (out, "number_to_affix (\"meta_%s\", %d);\n",
			    rule -> meta_nonterminal, val -> v.number);
		         return;
		       };
		    case tuple_type:
		       { fprintf (out, "new_affixnode (\"meta_%s\");\n",
				rule -> meta_nonterminal);
			 fprintf (out, "\t  new -> defined = 1;\n");
			 fprintf (out, "\t  new -> hasval = 1;\n");
			 generate_tuple_creator ("new -> value", val, 0);
			 return;
		       };
		  };
	     };
	  fprintf (out, "new_affixnode (\"meta_%s\");\n",
			rule -> meta_nonterminal);
	  fprintf (out, "\t  new -> defined = 1;\n");
	  fprintf (out, "\t  new -> mfunc = rec_%s_value;\n",
			rule -> meta_nonterminal);
	};

private void generate_creator_rule (meta_rule rule)
	{ if (rule -> ext) return;
	  if (!rule -> reachable) return;
	  fprintf (out, "private affixnode make_%s_value ()\n",
			rule -> meta_nonterminal);
	  fprintf (out, "\t{ affixnode new = ");
	  generate_creator_body (rule);
	  fprintf (out, "\t  return (new);\n");
	  fprintf (out, "\t};\n\n");
	};

private void generate_creator_rules ()
	{ int i;
	  for (i = 0; i < nr_of_meta_rules; i++)
	     generate_creator_rule (all_meta_rules[i]);
	};

/*
   Generate recognizer routines
*/
private char *meta_rule_type (meta_rule rule)
	{ switch (rule -> type)
	     { case string_type: return ("string");
	       case integer_type: return ("number");
	       case tuple_type: return ("compos");
	       case any_type: return ("any");
	       default: return ("undefined");
	     };
	};

private void generate_qstack_mcheck (meta_rule rule, int i, int body)
	{ char *indent = (body)?"":"    ";
	  fprintf (out, "\t   %sif (lqptr != qptr)\n", indent);
	  fprintf (out, "\t   %s   { fprintf (stderr,\
\"qstack inconsistency detected in metarule %s, alt %d, rec%s\\n\");\n",
		indent, rule -> meta_nonterminal, i+1, (body)?"body":"");
	  fprintf (out, "\t   %s     exit (4);\n", indent);
	  fprintf (out, "\t   %s   };\n", indent);
	};

private void generate_rec_affix (affix a, char *name, int *nrpushes)
	{ fprintf (out, "\t\tpushv (%s);\n", name);
	  switch (a -> tag)
	     { case tag_affix_nonterminal:
		  { meta_rule def = a -> u.var -> def;
		    fprintf (out, "\t\tpushq (rec_%s_value);\n",
				def -> meta_nonterminal);
		    *nrpushes += 2;
		  };
		  break;
	       case tag_affix_terminal:
		  { fprintf (out, "\t\tpushs (");
		    output_string (out, a -> u.string);
		    fprintf (out, ");\n");
		    fprintf (out, "\t\tpushq (rec_parse_terminal);\n");
		    *nrpushes += 3;
		  };
		  break;
	       case tag_affix_number:
		  { fprintf (out, "\t\tpushi (%d);\n", a -> u.number);
		    fprintf (out, "\t\tpushq (rec_parse_number);\n");
		    *nrpushes += 3;
		  };
		  break;
	       case tag_affix_set:
		  { fprintf (out, "\t\tpushs (");
		    output_string (out, a -> u.aset -> string);
		    fprintf (out, ");\n");
		    fprintf (out, "\t\tpushq (rec_parse");
		    if (a -> u.aset -> kind & non) fprintf (out, "_non");
		    fprintf (out, "_set");
		    if (a -> u.aset -> kind & star) fprintf (out, "_star");
		    if (a -> u.aset -> kind & plus) fprintf (out, "_plus");
		    if (a -> u.aset -> kind & strict) fprintf (out, "_strict");
		    fprintf (out, ");\n");
		    *nrpushes += 3;
		  };
	     };
	};

private void generate_recbody_affix (affix a, int *nrpushes)
	{ switch (a -> tag)
	     { case tag_affix_nonterminal:
		  { meta_rule def = a -> u.var -> def;
		    fprintf (out, "\t\tpushq (recbody_%s_value);\n",
				def -> meta_nonterminal);
		    *nrpushes += 1;
		  };
		  break;
	       case tag_affix_terminal:
		  { fprintf (out, "\t\tpushs (");
		    output_string (out, a -> u.string);
		    fprintf (out, ");\n");
		    fprintf (out, "\t\tpushq (recbody_parse_terminal);\n");
		    *nrpushes += 2;
		  };
		  break;
	       case tag_affix_number:
		  { fprintf (out, "\t\tpushi (%d);\n", a -> u.number);
		    fprintf (out, "\t\tpushq (recbody_parse_number);\n");
		    *nrpushes += 2;
		  };
		  break;
	       case tag_affix_set:
		  { fprintf (out, "\t\tpushs (");
		    output_string (out, a -> u.aset -> string);
		    fprintf (out, ");\n");
		    fprintf (out, "\t\tpushq (recbody_parse");
		    if (a -> u.aset -> kind & non) fprintf (out, "_non");
		    fprintf (out, "_set");
		    if (a -> u.aset -> kind & star) fprintf (out, "_star");
		    if (a -> u.aset -> kind & plus) fprintf (out, "_plus");
		    if (a -> u.aset -> kind & strict) fprintf (out, "_strict");
		    fprintf (out, ");\n");
		    *nrpushes += 2;
		  };
	     };
	};

private void generate_rec_single (affix a)
	{ int nrpushes = 0;
	  generate_rec_affix (a, "v", &nrpushes);
	  fprintf (out, "\t\tcallq ();\n");
	  fprintf (out, "\t\tpop (%d);\n", nrpushes);
	};

private void generate_rec_compos (affix_list affs)
	{ char buffer[40];
	  int i;
	  int nrpushes = 0;
	  fprintf (out, "\t\tif (v -> v.co.nr == %d)\n", affs -> nrofas);
	  fprintf (out, "\t\t  {\n");
	  for (i = affs -> nrofas - 1; 0 <= i; i--)
	     { sprintf (buffer, "v -> v.co.vals[%d]", i);
	       generate_rec_affix (affs -> as[i], buffer, &nrpushes);
	     };
	  fprintf (out, "\t\t     callq ();\n");
	  fprintf (out, "\t\t     pop (%d);\n", nrpushes);
	  fprintf (out, "\t\t   };\n");
	};

private void generate_rec_concat (meta_rule rule, affix_list affs)
	{ int i;
	  int nrpushes = 1;
	  if (rule -> type == string_type)
	     { fprintf (out, "\t\tmiptr = v -> v.string;\n");
	       fprintf (out, "\t\tpushq (meta_endofsentence);\n");
	     }
	  else
	     { fprintf (out, "\t\tmtotal = v -> v.number;\n");
	       fprintf (out, "\t\tpushq (meta_endofnumber);\n");
	     };
	  for (i=affs -> nrofas - 1; 0 <= i; i--)
	     generate_recbody_affix (affs -> as[i], &nrpushes);
	  fprintf (out, "\t\tcallq ();\n");
	  fprintf (out, "\t\tpop (%d);\n", nrpushes);
	};

private void generate_rec_expr (meta_rule rule, int i, expr e)
	{ if (qstackflag) fprintf (out, "\t     { cont *lqptr = qptr;\n");
	  else fprintf (out, "\t     {\n");
	  if (e == expr_nil)
	     { fprintf (out, "\t       pushv (v);\n");
	       fprintf (out, "\t       pushq (rec_empty_value);\n");
	       fprintf (out, "\t       callq ();\n");
	       fprintf (out, "\t       pop (2);\n");
	     }
	  else
	     switch (e -> tag)
	        { case tag_single: generate_rec_single (e -> u.single); break;
	          case tag_compos: generate_rec_compos (e -> u.compos); break;
	          case tag_concat: generate_rec_concat (rule, e -> u.concat);
				   break;
	        };
	  if (qstackflag) generate_qstack_mcheck (rule, i, 0);
	  fprintf (out, "\t     };\n");
	};

private void generate_rec_alts (meta_rule rule)
	{ int i;
	  for (i=0; i < rule -> meta_alts -> nrofas; i++)
	     generate_rec_expr (rule, i, rule -> meta_alts -> as[i] -> e);
	};

private void generate_recognizer_body (meta_rule rule)
	{ if ((rule -> kind == single_meta_value) &&
	      (rule -> type == string_type))
	     { fprintf (out, "\t    { if (strcmp (v -> v.string, ");
	       output_string (out, rule -> mvalue -> v.string);
	       fprintf (out, ") == 0) callq ();\n");
	     }
	  else if ((rule -> kind == single_meta_value) &&
		   (rule -> type == integer_type))
	     { fprintf (out, "\t    { if (v -> v.number == %d) callq ();\n",
			rule -> mvalue -> v.number);
	     }
	  else
	     { fprintf (out, "\t    {\n");
	       generate_rec_alts (rule);
	     };
	  fprintf (out, "\t    };\n");
	};

private void generate_recognizer_rule (meta_rule rule)
	{ if (rule -> ext) return;
	  if (!rule -> reachable) return;
	  fprintf (out, "private void rec_%s_value ()\n",
			rule -> meta_nonterminal);
	  fprintf (out, "\t{ valuenode v = popv ();\n");
	  if (rule -> type == any_type) fprintf (out, "\t  callq ();\n");
	  else
	     { fprintf (out, "\t  if (v -> type == undefinedtype) callq ();\n");
	       fprintf (out, "\t  else if (v -> type == %stype)\n",
			meta_rule_type (rule));
	       generate_recognizer_body (rule);
	     };
	  fprintf (out, "\t  pushv (v);\n");
	  fprintf (out, "\t  pushq (rec_%s_value);\n",
			rule -> meta_nonterminal);
	  fprintf (out, "\t};\n\n");
	};

private void generate_recognizer_rules ()
	{ int i;
	  for (i = 0; i < nr_of_meta_rules; i++)
	     generate_recognizer_rule (all_meta_rules[i]);
	};

/*
   Generate parser routines
*/
private void generate_recbody_concat (affix_list affs, int *nrpushes)
	{ int i;
	  for (i=affs -> nrofas - 1; 0 <= i; i--)
	     generate_recbody_affix (affs -> as[i], nrpushes);
	};

private void generate_parser_expr (expr e, int *nrpushes)
	{ if (e == expr_nil) return;
	  switch (e -> tag)
	     { case tag_single: 
		  generate_recbody_affix (e -> u.single, nrpushes);
		  break;
	       case tag_concat:
		  generate_recbody_concat (e -> u.concat, nrpushes);
		  break;
	       default: panic ("inconsistency in meta rule generation");
	     };
	};

private void generate_parser_alt (meta_rule rule, int i, expr e)
	{ int nrpushes = 0;
	  if (qstackflag) fprintf (out, "\t { cont *lqptr = qptr;\n");
	  else fprintf (out, "\t {\n");
	  generate_parser_expr (e, &nrpushes);
	  fprintf (out, "\t   callq ();\n");
	  fprintf (out, "\t   pop (%d);\n", nrpushes);
	  if (qstackflag) generate_qstack_mcheck (rule, i, 1);
	  fprintf (out, "\t };\n");
	};

private void generate_parser_alts (meta_rule rule)
	{ int i;
	  meta_alt_list alts = rule -> meta_alts;
	  if (alts == meta_alt_list_nil) fprintf (out, "\t  callq ();\n");
	  else
	     for (i=0; i < alts -> nrofas; i++)
		generate_parser_alt (rule, i, alts -> as[i] -> e);
	};

private void generate_parser_rule (meta_rule rule)
	{ if (rule -> ext) return;
	  if (!rule -> reachable) return;
	  if (rule -> type == tuple_type) return;
	  fprintf (out, "private void recbody_%s_value ()\n",
			rule -> meta_nonterminal);
	  fprintf (out, "\t{\n");
	  generate_parser_alts (rule);
	  fprintf (out, "\t  pushq (recbody_%s_value);\n",
			rule -> meta_nonterminal);
	  fprintf (out, "\t};\n\n");
	};

private void generate_parser_rules ()
	{ int i;
	  for (i = 0; i < nr_of_meta_rules; i++)
	     generate_parser_rule (all_meta_rules[i]);
	};

void generate_meta_rules ()
	{ warning ("coding meta rules...");
	  generate_meta_rule_declarations ();
	  generate_creator_rules ();
	  generate_recognizer_rules ();
	  generate_parser_rules ();
	};
