/* rtslint.c - Lexicon interface part
 *
 * Copyright 2000 KUN.
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Library General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

/* $Id: reading.c,v 1.13 2001/10/17 10:39:31 ejv Exp $ */

#include <assert.h>
#include <stdio.h>
#include <stdlib.h>
#include <ctype.h>
#include <sys/types.h>
#include <setjmp.h>
#include <glib.h>
#include <string.h>
#include "lexicon.h"
#include "lexiconutil.h"

/*
** IEIEIE! Global alert!
** This is needed for error handling.
*/
jmp_buf lex_file_error_env;

/*
**------------------------------------------------------------------------------
** Debugging macros:
**------------------------------------------------------------------------------
*/

#ifdef DEBUG
#define DB(X) X
#else
#define DB(X)
#endif

typedef long INT_AFFIX;
typedef char* TEXT_AFFIX;

/*
** Structure for storing the lexicon affixes
**
** name: affix name
** bitset: the bitset associated with this affix
** is_nont: if false, then this affix is a terminal, otherwise it's a
**          nonterminal
** lhs[]: the indexes of the left-hand-sides (affix nonterminals) that this
**        affix belongs to.
*/
typedef struct {
    char *name;
    SET bitset;
    gboolean is_nont;
    size_t nr_lhsses;
    off_t *lhs;
} SET_AFFIX;

/* Structure for storing the nonterminal parameters */
enum LexParamTypes {
    LexParamLattice = 1,
    LexParamText = 2,
    LexParamInt = 3
};

typedef struct {
    char type;
    off_t index;
} PARAMETER;

/* Structure for storing the lexicon rules (nonterminals) */
typedef struct {
    char *name;
    int arity;
    PARAMETER *pars;
} NONTERMINAL;

/* Structure for storing the entries */
typedef struct {
    unsigned nontnr;
    PENALTY penalty;
    unsigned *pars;
} ENTRY;

/* Generic structure for storing pairs */
typedef struct {
    unsigned index;
    int pointer;
} PAIR;


/*
**------------------------------------------------------------------------------
** Lexicon structure
**------------------------------------------------------------------------------
*/

struct LEXICON {
    Trie* trie_buf;
    size_t trie_size;

    size_t nr_entrylist_pairs;
    PAIR *entrylist_pair;
    
    size_t nr_nonterminals;
    NONTERMINAL **nonterminal;
    
    size_t nr_entries;
    ENTRY **entry;
    
    size_t nr_int_affixes;
    INT_AFFIX *int_affix;
    
    size_t nr_text_affixes;
    TEXT_AFFIX *text_affix;
    
    size_t nr_set_affixes;
    SET_AFFIX **set_affix;
}; /* typedef'd in rtslint.h */


/*
**------------------------------------------------------------------------------
** Lexicon file access routines
**------------------------------------------------------------------------------
*/

static char
lexfile_read_byte(FILE* lex_file)
{
    int c;

    if ((c = fgetc(lex_file)) == EOF) {
        DB(fprintf(stderr, "lexfile_read_byte"));
        longjmp(lex_file_error_env, 666);
    }

    return (char)c;
}

gboolean
lexfile_read_bool(FILE* lex_file)
{
    char c;

    c = lexfile_read_byte(lex_file);

    return c == 'T';
}

static char*
lexfile_read_string(FILE* lex_file)
{
    GString *tmp = g_string_sized_new(10);
    char c;
    char* res;

    while((c = lexfile_read_byte(lex_file)) != 0) {
        tmp = g_string_append_c(tmp, c);
    }

    res = tmp->str;
    g_string_free(tmp, FALSE);
    return res;
}


static int
lexfile_read_word(FILE* lex_file)
{
    int i = lexfile_read_byte(lex_file);
    int j = lexfile_read_byte(lex_file);

    return (i << 8) | j;
}

static SET
lexfile_read_set(FILE* lex_file)
{
    SET set;

    if (fread(&set, sizeof(set), 1, lex_file) != 1) {
        DB(fprintf(stderr, "lexfile_read_set"));
        longjmp(lex_file_error_env, 666);
    }

    return set;
}

static off_t
lexfile_read_index(FILE* lex_file)
{
    off_t res;

    if (fread(&res, sizeof(res), 1, lex_file) != 1) {
        DB(fprintf(stderr, "lexfile_read_index"));
        longjmp(lex_file_error_env, 666);
    }

    return res;
}

static size_t
lexfile_read_size(FILE* lex_file)
{
    size_t res;

    if (fread(&res, sizeof(res), 1, lex_file) != 1) {
        DB(fprintf(stderr, "lexfile_read_size"));
        longjmp(lex_file_error_env, 666);
    }

    return res;
}

static PENALTY
lexfile_read_penalty(FILE* lex_file)
{
    PENALTY res;

    if (fread(&res, sizeof(res), 1, lex_file) != 1) {
        DB(fprintf(stderr, "lexfile_read_penalty"));
        longjmp(lex_file_error_env, 666);
    }

    return res;
}


/*
**------------------------------------------------------------------------------
** INT affixes
**------------------------------------------------------------------------------
*/

static void
lexfile_read_int_affixes(FILE* lex_file, LEXICON* lex)
{
    off_t i;
    size_t nr = lexfile_read_size(lex_file);

    lex->nr_int_affixes = nr;

    DB(printf("reading %d INT affixes...\n", nr));

    if (nr) {
        lex->int_affix = (INT_AFFIX *) GetMem(sizeof(INT_AFFIX) * nr,
                                               "lex->int_affix");
  
        for (i = 0; i < nr; i++) {
            lex->int_affix[i] = lexfile_read_set(lex_file);
        }
    } else {
        lex->int_affix = NULL;
    }
}

static void
lexfile_free_int_affixes(LEXICON* lex)
{
    size_t nr = lex->nr_int_affixes;

    if (nr) {
        FreeMem(lex->int_affix, "lex->int_affix");
    }
}


/*
**------------------------------------------------------------------------------
** TEXT affixes
**------------------------------------------------------------------------------
*/

static void
lexfile_read_text_affixes(FILE* lex_file, LEXICON* lex)
{
    int i;
    TEXT_AFFIX input;
    size_t nr = lexfile_read_size(lex_file);

    lex->nr_text_affixes = nr;

    DB(printf("reading %d TEXT affixes...\n", nr));

    if (nr) {
        lex->text_affix = (TEXT_AFFIX*) GetMem(sizeof(TEXT_AFFIX) * nr,
                                        "lex->text_affix");

        for (i = 0; i < nr; i++) {
            input = lexfile_read_string(lex_file);
            lex->text_affix[i] = input;
            DB(printf("\tread affix: \"%s\"\n", input));
        }
    } else {
        lex->text_affix = NULL;
    }
}

static void
lexfile_free_text_affixes(LEXICON* lex)
{
    off_t i;

    if (lex->nr_text_affixes > 0) {
        for (i = 0; i < lex->nr_text_affixes; i++) {
            FreeMem(lex->text_affix[i], "TEXT_AFFIX");
        }

        FreeMem(lex->text_affix, "lex->text_affix");
    }
}


/*
**------------------------------------------------------------------------------
** Affix names
**------------------------------------------------------------------------------
*/

static void
lexfile_read_set_affixes(FILE* lex_file, LEXICON* lex)
{
    int i;
    SET_AFFIX* set_affix;

    size_t nr = lexfile_read_size(lex_file);
    lex->nr_set_affixes = nr;

    DB(printf("reading %d SET affixes...\n", nr));

    if (nr) {
        lex->set_affix = (SET_AFFIX **) GetMem(sizeof(SET_AFFIX*) * nr,
                                               "lex->set_affixes");
        for(i = 0; i < nr; i++) {
            int j;
            off_t nr_lhsses;

            set_affix = (SET_AFFIX*) GetMem(sizeof(SET_AFFIX), "SET_AFFIX");
            set_affix->name = lexfile_read_string(lex_file);
            DB(printf("\tread affix %d: \"%s\"\n", i, set_affix->name));
            set_affix->bitset = lexfile_read_set(lex_file);
            set_affix->is_nont = lexfile_read_bool(lex_file);

            nr_lhsses = lexfile_read_size(lex_file);
            set_affix->nr_lhsses = nr_lhsses;
            if (nr_lhsses) {
                set_affix->lhs = (off_t*) GetMem(sizeof(off_t) * nr_lhsses,
                                                 "SET_AFFIX->lhs");
                for (j = 0; j < nr_lhsses; j++) {
                    set_affix->lhs[j] = lexfile_read_index(lex_file);
                    DB(printf("\t\twith LHS: %d\n", (int)(set_affix->lhs[j])));
                }
            } else {
                set_affix->lhs = NULL;
            }

            lex->set_affix[i] = set_affix;
        }
    } else {
        lex->set_affix = NULL;
    }
}

static void
lexfile_free_set_affixes(LEXICON* lex)
{
    off_t i;
    size_t nr = lex->nr_set_affixes;

    if (nr) {
        for (i = 0; i < nr; i++) {
            FreeMem(lex->set_affix[i], "SET_AFFIX");
        }

        FreeMem(lex->set_affix, "lex->set_affix");
    }
}


/*
**------------------------------------------------------------------------------
** Nonterminal names
**------------------------------------------------------------------------------
*/

static void
lexfile_read_nonterminals(FILE* lex_file, LEXICON* lex)
{
    NONTERMINAL* nonterminal;
    off_t i, j;
    char* nont_name;

    size_t nr = lexfile_read_size(lex_file);
    lex->nr_nonterminals = nr;

    DB(printf("reading %d nonterminals\n", nr));

    if (nr) {
        lex->nonterminal = (NONTERMINAL**) GetMem(sizeof(NONTERMINAL*) * nr,
                                                  "lex->nonterminal");
        memset(lex->nonterminal, 0, sizeof(NONTERMINAL*) * nr);

        for (i = 0; i < nr; i++) {
            nont_name = lexfile_read_string(lex_file);

            if (nont_name[0] == '\0') {
                /* empty nonterminal entry -- this means this nonterminal is
                   not a lexicon nonterminal */
                DB(printf("reading nonterminal %d: syntax nonterminal\n", (int)i));
                free(nont_name);
                nonterminal = NULL;
            } else {
                nonterminal = (NONTERMINAL*) GetMem(sizeof(NONTERMINAL),
                                                    "NONTERMINAL");
                nonterminal->name = nont_name;
                DB(printf("reading nonterminal %d: \"%s/", (int)i, nonterminal->name));
                nonterminal->arity = lexfile_read_word(lex_file);
                DB(printf("%d\"\n", nonterminal->arity));
    
                if (nonterminal->arity) {
                    nonterminal->pars = (PARAMETER*) GetMem(sizeof(PARAMETER)
                                                            * nonterminal->arity,
                                                            "PARAMETERS");
    
                    for (j = 0; j < nonterminal->arity; j++) {
                        char type = lexfile_read_byte(lex_file);
                        (nonterminal->pars[j]).type = type;
                        switch (type) {
                            case LexParamLattice:
                                (nonterminal->pars[j]).index =
                                    lexfile_read_index(lex_file);
                                break;
                            case LexParamText:
                            case LexParamInt:
                                break;
                            default:
                                DB(fprintf(stderr, "lexfile_read_nonterminals: Unknown parameter type 0x%x\n", type));
                                longjmp(lex_file_error_env, 666);
                        }
                    }
                } else {
                    nonterminal->pars = NULL;
                }
            }

            lex->nonterminal[i] = nonterminal;
        }
    } else {
        lex->nonterminal = NULL;
    }
}

static void
lexfile_free_nonterminals(LEXICON* lex)
{
    off_t i;
    size_t nr = lex->nr_nonterminals;

    if (nr) {
        for (i = 0; i < nr; i++) {
            if (lex->nonterminal[i]) {
                free((lex->nonterminal[i])->name);
                FreeMem((lex->nonterminal[i])->pars, "PARAMETERS");
            }
        }

        FreeMem(lex->nonterminal, "lex->nonterminal");
    }
}

/*
**------------------------------------------------------------------------------
** Trie alloc/read/free routines:
**------------------------------------------------------------------------------
*/

#define TS_LENGTH sizeof(long)

static void
lexfile_read_trie_size(FILE* lex_file, LEXICON* lex)
{
    unsigned long ptr;

    if (fread((char *)&ptr, TS_LENGTH, 1, lex_file) != 1) {
        longjmp(lex_file_error_env, 666);
    }

    DB(printf("lexfile_read_trie_size: size = %lu bytes\n", ptr));
    lex->trie_size = ptr;
}

static void
lexfile_read_trie(FILE* lex_file, LEXICON* lex)
{
    lexfile_read_trie_size(lex_file, lex);
    lex->trie_buf = GetMem(lex->trie_size, "trie");
  
    if (fread(lex->trie_buf, 1, lex->trie_size, lex_file) != lex->trie_size) {
        longjmp(lex_file_error_env, 666);
    }

    DB(printf("lexfile_read_trie: %d bytes\n", lex->trie_size));
}

static void
lexfile_free_trie(LEXICON* lex)
{
  FreeMem(lex->trie_buf, "trie");
}

Trie*
get_lexicon_trie(const LEXICON* lex)
{
    assert(lex);
    return lex->trie_buf;
}


/*
**------------------------------------------------------------------------------
** Lexicon entries handling
**------------------------------------------------------------------------------
*/

static void
lexfile_read_entries(FILE* lex_file, LEXICON* lex)
{
    ENTRY* entry;
    int arity;
    int i, j;
    size_t nr = lexfile_read_size(lex_file);
   
    lex->nr_entries = nr;
    lex->entry = (ENTRY**) GetMem(sizeof(ENTRY*) * nr, "lex->entry");

    DB(printf("-- There are %d entries\n", nr));

    for (i = 0; i < nr; i++) {
        entry = (ENTRY *) GetMem(sizeof(ENTRY), "ENTRY");

        entry->nontnr = lexfile_read_index(lex_file);
        entry->penalty = lexfile_read_penalty(lex_file);

        arity = lex->nonterminal[entry->nontnr]->arity;
        DB(printf("Entry %d: %s", i, lex->nonterminal[entry->nontnr]->name));
        if (arity) {
            entry->pars = GetMem(sizeof(off_t) * arity, "entry->pars");

            DB(printf("("));
            for (j = 0; j < arity; j++) {
                entry->pars[j] = lexfile_read_index(lex_file);
#ifdef DEBUG
                printf("\"%s\"", lex->set_affix[entry->pars[j]]->name);
                if ((j + 1) != arity) {
                    printf(", ");
                }
#endif // DEBUG
            }
            DB(printf(")"));
        } else {
            entry->pars = NULL;
        }
        DB(printf("\n"));

        lex->entry[i] = entry;
    }
}

static void
lexfile_free_entries(LEXICON* lex)
{
    off_t i;
    size_t nr = lex->nr_entries;

    if(nr) {
        for (i = 0; i < nr; i++) {
            FreeMem(lex->entry[i], "ENTRY");
        }

        FreeMem(lex->entry, "lex->entries");
    }
}


/*
**------------------------------------------------------------------------------
** Entry list pairs
**------------------------------------------------------------------------------
*/

static void
lexfile_read_entry_lists_pairs(FILE* lex_file, LEXICON* lex)
{
    off_t i;
    size_t nr = lexfile_read_size(lex_file);

    lex->nr_entrylist_pairs = nr;
    lex->entrylist_pair = (PAIR*) GetMem(sizeof(PAIR) * nr,
                                         "lex->entrylist_pair");

    for (i = 0; i < nr; i++) {
        lex->entrylist_pair[i].index = lexfile_read_index(lex_file);
        lex->entrylist_pair[i].pointer = lexfile_read_index(lex_file);
    }
}

static void
lexfile_free_entry_list_pairs(LEXICON* lex)
{
    if (lex->nr_entrylist_pairs) {
        FreeMem(lex->entrylist_pair, "lex->entrylist_pair");
    }
}


/*
**------------------------------------------------------------------------------
** Function:
**	gboolean lexicon_affix_belongs_to_lhs(LEXICON* lex, off_t aff_nr,
**                                            off_t lhs_nr)
**
** Description:
**	Checks if affix aff_nr belongs to nontemrinal lhs_nr.
**
** Return value:
**	true if check succeeds, false otherwise.
**------------------------------------------------------------------------------
*/

gboolean
lexicon_affix_belongs_to_lhs(LEXICON* lex, off_t aff_nr, off_t lhs_nr)
{
    SET_AFFIX* aff_entry;
    unsigned i;

    assert(lhs_nr <= lex->nr_set_affixes);
    assert(aff_nr <= lex->nr_set_affixes);

    aff_entry = lex->set_affix[aff_nr];
    for (i = 0; i < aff_entry->nr_lhsses; ++i) {
        if ((aff_entry->lhs[i]) == lhs_nr) {
            return TRUE;
        }
    }

    return FALSE;
}

/*
**------------------------------------------------------------------------------
** Function:
**	int try_advance_to_next_entry_in_list(const LEXICON* lex, off_t* lst)
**
** Description:
**	replaces the list index by the next entry's list index.
**
** Return value:
**	0 (false) if there are no more entries;
**	true otherwise.
**------------------------------------------------------------------------------
*/

gboolean
try_advance_to_next_entry_in_list(const LEXICON* lex, off_t* lst)
{
    assert(lst);
    *lst = lex->entrylist_pair[*lst].pointer;
    return *lst != -1;
}

void
get_params_from_entry_in_list(const LEXICON* lex, off_t entry_list,
			      off_t *nontnr, unsigned *arity,
			      PENALTY *penalty, const PARAM **params)
{
    char type;
    off_t i, par_idx;
    ENTRY* cur_entry = lex->entry[lex->entrylist_pair[entry_list].index];
    NONTERMINAL* nonterminal;
    PARAM* cur_par;
    *nontnr = cur_entry->nontnr;
    nonterminal = lex->nonterminal[*nontnr];
    *arity = nonterminal->arity;
    *penalty = cur_entry->penalty;

    if (!(*arity)) {
        *params = NULL;
        return;
    }

    cur_par = (PARAM*) GetMem(sizeof(PARAM) * (*arity), "state");
    *params = cur_par;

    for (i = 0; i < (*arity); i++) {
        type = (nonterminal->pars[i]).type;
        par_idx = cur_entry->pars[i];
        switch (type) {
            case LexParamInt:
                cur_par->kind = IntKind;
                cur_par->value.int_par = lex->int_affix[par_idx];
                break;
            case LexParamText:
                cur_par->kind = TextKind;
                cur_par->value.text_par = lex->text_affix[par_idx];
                break;
            case LexParamLattice:
                cur_par->kind = SetKind;
                cur_par->value.set_par = lex->set_affix[par_idx]->bitset;
                break;
            default:
                DB(fprintf(stderr, "get_params_from_entry_in_list: Error: unknown parameter type\n"));
                longjmp(lex_file_error_env, 666);
        }
        cur_par++;
    }
}


/*
**------------------------------------------------------------------------------
** Lexicon acces routines
**------------------------------------------------------------------------------
*/

size_t lexicon_get_nr_nonterminals(LEXICON* lex)
{
    assert(lex);

    return lex->nr_nonterminals;
}

char* lexicon_get_nont_name(LEXICON* lex, off_t nont_nr)
{
    assert(lex);
    assert((nont_nr >= 0) && (nont_nr < lex->nr_nonterminals));

    return lex->nonterminal[nont_nr]->name;
}

gboolean lexicon_nont_exists(LEXICON* lex, off_t nont_nr)
{
    assert(lex);

    if ((nont_nr < 0) || (nont_nr > lex->nr_nonterminals)) {
        return FALSE;
    }

    return (lex->nonterminal[nont_nr] != NULL);
}

size_t lexicon_get_nont_arity(LEXICON* lex, off_t nont_nr)
{
    assert(lex);
    assert((nont_nr >= 0) && (nont_nr < lex->nr_nonterminals));

    return lex->nonterminal[nont_nr]->arity;
}

off_t
find_lex_nonterminal_with_arity(LEXICON* lex, char const *nname, int arity)
/* Simple but expensive search, for now.
** Returns nontnr, or -1 if not found.
*/
{
    size_t nr_l_n = lexicon_get_nr_nonterminals(lex);
    off_t retval;
    for (retval = 0; retval < nr_l_n; retval++) {
        if (lexicon_nont_exists(lex, retval)
	    && ((lex->nonterminal[retval])->arity == arity)
            && !strcmp((lex->nonterminal[retval])->name, nname)) {
                return retval;
        }
    }
    return -1;  /* not found */
}

char* lexicon_get_nont_param_name(LEXICON* lex, off_t nont_nr, off_t param_nr)
{
    off_t idx;
    char type;

    assert(lex);
    assert((nont_nr >= 0) && (nont_nr < lex->nr_nonterminals));
    assert((param_nr >= 0) && (param_nr < lex->nonterminal[nont_nr]->arity));

    idx = lex->nonterminal[nont_nr]->pars[param_nr].index;
    type = lex->nonterminal[nont_nr]->pars[param_nr].type;

    switch (type) {
        case LexParamText:
            return "TEXT";
        case LexParamInt:
            return "INT";
        case LexParamLattice:
            return lex->set_affix[idx]->name;
    }

    /* Should be unreachable: */
    return "Unknown type";
}

off_t
lexicon_get_nont_param_nr(LEXICON* lex, off_t nont_nr, off_t param_nr)
{
    assert(lex);
    assert((nont_nr >= 0) && (nont_nr < lex->nr_nonterminals));
    assert((param_nr >= 0) && (param_nr < lex->nonterminal[nont_nr]->arity));

    return lex->nonterminal[nont_nr]->pars[param_nr].index;
}

size_t lexicon_get_nr_setaffixes(LEXICON* lex)
{
    assert(lex);

    return lex->nr_set_affixes;
}

char* lexicon_get_setaffix_name(LEXICON* lex, off_t aff_nr)
{
    assert(lex);
    assert(aff_nr >= 0);
    assert(aff_nr < lex->nr_set_affixes);

    return lex->set_affix[aff_nr]->name;
}

SET lexicon_get_setaffix_bitset(LEXICON* lex, off_t aff_nr)
{
    assert (lex);
    assert(aff_nr >= 0);
    assert(aff_nr < lex->nr_set_affixes);

    return lex->set_affix[aff_nr]->bitset;
}

gboolean lexicon_get_setaffix_nont_flag(LEXICON* lex, off_t aff_nr)
{
    assert (lex);
    assert(aff_nr >= 0);
    assert(aff_nr < lex->nr_set_affixes);

    return lex->set_affix[aff_nr]->is_nont;
}

size_t lexicon_get_setaffix_nr_lhsses(LEXICON* lex, off_t aff_nr)
{
    assert (lex);
    assert(aff_nr >= 0);
    assert(aff_nr < lex->nr_set_affixes);

    return lex->set_affix[aff_nr]->nr_lhsses;
}

char* lexicon_get_setaffix_lhs_name(LEXICON* lex, off_t aff_nr, off_t lhs_nr)
{
    off_t affix_idx;

    assert (lex);
    assert(aff_nr >= 0);
    assert(aff_nr < lex->nr_set_affixes);
    assert(lhs_nr >= 0);
    assert(lhs_nr < lex->set_affix[aff_nr]->nr_lhsses);

    affix_idx = lex->set_affix[aff_nr]->lhs[lhs_nr];

    return lexicon_get_setaffix_name(lex, affix_idx);
}


/*
**------------------------------------------------------------------------------
** Frees all lexicon tables
**------------------------------------------------------------------------------
*/

void
lexicon_free(LEXICON* lex)
{
    lexfile_free_trie(lex);
    lexfile_free_set_affixes(lex);
    lexfile_free_text_affixes(lex);
    lexfile_free_int_affixes(lex);
    lexfile_free_nonterminals(lex);
    lexfile_free_entries(lex);
    lexfile_free_entry_list_pairs(lex);

    FreeMem(lex, "LEXICON");
}


/*
**------------------------------------------------------------------------------
** lexicon initialization
**------------------------------------------------------------------------------
*/

LEXICON*
lexicon_new(FILE* lex_file)
{
    LEXICON* res = (LEXICON*) GetMem(sizeof(LEXICON), "LEXICON");

    if (!setjmp(lex_file_error_env)) {
        lexfile_read_trie(lex_file, res);
        lexfile_read_int_affixes(lex_file, res);
        lexfile_read_text_affixes(lex_file, res);
        lexfile_read_set_affixes(lex_file, res);
        lexfile_read_nonterminals(lex_file, res);
        lexfile_read_entries(lex_file, res);
        lexfile_read_entry_lists_pairs(lex_file, res);

        return res;
    } else {
        /*
        ** If an read error occurs, we return NULL.
        */

        return NULL;
    }
}

