/* Code for retreiving and printing the memory usage of the current process.
 *
 * Copyright 2000, KUN
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Library General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

/* $Id: meminfo.c,v 1.2 2001/10/17 13:13:16 ejv Exp $ */

#ifndef meminfo_h
#define meminfo_h

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif /* HAVE_CONFIG_H */

#include <unistd.h>
#include <stdlib.h>
#include <stdio.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>

#if defined(sun) && (defined(__SVR4) || defined(__svr4__))
// OS is Solaris

#include <procfs.h>

static void
get_mem_info(pid_t pid, long* vsize, long* rss)
{
    psinfo_t proc_info;
    int fd;
    char filename[2048];
    sprintf(filename, "/proc/%d/psinfo", (int)pid);
    *vsize = -1;
    *rss = -1;

    if ((fd = open(filename, O_RDONLY)) < 0) {
        return;
    }

    if (read(fd, &proc_info, sizeof(psinfo_t)) != sizeof(psinfo_t)) {
        close(fd);
        return;
    }

    close(fd);

    *vsize = proc_info.pr_size;
    *rss = proc_info.pr_rssize;
}
#else

#ifdef __linux__
// OS is Linux

#include <sys/user.h>                   // for PAGE_SHIFT
#include <ctype.h>

static void
get_mem_info(pid_t pid, long *vm_size, long *res_size)
{
    char proc_fname[64];
    FILE *pfile;
    unsigned int vm, res;
    int ix, ch;

    /* set defaults */
    *vm_size = -2;
    *res_size = -2;

    /* try open current process psinfo */
    sprintf (proc_fname, "/proc/%ld/stat", (long) pid);
    pfile = fopen (proc_fname, "r");
    if (pfile == NULL) {
        fprintf(stderr, "could not open '%s'\n", proc_fname);
        return;
    };

    /* Skip 22 fields */
    for (ix = 0; ix < 22; ix++) {
        do {
            ch = fgetc (pfile);
            if (ch == EOF) return;        /* Too bad */
        } while (!isspace (ch));
    }

    /* read vm and rss */
    if (fscanf (pfile, "%u", &vm) != 1) return;
    if (fscanf (pfile, "%u", &res) != 1) return;

    /* convert them into KB: see man 4 procfs */
    *vm_size = (long) (vm / 1024);
    *res_size = (long) ((res + 3) << (PAGE_SHIFT - 10));  /* note */
};

#else
#ifdef HAVE_LIBKVM

/* OS is *BSD */
#include <limits.h>
#include <fcntl.h>
#include <kvm.h>
#include <paths.h>
#include <sys/param.h>
#include <sys/sysctl.h>
#include <sys/user.h>

inline void
get_mem_info(pid_t pid, long* vm_size, long* res_size)
{
    char errbuf[_POSIX2_LINE_MAX];
    kvm_t *kd;
    int count;
    struct kinfo_proc *proc_info;
    int vm, rss;

    /* set defaults */
    *vm_size = -2;
    *res_size = -2;

    /* open kvm interface */
#if defined(__OpenBSD__) || defined(__NetBSD__)
    kd = kvm_openfiles (NULL, NULL, NULL, O_RDONLY, errbuf);
#else /* assume FreeBSD */
    kd = kvm_openfiles (_PATH_DEVNULL, _PATH_DEVNULL, _PATH_DEVNULL,
                        O_RDONLY, errbuf);
#endif
    if (kd == NULL) {
        return;
    };

    /* ask process info of this process */
    proc_info = kvm_getprocs (kd, KERN_PROC_PID, pid, &count);
    if (count != 1) {
        return;
    }

    /* pick vm and rss_size out of kernel structures */
    vm = proc_info->kp_eproc.e_vm.vm_dsize +
         proc_info->kp_eproc.e_vm.vm_ssize +
         proc_info->kp_eproc.e_vm.vm_tsize;
    rss = proc_info->kp_eproc.e_vm.vm_rssize;
    kvm_close (kd);
    *vm_size = vm * getpagesize() / 1024;
    *res_size = rss * getpagesize() / 1024;
};

#else
// Unknown OS

inline void
get_mem_info(pid_t pid, long* vsize, long* rss)
{
    *vsize = -1;
    *rss = -1;
}
#endif
#endif
#endif

void 
log_memusage()
{
#ifndef WIN32
    long vsize, rss;
    pid_t my_pid = getpid();

    get_mem_info(my_pid, &vsize, &rss);

    if ((vsize >= 0) && (rss >= 0)) {
        printf("memory: %ldKB total, %ldKB in RAM\n", vsize, rss);
    }
#endif /* WIN32 */
}

#endif // meminfo_h
