// Implementation of the Options class.
//
// Copyright 2001, KUN.
//
// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU Library General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.

// $Id: options.cc,v 1.8 2001/09/13 11:56:19 ejv Exp $

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <string.h>
#include <iostream.h>
#ifdef HAVE_GETOPT_H
#include <getopt.h>
#else /* HAVE_GETOPT_H */
#ifdef HAVE_GETOPT
#include <unistd.h>
#endif /* HAVE_GETOPT */
#endif /* HAVE_GETOPT_H */
#include <stdlib.h>
#include <string.h>

#include "options.h"

void Options::show_synopsis()
{
   cerr << "Usage: " << prog_name
	<< " [ -f ] [ -h ] [ -v ] [ -V ] [ -o <out-file> ] <lif-file>" << endl;
}

void Options::show_help()
{
#ifdef HAVE_GETOPT_LONG
   cerr << "Options are:" << endl
	<< "    -h    --help           show help" << endl
	<< "    -v    --verbose        be verbose" << endl
	<< "    -V    --version        show version" << endl
	<< "    -o    --output         specify output file" << endl
        << "    -f    --force          force compilation" << endl
	<< "Arguments are:" << endl
	<< "    <lif-file>             name of input lif file" << endl
	<< "    <out-file>             name of output file" << endl;
#else
   cerr << "Options are:" << endl
	<< "    -h                     show help" << endl
	<< "    -v                     be verbose" << endl
	<< "    -V                     show version" << endl
	<< "    -o                     specify output file" << endl
        << "    -f                     force compilation" << endl
	<< "Arguments are:" << endl
	<< "    <in-file>              name of input file" << endl
	<< "    <out-file>             name of output file" << endl;
#endif // HAVE_GETOPT_LONG
}

void Options::say_hello()
{
  clog	<< "This is " << prog_name << ", version " << version << endl;
}

void Options::show_version()
{
   cerr << "This is " << prog_name << ", version " << version << endl;
}

Options::Options(int argc, char* const argv[]):
        outfile(""),
        lex_interface_file("")
{
    // Initialize options
    force_compile = false;
    verbose_option = false;

    int errflg = 0;
    char c;
    char* opt_str = "vVo:hf";

#ifdef HAVE_GETOPT_LONG
    int option_index = 0;
    struct option long_options[] = {
        {"verbose", 0, 0, 0},
        {"version", 0, 0, 0},
        {"output-file", 2, 0, 'o'},
        {"help", 0, 0, 0},
        {"force", 0, 0, 0},
        {0, 0, 0, 0}
    };
#endif

#ifdef HAVE_GETOPT_LONG
    while ((c = getopt_long(argc, argv, opt_str, long_options, &option_index)) != EOF) {
#else
    while ((c = getopt(argc, argv, opt_str)) != EOF) {
#endif
        switch (c) {
            case 'v':
                verbose_option = true;
                break;
            case 'V':
                show_version();
                exit(0);
            case 'h':
                show_help();
                exit(0);
            case 'o':
                outfile = string(optarg);
                break;
            case 'f':
                force_compile = true;
                break;
            default:
                errflg++;
                cerr << "Unknown option \"" << c << "\"" << endl;
                break;
        }
    }

    if (errflg) {
        show_synopsis();
        exit(1);
    }

    if (optind < argc) {
        lex_interface_file = argv[optind++];
    } else {
        cerr << "No lif file specified." << endl;
        exit(1);
    }

    if (outfile.length() == 0) {
        outfile = substitute_extension(lex_interface_file, lif_ext, lex_ext);
    }

    if (verbose_option) {
        clog << "output file: \"" << outfile << "\"" << endl;
    }
}

//------------------------------------------------------------------------------
// Function:
//	void t_file(char* str, const char* ext1, const char* ext2)
//
// Description:
//	Strip extension ext1, where ext1 present, and
//	add extension ext2
//------------------------------------------------------------------------------

string Options::substitute_extension(string str, string ext1, string ext2)
{
    string res = str;

    if (res.length() > ext1.length()) {
        if (str.compare(ext1, str.length()-ext1.length(), ext1.length()) == 0) {
            res.erase(str.length()-ext1.length(), ext1.length());
        }
    }

    return res + ext2;
}

