// Nonterminal class definition for storing nonterminals with parameters.
//
// Copyright 2001, KUN
//
// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU Library General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.

// $Id: nonttab.h,v 1.5 2001/03/30 12:46:05 ejv Exp $


#ifndef NONTTAB_H
#define NONTTAB_H

#include <iostream.h>
#include <string>

#include "paramtab.h"
#include "id.h"

typedef ID HeadID;

//------------------------------------------------------------------------------
// struct Nonterm
//
// Description:
// A nonterminal is a head with a list of parameters.
//------------------------------------------------------------------------------

struct Nonterm {
    Nonterm();
    Nonterm(HeadID head, pParam params);
    void print(ostream& os) const;
    string to_string() const;

    HeadID head;
    pParam params;
};

inline
Nonterm::Nonterm(HeadID hd, pParam pars) {
    head = hd;
    params = pars;
}

inline string Nonterm::to_string() const
{
    string res = idtable.to_string(head);

    if (params != 0) {
        res = res + '(' + params->elem.to_string();
        pParam p = params->next;

        while (p != 0) {
            res = res + ',' + p->elem.to_string();
            p = p->next;
        }

        res += ')';
    }

    return res;
}

inline void
Nonterm::print(ostream& os) const {
    os << to_string();
}

inline ostream&
operator <<(ostream& os, const Nonterm& nonterm) {
    nonterm.print(os);
    return os;
}

inline bool
operator ==(const Nonterm& nonterm1, const Nonterm& nonterm2) {
    if (nonterm1.head != nonterm2.head) return false;
    if (nonterm1.params != nonterm2.params) return false;
    if (!nonterm1.params || !nonterm2.params) return false;
    return *(nonterm1.params) == *(nonterm2.params);
}

inline bool
operator <(const Nonterm& nonterm1, const Nonterm& nonterm2) {
    string s1 = idtable.to_string(nonterm1.head);
    string s2 = idtable.to_string(nonterm2.head);
    if (s1 < s2) return true;
    if (s1 > s2) return false;
    if (nonterm1.params == nonterm2.params) return true;
    if (!(nonterm1.params) || !(nonterm2.params)) return false;
    if (nonterm1.params->length() < nonterm2.params->length()) return true;
    if (nonterm1.params->length() > nonterm2.params->length()) return false;
    return *(nonterm1.params) < *(nonterm2.params);
}

struct NontermLT {
    bool operator ()(const Nonterm& nonterm1, const Nonterm& nonterm2) const;
};

inline bool
NontermLT::operator ()(const Nonterm& nonterm1, const Nonterm& nonterm2) const {
    if (nonterm1.head < nonterm2.head)
        return true;
    else if (nonterm1.head > nonterm2.head)
        return false;
    else
        return nonterm1.params < nonterm2.params;
}


//------------------------------------------------------------------------------
// class NontermTable
//
// Description:
// Table containing all nonterminals. Equal nonterminals are
// are stored only once.
//------------------------------------------------------------------------------

typedef const Nonterm*  pNonterm;
typedef set<Nonterm, NontermLT> NontermTableBase;

class NontermTable : NontermTableBase
{
    public:
        NontermTable(void);

        pNonterm enter(HeadID head, pParam params);
        unsigned requested(void) const;
        unsigned allocated(void) const;

        void dump();

    private:
        unsigned nr_requested;
};

inline NontermTable::NontermTable()
{
    nr_requested = 0;
}

inline pNonterm
NontermTable::enter(HeadID head, pParam params)
{
    assert(head != unknown_id);
    nr_requested++;
    pair<iterator, bool> p = insert(Nonterm(head, params));
    return &*p.first;
}

inline unsigned
NontermTable::requested() const
{
    return nr_requested;
}

inline unsigned
NontermTable::allocated() const
{
    return size();
}

inline void
NontermTable::dump() {
    for (NontermTable::const_iterator i = begin(); i != end(); ++i) {
        cout << "NontermTable::dump() \"" << *i << "\"" << endl;
    }
}

#endif // NONTTAB_H
