// Implementation of the methods of the LexNontermList class.
//
// Copyright 2001, KUN
//
// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU Library General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.

// $Id: lexnontermlist.cc,v 1.9 2001/10/08 15:07:23 ejv Exp $


#include "lexnontermlist.h"
#include <iostream.h>
#include "lexfileio.h"

#include "lexaffixnamelist.h"
extern LexAffixNameList lex_affixname_list;
#include "lextextaffixlist.h"
extern LexTextAffixList lex_textaffix_list;
#include "lexintaffixlist.h"
extern LexIntAffixList lex_intaffix_list;

void
LexNontermList::dump() {
    unsigned nont_list_size = get_max_idx() + 1;
    cout << "---- LexNontermList::dump(" << nont_list_size << ")" << endl;

    for (unsigned idx = 0; idx < nont_list_size; ++idx) {
        LexNontMap::iterator i = get_entry_by_idx(idx);

        if (i == nonterms.end()) {
            cout << "\tindex " << idx << " is a grammar nonterminal" << endl;
        } else {
            cout << "\t" << (i->second).first;
            cout << ":\tname " << (i->first).first << endl;
            cout << "\t\tarity " << (i->first).second << endl;

            if ((i->second).second) {
                pParam p = (i->second).second;

                cout << "\t\taffix indexes";

                while (p) {
                    cout << " ";
                    cout << get_param_idx(p->elem);

                    p = p->next;
                }

                cout << endl;
            }
        }
    }

    cout << "----" << endl;
}

int
LexNontermList::get_param_idx(Param p) {
    ParamType type = p.type;
    
    switch(type) {
        case IdType:
            return lex_affixname_list.get_index(p.value.id);
        case SetType: {
                string set_str(convert_to_string(p.value.set));
                return lex_affixname_list.get_index_of_set(set_str);
            }
        case TextType:
            return lex_textaffix_list.get_index(p.value.id);
        case IntType:
            return lex_intaffix_list.get_index(p.value.number);
        case AnyText:
            return -2;
        case AnyInt:
            return -3;
        default:
            cerr << "Internal error in LexNontermList::get_param_idx: Any{Text,Int}" << endl;
            exit(1);
    }
    
    // Never reached:
    return 0;
}

enum LexParamTypes {
    LexParamLattice = 1,
    LexParamText = 2,
    LexParamInt = 3
};

LexNontMap::iterator
LexNontermList::get_entry_by_idx(unsigned idx) {
    LexNontMap::iterator i = nonterms.begin();

    while (i != nonterms.end()) {
        if ((i->second).first == idx) {
            return i;
        }

        ++i;
    }

    return nonterms.end();
}

unsigned
LexNontermList::get_max_idx() {
    unsigned nont_list_size = 0;

    for (LexNontMap::iterator i = nonterms.begin(); i != nonterms.end(); ++i) {
        if (i->second.first > nont_list_size) {
            nont_list_size = i->second.first;
        }
    }

    return nont_list_size;
}

ofstream&
operator <<(ofstream& ofs, LexNontermList& l)
{
    return l.print(ofs);
}

ofstream& LexNontermList::print(ofstream& ofs)
{
    // first calculate the max index, the array is this index+1
    unsigned nont_list_size = get_max_idx() + 1;
    write_lex_size(ofs, nont_list_size);

    for (unsigned idx = 0; idx < nont_list_size; ++idx) {
        LexNontMap::iterator i = get_entry_by_idx(idx);
        if (i == nonterms.end()) {
            write_lex_byte(ofs, '\0');
        } else {
            write_lex_string(ofs, i->first.first.c_str());
            write_lex_word(ofs, i->first.second);

            if ((i->second).second) {
                pParam p = (i->second).second;

                while (p) {
                    Param par = p->elem;
                    ParamType type = par.type;
                    
                    switch(type) {
                        case IdType:
                        case SetType:
                            write_lex_byte(ofs, LexParamLattice);
                            write_lex_index(ofs, get_param_idx(par));
                            break;
                        case TextType:
                        case AnyText:
                            write_lex_byte(ofs, LexParamText);
                            break;
                        case IntType:
                        case AnyInt:
                            write_lex_byte(ofs, LexParamInt);
                            break;
                        default:
                            cerr << "Internal error in LexNontermList::<<: unknown parameter type" << endl;
                            exit(1);
                    }

                    p = p->next;
                }
            }
        }
    }

    return ofs;
}

bool LexNontermList::has_changed(LEXICON* old_lex)
{ 
#ifdef CMP_DEBUG
    cerr << "comparing nonterminals" << endl;
#endif

    for (unsigned idx = 0; idx < nonterms.size(); ++idx) {
#ifdef CMP_DEBUG
        cerr << "nonterminal " << idx << ":" << endl;
#endif // CMP_DEBUG
        LexNontMap::iterator i = get_entry_by_idx(idx);

        if (lexicon_nont_exists(old_lex, idx) && (i == nonterms.end())) {
            return true;
        }

        if (!lexicon_nont_exists(old_lex, idx) && (i != nonterms.end())) {
            return true;
        }

        if (lexicon_nont_exists(old_lex, idx) && (i != nonterms.end())) {
            string old_name = lexicon_get_nont_name(old_lex, idx);
#ifdef CMP_DEBUG
            cerr << "comparing (old,new) (" << old_name << "," << i->first.first
                << ")" << endl;
#endif // CMP_DEBUG

            if (i->first.first != old_name) {
#ifdef CMP_DEBUG
                cerr << "nonterminal " << idx << " was " << old_name;
                cerr << " and is " << i->first.first << endl;
#endif // CMP_DEBUG
                return true;
            }

            size_t old_arity = lexicon_get_nont_arity(old_lex, idx);
            if (i->first.second != old_arity) {
#ifdef CMP_DEBUG
                cerr << "arity of nonterminal " << old_name << " has changed from ";
                cerr << old_arity << " to " << i->first.second << endl;
#endif // CMP_DEBUG
                return true;
            }

            if ((i->second).second) {
                pParam p = (i->second).second;
                off_t old_param_idx = 0;

                while (p) {
                    string old_param_name = lexicon_get_nont_param_name(old_lex, idx, old_param_idx);
                    if (has_changed(p->elem, old_param_name)) {
#ifdef CMP_DEBUG
                        cerr << "parameter " << old_param_idx + 1;
                        cerr << " of nonterminal " << old_name;
                        cerr << " with index " << idx;
                        cerr << " has changed from " << old_param_name;
                        cerr << " to something else" << endl;
#endif // CMP_DEBUG
                        return true;
                    }

                    p = p->next;
                    old_param_idx++;
                }
            }
        }
    }

    return false;
}

bool LexNontermList::has_changed(const Param par, string old_name)
{
    ParamType type = par.type;
    
    switch(type) {
        case IdType:
            return idtable.to_string(par.value.id) != old_name;
        case SetType:
            return convert_to_string(par.value.set) != old_name;
        case TextType:
        case AnyText:
            return old_name != "TEXT";
        case IntType:
        case AnyInt:
            return old_name != "INT";
        default:
            cerr << "Internal error in LexNontermList::get_param_idx: Any{Text,Int}" << endl;
            exit(1);
    }
}

void
LexNontermList::add_nonterm(string name, pParam params, unsigned rule_nr) {
    unsigned arity = calculate_arity(params);
    LexNontWithArity el = LexNontWithArity(name, arity);
    LexNontMap::iterator i = nonterms.find(el);

    if (i == nonterms.end()) {
        LexNontInfo info = LexNontInfo(rule_nr, params);
        nonterms[el] = info;
    }
}

bool
LexNontermList::is_defined(string nonterm, pParam params)
{
    unsigned arity = calculate_arity(params);
    LexNontWithArity el = LexNontWithArity(nonterm, arity);
    LexNontMap::iterator position = nonterms.find(el);

    return (position != nonterms.end());
}

unsigned
LexNontermList::calculate_arity(pParam params) {
    register pParam p = params;
    register unsigned count = 0;

    while(p) {
        count++;
        p = p->next;
    }

    return count;
}

unsigned
LexNontermList::get_index(Entry e) {
    unsigned arity = calculate_arity((e.nonterm)->params);
    string head_str = idtable.to_string((e.nonterm)->head);
    LexNontWithArity el = LexNontWithArity(head_str, arity);
    LexNontMap::iterator i = nonterms.find(el);

    if (i == nonterms.end()) {
        cerr << "internal error in LexNontermList::get_index: entry not in table" << endl;
        cerr << "Entry: " << e << " (" << head_str << "/" << arity << ")" << endl;
        cerr << "-- Entry dump:" << endl;
        dump();
        cerr << "-- End entry dump" << endl;
        assert(0);
        exit(1);
    }

    return (i->second).first;
}

