/*
 * I/O bufferring routines
 * 
 * Copyright 2000 KUN.
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Library General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

/* $Id: rtsbuffer.c,v 1.5 2001/10/17 10:39:28 ejv Exp $ */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif /* HAVE_CONFIG_H */

#include <stdio.h>
#include <string.h>

#include "rtsbuffer.h"
#include "rtsutil.h"

#define BUFSIZE	(64 * 1024 - 32)
#define BUFINC	(64 * 1024)

/*
** Private
*/

static struct
{
  Buffer *buffer;
} freeList;

Buffer *NewBuffer(void)
{
  Buffer *buffer;

  if (freeList.buffer == NULL)
  {
    buffer = (Buffer *)GetMem(sizeof(Buffer), "buffer");
    buffer->space = (char *)GetMem(BUFSIZE, "buffer");
    buffer->size = BUFSIZE;
  }
  else
  {
    buffer = freeList.buffer;
    freeList.buffer = buffer->next;
  };
  ResetBuffer(buffer);
  return buffer;
}

void FreeBuffer(Buffer *buffer)
{
  Assert(buffer != NULL, "FreeBuffer", "buffer is null");
  if (buffer->size > BUFSIZE)
  {
    buffer->space = (char *)ResizeMem((void *)buffer->space, BUFSIZE, "buffer");
    buffer->size = BUFSIZE;
  };
  buffer->next = freeList.buffer;
  freeList.buffer = buffer;
}

void InitBuffer(void)
{
  freeList.buffer = NULL;
}

void EndBuffer(void)
{
  Buffer *buffer;

  buffer = freeList.buffer;
  while (buffer != NULL)
  {
    Buffer *next;

    next = buffer->next;
    FreeMem((void *)buffer->space, "buffer");
    FreeMem((void *)buffer, "buffer");
    buffer = next;
  };
}

void CommitBuffer(Buffer *buffer)
{
  buffer->commit.pos = buffer->current.pos;
  buffer->commit.free = buffer->current.free;
}

void TruncBuffer(Buffer *buffer)
{
  buffer->current.pos = buffer->commit.pos;
  buffer->current.free = buffer->commit.free;
  buffer->current.pos[0] = '\0';
}

void ResetBuffer(Buffer *buffer)
{
  int free;
  char *space;

  if (buffer->size != BUFSIZE)
  {
    buffer->space = (char *)ResizeMem((void *)buffer->space, BUFSIZE, "buffer");
    buffer->size = BUFSIZE;
  };
  free = buffer->size - 1;
  space = buffer->space;
  space[0] = '\0';
  buffer->commit.pos = space;
  buffer->commit.free = free;
  buffer->current.pos = space;
  buffer->current.free = free;
}

void ResizeBuffer(Buffer *buffer, int size)
{
  if (size > buffer->size)
  {
    buffer->space = (char *)ResizeMem((void *)buffer->space, size, "buffer");
    buffer->size = size;
  };
}

static void ExtendBuffer(Buffer *buffer, int size)
{
  int newSize;
  char *oldSpace;
  char *newSpace;

  newSize = buffer->size + size;
  oldSpace = buffer->space;
  newSpace = (char *)ResizeMem((void *)oldSpace, newSize, "buffer");
  buffer->space = newSpace;
  buffer->size = newSize;
  buffer->current.free += size;
  buffer->commit.free += size;
  buffer->current.pos = (buffer->current.pos - oldSpace) + newSpace;
  buffer->commit.pos = (buffer->commit.pos - oldSpace) + newSpace;
}

void PutStrBuffer(const char *s, Buffer *buffer)
{
  int len;

  len = strlen(s);
  if (buffer->current.free < len)
    ExtendBuffer(buffer, BUFINC + len);
  strcpy(buffer->current.pos, s);
  buffer->current.pos += len;
  buffer->current.free -= len;
}

void PutBlBuffer(const int len, Buffer *buffer)
{
  if (buffer->current.free < len)
    ExtendBuffer(buffer, BUFINC + len);
  sprintf(buffer->current.pos, "%*s", len, "");
  buffer->current.pos += len;
  buffer->current.free -= len;
}

void PutChBuffer(const int c, Buffer *buffer)
{
  if (buffer->current.free < 1) 
    ExtendBuffer(buffer, BUFINC + 1);
  buffer->current.pos[0] = c;
  buffer->current.pos[1] = '\0';
  buffer->current.pos++;
  buffer->current.free--;
}

void PutIntBuffer(const int i, Buffer *buffer)
{
  char buf[40];
  int len;

  sprintf(buf, "%d", i);
  len = strlen(buf);
  if (buffer->current.free < len)
    ExtendBuffer(buffer, BUFINC + len);
  strcpy(buffer->current.pos, buf);
  buffer->current.pos += len;
  buffer->current.free -= len;
}

void FlushBuffer(Buffer *buffer, FILE *file)
{
  fputs(buffer->space, file);
}

int BufferUsed(Buffer *buffer)
{
  return buffer->current.pos - buffer->space + 1;
}
