/*
 * This C file contains the interpreter for the instructions generated by gra2o.
 *
 * Copyright 2000 KUN.
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Library General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

/* $Id: rtsagfl.c,v 1.70 2002/01/17 15:01:36 pspiertz Exp $ */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif /* HAVE_CONFIG_H */

#include <stdio.h>
#include <stdlib.h>
#include <limits.h>
#include <assert.h>
#include <sys/types.h>
#include <unistd.h>
#include <string.h>
#include <glib.h>

#include "rtsopt.h"
#include "rtscode.h"
#include <lexicon.h>
#include "rtslex.h"
#include "rtsio.h"
#include "rtsutil.h"
#include "rtsagfl.h"
#include "rtstime.h"

#ifdef GEN_RTS
#include "rtsgen.h"
#endif /* GEN_RTS */

#ifdef PMRTS
#include "posmemo.h"
#endif /* PMRTS */

#ifdef DEBUGRTS
#include "rtsdebug.h"
#endif /* DEBUGRTS */

#ifdef DEBUG_INTERPRETER
#  undef DEBUG_TRANSLATE
#  define TRACE_MATCH
#  undef TRACE_LEX_MATCH
#  define DB(x)	x
#else /* DEBUG_INTERPRETER */
#  define DB(x)
#endif /* DEBUG_INTERPRETER */


/*
//------------------------------------------------------------------------------
// Cells
//
// definition of data/code/cel/union Cel moved here from rtscode.h
// A cell should fit in one machine word.
//------------------------------------------------------------------------------
*/

typedef long ARG;
typedef union Cel CODE;
typedef union Cel DATA;
typedef union Cel
{
    void* action;
    ARG arg;
    VALUE val;
    CODE* code;
    DATA* data;
    StateIndicator input_state;         /* these 3 not in generated code */
    Transition *input_transition;
    PosMemo pmprod;
    PENALTY penalty;
} cel;

extern CODE root;	/* start label of code to be interpreted */


/*
//------------------------------------------------------------------------------
// Private globals
//------------------------------------------------------------------------------
*/

typedef struct
{
    long nr_parses;                     /* nr of parses */
    long nr_penalties;                  /* nr of penalties of last parsing */
} ParseResult;

static ParseResult parse_result;

static void
reset_parse_result()
{
    parse_result.nr_parses = 0;
    parse_result.nr_penalties = 0;
}

static void
incr_nr_parses()
{
    parse_result.nr_parses++;
}

long
get_nr_parses()
{
    return parse_result.nr_parses;
}

static long
get_nr_penalties()
{
    return parse_result.nr_penalties;
}

static int pos_is_visited[MAXINPUT];
#define MARK_POS_VISITED(pos)	pos_is_visited[pos] = 1

static void clear_visited_admin(Trellis *trellis)
{
    StateNode ** row = GET_TRELLIS_STATE_ROW(trellis);
    long pos_index;

    for (pos_index = trellis->length - 1; pos_index >= 0; pos_index--) {
        if (row[pos_index] != NULL) {
            pos_is_visited[pos_index] = 0;
        } else {
            pos_is_visited[pos_index] = -1; /* no token at this pos */
        }
    }
}

#if 0
static int
get_visited_length(int length)
{
    long pos_index;
    long first_pos_unvisited = length;

    for (pos_index = length - 1; pos_index >= 0; pos_index--) {
        if (pos_is_visited[pos_index] >= 0) {/* <0 means no token starts here */
            if (pos_is_visited[pos_index] > 0) {
                return first_pos_unvisited;
            } else {
                first_pos_unvisited = pos_index;
            }
        }
    }

    return 0;
}
#endif

/*
//------------------------------------------------------------------------------
// Private prototypes
//------------------------------------------------------------------------------
*/

#if 0
static void print_time_out_reached(long max);
static void print_max_parses_reached(long max);
#endif
static void start_printing(void);
static void stop_printing(void);
static void show_totals(Trellis *trellis);


/*
//------------------------------------------------------------------------------
// Time limit.
//
// The time-out is checked for each succeeding match of a terminal.
// For efficiency reasons, we only check the clock once every
// MAX_TICKS times. High MAX_TICKS increases performance, but
// reduces timer resolution.
//------------------------------------------------------------------------------
*/

static int
have_time_limit(void)
{
    return max_parsetime < LONG_MAX;
}

static void
set_time_limit(void)
{
    set_time_out(max_parsetime);
}


/*
//------------------------------------------------------------------------------
// Printing
//------------------------------------------------------------------------------
*/

static void
print_parsed_sentence(void)
{
    if (lexer_stats_option) {
        const char* input_text = get_input_text();
        printf("%s$EOS$\n\n", input_text);
    }
}

static void
start_printing(void)
{
    if (parsing_stats_option) {
        long nr_parses = get_nr_parses();
        long nr_penalties = get_nr_penalties();
        Time parse_time = get_parse_time();

        current_parse_printf("parsing %ld", nr_parses);
        current_parse_printf(" time %.3f", parse_time);
        current_parse_printf(" penalty %ld\n", nr_penalties);
    }
}

static void
stop_printing(void)
{
    if (!no_output) {
        current_parse_add_char('\n');
    }
}

static void
show_totals(Trellis *trellis)
{
    long nr_parses = get_nr_parses();

    if (total_stats_option) {
        Time scan_time = get_total_scan_time();
        Time parse_time = get_total_parse_time();
        Time print_time = get_total_print_time();
        Time total_time = scan_time + parse_time + print_time;

        if(max_parses == LONG_MAX) {
            printf("total number of parsings %ld (max unlimited) \n",
                   nr_parses);
        } else {
            printf("total number of parsings %ld (max %ld) \n",
                   nr_parses, max_parses);
        }

        printf("total scan time %.3f\n", scan_time);
        printf("total parse time %.3f\n", parse_time);
        printf("total print time %.3f\n", print_time);
        printf("total overall time %.3f\n", total_time);
    }

#ifdef DEBUG_NEGMEMO
    if (neg_memo_option) {
        print_negmemo_table(trellis, TRUE);
    }
#endif /* DEBUG_NEGMEMO */

#ifdef DEBUG_POSMEMO
    if (pos_memo_option) {
        posmemo_dump_table(trellis);
    }
#endif /* DEBUG_POSMEMO */

#ifdef PROFILING
    if ((prof_count > 0) && reset_profile && !no_output) {
        ShowProfile("sentence");
    }
#endif
    fflush(stdout);
}

#if 0
static void
print_max_parses_reached(long max)
{
    if (verbose) {
        printf("** Break: maximum number of parses (%ld) reached\n", max);
    }
}

static void
print_time_out_reached(long max)
{
    if (show_timebreak_option) {
        printf("** Break: time limit (%ld seconds) exceeded\n", max);
    }
}
#endif


/*
//------------------------------------------------------------------------------
// Stack frame layout.
//      The interpreter uses two different stack layouts: one for the first
//      pass, and one for the second pass.
//
//      The first pass frame consists of:
//          - constants pushed by the CALL instruction (the elements with a
//            positive offset);
//          - a fixed number of entries filled in by the CREATE instruction (the
//            elements with a negative offset);
//          - zero or more formals;
//          - zero or more locals;
//          - zero or more entries for pointers to the sons of this rule.
//      The last three are untouched by the CREATE instruction.
//
//      The second pass frame consists of:
//          - constants pushed by the PRINT_SON instruction
//          - parameters for the second pass function.
//
//      Both a stack pointer and a frame pointer are used. The execution model
//      allows to continue execution in another frame XXX
//
//------------------------------------------------------------------------------
*/

#define STACK_SIZE	(64 * 1024)

static DATA* stack = 0;

#define PASS2_FRAME_HEAD_LEN    (3)     /* length of the second pass frame
                                         * head */
#define PASS2_PASS2FP           (3)     /* previous second pass frame pointer */
#define PASS2_FP                (2)     /* previous rule frame pointer */
#define PASS2_RET_ADDR          (1)     /* return address */

#define FRAME_HEAD_LEN          (3)     /* length of the frame head */
#define NONT_NR_OFF             (3)
#define SUCC_ADDR_OFF           (2)     /* success continuation address */
#define FAIL_ADDR_OFF           (1)     /* fail continuation address */
#define OLD_FP                  (0)     /* pointer to parent frame */
#define NR_FORMALS_OFF         (-1)
#define NR_LOCALS_OFF          (-2)
#define NR_SONS_OFF            (-3)
#define PASS2_CODE             (-4)     /* second pass code pointer */
#define ISTATE                 (-5)     /* input state at start of rule */
#define NR_SUCCS_OFF           (-6)     /* nr of successes */
#define PMPRODPTR              (-7)     /* pointer to current positive memo
                                         * production */
#define ALTNR                  (-8)     /* alternative number */
#define NR_SONS_DONE           (-9)     /* nr of sons which have been called */
#define PENORIG                (-10)    /* penalty level at start of rule */
#define VAR_OFFSET             (11)     /* offset for variable frame part */

#define NR_FORMALS      (fp[NR_FORMALS_OFF].arg)
#define NR_LOCALS       (fp[NR_LOCALS_OFF].arg)
#define VARIABLE(nr)    (fp[- (VAR_OFFSET + nr)])

#define NR_SONS         (fp[NR_SONS_OFF].arg)
#define NONT_NR         (fp[NONT_NR_OFF].arg)
#define SON_FP(nr)      (fp[-(VAR_OFFSET + NR_FORMALS + NR_LOCALS + nr)].data)

#define START_INPUT_STATE (fp[ISTATE].input_state)


/*
//------------------------------------------------------------------------------
// Accessing the memo values.
//
// Macro:
//	MEMO(no)
// Description:
//	Convert memo with number to to pointer to value of memo.
//------------------------------------------------------------------------------
*/

#define NEG_MEMO_BLOCK_VAL (LONG_MAX)
#define NO_PENALTY (NEG_MEMO_BLOCK_VAL - 1)
#define NEG_MEMO_UNKNOWN_VAL (-1)
#define NEG_MEMO(no) (((fp[ISTATE].input_state)->neg_memos)[no])

#define negmemo_is_unknown(nr) (NEG_MEMO(nr) == NEG_MEMO_UNKNOWN_VAL)
#define negmemo_is_blocked(nr,pen) (NEG_MEMO(nr) > pen)
#define negmemo_set_blocked(nr,pen) (NEG_MEMO(nr) = pen + 1)
#define negmemo_set_succeeded(nr,pen)   \
{                                       \
    PENALTY diff = NO_PENALTY - pen;    \
    if (NEG_MEMO(nr) > diff) {          \
        NEG_MEMO(nr) = diff;            \
    }                                   \
}

unsigned long memo_enabled(long rule_nr)
{
    return memo_enable_table[rule_nr];
}


/*
** The following routine is for debugging and statistical purposes:
*/

void
print_negmemo_table(Trellis *trellis, gboolean skip_unknown)
{
    StateNode** state_row = GET_TRELLIS_STATE_ROW(trellis);
    int node_nr;
    int alt_nr;
    gboolean* empty_rule;
    gboolean* empty_node;

    long** overview = (long**) GetMem(sizeof(long*) * trellis->length, "overview[]");

    printf ("print_negmemo_table: there are %u negative memos.\n", get_nr_neg_memos());
    /* Build the table: */
    for(node_nr = 0; node_nr < trellis->length; node_nr++) {
        StateNode* state = *state_row++;

        if(!state) {
            overview[node_nr] = NULL;
        } else {
            NegMemo* neg_memo_vec = state->neg_memos;

            if(!neg_memo_vec) {
                overview[node_nr] = NULL;
            } else {
                overview[node_nr] = (long*) GetMem(sizeof(long) * get_nr_neg_memos(), "overview[][]");

                for (alt_nr = 0; alt_nr < get_nr_neg_memos(); alt_nr++) {
                    overview[node_nr][alt_nr] = neg_memo_vec[alt_nr];
                }
            }
        }
    }

    /* printed table compression */
    empty_rule = (gboolean*) GetMem(sizeof(gboolean) * get_nr_neg_memos(), "empty_rule");
    for (alt_nr = 0; alt_nr < get_nr_neg_memos(); alt_nr++) {
        empty_rule[alt_nr] = TRUE;
        node_nr = 0;

        while ((node_nr < trellis->length) && (empty_rule[alt_nr])) {
            if (overview[node_nr]) {
                switch (overview[node_nr][alt_nr]) {
                    case NEG_MEMO_UNKNOWN_VAL:
                        empty_rule[alt_nr] = skip_unknown;
                        break;
                    case NEG_MEMO_BLOCK_VAL:
                        empty_rule[alt_nr] = FALSE;
                        break;
                    default:
                        empty_rule[alt_nr] = FALSE;
                }
            }

            node_nr++;
        }
    }
    empty_node = (gboolean*) GetMem(sizeof(gboolean) * trellis->length, "empty_node");
    for (node_nr = 0; node_nr < trellis->length; node_nr++) {
        empty_node[node_nr] = TRUE;
        alt_nr = 0;

        while ((alt_nr < get_nr_neg_memos())
               && (empty_node[node_nr])
               && (overview[node_nr])) {
            switch (overview[node_nr][alt_nr]) {
                case NEG_MEMO_UNKNOWN_VAL:
                    empty_node[node_nr] = skip_unknown;
                    break;
                case NEG_MEMO_BLOCK_VAL:
                    empty_node[node_nr] = FALSE;
                    break;
                default:
                    empty_node[node_nr] = !overview[node_nr][alt_nr];
            }

            alt_nr++;
        }
    }

    /* actually show it: */
    /* first the table */
    for (alt_nr = 0; alt_nr < get_nr_neg_memos(); alt_nr++) {
        if (!empty_rule[alt_nr]) {
            printf("%3d|", alt_nr);

            for (node_nr = 0; node_nr < trellis->length; node_nr++) {
                if (!empty_node[node_nr]) {
                    switch (overview[node_nr][alt_nr]) {
                        case NEG_MEMO_BLOCK_VAL:
                            printf("          b");
                            break;
                        case NEG_MEMO_UNKNOWN_VAL:
                            printf("          u");
                            break;
                        default:
                            printf(" %10ld", overview[node_nr][alt_nr]);
                    }
                }
            }
            printf("\n");
        }
    }
    /* then a neat line below it */
    printf("---+");
    for (node_nr = 0; node_nr < trellis->length; node_nr++) {
        if (!empty_node[node_nr]) {
            printf("-----------");
        }
    }
    /* and of course the numbers */
    printf("\n   |");
    for (node_nr = 0; node_nr < trellis->length; node_nr++) {
        if (!empty_node[node_nr]) {
            printf(" %10d", node_nr);
        }
    }
    printf("\n");

    /* free the space: */
    for (node_nr = 0; node_nr < trellis->length; node_nr++) {
        if (overview[node_nr]) {
            FreeMem(overview[node_nr], "overview[][]");
        }
    }
    FreeMem(overview, "overview[]");
    FreeMem(empty_rule, "empty_rule");
    fflush(stdout);
}


#define STATE_GET_LENGTH(s) (s ? STATE_POS(s) : UINT_MAX)

#ifdef PMRTS
/*
//------------------------------------------------------------------------------
// Posmemo EOS check:
//------------------------------------------------------------------------------
*/

long* prodpens;
long prodpens_nr;
long prodpens_max;

static void prodpen_new(unsigned max)
{
    unsigned i;
    prodpens_nr = 0;
    prodpens_max = max;
    prodpens = (long*) GetMem(sizeof(long) * prodpens_max, "prodpens");
    for (i = 0; i < prodpens_max; ++i) {
        prodpens[i] = NO_PENALTY;
    }
}

static void prodpen_delete()
{
    FreeMem(prodpens, "prodpens");
}

static void prodpen_add(PENALTY pen)
{
    long i;
    for (i = 0; i < prodpens_max; ++i) {
        if (prodpens[i] > pen) {
            long j;
            for (j = prodpens_max - 2; j >= i; --j) {
                prodpens[j+1] = prodpens[j];
            }
            prodpens[i] = pen;
            break;
        }
    }
    if (prodpens_nr < prodpens_max) { prodpens_nr++; }
}

static PENALTY prodpen_get_worst()
{
//int i;for(i=0;i<prodpens_max;i++)fprintf(stderr,"%3d: %15ld\n",i,prodpens[i]);
    if (!prodpens_nr) return -1;

    if (best_parsings) {
        return prodpens[0];
    } else {
        return prodpens[prodpens_nr - 1];
    }
}

static PENALTY get_worst_penalty(unsigned max, PMPROD* entry)
{
    long worst_pen;
    gboolean found_candidate = FALSE;
    prodpen_new(max);

    while (entry) {
        StateIndicator res_state = posmemo_get_input_state(entry);

        if (state_has_eos_transition(res_state)) {
            prodpen_add(posmemo_get_penalty(entry));
            found_candidate = TRUE;
        }

        entry = posmemo_get_next_prod(entry);
    }

    worst_pen = prodpen_get_worst();
    prodpen_delete();

    if (found_candidate) {
        return worst_pen;
    } else {
        return -1;
    }
}

static PENALTY get_longest_worst_penalty(unsigned max, PMPROD* entry)
{
    long worst_pen;
    unsigned best_length = 0;
    gboolean found_candidate = FALSE;
    prodpen_new(max);

    while (entry) {
        StateIndicator res_state = posmemo_get_input_state(entry);
        DB(fprintf(stderr, "get_longest_worst_penalty: penalty %ld len %d\n",
                   posmemo_get_penalty(entry), STATE_GET_LENGTH(res_state)));

        if (STATE_GET_LENGTH(res_state) == best_length) {
            prodpen_add(posmemo_get_penalty(entry));
            found_candidate = TRUE;
        } else if (STATE_GET_LENGTH(res_state) > best_length) {
            prodpen_delete();
            prodpen_new(max);
            best_length = STATE_GET_LENGTH(res_state);
            prodpen_add(posmemo_get_penalty(entry));
            found_candidate = TRUE;
        }

        entry = posmemo_get_next_prod(entry);
    }

    worst_pen = prodpen_get_worst();
    prodpen_delete();

    if (found_candidate) {
        return worst_pen;
    } else {
        return -1;
    }
}
#endif /* PMRTS */


/*
//------------------------------------------------------------------------------
// Instruction switching macros. 
//
// Macro:
//	NEXT
// Description:
//	Execute next instruction. Assume newpc and label are set correctly.
//
// Macro:
//	GOTO(addr)
// Description:
//	Go to instruction at address addr.
//------------------------------------------------------------------------------
*/

#define NEXT					\
{						\
    pc = newpc;					\
    goto *label;				\
}

#define GOTO(x)					\
{						\
    CODE* _x = (x);	        		\
    label = _x->action;				\
    pc = _x;					\
    goto *label;				\
}


/************
 *
 */

#define PUSH_VALUE(x)                           \
{                                               \
    sp->val = x;                                \
    sp--;                                       \
}

#define PUSH_ADDR(x)                            \
{                                               \
    sp->code = x;                               \
    sp--;                                       \
}

#define POP_VALUE(x)                            \
{                                               \
    sp++;                                       \
    x = sp[0].val;                              \
}

#define POP_ADDR(x)                             \
{                                               \
    sp++;                                       \
    x = sp[0].code;                             \
}

#ifdef DEBUG_INTERPRETER
#ifndef PMRTS
#define logexec(x) { \
    fprintf(stderr, "logexec: %s fp=%p sp=%p state=%p\n", \
            translation[x].name, fp, sp, i_state);\
}
#define logfail(x) { \
    fprintf(stderr, "logfail: %s\n", translation[x].name); \
}
#else /* PMRTS */
#define logexec(x) { \
    fprintf(stderr, "logexec(%c): %s fp=%p sp=%p state=%p\n", \
            analyse_stage?'A':'G', \
            translation[x].name, fp, sp, i_state);\
}
#define logfail(x) { \
    fprintf(stderr, "logfail(%c): %s\n", \
            analyse_stage?'A':'G', \
            translation[x].name); \
}
#endif /* PMRTS */
#else /* DEBUG_INTERPRETER */
#define logexec(x)
#define logfail(x)
#endif /* DEBUG_INTERPRETER */

#ifdef DEBUGRTS
#define DBRTS(x) x
#else /* DEBUGRTS */
#define DBRTS(x)
#endif /* DEBUGRTS */

/*
//------------------------------------------------------------------------------
// Function:
//	StateIndicator agfl_interpret(enum Runlevel runlevel,
//	                              Trellis* trellis,
//	                              StateIndicator start_state)
//
// Description:
//	The AGFL interpreter. The interpreter can be called with
//	the following runlevels, selecting the appropriate functionality:
//	* Translate
//		Initialization mode replacing numbers of instructions with 
//		labels of instruction. Performed only once at initialization.
//	* Run
//		Parsing mode; execute grammar.
//	* Statistics
//		Display statistics of instructions (define STATISTICS).
//------------------------------------------------------------------------------
*/

enum Runlevel { Translate, Run, Statistics };

static StateIndicator
agfl_interpret(enum Runlevel runlevel,
               Trellis* trellis,
               StateIndicator start_state)
{

/*
//------------------------------------------------------------------------------
// Instruction opcodes.
//
// The interpreter implements the following instructions. The number of
// each instruction must correspond to the opcode number defined in the
// machdep file.
//
// The instruction size is the number of words occupied by the instruction
// sequence (opcode + parameters).
//------------------------------------------------------------------------------
*/

    enum instruction {
        unknown = 0,
        create = 1,
        cont,
        fail,

        call,
        ucall,

        init_i,
        init_s,
        init_t,
        init_v,

        adjust_i,
        adjust_s,
        adjust_t,
        adjust_v,

        uadjust_i,
        uadjust_s,
        uadjust_t,
        uadjust_v,

        restrict_i,
        restrict_s,
        restrict_t,
        restrict_v,

        urestrict_i,
        urestrict_s,
        urestrict_t,
        urestrict_v,

        match,
        umatch,
        skip_re,
        match_re,
        umatch_re,

        lex_match,
        lex_umatch,

        penalty,
        upenalty,

        commit,

        success,
        usuccess,
        end,
        pass2,

        retrn,
        push_f,
        push_l,
        print_son,

        print_abeg,
        print_aend,
        print_pbeg,
        print_psep,
        print_pend,

        print_val,
        print_term,
        print_lex,
        print_re,

        print_pen,

        print_n,
        print_il,
        print_sl,
        print_tl,
        print_vl,
        print_if,
        print_sf,
        print_tf,
        print_vf,

        tmemo,
        tsmemo,
        smemo_s,

        choice,
        uchoice,
        alt_marker,
        ualt_marker,

        root_call,
        root_create,

        no_lrec_marker,
        lrec_marker,

        trace
    };

    enum instruction_sz {
        create_sz = 4,
        cont_sz = 3,
        fail_sz = 1,

        call_sz = 4,
        ucall_sz = 2,

        init_i_sz = 3,
        init_s_sz = 3,
        init_t_sz = 3,
        init_v_sz = 3,

        adjust_i_sz = 4,
        adjust_s_sz = 4,
        adjust_t_sz = 4,
        adjust_v_sz = 4,

        uadjust_i_sz = 3,
        uadjust_s_sz = 3,
        uadjust_t_sz = 3,
        uadjust_v_sz = 3,

        restrict_i_sz = 4,
        restrict_s_sz = 4,
        restrict_t_sz = 4,
        restrict_v_sz = 4,

        urestrict_i_sz = 2,
        urestrict_s_sz = 2,
        urestrict_t_sz = 2,
        urestrict_v_sz = 2,

        match_sz = 4,
        umatch_sz = 1,
        skip_re_sz = 4,
        match_re_sz = 4,
        umatch_re_sz = 1,

        lex_match_sz = 3,
        lex_umatch_sz = 2,

        penalty_sz = 3,
        upenalty_sz = 2,

        commit_sz = 2,

        success_sz = 2,
        usuccess_sz = 1,
        end_sz = 3,
        pass2_sz = 3,

        retrn_sz = 1,
        push_f_sz = 2,
        push_l_sz = 2,
        print_son_sz = 2,

        print_abeg_sz = 2,
        print_aend_sz = 1,
        print_pbeg_sz = 1,
        print_psep_sz = 1,
        print_pend_sz = 1,

        print_val_sz = 2,
        print_term_sz = 2,
        print_lex_sz = 2,
        print_re_sz = 2,

        print_pen_sz = 2,

        print_n_sz = 2,
        print_il_sz = 2,
        print_sl_sz = 3,
        print_tl_sz = 2,
        print_vl_sz = 1,
        print_if_sz = 2,
        print_sf_sz = 3,
        print_tf_sz = 2,
        print_vf_sz = 1,

        tmemo_sz = 3,
        tsmemo_sz = 3,
        smemo_s_sz = 2,

        choice_sz = 3,
        uchoice_sz = 2,
        alt_marker_sz = 2,
        ualt_marker_sz = 1,

        root_call_sz = 4,
        root_create_sz = 4,

        no_lrec_marker_sz = 2,
        lrec_marker_sz = 2,

        trace_sz = 2
    };

/*
**------------------------------------------------------------------------------
** Opcode decoding table.
**
** The instruction translation table defines for each instruction its name,
** the label of its code and its length. If STATISTICS is defined,
** counters for execution, failure and looping are allocated for each
** instruction.
** 
** Note: The offset of each instruction in the table must correspond to
** the opcode number defined in the machdep file.
**------------------------------------------------------------------------------
*/

    typedef struct {
        char	*name;
        void	*label;
        int	length;
    } opcode;

    static opcode translation[] =
    { 
        {NULL, 0, 0}, 

        {"CREATE", &&CREATE, create_sz},
        {"CONT", &&CONT, cont_sz},
        {"FAIL", &&FAIL, fail_sz},
        
        {"CALL", &&CALL, call_sz},
        {"UCALL", &&UCALL, ucall_sz},

        {"INIT_I", &&INIT_I, init_i_sz},
        {"INIT_S", &&INIT_S, init_s_sz},
        {"INIT_T", &&INIT_T, init_t_sz},
        {"INIT_V", &&INIT_V, init_v_sz},

        {"ADJUST_I", &&ADJUST_I, adjust_i_sz},
        {"ADJUST_S", &&ADJUST_S, adjust_s_sz},
        {"ADJUST_T", &&ADJUST_T, adjust_t_sz},
        {"ADJUST_V", &&ADJUST_V, adjust_v_sz},

        {"UADJUST_I", &&UADJUST_I, uadjust_i_sz},
        {"UADJUST_S", &&UADJUST_S, uadjust_s_sz},
        {"UADJUST_T", &&UADJUST_T, uadjust_t_sz},
        {"UADJUST_V", &&UADJUST_V, uadjust_v_sz},

        {"RESTRICT_I", &&RESTRICT_I, restrict_i_sz},
        {"RESTRICT_S", &&RESTRICT_S, restrict_s_sz},
        {"RESTRICT_T", &&RESTRICT_T, restrict_t_sz},
        {"RESTRICT_V", &&RESTRICT_V, restrict_v_sz},

        {"URESTRICT_I", &&URESTRICT_I, urestrict_i_sz},
        {"URESTRICT_S", &&URESTRICT_S, urestrict_s_sz},
        {"URESTRICT_T", &&URESTRICT_T, urestrict_t_sz},
        {"URESTRICT_V", &&URESTRICT_V, urestrict_v_sz},

        {"MATCH", &&MATCH, match_sz},
        {"UMATCH", &&UMATCH, umatch_sz},
        {"SKIP_RE", &&SKIP_RE, skip_re_sz},
        {"MATCH_RE", &&MATCH_RE, match_re_sz},
        {"UMATCH_RE", &&UMATCH_RE, umatch_re_sz},

        {"LEX_MATCH", &&LEX_MATCH, lex_match_sz},
        {"LEX_UMATCH", &&LEX_UMATCH, lex_umatch_sz},

        {"PENALTY", &&PENALTY, penalty_sz},
        {"UPENALTY", &&UPENALTY, upenalty_sz},

        {"COMMIT", &&COMMIT, commit_sz},

        {"SUCCESS", &&SUCCESS, success_sz},
        {"USUCCESS", &&USUCCESS, usuccess_sz},
        {"END", &&END, end_sz},
        {"PASS2", &&PASS2, pass2_sz},

        {"RETRN", &&RETRN, retrn_sz},
        {"PUSH_F", &&PUSH_F, push_f_sz},
        {"PUSH_L", &&PUSH_L, push_l_sz},
        {"PRINT_SON", &&PRINT_SON, print_son_sz},

        {"PRINT_ABEG", &&PRINT_ABEG, print_abeg_sz},
        {"PRINT_AEND", &&PRINT_AEND, print_aend_sz},
        {"PRINT_PBEG", &&PRINT_PBEG, print_pbeg_sz},
        {"PRINT_PSEP", &&PRINT_PSEP, print_psep_sz},
        {"PRINT_PEND", &&PRINT_PEND, print_pend_sz},

        {"PRINT_VAL", &&PRINT_VAL, print_val_sz},
        {"PRINT_TERM", &&PRINT_TERM, print_term_sz},
        {"PRINT_LEX", &&PRINT_LEX, print_lex_sz},
        {"PRINT_RE", &&PRINT_RE, print_re_sz},

        {"PRINT_PEN", &&PRINT_PEN, print_pen_sz},

        {"PRINT_N", &&PRINT_N, print_n_sz},
        {"PRINT_IL", &&PRINT_IL, print_il_sz},
        {"PRINT_SL", &&PRINT_SL, print_sl_sz},
        {"PRINT_TL", &&PRINT_TL, print_tl_sz},
        {"PRINT_VL", &&PRINT_VL, print_vl_sz},
        {"PRINT_IF", &&PRINT_IF, print_if_sz},
        {"PRINT_SF", &&PRINT_SF, print_sf_sz},
        {"PRINT_TF", &&PRINT_TF, print_tf_sz},
        {"PRINT_VF", &&PRINT_VF, print_vf_sz},

        {"TMEMO", &&TMEMO, tmemo_sz},
        {"TSMEMO", &&TSMEMO, tsmemo_sz},
        {"SMEMO_S", &&SMEMO_S, smemo_s_sz},

        {"CHOICE", &&CHOICE, choice_sz},
        {"UCHOICE", &&UCHOICE, uchoice_sz},
        {"ALT_MARKER", &&ALT_MARKER, alt_marker_sz},
        {"UALT_MARKER", &&UALT_MARKER, ualt_marker_sz},

        {"ROOT_CALL", &&ROOT_CALL, root_call_sz},
        {"ROOT_CREATE", &&ROOT_CREATE, root_create_sz},

        {"NO_LREC_MARKER", &&NO_LREC_MARKER, no_lrec_marker_sz},
        {"LREC_MARKER", &&LREC_MARKER, lrec_marker_sz},

        {"TRACE", &&TRACE, trace_sz},

        {NULL, 0, 0}
    };

/*
//------------------------------------------------------------------------------
// Interpreter registers.
//
//	pc
//		program counter; contains address of current instruction.
//
//	newpc
//		new program counter; contains address of next instruction.
//
//	label
//		contains label of implementation code of next instruction.
//
//	sp
//		stack pointer; points to next free position on stack
//
//	fp
//		frame pointer; points to current stack frame.
//
//	i_state
//		state pointer pointing to current input state
//
//	i_transition
//		points to current transition to be matched
//		(only used during LEX_MATCH/LEX_UMATCH)
//
//	penfailed
//		gets incremented on failing penalty, and saved for some
//		reason in the stack frame	(Erik, October 16 1999)
//	penleft
//		the number of penalties that the current parse can still
//		endure without failing		(Erik, October 16 1999)
//
//	indent
//	have_cancel_token
//		indicates whether cancellation token ahead in input
//------------------------------------------------------------------------------
*/

    register CODE* pc = 0;
    register CODE* newpc = 0; 
    register void* label = 0;
    register DATA* fp = 0;
    DATA* pass2fp = 0;
    register DATA* sp = 0;
    StateIndicator i_state = 0;
    ARG current_terminal = 0;

#ifdef GEN_RTS
    long chosen_freq = 0;
    long alt_depth = 0;

#ifdef DEBUG_GENERATOR
#define ALT_DEPTH_RESET (alt_depth = 0)
#define ALT_DEPTH_CHECK (assert(alt_depth >= 0))
#define ALT_DEPTH_DECREASE \
{                                                                           \
    alt_depth--;                                                            \
    fprintf(stderr,"ALT_DEPTH_DECREASE: alt_depth now %ld\n",alt_depth);    \
}
#define ALT_DEPTH_INCREASE \
{                                                                           \
    alt_depth++;                                                            \
    fprintf(stderr,"ALT_DEPTH_INCREASE: alt_depth now %ld\n",alt_depth);    \
}
#else /* DEBUG_GENERATOR */
#define ALT_DEPTH_RESET (alt_depth = 0)
#define ALT_DEPTH_CHECK (assert(alt_depth >= 0))
#define ALT_DEPTH_DECREASE (alt_depth--)
#define ALT_DEPTH_INCREASE (alt_depth++)
#endif /* DEBUG_GENERATOR */
#endif /* GEN_RTS */

    register Transition *i_transition = 0;

    PENALTY penleft = 0;
    PENALTY penalty_limit = 0;

    int indent = 0;

#ifdef PMRTS
    gboolean analyse_stage = TRUE;
#endif /* PMRTS */

/*
//------------------------------------------------------------------------------
// Initializing the interpreter.
//
// Description:
//	In the generated code, each instruction number is replaced with
//	the address of the label of the code implementing the instruction.
//
//	The call, fcall, save_f, restore_f, save_l, restore_l, penalty,
//	and upenalty instructions have specialized versions for common cases,
//	and generalized versions of other cases.
//
//	For the cont instruction, either the code for printing the
//	transduction or printing the parse tree is selected.
//
//	For the prof_exec and prof_succ instruction, the profile counter
//	is stored in the code. In the profile table, pointers to the
//	storage locations in the code are filled in.
//
//	If the programs mode is not very verbose, the arguments of print_pen,
//	and print_abeg are overwritten with 0, disabling the printing
//	of penalties and alternative numbers in the parse tree.
//------------------------------------------------------------------------------
*/

    switch (runlevel) {
        case Translate: {
            long instr;

            pc = &root;
            while ((instr = pc[0].arg) > 0) {
#ifdef DEBUG_TRANSLATE
                fprintf(stderr, "translating %s\n", translation[instr].name);
#endif /* DEBUG_TRANSLATE */
                switch (instr) {
                    case (long) cont:
                        if (!transduce_option) {
                            pc[1].arg = pc[2].arg;
                        }
                        pc[0].action = translation[cont].label;
                        pc += cont_sz;
                        break;

                    case (long) pass2:
                        if (!transduce_option) {
                            pc[1].arg = pc[2].arg;
                        }
                        pc[0].action = translation[pass2].label;
                        pc += pass2_sz;
                        break;

/*
** The type and number of the terminal (skip, match, lexicon, grammar) have
** been encoded in state->terminal. The same encoding should be applied to the
** terminal numbers in the match and un-match instructions.  Notice that the
** nonterminal number and arity of each LEX_MATCH and LEX_UMATCH have already
** been coded by the agfl compiler.
*/

#ifndef GEN_RTS
                    case match:
                        pc[0].action = translation[instr].label;
                        pc[1].arg = ENCODE_TERM(pc[1].arg);
                        pc += translation[instr].length;
                        break;

                    case match_re:
                        pc[0].action = translation[instr].label;
                        pc[1].arg = ENCODE_MATCH(pc[1].arg);
                        pc += translation[instr].length;
                        break;

                    case skip_re:
                        pc[0].action = translation[instr].label;
                        pc[1].arg = ENCODE_SKIP(pc[1].arg);
                        pc += translation[instr].length;
                        break;
#endif

/*
** Replace the number of each abstract instruction with the label of
** the code implementing the instruction, and advance the pc to the
** next instruction.
*/

                    default:
                        pc[0].action = translation[instr].label;
                        pc += translation[instr].length;
                        break;
                }
            }
            return NULL;
        }

/*
//------------------------------------------------------------------------------
// Printing the instruction execution counters.
//------------------------------------------------------------------------------
*/

        case Statistics: {
            rtsBug("agfl_interpret", "Statistics requested");
        }

/*
//------------------------------------------------------------------------------
// Interpret the compiled grammar.
//
// Description:
//	Set up the interpreters registers and jump to the first instruction.
//
// Note:
//	The interpreter should have been initialized and reseted.
//------------------------------------------------------------------------------
*/

        case Run: {
        /*
        ** Initialize interpreters registers
        */
            penleft = NO_PENALTY;
            stack = malloc(STACK_SIZE * sizeof(cel));
            fp = stack + STACK_SIZE - 1 - FRAME_HEAD_LEN;
            sp = fp;
            pc = &root;

#ifndef GEN_RTS
            if (segment_mode && (start_state != NULL)) {
                i_state = start_state;
            } else {
                i_state = GET_FIRST_STATE_INDICATOR(trellis);
            }
#else /* GEN_RTS */
            i_state = NULL;
            choice_admin_init();
            chosen_freq = 0;
            ALT_DEPTH_RESET;
#endif /* GEN_RTS */

            indent = 0;

            /*
            ** Start interpreter
            */
            start_parse_time();
            goto *(pc[0].action);
        }
    }
    rtsBug("agfl_interpret", "invalid runlevel");

/*
//------------------------------------------------------------------------------
// Implementation of abstract instructions.
//------------------------------------------------------------------------------
*/

CONT:
    {
#ifdef PMRTS
        DB(fprintf(stderr, "CONT: penleft = %ld, diff = %ld\n", penleft, NO_PENALTY - penleft));
        if (!(analyse_stage && memo_enabled(NONT_NR))) {
#endif
            unsigned i = NR_FORMALS;
            CODE* routine = pc[1].code;

            fp[NR_SUCCS_OFF].arg++;               /* for commit: parsing found */

            logexec(cont);

            sp -= 2;
            sp[2].code = pc + cont_sz;
            sp[1].data = fp; 
            /* push result parameters */
            while (i > 0) {
                i--;
                PUSH_VALUE(VARIABLE(i).val);
                DB(fprintf(stderr, "pushing value %lu\n", VARIABLE(i).val.set_par));
            }

            fp[PASS2_CODE].code = routine;
            DB(fprintf(stderr, "2nd pass routine @%p\n", routine));

            /* continue with the continuation... */
            newpc = fp[SUCC_ADDR_OFF].code;
            label = newpc->action;
            /* ...which assumes execution in the previous stack frame: */
            fp = fp[OLD_FP].data;
            DBRTS(debug_cont(NONT_NR));
            DB(fprintf(stderr, "CONTinuing into \"%s\"\n",
                       nonterm_names[NONT_NR]));

            NEXT;
#ifdef PMRTS
        } else {
            logexec(cont);

            fp[NR_SUCCS_OFF].arg++;               /* for commit: parsing found */

            posmemo_add_production(START_INPUT_STATE, NONT_NR,
                                   fp[PENORIG].penalty - penleft,
                                   NR_FORMALS, &(VARIABLE(NR_FORMALS - 1).val),
                                   i_state);

            newpc = pc + cont_sz;
            label = newpc->action;
            NEXT;
        }
#endif /* PMRTS */
    }

CREATE:
    {
        ARG nr_formals = pc[1].arg;
        ARG nr_locals = pc[2].arg;
        ARG nr_sons = pc[3].arg;

        logexec(create);

        sp[0].data = fp;                /* store old fp */
        fp = sp;

        sp -= nr_formals + nr_locals + nr_sons + VAR_OFFSET;
        NR_FORMALS = nr_formals;
        NR_LOCALS = nr_locals;
        NR_SONS = nr_sons;

        DB(fprintf(stderr, "CREATE(%ld, %ld, %ld)\n", NR_FORMALS, NR_LOCALS, NR_SONS));

        START_INPUT_STATE = i_state;
        fp[NR_SUCCS_OFF].arg = 0;
        fp[NR_SONS_DONE].arg = 0;
        fp[PENORIG].penalty = penleft;

        DB(fprintf(stderr, "CREATE: start_input_state = %p\n", i_state));

#ifdef PMRTS
        if (analyse_stage && memo_enabled(NONT_NR)) {
            if (posmemo_is_known(i_state, NONT_NR) &&
                posmemo_is_unblocked(i_state, NONT_NR)) {
                CODE* failcont = posmemo_get_failcont(fp[PMPRODPTR].pmprod);
                DB(fprintf(stderr, "CREATE: known & unblocked\n"));
                fp[PMPRODPTR].pmprod = NULL;
                GOTO(failcont);
            } else {
                fp[PMPRODPTR].pmprod = NULL;
            }
        }
#endif /* PMRTS */

        newpc = pc + create_sz;
        label = newpc->action;
        NEXT;
    }

#ifndef PMRTS
    /*
     * Instruction: FAIL (no positive memo)
     * Description: restore the stack frame of the parent rule and continue
     *              with the fail continuation.
     */
FAIL:
    {
        logexec(fail);

        newpc = fp[FAIL_ADDR_OFF].code; 
        label = newpc->action;

        sp = fp + FRAME_HEAD_LEN;

        fp = fp[OLD_FP].data;
        fp[NR_SONS_DONE].arg--;

        NEXT;
    }
#else /* PMRTS */
    /*
     * Instruction: FAIL (positive memo)
     * Description: current rule is done; fail if there are no successfull
     *              production, otherwise generate those productions by jumping
     *              to PM_GENERATE_PRODUCTION.
     */
FAIL:
    {
        logexec(fail);

        i_state = START_INPUT_STATE;

        DB(fprintf(stderr, "FAIL in \"%s\"...\n", nonterm_names[NONT_NR]));
        DB(fprintf(stderr, "  into \"%s\"\n", nonterm_names[(fp[OLD_FP].data)[NONT_NR_OFF].arg]));
        DB(fprintf(stderr, "FAIL: start_input_state = %p\n", i_state));
        if (analyse_stage && memo_enabled(NONT_NR)) {
            if (!posmemo_is_blocked(i_state, NONT_NR)) {
                DB(fprintf(stderr, "FAIL: \"%s\" is not blocked\n",
                           nonterm_names[NONT_NR]));
                if (fp[PMPRODPTR].pmprod) {
                    /* not the first time, restore penalty level... */
                    penleft += posmemo_get_penalty(fp[PMPRODPTR].pmprod);
                    /* ... and get next production */
                    fp[PMPRODPTR].pmprod =
                        posmemo_get_next_prod(fp[PMPRODPTR].pmprod);
                    if (fp[PMPRODPTR].pmprod != NULL) {
                        /* there are more productions  */
                        posmemo_set_failcont(fp[PMPRODPTR].pmprod, pc);
                        DB(fprintf(stderr,
                                   "FAIL: productions left, generating...\n"));
                        goto PM_GENERATE_PRODUCTION;
                    } else {
                        DB(fprintf(stderr,
                               "FAIL: no more productions left, failing...\n"));
                    }
                } else {
                    /* first time we generate... */
                    fp[PMPRODPTR].pmprod = posmemo_get_prod_ptr(i_state,
                                                                NONT_NR);
                    assert(fp[PMPRODPTR].pmprod); /* if this fails, the rule is
                                                     marked unknown, which Should
                                                     Not Happen (TM) */
                    posmemo_set_failcont(fp[PMPRODPTR].pmprod, pc);
                    goto PM_GENERATE_PRODUCTION;
                }
            } else {
                DB(fprintf(stderr, "FAIL: \"%s\" is blocked\n",
                           nonterm_names[NONT_NR]));
            }
        }

        /* should not be needed, but just to be sure: */
        penleft = fp[PENORIG].penalty;

        newpc = fp[FAIL_ADDR_OFF].code; 
        label = newpc->action;

        sp = fp + FRAME_HEAD_LEN;

        fp = fp[OLD_FP].data;
        fp[NR_SONS_DONE].arg--;

        NEXT;
    }

    /*
     * Instruction: none, helper.
     * Description: push formals to stack, set input state and penalty level,
     *              and continue with success continuation.
     */
PM_GENERATE_PRODUCTION:
    {
        unsigned nr_formals = NR_FORMALS;
        void* memod_formals = posmemo_get_formal_ptr(fp[PMPRODPTR].pmprod);

        sp -= 2;
        sp[2].code = pc;
        sp[1].data = fp; 
        /* push result parameters */
        sp -= nr_formals;
        bcopy(memod_formals, sp + 1, nr_formals * sizeof(VALUE));

        i_state = posmemo_get_input_state(fp[PMPRODPTR].pmprod);
        penleft -= posmemo_get_penalty(fp[PMPRODPTR].pmprod);

        /* continue with the continuation... */
        newpc = fp[SUCC_ADDR_OFF].code;
        label = newpc->action;
        /* ...which assumes execution in the previous stack frame: */
        fp = fp[OLD_FP].data;

        NEXT;
    }
#endif /* PMRTS */

    /*
     * Instruction: CALL(rule number, start point, fail continuation, success
     *                   continuation)
     * Description: call to a rule by building the pre-call stack part and jump
     *              to the starting point. With positive memoization extra
     *              checks are done: in analyse mode: check if we already know
     *              the productions and if so, jump to the appropriate FAIL
     *              instruction to produce the results; if blocked (no
     *              productions) just fail; if unknown enter the rule.
     *              If not in analyse mode, see if the rule being called is
     *              blocked for the current penalty level; if so, fail,
     *              otherwise do the call.
     */
CALL:
    {
        ARG rule_nr = pc[1].arg;
        CODE* jmp_point = pc[2].code;
        CODE* fail_addr = pc[3].code;
        CODE* succ_addr = pc + call_sz;

        logexec(call);
        DBRTS(debug_call(rule_nr));

        DB(fprintf(stderr, "CALLing \"%s\" from \"%s\"\n",
                   nonterm_names[rule_nr], nonterm_names[NONT_NR]));
if (have_time_limit() && have_time_out()) {
    GOTO(pc[3].code); };

#ifdef PMRTS
        if (analyse_stage && memo_enabled(rule_nr)) {
            if (posmemo_is_blocked(i_state, rule_nr)) {
                DB(fprintf(stderr, "CALL: \"%s\" is blocked\n",
                           nonterm_names[rule_nr]));
                GOTO(fail_addr);
            } else if (posmemo_is_known(i_state, rule_nr)) {
                DB(fprintf(stderr, "CALL: \"%s\" is known\n",
                           nonterm_names[rule_nr]));
            } else { /* positive memo is unknown */
                posmemo_set_blocked(i_state, rule_nr);
                DB(fprintf(stderr,
                           "CALL: \"%s\" is unknown, changing to blocked\n",
                           nonterm_names[rule_nr]));
            }
        } else {
            if (!analyse_stage && memo_enabled(rule_nr) &&
                (posmemo_is_unknown(i_state, rule_nr)
                 || posmemo_is_blocked_for_penlevel(i_state, rule_nr,
                                                    penleft - penalty_limit))) {
                GOTO(fail_addr);
            }
        }
#endif /* PMRTS */

        sp -= FRAME_HEAD_LEN;
        sp[FAIL_ADDR_OFF].code = fail_addr;
        sp[SUCC_ADDR_OFF].code = succ_addr;
        sp[NONT_NR_OFF].arg = rule_nr;

        assert (fp[NR_SONS_DONE].arg >= 0);
        assert (fp[NR_SONS_DONE].arg < NR_SONS);

        DB(fprintf(stderr, "sons done: %ld, son_fp = %p\n", fp[NR_SONS_DONE].arg, fp + fp[NR_SONS_DONE].arg));
        SON_FP(fp[NR_SONS_DONE].arg) = sp;
        fp[NR_SONS_DONE].arg++;

#if PMRTS
        if (analyse_stage && memo_enabled(NONT_NR)) { sp[PMPRODPTR].pmprod = posmemo_get_prod_ptr(i_state, rule_nr); }
#endif

        newpc = jmp_point;
        label = newpc->action;
        NEXT;
    }

    /*
     * Instruction: UCALL(nr formals)
     * Description: clean up previous result parameters from the stack and 
     *              jump to the continuation of the child rule.
     */
UCALL:
    {
        ARG nr_formals = pc[1].arg;

        logexec(ucall);
        DB(fprintf(stderr, "UCALL(%ld)\n", nr_formals));

        sp += nr_formals;
        newpc = sp[2].code;
        fp = sp[1].data;
        sp += 2;

        DBRTS(debug_ucall(NONT_NR));

        label = newpc->action;
        NEXT;
    }

/*
 *------------------------------------------------------------------------------
 *
 * Instructions on variables:
 *
 *------------------------------------------------------------------------------
 */

    /*
     * Instruction: INIT_{IST}(position, value)
     * Description: initialize INT/SET/TEXT variable at position (the position
     *              is relative to the frame pointer plus fixed part offset)
     *              to the specified value.
     */
INIT_I:
    {
        ARG pos = pc[1].arg;
        VALUE val = pc[2].val;

        logexec(init_i);

        VARIABLE(pos).val = val;

        /* assumption: init_i_sz == init_s_sz == init_t_sz */
        newpc = pc + init_i_sz;
        label = newpc->action;
        NEXT;
    }

INIT_S:
    {
        ARG pos = pc[1].arg;
        VALUE val = pc[2].val;

        logexec(init_s);

        VARIABLE(pos).val = val;

        /* assumption: init_i_sz == init_s_sz == init_t_sz */
        newpc = pc + init_i_sz;
        label = newpc->action;
        NEXT;
    }
INIT_T:
    {
        ARG pos = pc[1].arg;
        VALUE val = pc[2].val;

        logexec(init_v);

        VARIABLE(pos).val = val;

        /* assumption: init_i_sz == init_s_sz == init_t_sz */
        newpc = pc + init_i_sz;
        label = newpc->action;
        NEXT;
    }

INIT_V:
    {
        /* unimplemented yet */
        goto ILLEGAL;
    }

ADJUST_I:
    {
        ARG target = pc[1].arg;
        ARG param = pc[2].arg;
        CODE* fail_addr = pc[3].code;
        VALUE val = sp[param].val;
        VALUE orig = VARIABLE(target).val;

        logexec(adjust_i);

        if (val.int_par == TOP_INT) {
            /* save the original, so the restore will put the right value back
             * (and not TOP_INT).
             */
            sp[param].val = orig;
        } else if (orig.int_par == TOP_INT) {
            sp[param].val = orig;
            VARIABLE(target).val = val;
        } else if (orig.int_par != val.int_par) {
            GOTO(fail_addr);
        }

        newpc = pc + adjust_i_sz;
        label = newpc->action;
        NEXT;
    }

ADJUST_S:
    {
        ARG target = pc[1].arg;
        ARG param = pc[2].arg;
        CODE* fail_addr = pc[3].code;
        VALUE val = sp[param].val;
        VALUE orig = VARIABLE(target).val;
        VALUE result;

        logexec(adjust_s);

        result.set_par = orig.set_par & val.set_par;
DB(fprintf(stderr, "ADJUST_S: %ld <- (orig) %ld & (res) %ld\n", result.set_par, orig.set_par, val.set_par));

        if (result.set_par == 0) {
            /* intersection of the sets is empty -- fail */
            GOTO(fail_addr);
        }

        /* push original value */
        sp[param].val = orig;

        /* assign intersection to variable */
        VARIABLE(target).val = result;

        newpc = pc + adjust_s_sz;
        label = newpc->action;
        NEXT;
    }

ADJUST_T:
    {
        ARG target = pc[1].arg;
        ARG param = pc[2].arg;
        CODE* fail_addr = pc[3].code;
        VALUE val = sp[param].val;
        VALUE orig = VARIABLE(target).val;

        if (val.text_par == TOP_TEXT) {
            sp[param].val = orig;
        } else if (orig.text_par == TOP_TEXT) {
            sp[param].val = orig;
            VARIABLE(target).val = val;
        } else if (strcmp(orig.text_par, val.text_par) == 0) {
            sp[param].val = orig;
        } else {
            GOTO(fail_addr);
        }

        newpc = pc + restrict_t_sz;
        label = newpc->action;
        NEXT;
    }

ADJUST_V:
    {
        goto ILLEGAL;
    }

UADJUST_I:
    {
        ARG target = pc[1].arg;
        ARG param = pc[2].arg;
        VALUE val = sp[param].val;
        
        logexec(uadjust_i);

        VARIABLE(target).val = val;

        newpc = pc + uadjust_s_sz;
        label = newpc->action;
        NEXT;
    }
UADJUST_S:
    {
        ARG target = pc[1].arg;
        ARG param = pc[2].arg;
        VALUE val = sp[param].val;
        
        logexec(uadjust_s);

        VARIABLE(target).val = val;

        newpc = pc + uadjust_s_sz;
        label = newpc->action;
        NEXT;
    }
UADJUST_T:
    {
        ARG target = pc[1].arg;
        ARG param = pc[2].arg;
        VALUE val = sp[param].val;
        
        logexec(uadjust_t);

        VARIABLE(target).val = val;

        newpc = pc + uadjust_s_sz;
        label = newpc->action;
        NEXT;
    }

UADJUST_V:
    {
        goto ILLEGAL;
    }

RESTRICT_I:
    {
        ARG target = pc[1].arg;
        VALUE val = pc[2].val;
        CODE* fail_addr = pc[3].code;
        VALUE orig = VARIABLE(target).val;

        logexec(restrict_i);

        if (val.int_par == TOP_INT) {
            /* this should not be generated, but to be sure: */

            /* save original value */
            PUSH_VALUE(orig);
        } else if (orig.int_par == TOP_INT) {
            /* save original value */
            PUSH_VALUE(orig);

            /* assign new value to variable */
            VARIABLE(target).val = val;
        } else if (orig.int_par != val.int_par) {
            GOTO(fail_addr);
        }

        newpc = pc + adjust_i_sz;
        label = newpc->action;
        NEXT;
    }

RESTRICT_S:
    {
        ARG target = pc[1].arg;
        VALUE val = pc[2].val;
        CODE* fail_addr = pc[3].code;
        VALUE orig = VARIABLE(target).val;
        VALUE result;

        logexec(restrict_s);

        result.set_par = orig.set_par & val.set_par;

        if (result.set_par == 0) {
            /* intersection of the sets is empty -- fail */
            GOTO(fail_addr);
        }

        /* push original value */
        PUSH_VALUE(orig);

        /* assign intersection to variable */
        VARIABLE(target).val = result;

        newpc = pc + restrict_s_sz;
        label = newpc->action;
        NEXT;
    }

RESTRICT_T:
    {
        ARG target = pc[1].arg;
        VALUE val = pc[2].val;
        CODE* fail_addr = pc[3].code;
        VALUE orig = VARIABLE(target).val;

        logexec(restrict_t);

        if (val.text_par == TOP_TEXT) {
            PUSH_VALUE(orig);
        } else if (orig.text_par == TOP_TEXT) {
            PUSH_VALUE(orig);
            VARIABLE(target).val = val;
        } else if (strcmp(orig.text_par, val.text_par) == 0) {
            PUSH_VALUE(orig);
        } else {
            GOTO(fail_addr);
        }

        newpc = pc + restrict_t_sz;
        label = newpc->action;
        NEXT;
    }

RESTRICT_V:
    {
        goto ILLEGAL;
    }

URESTRICT_I:
    {
        ARG target = pc[1].arg;
        
        logexec(urestrict_i);

        POP_VALUE(VARIABLE(target).val);

        newpc = pc + urestrict_s_sz;
        label = newpc->action;
        NEXT;
    }
URESTRICT_S:
    {
        ARG target = pc[1].arg;
        
        logexec(urestrict_s);

        POP_VALUE(VARIABLE(target).val);

        newpc = pc + urestrict_s_sz;
        label = newpc->action;
        NEXT;
    }
URESTRICT_T:
    {
        ARG target = pc[1].arg;
        
        logexec(urestrict_t);

        POP_VALUE(VARIABLE(target).val);

        newpc = pc + urestrict_s_sz;
        label = newpc->action;
        NEXT;
    }

URESTRICT_V:
    {
        goto ILLEGAL;
    }

/*
 *------------------------------------------------------------------------------
 *
 * Optimization instructions
 *
 *------------------------------------------------------------------------------
 */

    /*
     * Instruction: TMEMO(nr, fail addr)
     * Description: Test memo nr, if blocked for current or lower penalty level
     *              then goto fail addr, otherwise continue.
     */
TMEMO:
    {
        ARG memo_nr = pc[1].arg;
        CODE* fail_addr = pc[2].code;
        long memoval;

        logexec (tmemo);
        DBRTS(debug_tmemo());

        memoval = NEG_MEMO(memo_nr);
        if (penleft < memoval) {
            /* blocked for this penalty level */
            logfail(tmemo);
            newpc = fail_addr;
            label = newpc->action;
            NEXT;
        }

        newpc = pc + tmemo_sz;
        label = newpc->action;
        NEXT;
    }

    /*
     * Instruction: TSMEMO(nr, fail addr)
     * Description: Test and set memo: same as TMEMO, but if unknown set memo
     *              nr to blocked and continue.
     */
TSMEMO:
    {
        ARG memo_nr = pc[1].arg;

        logexec(tsmemo);
        DBRTS(debug_tsmemo());

        if (negmemo_is_blocked(memo_nr, penleft)) {
            logfail(tsmemo);
            newpc = pc[2].code;
            label = newpc->action;
            NEXT;
        }

        if (negmemo_is_unknown(memo_nr)) {
            negmemo_set_blocked(memo_nr, penleft);
        } else {
        }

        newpc = pc + tsmemo_sz;
        label = newpc->action;
        NEXT;
    }

    /*
     * Instruction: SMEMO_S(nr)
     * Description: Set memo nr to succeeded for current penalty level.
     */
SMEMO_S:
    {
        ARG memo_nr = pc[1].arg;

        logexec(smemo_s);
        DBRTS(debug_smemo_s());
#ifdef DEBUG_INTERPRETER
        fprintf(stderr, "SMEMO_S(%lu) at pos %u\n", memo_nr, STATE_POS(i_state));
#endif // DEBUG_INTERPRETER
        assert(!negmemo_is_unknown(memo_nr));
        
        negmemo_set_succeeded(memo_nr, penleft);

        newpc = pc + smemo_s_sz;
        label = newpc->action;
        NEXT;
    }


/*
 *------------------------------------------------------------------------------
 *
 * Input matching instructions:
 *
 *------------------------------------------------------------------------------
 */

/*
// Matching of grammar terminals, regexps matches and skips
// is performed by the same code. In the initialization phase
// of the interpreter, the terminal type has been encoded in
// the first argument of the instruction.
*/

MATCH:
MATCH_RE:
SKIP_RE:
    {
        Transition* curr_transition;
        Terminal terminal = pc[1].arg;		/* get token to be matched */

        logexec(match);

#ifndef GEN_RTS
        curr_transition = GET_STATE_TRANSLIST(trellis, i_state,
                DECODE_TERM_OR_RE_CLASS(terminal));

        DBRTS(debug_match());
#ifdef TRACE_MATCH
        fprintf(stderr, "MATCH: %ld == \"%s\"\n", DECODE_TERM_NUMBER(terminal), term_names[DECODE_TERM_NUMBER(terminal)]);
#endif /* TRACE_MATCH */
        while (curr_transition != NULL) {	/* do we have another entry? */
#ifdef TRACE_MATCH
            fprintf(stderr, "state_has_eos_transition == %d\n", state_has_eos_transition(i_state));
            fprintf(stderr, "c->t == %lu, t == %lu\n", curr_transition->terminal, terminal);
#endif /* TRACE_MATCH */
            if (curr_transition->terminal != terminal) {
                /* This entry doesn't match, try next one */
                curr_transition = curr_transition->next;
            } else {
                unsigned cur_pos = STATE_POS(i_state);

                MARK_POS_VISITED(cur_pos);

                /* save input position on stack */
                sp[0].input_state = i_state;
                sp--;
#ifdef TRACE_MATCH
                fprintf(stderr, "saved state = %p\n", i_state);
                fprintf(stderr, "at local %ld = %p\n", pc[2].arg, &VARIABLE(pc[2].arg));
#endif /* TRACE_MATCH */

                /* save entry */
                VARIABLE(pc[2].arg).input_transition = curr_transition;
                i_state = TRANSITION_DEST_STATE_INDICATOR(curr_transition,
                                                          trellis);
#ifdef TRACE_MATCH
                fprintf(stderr, "new state = %p\n", i_state);
#endif /* TRACE_MATCH */

                newpc = pc + match_sz;
                label = newpc->action;
                NEXT;
            }
        }
#ifdef TRACE_MATCH
        fprintf(stderr, "MATCH: no more transitions\n");
#endif /* TRACE_MATCH */

#else /* GEN_RTS */
        gen_output_add_token(term_names[terminal]);

        newpc = pc + match_sz;
        label = newpc->action;
        NEXT;
#endif /* GEN_RTS */

        /* If all transitions are tried, nothing is left, so fail */

        GOTO(pc[3].code);
    }

    UMATCH:
    UMATCH_RE:
    {
        logexec(umatch);
        DBRTS(debug_umatch());

#ifndef GEN_RTS
        i_state = sp[1].input_state;	/* restore previous input position */
        sp += 1;
#ifdef TRACE_MATCH
        fprintf(stderr, "restored state = %p\n", i_state);
#endif /* TRACE_MATCH */
#else /* GEN_RTS */
        gen_output_remove_token();
#endif /* GEN_RTS */

        newpc = pc + umatch_sz;
        label = newpc->action;
        NEXT; 
    }

#ifndef GEN_RTS
LEX_MATCH:
    {
        PENALTY pen_gain;

        current_terminal = pc[1].arg;   /* get lexicon nont to be matched */
        logexec(lex_match);

        sp -= 2;
        sp[1].input_state = i_state;        /* save current input position */
        sp[2].code = pc;                    /* save current program counter */

        i_transition = GET_STATE_TRANSLIST(trellis,i_state,
                                           DECODE_NONT_CLASS(current_terminal));

LEX_MATCH_AGAIN:
        DBRTS(debug_lex_match());

        if (i_transition != NULL) {     /* do we have another entry? */
            if (hybrid_parsing_option) {
                pen_gain = TRANSITION_PENALTY(i_transition);
            } else {
                pen_gain = 0;
            }
        } else {
            pen_gain = LONG_MAX;        /* no transition -> block next if */
        }
        penleft -= pen_gain;

        if (penleft > penalty_limit) {  /* we can go on */
            int i, arity;
            PARAM const * val_ptr;
            unsigned cur_pos;

            assert(i_transition->terminal == current_terminal);

            /* restrict formals to lexicon entry parameters */
            val_ptr = TRANSITION_PARAMS(i_transition);
            arity = DECODE_NONT_ARITY(current_terminal);
            for (i = arity - 1; i >= 0; --i) {
#ifdef DEBUG_INTERPRETER
                fprintf(stderr, "pushing value %lu\n", val_ptr[i].value.set_par);
#endif // DEBUG_INTERPRETER
                PUSH_VALUE(val_ptr[i].value);
            }

            cur_pos = STATE_POS(i_state);

            SON_FP(fp[NR_SONS_DONE].arg) = (DATA*) i_transition;
            fp[NR_SONS_DONE].arg++;

            MARK_POS_VISITED(cur_pos);
            i_state = TRANSITION_DEST_STATE_INDICATOR(i_transition,trellis);

            newpc = pc + lex_match_sz;
            label = newpc->action;
            NEXT;
        } else {
            DBRTS(debug_lex_match_fail());
            penleft += pen_gain;                /* restore penalty level */
            i_state = sp[1].input_state;        /* restore input position */
            sp += 2;
            GOTO(pc[2].code);                    /* and fail */
        }
    }

LEX_UMATCH:
    {
        int arity;

        current_terminal = pc[1].arg;   /* get lexicon nont to be matched */

        logexec(lex_umatch);

        fp[NR_SONS_DONE].arg--;
        (DATA*) i_transition = SON_FP(fp[NR_SONS_DONE].arg);

        arity = DECODE_NONT_ARITY(current_terminal);
        if (hybrid_parsing_option) {
            /* restore penalty level */
            penleft += TRANSITION_PENALTY(i_transition);
        }

        sp += arity;                        /* remove previous results from
                                             * the stack
                                             */
        pc = sp[2].code;                    /* restore old pc */
        i_transition = i_transition->next;
        goto LEX_MATCH_AGAIN;               /* try to match next entry */
    }

#else /* GEN_RTS */

LEX_MATCH:
    {
        logexec(lex_match);
        goto ILLEGAL;
    }

LEX_UMATCH:
    {
        logexec(lex_match);
        goto ILLEGAL;
    }
#endif /* GEN_RTS */


/*
 *------------------------------------------------------------------------------
 *
 * Additional flow control instructions (penalty/commit):
 *
 *------------------------------------------------------------------------------
 */

PENALTY:
    {
        ARG arg = pc[1].arg;
        CODE* fail_addr = pc[2].code;

        logexec(penalty);

        penleft -= arg;
        DB(fprintf(stderr, "PENALTY(%ld): penleft <- %ld, penalty_limit = %ld\n", arg, penleft, penalty_limit));

        if (penleft <= penalty_limit) {/* gained too much penalty to continue */
            penleft += arg;
            GOTO(fail_addr);
        }

        newpc = pc + penalty_sz;
        label = newpc->action;
        NEXT;
    }

UPENALTY:
    {
        ARG arg = pc[1].arg;

        logexec(upenalty);

        penleft += arg;
        DB(fprintf(stderr, "UPENALTY(%ld): penleft <- %ld, penalty_limit = %ld\n", arg, penleft, penalty_limit));

        newpc = pc + upenalty_sz;
        label = newpc->action;
        NEXT;
    }

COMMIT:
    {
        logexec(commit);

        if (fp[NR_SUCCS_OFF].arg) {            /* continuation taken (SUCCESS)? */
            GOTO(pc[1].code);		/* skip other alternatives */
        }

        newpc = pc + commit_sz;
        label = newpc->action;
        NEXT;				/* try other alts for lower penlevel */
    }


/*
 *------------------------------------------------------------------------------
 *
 * Instructions for the start rule:
 *
 *------------------------------------------------------------------------------
 */

SUCCESS:
    {
#ifndef GEN_RTS
        unsigned length;
        long nr_pens = LONG_MAX - penleft - 1;

        logexec(success);

#ifdef PMRTS
        assert(!analyse_stage);
#endif /* PMRTS */

        if (penleft <= penalty_limit) {
fprintf(stderr, "skipping parsing (%ld <= %ld)\n", penleft, penalty_limit);
            GOTO(pc[1].code);               /* skip this parsing */
        }

        parse_result.nr_penalties = nr_pens;
        length = STATE_GET_LENGTH(i_state);

        stop_parse_time();
        incr_nr_parses();
        current_parse_set_prio_and_add(parse_result.nr_penalties,
                                       i_state,
                                       length);
        start_print_time();
        start_printing();
#else /* GEN_RTS */
        logexec(success);
        incr_nr_parses();
#endif /* GEN_RTS */

        newpc = pc + success_sz;
        label = newpc->action;
        NEXT;
    }

USUCCESS:
    {
        logexec(usuccess);

#ifndef GEN_RTS
        stop_print_time();
        stop_printing();
        parse_results_start_new();
        start_parse_time();
#endif /* GEN_RTS */

        newpc = pc + usuccess_sz;
        label = newpc->action;
        NEXT;
    }

END:
    {
        StateIndicator new_start_node;

#ifndef GEN_RTS
        stop_parse_time();
        DB(fprintf(stderr, "END: current time is %.3fs.\n", get_parse_time()));
        print_parsed_sentence();
        parse_results_dump();
        show_totals(trellis);

        /* exit interpreter */
        if (!segment_mode) {
            /* Not in paragraph-mode, so stop. */
            new_start_node = NULL;
        } else {
            if (have_time_limit() && have_time_out()) {
                /* we have reached the time limit, see what we can do now...: */

                if (parse_results_get_nr_parses()) {
                    /* we found at least one parsing, so start after that one */
                    new_start_node = get_statenode_from_first_parse();
                    if (new_start_node && state_has_eos_transition(new_start_node)) {
                        /* hey, we're at the end of the sentence, so we can stop! */
                        new_start_node = NULL;
                    }
                } else {
                    /* we didn't even find 1 single parsing, bail out now! */
                    new_start_node = get_shortest_transition(trellis, START_INPUT_STATE);
                }
            } else {
                /* We found a good segment, so start after it the next time: */
                new_start_node = get_statenode_from_first_parse();

                if (!new_start_node) {
                    new_start_node = get_shortest_transition(trellis, START_INPUT_STATE);
                } else if (state_has_eos_transition(new_start_node)) {
                    /* hey, we're at the end of the sentence, so we can stop! */
                    new_start_node = NULL;
                }
            }
        }

        parse_results_destroy();
#else /* GEN_RTS */
#ifdef DEBUG
        choice_admin_dump_table();
#endif
        choice_admin_destroy();
#endif /* GEN_RTS */

        free(stack);
        return new_start_node;
    }

/*
 *------------------------------------------------------------------------------
 * Transduction
 *------------------------------------------------------------------------------
 */

PASS2:
    {
        logexec(pass2);

#ifndef GEN_RTS
        if (no_output) {
            newpc = pc + pass2_sz;
            label = newpc->action;
            NEXT;
        } else {
            sp -= PASS2_FRAME_HEAD_LEN;
            sp[PASS2_PASS2FP].data = 0;
            sp[PASS2_FP].data = fp;
            sp[PASS2_RET_ADDR].code = pc + pass2_sz;
            pass2fp = sp;

            newpc = pc[1].code;
            label = newpc->action;
            NEXT; 
        }
#else /* GEN_RTS */
        gen_output_show();
        newpc = pc + pass2_sz;
        label = newpc->action;
        NEXT;
#endif /* GEN_RTS */
    }

RETRN:
    {
        DATA* new_pass2fp;
        unsigned nr_formals;

        logexec(retrn);
        nr_formals = NR_FORMALS;
        DB(fprintf(stderr, "RETRN: nr_formals = %u\n", nr_formals));

        newpc = pass2fp[PASS2_RET_ADDR].code;
        fp = pass2fp[PASS2_FP].data;
        new_pass2fp = pass2fp[PASS2_PASS2FP].data;
        sp += PASS2_FRAME_HEAD_LEN + nr_formals;
        pass2fp = new_pass2fp;

        label = newpc->action;
        NEXT;
    }

PUSH_F:
    {
        ARG formal = pc[1].arg;
        VALUE formal_value = pass2fp[PASS2_FRAME_HEAD_LEN + formal + 1].val;

        logexec(push_f);
DB(fprintf(stderr, "formal == %ld, value == %lu\n", formal, formal_value.set_par));

        PUSH_VALUE(formal_value);

        newpc = pc + push_f_sz;
        label = newpc->action;
        NEXT;
    }

PUSH_L:
    {
        ARG local = pc[1].arg + NR_FORMALS;

        logexec(push_l);

        PUSH_VALUE(VARIABLE(local).val);
DB(fprintf(stderr, "local == %ld, value == %lu\n", local, VARIABLE(local).val.set_par));
        newpc = pc + push_l_sz;
        label = newpc->action;
        NEXT;
    }

PRINT_SON:
    {
        ARG son_nr = pc[1].arg;

        logexec(print_son);

        sp -= PASS2_FRAME_HEAD_LEN;
        sp[PASS2_PASS2FP].data = pass2fp;
        sp[PASS2_FP].data = fp;
        sp[PASS2_RET_ADDR].code = pc + print_son_sz;

        pass2fp = sp;
        fp = SON_FP(son_nr);

        DB(fprintf(stderr, "printing son %ld: \"%s\"\n", son_nr,
                   nonterm_names[NONT_NR]));
        DB(fprintf(stderr, "jump to 2nd pass code @%p\n", fp[PASS2_CODE].code));

        newpc = fp[PASS2_CODE].code;
        label = newpc->action;
        NEXT;
    }

PRINT_ABEG:
    {
        char* alt_text = pc[1].val.text_par;

        logexec(print_abeg);

        if (label_bracket) {
            if (alt_text && parsing_stats_option) {
                current_parse_printf("{%s:", alt_text);
            } else {
                current_parse_add_char('{');
            }
        } else {
            if (alt_text && parsing_stats_option) {
                current_parse_printf(" <alt %s>\n", alt_text);
            } else {
                current_parse_add_char('\n');
            }

            indent += 2;
        }

        newpc = pc + print_abeg_sz;
        label = newpc->action;
        NEXT;
    }

/*
** PRINT_AEND: begin alternative.
** if labelled bracket output, then print close curly brace;
** decrement indentation otherwise
*/

PRINT_AEND:
    {
        logexec(print_aend);

        if (label_bracket) {
            current_parse_add_char('}');
        } else {
            indent -= 2;
        }

        newpc = pc + print_aend_sz;
        label = newpc->action;
        NEXT;
    }

PRINT_PBEG:
    {
        logexec(print_pbeg);

        current_parse_add_char('(');
        
        newpc = pc + print_pbeg_sz;
        label = newpc->action;
        NEXT;
    }

PRINT_PSEP:
    {
        logexec(print_psep);

        current_parse_add_char(',');
        if (!label_bracket) {
            current_parse_add_char(' ');
        }
        
        newpc = pc + print_psep_sz;
        label = newpc->action;
        NEXT;
    }

PRINT_PEND:
    {
        logexec(print_pend);

        current_parse_add_char(')');
        
        newpc = pc + print_pend_sz;
        label = newpc->action;
        NEXT;
    }


PRINT_VAL:
    {
        char* text = pc[1].val.text_par;

        logexec(print_val);

        if (transduce_option) {
            current_parse_add_string(text);
        } else {
            rtsBug("agfl_interpret", "unexpected PRINT_VAL in tree");
        }

        newpc = pc + print_val_sz;
        label = newpc->action;
        NEXT;
    }

/*
** PRINT_T: print terminal
*/

PRINT_TERM:
    {
        Transition *p_transition = VARIABLE(pc[1].arg).input_transition;
        const char* text = p_transition->text;

        logexec(print_term);

        if (transduce_option) {
            current_parse_add_string(text);
            if (IS_LASTPART(p_transition))
                current_parse_add_char(' ');
        } else {
            int prefix = 0;
            int suffix = 0;

            switch (get_transition_lex_type(p_transition)) {
                case Infix:
                    suffix = 1; 
                case Prefix:
                    prefix = 1;
                    break;
                case Suffix:
                    suffix = 1;
                    break;
                default:
                    break;
            }

            if (!label_bracket) {
                current_parse_add_space(indent);
            }

            if (suffix) {
                current_parse_add_string("\"-");
            } else {
                current_parse_add_char('\"');
            }

            current_parse_add_string(text);

            if (prefix) {
                current_parse_add_string("-\"");
            } else {
                current_parse_add_char('\"');
            }

            if (!label_bracket) {
                current_parse_add_char('\n');
            }
        }

        newpc = pc + print_term_sz;
        label = newpc->action;
        NEXT;
    }

PRINT_LEX:
    {
        char const * text;
        Transition* p_transition = (Transition*) SON_FP(pc[1].arg);

        logexec(print_lex);

        text = p_transition->text;

        if (transduce_option) {
            current_parse_add_string(text);
            if (IS_LASTPART(p_transition)) {
                current_parse_add_char(' ');
            }
        } else {
            gboolean prefix = FALSE;
            gboolean suffix = FALSE;

            switch (get_transition_lex_type(p_transition)) {
                case Infix:
                    suffix = TRUE;
                case Prefix:
                    prefix = TRUE;
                    break;
                case Suffix:
                    suffix = TRUE;
                    break;
                default:
                    break;
            }

            if (!label_bracket) {
                current_parse_add_space(indent);
            }

            if (suffix) {
                current_parse_add_string("\"-");
            } else {
                current_parse_add_char('\"');
            }

            current_parse_add_string(text);

            if (prefix) {
                current_parse_add_string("-\"");
            } else {
                current_parse_add_char('\"');
            }

            if (!label_bracket) {
                current_parse_add_char('\n');
            }
        }

        newpc = pc + print_lex_sz;
        label = newpc->action;
        NEXT;
    }

PRINT_RE:
    {
        const char *text;
        Transition *p_transition;

        logexec(print_re);

        p_transition = VARIABLE(pc[1].arg).input_transition;
        text = p_transition->text;

        if (transduce_option) {
            current_parse_add_string(text);
            if (IS_LASTPART(p_transition)) {
                current_parse_add_char(' ');
            }
        } else {
            if (!label_bracket) {
                current_parse_add_space(indent);
            }

            current_parse_add_char('\"');
            current_parse_add_string(text);

            if (!label_bracket) {
                current_parse_add_string("\"\n");
            } else {
                current_parse_add_char('\"');
            }
        }

        newpc = pc + print_re_sz;
        label = newpc->action;
        NEXT;
    }

PRINT_PEN:
    {
        int pen = pc[1].arg;

        logexec(print_pen);

        if (parsing_stats_option) {
            if (!label_bracket && !transduce_option) {
                current_parse_add_space(indent);

                if (pen == 1) {
                    current_parse_add_string("$PENALTY");
                } else {
                    current_parse_add_string("$PENALTY(");
                    current_parse_add_int(pen);
                    current_parse_add_char(')');
                }

                current_parse_add_char('\n');
            }
        }

        newpc = pc + print_pen_sz;
        label = newpc->action;
        NEXT;
    }

/*
** PRINT_N: print nonterminal name
*/

PRINT_N:
    {
        char* text = pc[1].val.text_par;

        logexec(print_n);

        if (!label_bracket)
            current_parse_add_space(indent);
        current_parse_add_string(text);

        newpc = pc + print_n_sz;
        label = newpc->action;
        NEXT;
    }

PRINT_IF:
    {
        ARG formal = pc[1].arg;
        VALUE val = pass2fp[PASS2_FRAME_HEAD_LEN + formal + 1].val;

        logexec(print_if);

        print_integer_affix(val.int_par, TRUE);

        newpc = pc + print_if_sz;
        label = newpc->action;
        NEXT;
    }

PRINT_IL:
    {
        ARG local = pc[1].arg;
        VALUE val = VARIABLE(NR_FORMALS + local).val;

        logexec(print_il);

        print_integer_affix(val.int_par, TRUE);

        newpc = pc + print_il_sz;
        label = newpc->action;
        NEXT;
    }

PRINT_SF:
    {
        ARG formal = pc[1].arg;
        VALUE val = pass2fp[PASS2_FRAME_HEAD_LEN + formal + 1].val;
        ARG domain = pc[2].arg;

        logexec(print_sf);

        print_set_affix(val.set_par, domain, TRUE);

        newpc = pc + print_sf_sz;
        label = newpc->action;
        NEXT;
    }

PRINT_SL:
    {
        ARG local = pc[1].arg;
        VALUE val = VARIABLE(NR_FORMALS + local).val;
        ARG domain = pc[2].arg;

        logexec(print_sl);

        print_set_affix(val.set_par, domain, TRUE);

        newpc = pc + print_sl_sz;
        label = newpc->action;
        NEXT;
    }

PRINT_TF:
    {
        ARG formal = pc[1].arg;
        VALUE val = pass2fp[PASS2_FRAME_HEAD_LEN + formal + 1].val;

        logexec(print_tf);

        print_text_affix(val.text_par, TRUE);

        newpc = pc + print_tf_sz;
        label = newpc->action;
        NEXT;
    }

PRINT_TL:
    {
        ARG local = pc[1].arg;
        VALUE val = VARIABLE(NR_FORMALS + local).val;

        logexec(print_tl);

        print_text_affix(val.text_par, TRUE);

        newpc = pc + print_tl_sz;
        label = newpc->action;
        NEXT;
    }

PRINT_VF:
    {
        logexec(print_vf);

        assert(!"PRINT_VF not implemented");

        newpc = pc + print_vf_sz;
        label = newpc->action;
        NEXT;
    }

PRINT_VL:
    {
        logexec(print_vl);

        assert(!"PRINT_VL not implemented");

        newpc = pc + print_vl_sz;
        label = newpc->action;
        NEXT;
    }


/*
//------------------------------------------------------------------------------
// Generateve grammar instructions:
//------------------------------------------------------------------------------
*/

CHOICE:
#ifndef GEN_RTS
    {
        logexec(choice);
        goto ILLEGAL;
    }
#else /* GEN_RTS */
#define NR_TRIES_PER_ALTGROUP 1
    {
        ARG choice_nr = pc[1].arg;
        ARG nr_alts = pc[2].arg;

        logexec(choice);
        ALT_DEPTH_CHECK;

        if (get_nr_parses()) {
            /* a sentence is generated, we can stop */
            choice_admin_remove(alt_depth, choice_nr);
            chosen_freq = -2;
        } else if (choice_admin_get_nr_tries(alt_depth, choice_nr) >= (NR_TRIES_PER_ALTGROUP * nr_alts)) {
            /* nr_alts == nr of tries, which all failed, so fail */
            choice_admin_remove(alt_depth, choice_nr);
            chosen_freq = -3;
        } else {
            /* failed to generate a sentence, but not everything is tried */
            PUSH_ADDR(pc);
            chosen_freq = rand() % nr_alts;
//            chosen_freq = choice_admin_get_nr_tries(alt_depth, choice_nr);
            choice_admin_incr_nr_tries(alt_depth, choice_nr);
        }

        newpc = pc + choice_sz;
        label = newpc->action;
        NEXT;
    }
#endif /* GEN_RTS */

UCHOICE:
#ifndef GEN_RTS
    {
        logexec(uchoice);
        goto ILLEGAL;
    }
#else /* GEN_RTS */
    {
        ARG choice_nr = pc[1].arg;

        logexec(uchoice);

        if (((chosen_freq == -2) || (chosen_freq == -3)) && !choice_admin_get_nr_tries(alt_depth, choice_nr)) {
            newpc = pc + uchoice_sz;
            label = newpc->action;
            NEXT;
        } else {
            CODE* choice_addr;
            POP_ADDR(choice_addr);
            assert(choice_addr);
            GOTO(choice_addr);
        }
    }
#endif /* GEN_RTS */

ALT_MARKER:
#ifndef GEN_RTS
    {
        logexec(alt_marker);

        goto ILLEGAL;
    }
#else /* GEN_RTS */
    {
        ARG my_freq = 1;
        CODE* next_alt = pc[1].code;

        logexec(alt_marker);
        ALT_DEPTH_CHECK;

        if ((chosen_freq < my_freq) && (chosen_freq >= 0)) {
            /* we're gonna generate for this alternative */
            chosen_freq = -1;
            ALT_DEPTH_INCREASE;

            newpc = pc + alt_marker_sz;
            label = newpc->action;
            NEXT;
        } else if (chosen_freq >= 0) {
            /* this alternative is not chosen, goto next one */
            chosen_freq -= my_freq;
        } else {
            /* we're done, so skip this alternative */
        }

        newpc = next_alt;
        label = newpc->action;
        NEXT;
    }
#endif /* GEN_RTS */

UALT_MARKER:
#ifndef GEN_RTS
    {
        logexec(ualt_marker);
        goto ILLEGAL;
    }
#else /* GEN_RTS */
    {
        logexec(ualt_marker);
        ALT_DEPTH_CHECK;

        ALT_DEPTH_DECREASE;

        newpc = pc + ualt_marker_sz;
        label = newpc->action;
        NEXT;
    }
#endif /* GEN_RTS */

ROOT_CREATE:
#ifdef PMRTS
    {
        ARG nr_formals = pc[1].arg;
        ARG nr_locals = pc[2].arg;
        ARG nr_sons = pc[3].arg;

        logexec(root_create);
        DBRTS(debug_create(nr_formals, nr_locals, nr_sons));

        sp[0].data = fp;                /* store old fp */
        fp = sp;

        sp -= nr_formals + nr_locals + nr_sons + VAR_OFFSET;
        NR_FORMALS = nr_formals;
        NR_LOCALS = nr_locals;
        NR_SONS = nr_sons;

        START_INPUT_STATE = i_state;
        fp[NR_SUCCS_OFF].arg = 0;
        fp[NR_SONS_DONE].arg = 0;
        fp[PENORIG].penalty = penleft;

        DB(fprintf(stderr, "ROOT_CREATE: start_input_state = %p\n", i_state));

        fp[PMPRODPTR].pmprod = NULL;

        newpc = pc + root_create_sz;
        label = newpc->action;
        NEXT;
    }
#else /* PMRTS */
    {
        goto CREATE;
    }
#endif /* PMRTS */

ROOT_CALL:
#ifdef PMRTS
    {
        ARG rule_nr = pc[1].arg;
        CODE* jmp_point = pc[2].code;
        CODE* fail_addr = pc[3].code;

        logexec(root_call);

        DB(fprintf(stderr, "ROOT_CALL: current time is %.3fs.\n",
                   get_parse_time()));
        if (posmemo_is_unblocked(START_INPUT_STATE, rule_nr) &&
            posmemo_is_known(START_INPUT_STATE, rule_nr)) {
            PosMemo memo;
            PENALTY worst_penalty;
            unsigned nr_prods = posmemo_count_prod(START_INPUT_STATE, rule_nr);
            DB(fprintf(stderr, "after analyse stage: %d entries\n", nr_prods));
            memo = posmemo_get_prod_ptr(START_INPUT_STATE, rule_nr);
            if (segment_mode) {
                worst_penalty = get_longest_worst_penalty(nr_prods, memo);
            } else {
                worst_penalty = get_worst_penalty(nr_prods, memo);
            }
            DB(fprintf(stderr, "get_worst_penalty(...) -> %ld\n",
                       worst_penalty));
            if (worst_penalty < 0) {
                GOTO(fail_addr);
            } else {
                if (worst_penalty != NO_PENALTY) {
                    worst_penalty++;
                }
                penalty_limit = NO_PENALTY - worst_penalty;
            }
            /* Ok, this is dirty, but the rule we have called in the
             * analyse_stage has a stack frame in which it has marked that it
             * should yield memo productions. What we do is forcefully remove
             * the stack frame and (re-) call the rule, but now with
             * analyse_stage=FALSE.
             */
            analyse_stage = FALSE;

            /* restore penleft */
            penleft = NO_PENALTY;

            /* remove the frame: */
            fp[NR_SONS_DONE].arg--;
            sp = SON_FP(fp[NR_SONS_DONE].arg) + FRAME_HEAD_LEN;

            /* set the i_state to the beginning of the input: */
            i_state = START_INPUT_STATE;

            restart_parse_time();
        } else if (posmemo_is_unknown(START_INPUT_STATE, rule_nr)) {
            /* first try */
            posmemo_set_blocked(START_INPUT_STATE, rule_nr);
        }

        sp -= FRAME_HEAD_LEN;
        sp[FAIL_ADDR_OFF].code = fail_addr;
        if (analyse_stage) {
            sp[SUCC_ADDR_OFF].code = pc;
        } else {
            sp[SUCC_ADDR_OFF].code = pc + root_call_sz;
        }

        SON_FP(fp[NR_SONS_DONE].arg) = sp;
        fp[NR_SONS_DONE].arg++;

        sp[NONT_NR_OFF].arg = rule_nr;

        DB(fprintf(stderr, "Starting parsing with penleft = %ld, penalty_limit = %ld, diff = %ld\n", penleft, penalty_limit, NO_PENALTY - penalty_limit));

        newpc = jmp_point;
        label = newpc->action;
        NEXT;
    }
#else /* PMRTS */
    {
        logexec(root_call);

        goto ILLEGAL;
    }
#endif /* PMRTS */

NO_LREC_MARKER:
    {
        logexec(no_lrec_marker);

        goto ILLEGAL;
    }

LREC_MARKER:
    {
        logexec(lrec_marker);

        goto ILLEGAL;
    }

#ifdef DEBUGRTS
TRACE:
    {
        debug_set_trace_data(sp[1].data);

        newpc = pc + trace_sz;
        label = newpc->action;
        NEXT;
    }
#else /* DEBUGRTS */
TRACE:
    {
        logexec(trace);

        goto ILLEGAL;
    }
#endif /* DEBUGRTS */


/*
//------------------------------------------------------------------------------
// Illegal instructions
//------------------------------------------------------------------------------
*/

ILLEGAL:
    {
        rtsBug("agfl_interpret", "rts: illegal instruction");
    }

        free(stack);
  return NULL;
}

/*
//------------------------------------------------------------------------------
// Parsing interface
//------------------------------------------------------------------------------
*/

static void
reset_parser(Trellis* trellis)
{
    reset_parse_result();
    parse_results_init();
    parse_results_start_new();
#ifndef GEN_RTS
    clear_visited_admin(trellis);
#endif /* GEN_RTS */
}

StateIndicator
parse_trellis(Trellis* trellis, StateIndicator start_node)
{
    reset_parser(trellis);

    if (have_time_limit()) {
#ifdef DEBUG_TIME_OUT
        fprintf(stderr, "setting timeout to %ld secs\n", max_parsetime);
#endif

        set_time_limit();
    }

#ifdef DEBUG_NEGMEMO
    if (neg_memo_option || directors_option) {
        print_negmemo_table(trellis, FALSE);
    }
#endif /* DEBUG_NEGMEMO */

    return agfl_interpret(Run, trellis, start_node);
}

/*
//------------------------------------------------------------------------------
// Prelude and postlude of module
//------------------------------------------------------------------------------
*/

void
init_parser()
{
#ifdef PMRTS
    posmemo_init();
#endif /* PMRTS */
    agfl_interpret(Translate, NULL, NULL);
}

void
end_parser()
{
#ifdef PMRTS
    posmemo_done();
#endif /* PMRTS */
}
