/*
   File: lxcn_input.c
   Defines the various lexeme types and holds the character tables while lexing

   Copyright 2009 Radboud University of Nijmegen
 
   This program is free software: you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation, either version 3 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program.  If not, see <http://www.gnu.org/licenses/>.

   CVS ID: "$Id$
*/
#ifdef HAVE_CONFIG_H
#include <config.h>
#endif /* HAVE_CONFIG_H */

/* standard includes */
#include <stdio.h>
#include <ctype.h>
#include <string.h>

/* abase includes */
#include <abase_error.h>

/* local includes */
#include "lxcn_input.h"

/*------------------------------------------------------------------------------
// Global data
//----------------------------------------------------------------------------*/
static char white_space_table[256];
static char translate_table[256];
static int translate_penalty_table[256];

/*------------------------------------------------------------------------------
// Table functions
//----------------------------------------------------------------------------*/
static void set_table (char *table, char *str)
{ int ix;
  char c;
  if (str == NULL) abs_abort ("set_table", "called with NULL string");

  for (ix = 0; ix < 256; ix++) table[ix] = '\0';
  while ((c = *str++)) table[(int) ((unsigned char) c)] = c;
}

static void set_trans_table (char* table, int *pen_table, char* src, char* dst, int *penalties)
{ int ix;
  char c;
  if (src == NULL || dst == NULL)
    abs_abort ("set_trans_table", "called with NULL string");

  for (ix = 0; ix < 256; ix++) 
    { table[ix] = (char) ix;
      pen_table[ix] = 0;
    }

  while ((c = *src++))
    { table[(int) ((unsigned char) c)] = *dst++;
      if (penalties != NULL)
        pen_table[(int) ((unsigned char) c)] = *penalties++;
    }
}

/*------------------------------------------------------------------------------
// Token types and markers
//----------------------------------------------------------------------------*/

int lxcn_marker_from_lex_type (LexemeType lex_type)
{ switch (lex_type)
    { case Prefix:	 return (LexemePrefixBit);
      case Infix:	 return (LexemePrefixBit | LexemeSuffixBit);
      case Suffix:	 return (LexemeSuffixBit);
      case MultiToken:	 return (LexemeMultiWordBit);
      default:		 return (0);
    };
}

LexemeType lxcn_lex_type_from_marker (int marker)
{ switch (marker & 7)
    { case LexemePrefixBit: return (Prefix);
      case LexemeSuffixBit: return (Suffix);
      case LexemePrefixBit | LexemeSuffixBit: return (Infix);
      case LexemeMultiWordBit: return (MultiToken);
      default: return (SingleToken);
    };
}

void lxcn_print_lexeme (char *lexeme, LexemeType type)
{ char *ptr = lexeme;
  int marker = lxcn_marker_from_lex_type (type);
  int hyph_end = (marker & LexemePrefixBit);
  int hyph_beg = (marker & LexemeSuffixBit);
  if (hyph_beg) abs_printf ("-");
  while (*ptr)
    { switch (*ptr)
	{ case '\n': abs_printf ("\\n"); break;
	  case '\r': abs_printf ("\\r"); break;
	  case '\t': abs_printf ("\\t"); break;
	  case '\\': abs_printf ("\\\\"); break;
	  /* case '-': abs_printf ("\\-"); break; soft hyphen is @ anyway */
	  default:
	    if (('\0' < *ptr) && (*ptr < ' ')) abs_printf ("\\%03o", (unsigned) (*ptr));
	    else abs_printf ("%c", *ptr);
	};
      ptr++;
    };
  if (hyph_end) abs_printf ("-");
}

int lxcn_is_eos (char c)
{ return (c == 0);
}

int lxcn_is_white_space (char c)
{ return (white_space_table[(int) ((unsigned char) c)] != '\0');
}

char lxcn_translate (char c)
{ return (translate_table[(int) ((unsigned char) c)]);
}

int lxcn_translate_penalty (char c)
{ return (translate_penalty_table[(int) ((unsigned char) c)]);
}

void lxcn_init_char_tables (char *white_space,
			    char *translate_src, char *translate_dst,
			    int *translate_penalties)
{ set_table (white_space_table, white_space);
  set_trans_table (translate_table, translate_penalty_table,
		   translate_src, translate_dst, translate_penalties);
}
