/*
   File: options.c
   Parses options to agfl-lexgen

   Copyright 2009-2010 Radboud University of Nijmegen

   This program is free software: you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation, either version 3 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program.  If not, see <http://www.gnu.org/licenses/>.

   CVS ID: "$Id$"
*/

/* system includes */
#include <stdio.h>
#include <string.h>

/* libabase includes */
#include <abase_error.h>
#include <abase_version.h>
#include <abase_dstring.h>
#include <abase_memalloc.h>

/* Local includes */
#include "options.h"

/* exported options */
static int show_version;
int affix_hash_size;
int param_hash_size;
int fact_hash_size;
int lexgen_forced;
int generate_info;
int generate_dump;
int verbose;

/* exported pragmats from lif file */
int hyphen_convention_active;
int hybrid_parsing;
char *encoding;

/* exported file names */
char *lexicon_fname;
char *dump_fname;
char *info_fname;
char *basename;

static void init_options ()
{ show_version = 0;
  affix_hash_size = 65536;
  param_hash_size = 65536;
  fact_hash_size = 65536;
  lexgen_forced = 0;
  generate_info = 0;
  generate_dump = 0;
  verbose = 0;
  hyphen_convention_active = 0;
  hybrid_parsing = 0;
  encoding = NULL;
  lexicon_fname = NULL;
  dump_fname = NULL;
  info_fname = NULL;
  basename = NULL;
};

static void print_usage (int rh)
{ abs_message ("usage: agfl-lexgen [flags] grammarname [more_flags]");
  abs_message ("where flags and more_flags may be any of the following:");
  abs_message ("-rh:      provide research help");
  abs_message ("-h:       provide this help");
  abs_message ("-V:       show version");
  abs_message ("-v:       verbose");
  abs_message ("-f:       force lexicon generation");
  abs_message ("-gi:      generate lexicon information");
  abs_message ("-gd:      generate lexicon dump");
  abs_message ("-i ifile: output lexicon information in file 'ifile'");
  abs_message ("-d dfile: output lexicon dump in file 'dfile'");
  abs_message ("-o ofile: output lexicon in file 'ofile'");
  if (!rh) abs_exit (4);
  abs_message ("-ahs sz:  specify affix hash size");
  abs_message ("-phs sz:  specify param hash size");
  abs_message ("-fsh sz:  specify fact hash size");
  abs_exit (4);
};

static void syntax_error (char *syn_error)
{ abs_error (syn_error);
  print_usage (0);
};

#define streq(s1,s2) (strcmp((s1),(s2)) == 0)
static void scan_option (char *ptr, int *i, int argc, char **argv)
{ if (streq (ptr, "V")) show_version = 1;
  else if (streq (ptr, "v")) { show_version = 1; verbose++; }
  else if (streq (ptr, "h")) print_usage (0);
  else if (streq (ptr, "f")) lexgen_forced = 1;
  else if (streq (ptr, "rh")) print_usage (1);
  else if (streq (ptr, "gd")) generate_dump = 1;
  else if (streq (ptr, "gi")) generate_info = 1;
  else if (streq (ptr, "ahs"))
    { *i = *i + 1;
      if ((*i) < argc) affix_hash_size = atoi(argv[*i]);
      else syntax_error ("missing affix hash size");
    }
  else if (streq (ptr, "phs"))
    { *i = *i + 1;
      if ((*i) < argc) param_hash_size = atoi(argv[*i]);
      else syntax_error ("missing param hash size");
    }
  else if (streq (ptr, "fhs"))
    { *i = *i + 1;
      if ((*i) < argc) fact_hash_size = atoi(argv[*i]);
      else syntax_error ("missing fact hash size");
    }
  else if (streq (ptr, "i"))
    { *i = *i + 1;
      if ((*i) < argc)
	{ info_fname = abs_new_string (argv[*i], "scan_option");
	  generate_info = 1;
	}
      else syntax_error ("missing lexicon info file name");
    }
  else if (streq (ptr, "d"))
    { *i = *i + 1;
      if ((*i) < argc)
	{ dump_fname = abs_new_string (argv[*i], "scan_option");
	  generate_dump = 1;
	}
      else syntax_error ("missing lexicon info file name");
    }
  else if (streq (ptr, "o"))
    { *i = *i + 1;
      if ((*i) < argc) { lexicon_fname = abs_new_string (argv[*i], "scan_option"); }
      else syntax_error ("missing lexicon file name");
    }
  else
    { abs_error ("illegal command line option '-%s'", ptr);
      print_usage (0);
    };
};

static void try_report_version ()
{ if (!show_version) return;
  abs_message ("Agfl-lexgen, C Version %s, Copyright 2010, Radboud University of Nijmegen.",
	       AGFL_VERSION);
};

/* exported actions */
#define LEXICON_SUFFIX "blx"
#define INFO_SUFFIX "dlx"
#define DUMP_SUFFIX "dmp"
void parse_command_line (int argc, char **argv)
{ int ix;
  init_options ();
  for (ix = 1; ix < argc; ix++)
     { char *arg = argv[ix];
       if (arg[0] == '-') scan_option (arg + 1, &ix, argc, argv);
       else if (basename == NULL)
	 basename = abs_new_string (arg, "parse_command_line");
       else syntax_error ("too many source file names");
     };
  try_report_version ();

  /* Exit if no grammar name given */
  if (basename == NULL)
    { if (!show_version)
        abs_message ("no grammar name given, bailing out...");
      exit (0);
    };

  /* Try set info fname from grammar name */
  if (info_fname == NULL)
    info_fname = abs_new_fmtd_string ("parse_command_line", "%s.%s", basename, INFO_SUFFIX);

  /* Try set dump fname from grammar name */
  if (dump_fname == NULL)
    dump_fname = abs_new_fmtd_string ("parse_command_line", "%s.%s", basename, DUMP_SUFFIX);

  /* Try set lexicon output fname from grammar name */
  if (lexicon_fname == NULL)
    lexicon_fname = abs_new_fmtd_string ("parse_command_line", "%s.%s", basename, LEXICON_SUFFIX);
};
