/*
   File: arts_tracer.c
   Defines the tracer of the grammar.

   Copyright 2010 Radboud University of Nijmegen
 
   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.
 
   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU Library General Public License for more details.
 
   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.

*/

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif /* HAVE_CONFIG_H */

/* standard includes */
#include <stdio.h>
#include <stdlib.h>
#include <limits.h>
#include <assert.h>
#include <sys/types.h>
#include <string.h>

/* libabase includes */
#include <abase_error.h>
#include <abase_memalloc.h>
#include <abase_mm_alloc.h>

/* libarts includes */
#include "arts_ds.h"
#include "arts_interpreter.h"

#include "arts_tracer.h"

/*------------------------------------------------------------------------------
// Parsing call stack, for diagnostics.
//----------------------------------------------------------------------------*/
#define PARSE_CALL_STACKSIZE	2048

static int parse_call_stack[PARSE_CALL_STACKSIZE];
static int parse_call_stack_pointer;
static int saved_parse_call_stack[PARSE_CALL_STACKSIZE];
static int saved_parse_call_stack_pointer;

void clear_parse_stack()
{
    parse_call_stack_pointer = 0;
}

void push_parse_call(int ruleno)
{ if (parse_call_stack_pointer < PARSE_CALL_STACKSIZE-1)
    parse_call_stack[parse_call_stack_pointer++] = ruleno;
}

void pop_parse_call()
{ if (parse_call_stack_pointer > 0)
    parse_call_stack_pointer--;
}

void save_parse_call_stack()
{ int i;
  saved_parse_call_stack_pointer = parse_call_stack_pointer;
  for (i = 0; i < parse_call_stack_pointer; i++)
    saved_parse_call_stack[i] = parse_call_stack[i];
}

void print_parse_call_stack (State si)
{ int i;
  StateExtension e = NULL;

  if (si) e = GetStateExtension(si);
  for (i = 0; i < parse_call_stack_pointer; i++)
    { int nont = parse_call_stack[i];
      if (nont < 0) abs_printf (">%d ", nont);
      else abs_printf (">%s ", arts_ifd.nonterm_names[nont].str);
    }
}

void print_saved_parse_call_stack()
{ int i;

  for (i = 0; i < saved_parse_call_stack_pointer; i++)
    abs_printf(">%s ", arts_ifd.nonterm_names[saved_parse_call_stack[i]].str);
}

#if ENABLE_INTERACTIVE_TRACER

/*------------------------------------------------------------------------------
// Interactive grammar tracing
//----------------------------------------------------------------------------*/
static FILE *terminal;
static int trace_enabled = 1;

static void tracer_init()
{
    if (terminal == NULL) {
	terminal = fopen("/dev/tty", "r+");
    }
}

/* XXX copied from arts_interpreter.c */

#define NONT_NR_OFF		(-1)
#define NR_FORMALS_OFF		(-2)
#define NONT_NR			(fp[NONT_NR_OFF].arg)
#define NR_FORMALS		(fp[NR_FORMALS_OFF].arg)

int tracer_start_alt(int altnr, DATA *fp, Trellis trellis, State i_state)
{
    int rulenr;
    char *rulename;
    int fail = 0;
    static int NO_rule = -1;

    if (!trace_enabled)
	return 0;

    tracer_init();

    rulenr = NONT_NR;
    rulename = arts_ifd.nonterm_names[rulenr].str;
    
    if (NO_rule >= 0 && rulenr == NO_rule) {
	fail = 1;
    } else {
	NO_rule = -1;

	print_parse_call_stack(i_state);
	abs_printf("\n");
	fprintf(stderr, "> start alternative #%d (%s/%ld)\n", altnr, rulename, NR_FORMALS);
	print_state_transitions(trellis, i_state);
	fflush(NULL);

	fprintf(stderr, "Enter? (y/n/N) ");
	fflush(stderr);

	for (;;) {
	    int ch = fgetc(terminal);
	    if (ch == 'y') {
		fail = 0;
	    } else if (ch == 'n') {
		fail = 1;
	    } else if (ch == 'N') {
		fail = 1;
		NO_rule = rulenr;
		fprintf(stderr, "(skipping remaining alternatives of '%s'...)\n", rulename);
	    } else if (ch == '\n') {
		break;
	    } else if (ch == EOF) {
		trace_enabled = 0;
		break;
	    }
	}
    }

    return fail;
}

void tracer_end_alt(int altnr, DATA *fp, Trellis trellis, State i_state)
{
    if (!trace_enabled)
	return;

    abs_message("<   end alternative #%d (%s/%d)", altnr, arts_ifd.nonterm_names[NONT_NR].str, NR_FORMALS);
}

#endif /* ENABLE_INTERACTIVE_TRACER */
