/*
   File: arts_posmemo.c
   Positive memoization routines.

   Copyright 2005-2010 Radboud University of Nijmegen
 
   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.
 
   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU Library General Public License for more details.
 
   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.

   CVS ID: "$Id$"
*/

/* standard includes */
#ifdef HAVE_CONFIG_H
#include <config.h>
#endif /* HAVE_CONFIG_H */
#include <stdio.h>
#include <string.h>

/* libtrellis includes */
#include <trel_input.h>

/* libabase includes */
#include <abase_error.h>
#include <abase_memalloc.h>

/* libarts includes */
#include "arts_ds.h"
#include "arts_io.h"
#include "arts_lexer.h" 
#include "arts_posmemo.h"
#include "arts_interpreter.h"
#include "arts_hybrid.h"

#ifdef DEBUG_POSMEMO
#define DB_PM(x) x
#else
#define DB_PM(x)
#endif
#define LOOK_AT_HYBRID		1	/* off, because very slow due to
					   combinatorial explosion of
					   "onderscheidbare" PosMemos */

int posmemo_known_count;
int posmemo_unknown_count;
int posmemo_blocked_count;

/*
   Memory management code for posmemo
   Actual management has not yet been implemented
*/
static void* posmemo_getmem (size_t s)
{ return (abs_malloc (s, "posmemo_getmem"));
}

static void posmemo_freemem (void* ptr)
{ abs_free (ptr, "posmemo_freemem");
}

PosMemo posmemo_freelist;

inline
static PosMemo posmemo_from_freelist(void)
{
    if (posmemo_freelist != NULL) {
	PosMemo result = posmemo_freelist;
	posmemo_freelist = posmemo_freelist->next;
	return result;
    } else {
	return abs_malloc (sizeof(PMPROD), "posmemo_from_freelist");
    }
}

inline
static void posmemo_to_freelist(PosMemo pm)
{
    pm->next = posmemo_freelist;
    posmemo_freelist = pm;
}

static void free_posmemo_freelist()
{
    PosMemo first = posmemo_freelist;
    while (first != NULL) {
	PosMemo next = posmemo_freelist->next;
	abs_free(posmemo_freelist, "free_posmemo_freelist");
	first = next;
    }
    posmemo_freelist = NULL;
}

void posmemo_init()
{ /* init memory management, not implemented yet */
  posmemo_freelist = NULL;
  posmemo_known_count = 0;
  posmemo_unknown_count = 0;
  posmemo_blocked_count = 0;
}

void posmemo_done()
{ /* stop memory management, not implemented yet */
  free_posmemo_freelist();
  if (posmemo_known_count != 0)
    abs_message("posmemo_known_count = %d (should be 0)", posmemo_known_count);
  if (posmemo_unknown_count != 0)
    abs_message("posmemo_unknown_count = %d", posmemo_unknown_count);
  if (posmemo_blocked_count != 0)
    abs_message("posmemo_blocked_count = %d (should be 0)", posmemo_blocked_count);
}

/*
   Define access function for posmemo structures
   For each position in the input trellis a positive memo for each
   syntax nonterminal is maintained (using a vector indexed by nont nr).

   Two special denotations are used to denote the unknown and blocked
   positive memo. 
*/

/*
 * The table to map nonterminal numbers to nonterminal classes
 * (== lexicon terminal numbers, starting at 1)
 * had lots of free entries (viz. all rule numbers) where we can
 * store the rule type.
 *
 */

static int rule_is_terminal(int n)
{ int64 typebits = arts_ifd.lex_nont_nrs_table[n].ilval;
  return IS_TERMINAL(typebits) || IS_LEX(typebits);
}

/*
 * For position-independent rules, fetch a shared pointer instead
 * of one tied to the StateExtension. If the position-dependent
 * pointer is BLOCKED, fetch that one though, since that is locally
 * available information that improves efficiency.
 * (so far I haven't seen that happen, though)
 */

static PosMemo* posmemo_fetch (StateExtension i, int n)
{ assert(i);
  return (&(i->pos_memos[n]));
}

void posmemo_init_table_entry (PosMemo* entry)
{ *entry = POSMEMO_UNKNOWN;
  posmemo_unknown_count++;
}

int posmemo_is_unknown (StateExtension input_state, int nont_nr)
{ PosMemo *x = posmemo_fetch (input_state, nont_nr);
  return ((x != NULL) && ((*x) == POSMEMO_UNKNOWN));
}

int posmemo_is_known (StateExtension input_state, int nont_nr)
{ PosMemo *x = posmemo_fetch (input_state, nont_nr);
  return ((*x) != POSMEMO_UNKNOWN);
}

int posmemo_is_blocked (StateExtension input_state, int nont_nr)
{ PosMemo *x = posmemo_fetch (input_state, nont_nr);
  return ((*x) == POSMEMO_BLOCKED);
}

int posmemo_is_unblocked (StateExtension input_state, int nont_nr)
{ PosMemo *x = posmemo_fetch(input_state, nont_nr);
  return (((*x) != POSMEMO_BLOCKED) && ((*x) != POSMEMO_UNKNOWN));
}

void posmemo_set_blocked (StateExtension input_state, int nont_nr)
{ PosMemo *x = posmemo_fetch (input_state, nont_nr);
  if ((*x) != POSMEMO_UNKNOWN)
    abs_bug ("posmemo_set_blocked", "Posmemo is not unknown");
  *x = POSMEMO_BLOCKED;
  posmemo_unknown_count--;
  posmemo_blocked_count++;
}

/* Some abstraction of the access to the type bits */

#define NR_SONTYPE_BITS		(CHAR_BIT*sizeof(int))

#define GET_TYPE_BITS(posmemo)	\
&(posmemo -> variables [posmemo -> nr_variables + posmemo -> nr_sons].int_par)
#define SET_SON_AS_LEX(type_bits, ix) \
	(type_bits[(ix)/NR_SONTYPE_BITS] |= (1 << ((ix)%NR_SONTYPE_BITS)))
#define SON_IS_LEX(type_bits, ix) \
	(type_bits[(ix)/NR_SONTYPE_BITS] &  (1 << ((ix)%NR_SONTYPE_BITS)))

/*
   Dumping of posmemo.
*/
static void posmemo_dump_pmprod (PosMemo pm_prod)
{ int nr_formals = pm_prod -> nr_formals;
  int nr_variables = pm_prod -> nr_variables;
  int nr_sons = pm_prod -> nr_sons;
  int *type_bits = GET_TYPE_BITS(pm_prod);
  int nont_nr = pm_prod -> nont_nr;
  DATA *fdomain;
  int ix;

#if DEBUG_POSMEMO
  abs_printf ("PM %p: ", pm_prod);
#endif
  abs_printf ("-%3d: ",  trel_state_get_pos(pm_prod -> next_state));
  abs_printf ("%s/%d", arts_ifd.nonterm_names[nont_nr].str, nr_formals);

  if (nr_formals > 0)
  { abs_printf("(");
    fdomain = arts_ifd.nont_formals_domains[nont_nr].data;
    for (ix = 0; ix < nr_formals; ix++)
      { if (ix > 0) abs_printf(", ");
#if DEBUG_POSMEMO
        abs_printf("0x%llx=", (pm_prod -> variables)[ix].set_par);
#endif
        print_affix((pm_prod -> variables)[ix], (int) fdomain->ilval);
	fdomain++;
      }
    abs_printf(")");
  }

  abs_printf (" (%d) penalty=%d %c ", nont_nr, pm_prod -> penalty,
	      (pm_prod -> ambig != NULL)?'A':'N');
#if DEBUG_POSMEMO
  abs_printf (" -> ns=%p ", pm_prod -> next_state);
#endif
  for (ix = 0; ix < nr_sons; ix++)
    { PMPROD *son = (PMPROD *)(pm_prod -> variables[nr_variables + ix].ptr_par);
#if DEBUG_POSMEMO
      abs_printf("{%p} ", son);
#endif
      if (!SON_IS_LEX(type_bits, ix))
	abs_printf ("%d:%d ", son -> nont_nr, trel_state_get_pos(son -> next_state));
      else
	{ Transition trans = (Transition) son;
	  TransitionExtension ext = GetTransitionExtension(trans);
	  if (TERM_IS_FACT (ext -> terminal))
	    { abs_printf ("FACT ");
#if DEBUG_POSMEMO
	      abs_printf ("%p ", trans);
#endif
	      abs_printf ("%s", arts_ifd.nonterm_names[
				DECODE_NONT_NUMBER(ext->terminal)].str);
	    } else if (TERM_IS_NONT (ext -> terminal))
	    { abs_printf ("LEX ");
#if DEBUG_POSMEMO
	      abs_printf ("%p ", trans);
#endif
	      abs_printf ("%s \"%s\"", arts_ifd.nonterm_names[
				       DECODE_NONT_NUMBER(ext->terminal)].str,
				       trel_trans_get_text(trans));
	    } else
	    { abs_printf ("TRANS %p", trans);	/* should not happen */
	    }
	  abs_printf (":%d ", trel_state_get_pos(trel_trans_get_dest(trans)));
	};
    };
  if (pm_prod -> hybrid != NULL)
  { abs_printf(" / ");
    log_hyb_anchor(pm_prod -> hybrid);
  }
  abs_printf("\n");

}

void posmemo_print_tree2 (PosMemo pm_prod, int indent)
{ int nr_formals = pm_prod -> nr_formals;
  int nr_sons = pm_prod -> nr_sons;
  int nont_nr = pm_prod -> nont_nr;
  int *type_bits = GET_TYPE_BITS(pm_prod);
  DATA *fdomain;
  int ix, iy;

  /* Check for ambiguity: we may pass this point more than once */
  if (pm_prod -> prime != NULL)
    { PosMemo sibling;
      assert (pm_prod -> prime == pm_prod);
      if (try_lookup_sibling_in_ambi_stack (pm_prod, &sibling)) pm_prod = sibling;
      else push_ambi_stack (pm_prod -> prime);
    };

  for (iy = 0; iy < indent; iy++) abs_printf (". ");
  abs_printf ("%s", arts_ifd.nonterm_names[nont_nr].str);
  if (nr_formals > 0) {
    abs_printf("(");
    fdomain = arts_ifd.nont_formals_domains[nont_nr].data;
    for (ix = 0; ix < nr_formals; ix++)
      { if (ix > 0) abs_printf(", ");
	print_affix ((pm_prod -> variables)[ix], (int) fdomain->ilval);
	fdomain++;
      };
    abs_printf(")");
  }

  /* print terminal text, if applicable */
  abs_printf(" [%d]", pm_prod -> penalty);
  abs_printf("\n");

  for (ix = 0; ix < nr_sons; ++ix)
    { PMPROD *son = posmemo_get_son (pm_prod, ix);
      
      if (son == POSMEMO_UNKNOWN) {
	  abs_printf("son is POSMEMO_UNKNOWN\n");
	  continue;
      }

      if (SON_IS_LEX(type_bits, ix))
	{ Transition trans = (Transition) son;

          for (iy = -1; iy < indent; iy++) abs_printf ("  ");
	  print_transition (trans);
	  abs_printf ("\n");
	}
      else posmemo_print_tree2 (son, indent + 1);
    };
}

void posmemo_print_tree (PosMemo pm_prod)
{
    posmemo_print_tree2 (pm_prod, 0);
}

void posmemo_rdump_pmprod (PosMemo pm_prod, int indent)
{ int nr_formals = pm_prod -> nr_formals;
  int nr_variables = pm_prod -> nr_variables;
  int nr_sons = pm_prod -> nr_sons;
  int nont_nr = pm_prod -> nont_nr;
  int *type_bits = GET_TYPE_BITS(pm_prod);
  DATA *fdomain;
  int ix, iy;

  if (pm_prod == NULL)
    { abs_message ("PosMemo is NULL");
      return;
    }

  for (iy = 0; iy < indent; iy++) abs_printf (" ");
  abs_message (">PM %p: %s/%d (%d) penalty=%d -> ns=%p (pos %d) refcount=%d, %c",
	       pm_prod, arts_ifd.nonterm_names[nont_nr].str,
	       nr_formals, nont_nr, pm_prod -> penalty, pm_prod -> next_state, 
	       trel_state_get_pos(pm_prod -> next_state),
	       pm_prod -> refcount, (pm_prod -> prime != NULL)? 'P': 'N');
  fdomain = arts_ifd.nont_formals_domains[nont_nr].data;
  for (ix = 0; ix < nr_formals; ix++)
    { for (iy = 0; iy < indent; iy++) abs_printf (" ");
      abs_printf (">F%d: 0x%llx ", ix, (pm_prod -> variables)[ix].set_par);
      print_affix ((pm_prod -> variables)[ix], (int) fdomain->ilval);
      fdomain++;
      abs_printf ("\n");
    };
  for (ix = 0; ix < nr_sons; ++ix)
    { PMPROD *son = (PMPROD *)(pm_prod -> variables[nr_variables + ix].ptr_par);
      if (SON_IS_LEX(type_bits, ix))
	{ Transition trans;
	  TransitionExtension ext;
	  int nont_nr;
	  int arity;

	  trans = (Transition) son;
	  ext = GetTransitionExtension(trans);
	  nont_nr = DECODE_NONT_NUMBER (ext -> terminal);
	  arity = DECODE_NONT_ARITY (ext -> terminal);

          for (iy = 0; iy < indent + 3; iy++) abs_printf (" ");
	  if (TERM_IS_FACT (ext -> terminal))
	    abs_message (">FACT_%d %p %s/%d",
		    nont_nr,
		    trans,
		    arts_ifd.nonterm_names[DECODE_NONT_NUMBER(ext->terminal)].str,
		    arity);
	  else abs_message (">LEX_%d %p: %s/%d \"%s\"",
		  nont_nr,
		  son,
		  arts_ifd.nonterm_names[DECODE_NONT_NUMBER(ext->terminal)].str,
		  arity,
		  trel_trans_get_text(trans));
	}
      else posmemo_rdump_pmprod (son, indent + 3);
    };
}

/*
   The following code is the crux of the positive memoization.
   We enter this code when a positive memo has been constructed
   for this nonterminal, formals, penalty and next input position.

   The posmemo list is kept sorted by penalty. Ambiguous parses (having
   the same penalty) are indicated by being a member of the "ambig"
   list. Each of the ambiguous parses points to the first recognized
   equivalent parse with the prime pointer, and they all point to the
   same "next" posmemo. (These extra references are not counted in
   the refcount of the "next")

   While adding, a bound may be set to limit the number of parses;
   so while traversing we keep track of the number of parses with
   the same nonterminal, formals and next input position (in effect
   all parses with a different penalty).
*/

/*
   Two productions are equivalent if they have the same formals and next_state
   (They recognize the same span of input with the same resulting formals).
   Having the same number of formals is also a requirement, but it is
   implicitly met because the memos are of the same rule.

   If they also have the same penalty, they are ambiguous.
   Note that if they do not have the same penalty, they can not be ambiguous.

   It is also the single most-called function in the program.
   The inlined special-case copy of memcmp() is measurably faster
   than the library call, and there is no gcc-trickery needed to
   get an inlined version.
*/
static int equivalent_posmemo (PosMemo curr, PosMemo new_prod)
{ /* if (curr -> nr_formals != new_prod -> nr_formals) return 0; */
  if (curr -> next_state != new_prod -> next_state) return 0;
  {
    int n = new_prod -> nr_formals;
    Value *a = curr -> variables;
    Value *b = new_prod -> variables;

    while (n > 0) {
	if (a->set_par != b->set_par)	/* XXX assumes ->set_par is the largest member */
	    return 0;
	n--;
	a++;
	b++;
    }
  }
#if LOOK_AT_HYBRID
  if (arts_ifd.hash_production)
  { if (!eqv_hyb_anchor (curr -> hybrid, new_prod -> hybrid)) return 0;
  }
#endif /* LOOK_AT_HYBRID */
  /* kees likes to call this "on-onderscheidbaar" */
 return 1;
}

/*
   Two productions are completely equal if they have the same formals,
   locals, sons, next_state and penalty. Implicitly they also have the
   same initial state, since they are attached to the StateNode.
   That still leaves the possibility of 2 alternatives of a rule that
   are identical apart from their transduction.
   Therefore add ->pass2 to the test, which is basically unique for all
   alternatives in the grammar. (makes the ->nr_sons test unneeded)

   When absorb_equivalents is set, equivalent parsings are treated as equal parsings
*/
static int equal_posmemo (PosMemo curr, PosMemo new_prod)
{ if (!equivalent_posmemo (curr, new_prod)) return (0);
  if (arts_ifd.absorb_equivalents) return (1);

  return (curr -> pass2 == new_prod -> pass2 &&
	  (curr -> nr_sons == new_prod -> nr_sons) &&
	  (memcmp (curr -> variables, new_prod -> variables,
		   (new_prod -> nr_variables + new_prod -> nr_sons) * sizeof (Value)) == 0));
}

static void attach_posmemo (PosMemo memo, char *comment)
{
#if DEBUG_POSMEMO
  abs_message ("Attaching %s posmemo %p", comment, memo);
#endif
  memo -> refcount++;
}

static void delete_posmemo (PosMemo memo, char *comment);
void detach_posmemo (PosMemo memo, char *comment)
{
#if DEBUG_POSMEMO
  abs_message ("Detaching %s posmemo %p", comment, memo);
#endif
  if (--memo -> refcount <= 0) {
      delete_posmemo(memo, comment);
  }
}

static void really_delete_posmemo (PosMemo memo)
{ if (memo -> variables) posmemo_freemem (memo -> variables);
#if DEBUG_POSMEMO
    /*
     * Assumes incremental trellis/posmemo cleanup; not wholesale, which
     * simply ignores refcounts and recursion
     */
  assert(memo->refcount == 0);
#endif
  detach_hyb_anchor (memo -> hybrid, memo);
  //posmemo_freemem (memo);
  posmemo_to_freelist(memo); /* Put on freelist */

  posmemo_known_count--;
}

static void delete_posmemo (PosMemo memo, char *comment)
{ int *type_bits = GET_TYPE_BITS(memo);
  int nr_variables = memo -> nr_variables;
  int ix;
  DB_PM(abs_message ("Deleting %s posmemo %p", comment, memo);)
  for (ix = 0; ix < memo -> nr_sons; ++ix)
    { if (!SON_IS_LEX(type_bits, ix))
      { PosMemo son = (PosMemo)(memo -> variables[nr_variables + ix].ptr_par);
	detach_posmemo(son, "recursive");
      }
    };
  really_delete_posmemo (memo);
}

static PosMemo posmemo_add_sorted (PosMemo* pms, PosMemo new_prod)
{ PosMemo pred = NULL;
  PosMemo curr = *pms;
  int nr_parses = 0;
  int max_parses;

  /* Special case for the first posmemo to be entered, in case it was unknown */
  if (curr == POSMEMO_UNKNOWN)
    { *pms = new_prod;
      new_prod -> flags = 1;	/* Successful new insertion */
      posmemo_unknown_count--;
      return (new_prod);
    };

  if (rule_is_terminal(new_prod->nont_nr))
    max_parses = arts_ifd.max_terminal_parses;
  else
    max_parses = arts_ifd.max_posmemo_queue_length;

  /* Find the insertion point */
  DB_PM(abs_message ("Entering the posmemo sort with max parses = %d", max_parses);)
  /* First pass all parses that have a lower penalty than ours */
  while ((curr != NULL) && (curr -> penalty < new_prod -> penalty))
    { if (equivalent_posmemo (curr, new_prod))
	{ nr_parses++;
          if (nr_parses >= max_parses)
	    { detach_posmemo (new_prod, "new");
	      return (NULL);	/* equivalent to POSMEMO_BLOCKED */
	    }
	}
      pred = curr;
      curr = curr -> next;
    };

  DB_PM(abs_message ("Found %d equivalent parses with lower penalty", nr_parses);)

  /* Either curr == NULL at this point or curr -> penalty >= new_prod -> penalty */
  /* Note also that nr_parses < max_parses */
  assert ((curr == NULL) || (curr -> penalty >= new_prod -> penalty));
  assert (nr_parses < max_parses);
  while ((curr != NULL) && (curr -> penalty == new_prod -> penalty))
    { /* Pass over all parses who share the penalty */
      /* If you encounter an equivalent one, we have at least an equal or ambiguous parse */
      if (equivalent_posmemo (curr, new_prod))
	{ PosMemo prime = curr;
	  int length = 1;
	  DB_PM(abs_message ("Trying to insert new posmemo %p into ambiguous prime node %p",
		       new_prod, prime);)
	  /*
	     Iterate over the equivalent parses while checking if there is a
	     completely equal one; if so we can forget the new production
	  */
	  while (1)
	    { /* If you encounter an equal one, we're immediately done */
	      if (equal_posmemo (curr, new_prod))
		{
		  DB_PM(abs_message("New posmemo %p is equal to %p, returning NULL", new_prod, prime);)
		  detach_posmemo (new_prod, "new equal");
		  return (NULL);
		};

	      if (curr -> ambig == NULL) break;
	      curr = curr -> ambig;
	      length++;
	    };

	  /*
	     Limit the length of the equivalence list. Since there is
	     no criterion to decide which equivalent parse is "better"
	     (they all cover the same input), we only need the first
	     max_parses, since we are only going to print at most
	     max_parses combinations of equivalents.
	     (TODO: maybe use (nr_parses + length) rather than just length
	     to limit the list here; nr_parses includes other equivalent
	     parses.
	  */
	  if (length >= max_parses)
	    { detach_posmemo (new_prod, "new ambig");
	      return (NULL);
	    };

	  /* Add the new production to the end of the list of equivalents */
	  curr -> ambig = new_prod;
          new_prod -> flags = 1;	/* Successful new insertion */
	  new_prod -> prime = prime;
	  new_prod -> next = curr -> next;
	  prime -> prime = prime;
	  prime -> flags |= 1;		/* Mark the prime one as well */
	  DB_PM(abs_message("New posmemo %p has equivalent %p, returning old", new_prod, prime);)
	  return (prime);
	};

       pred = curr;
       curr = curr -> next;
    };

  /* Either curr == NULL at this point or curr -> penalty > new_prod -> penalty */
  /* Since nr_parses < max_parses still holds, we can insert the posmemo structure */
  assert((curr == NULL) || (curr -> penalty > new_prod -> penalty));
  assert((nr_parses < max_parses));
  DB_PM(abs_message ("Inserting new posmemo %p after %p before %p", new_prod, pred, curr);)
  new_prod -> next = curr;

  /* The question here is if we still need the next pointers on equivalent parses */
  if (pred == NULL) *pms = new_prod;
  else
    { PosMemo ambig;	/* update all next pointers of the predecessor's equivalents */
      for (ambig = pred; ambig != NULL; ambig = ambig -> ambig)
        ambig -> next = new_prod;
    };

  /* We have definitely added another parse, so remember */
  pred = new_prod;
  new_prod -> flags = 1;	/* Successful new insertion */
  nr_parses++;

  /* Scan the remainder of the posmemo list for parses that should be removed */
  while (curr != NULL)
    { if (equivalent_posmemo (curr, new_prod))
	{ /* another parse, check if we have to remove it */
	  nr_parses++;
	  if (nr_parses > max_parses)
	    { PosMemo ambig, next;
	      DB_PM(abs_message ("Deleting old equivalent posmemo %p after %p before %p",
			   curr, pred, curr -> next);)
	      /* Update the next pointers of the predecessor node and its equivalent nodes. */
	      for (ambig = pred; ambig != NULL; ambig = ambig -> ambig)
		ambig -> next = curr -> next;

	      /*
		 Naturally, if we delete the current production we should also
	         delete its equivalents.
	      */
	      for (ambig = curr; ambig != NULL; ambig = next)
		{ next = ambig -> ambig;
		  DB_PM(abs_message("detach_posmemo %p old equivalent of %p",
			  ambig, new_prod);)
		  detach_posmemo (ambig, "old equivalent");
		};

	      /* We're done */
	      return (new_prod);
	    };
	};
      pred = curr;
      curr = curr -> next;
    };

  /* Finally done */
  return (new_prod);
}

/*
 * Get the value of PMPRODPTR
 */
#include "arts_stackframe.h"

/*
   Add a positive memo
*/
int pointsIntoStack(void *p);
PosMemo posmemo_add_production (State input_state, int nont_nr, Penalty penalty,
				int nr_formals, int nr_locals, int nr_sons, Value* variables,
				State target_state, void *pass2,
				hyb_anchor anchor)
{ int nr_sontype_words;
  size_t variables_block_size;
  StateExtension ext;
  PosMemo *x;
  PosMemo new_memo;
  int *type_bits;
  int ix;

  ext = GetStateExtension(input_state);
  x = posmemo_fetch (ext, nont_nr);

  if (*x == POSMEMO_BLOCKED) {
      DB_PM(abs_message("posmemo_add_production: state=%p (%d.%c), nont_nr=%d: POSMEMO_BLOCKED",
	      input_state, trel_state_get_pos(input_state), trel_state_get_lex_state(input_state), nont_nr));
      detach_hyb_anchor(anchor, NULL);
      return NULL;
  }

  nr_sontype_words = (nr_sons + NR_SONTYPE_BITS-1)/NR_SONTYPE_BITS;
  variables_block_size = (nr_formals + nr_locals + nr_sons + nr_sontype_words) *
				 sizeof (Value);
  //new_memo = posmemo_getmem (sizeof (PMPROD));
  new_memo = posmemo_from_freelist();
  posmemo_known_count++;

  new_memo -> nont_nr = nont_nr;
  new_memo -> nr_formals = nr_formals;
  new_memo -> nr_variables = nr_formals + nr_locals;
  new_memo -> nr_sons = nr_sons;
  new_memo -> refcount = 1;		/* Assume one reference */
  new_memo -> flags = 0;		/* Although fresh born, not yet inserted */
  //printf("posmemo_add_production new memo %p anchor %p\n", new_memo, anchor);
  new_memo -> hybrid = anchor;
  if (anchor != NULL)
    arts_hyb_try_set_owner (anchor, new_memo);

  /* Allocate and fill a copy of the formals, locals, pointers to posmemo's of sons
     or trellis transition entries and a bitmap indicating the nature of the sons.
  */
  if (variables_block_size > 0)
    { new_memo -> variables = (Value *) posmemo_getmem (variables_block_size);

      /* Copy formals + locals */
      /* NOTE: variables 1 is at -1, 2 at -2, etc! */
      for (ix = 0; ix < nr_formals + nr_locals; ix++)
        new_memo -> variables[ix] = variables[-ix];

      /* Setup son typing admin */
      type_bits = GET_TYPE_BITS(new_memo);
      for (ix = 0; ix < nr_sontype_words; ix++) type_bits[ix] = 0;

      /* copy sons by copying PMPROD entries or transition pointers from frame */
      for (ix = 0; ix < nr_sons; ix++)
        { Value son = variables[-(nr_formals + nr_locals + ix)];
          if (pointsIntoStack((Value *)(son.ptr_par)))
	    { /* This son is a syntax nonterminal, copy its PMPROD entry */
	      Value sonpm = ((Value *)(son.ptr_par))[PMPRODPTR];
	      new_memo -> variables[nr_formals + nr_locals + ix] = sonpm;
	      attach_posmemo((PosMemo) sonpm.ptr_par, "posmemo_add_production");
	    }
          else
	    { /* copy the transition entry */
	      new_memo -> variables[nr_formals + nr_locals + ix] = son;	
	      SET_SON_AS_LEX(type_bits, ix);
	    };
        };
    }
  else new_memo -> variables = NULL;

  new_memo -> next_state = target_state;
  new_memo -> this_state = input_state;
  new_memo -> penalty = penalty;
  new_memo -> pass2 = pass2;

  new_memo -> ambig = NULL;
  new_memo -> prime = NULL;
  new_memo -> next = NULL;

#if LOOK_AT_HYBRID
  /*
   * By this time, the hybrid code has already been run, so any
   * bracket pairs have been cached.
   */
  if (arts_ifd.hash_production)
  { hash_hyb_anchor(anchor);
  }
#endif /* LOOK_AT_HYBRID */

#if DEBUG_POSMEMO
  abs_message ("Generated posmemo at input state: %p", input_state);
  posmemo_rdump_pmprod (new_memo, 0);
#endif
  return (posmemo_add_sorted (x, new_memo));
}

/*
   The following function promotes young born posmemos to acceptable
   and demotes acceptable ones to unacceptable
   XXX How does this mesh with sharing posmemos for COND rules?
*/
static void update_pm_and_equivalents (PosMemo pm)
{ PosMemo curr = pm;
  while (curr != NULL)
    { curr -> flags = (curr -> flags << 1) & 3;	/* 1 -> 2, 2 -> oblivion */
      curr = curr -> ambig;
    }
}

void posmemo_update (StateExtension input_state, int nont_nr)
{ PosMemo pm = *(posmemo_fetch (input_state, nont_nr));
  if (pm == POSMEMO_BLOCKED) return;
  while (pm != NULL)
    { update_pm_and_equivalents (pm);
      pm = pm -> next;
    };
}

#if 0
/*
 * This function is to be used only to remove ALL posmemos at once.
 * since it doesn't update any reference counts.
 */
void posmemo_free_vec (PosMemo* entry)
{ if ((*entry != POSMEMO_UNKNOWN) && (*entry != POSMEMO_BLOCKED))
    { PosMemo prod = *entry;
      while (prod != NULL)
        { PosMemo tail = (prod -> ambig != NULL)?prod -> ambig:prod -> next;
	  really_delete_posmemo(prod);
          prod = tail;
        };
      *entry = POSMEMO_UNKNOWN;
    };
}
#endif

/*
 * Use posmemo_detach_list() if you're removing posmemos incrementally.
 */
void posmemo_detach_list (PosMemo* entry)
{ if (*entry == POSMEMO_UNKNOWN)
    { /* No need to do anything */
    }
  else if (*entry == POSMEMO_BLOCKED)
    { *entry = POSMEMO_UNKNOWN;
      posmemo_blocked_count--;
      posmemo_unknown_count++;
    }
  else
    { PosMemo prod = *entry;
      while (prod != NULL)
        { PosMemo tail = (prod -> ambig != NULL)?prod -> ambig:prod -> next;
	  detach_posmemo(prod, "posmemo_detach_list");
          prod = tail;
        };
      *entry = POSMEMO_UNKNOWN;
      posmemo_unknown_count++;
    };
}

void* posmemo_get_pass2 (PosMemo prod)
{ assert (prod);
  return (prod -> pass2);
}

PosMemo posmemo_get_prod_ptr (StateExtension input_state, int nont_nr)
{ return (*(posmemo_fetch (input_state, nont_nr)));
}

void* posmemo_get_formal_ptr (PosMemo state)
{ assert (state);
  return (state -> variables);
}

Value posmemo_get_variable (PosMemo state, int nr)
{ assert (state);
  assert (nr < state -> nr_variables);
  return (state -> variables[nr]);
}

Value posmemo_get_local (PosMemo state, int nr)
{ assert (state);
  assert (state -> nr_formals + nr < state -> nr_variables);
  return (state -> variables[state -> nr_formals + nr]);
}

PosMemo posmemo_get_son (PosMemo state, int nr)
{ assert (state);
  assert (nr < state -> nr_sons);
  return ((PosMemo)(state -> variables[state -> nr_variables + nr].ptr_par));
}

Transition posmemo_get_son_transition (PosMemo state, int nr)
{ assert (state);
  assert (nr < state -> nr_sons);
  return ((Transition)(state -> variables[state -> nr_variables + nr].ptr_par));
}

Penalty posmemo_get_penalty (PosMemo state)
{ assert (state);
  return (state -> penalty);
}

int posmemo_is_acceptable (PosMemo curr)
{ assert (curr);
  return (curr -> flags & 3);
}

PosMemo posmemo_get_next_prod (PosMemo curr)
{ assert (curr);
  return (curr -> next);
}

State posmemo_get_next_input_state (PosMemo memo, State here)
{ State next;
  assert (memo);
  next = memo -> next_state;
  if (next == NULL)
      next = here;
  return next;
}

void posmemo_dump_pmprod_list (PMPROD* pmprod)
{ while (pmprod != NULL)
    { int ix;
      abs_message ("nont_nr = %d, pen = %d, nr_formals = %d, next_state = %d, refcount = %d",
        	   pmprod -> nont_nr, pmprod -> penalty, pmprod -> nr_formals,
		   trel_state_get_pos(pmprod -> next_state),
		   pmprod -> refcount);
      for (ix = 0; ix < pmprod -> nr_variables; ix++)
        abs_message ("\tformal %d value: %lx", ix, (pmprod -> variables)[ix].set_par);
      pmprod = pmprod -> next;
    };
}

void posmemo_dump_table (Trellis trellis)
{
  int node_nr, max_node_nr;
  int rule_nr;
  int *empty_rule;
  int *empty_node;
  State state;
  int trellis_length = trel_get_length(trellis) * NUM_LEX_STATES;
  int **overview = (int**) abs_calloc (trellis_length, sizeof(int*),
				       "posmemo_dump_table: overview[]");

  abs_printf ("Posmemos shown by start position, rule number, then penalty. Legend:\n");
  abs_printf ("start-end: name/#formals(formals, ...) (rulenumber) penalty=N Equiv/Not [sons:] nont_nr:end_pos ...\n");
  /* Build the table: */
  node_nr = 0;
  state = trel_get_initial_state(trellis);
  while (state != NULL)
    { if (!state) overview[node_nr] = NULL;
      else
        { StateExtension sext = GetStateExtension(state);
	  PosMemo* pma = sext -> pos_memos;

          if (!pma)
	    { overview[node_nr] = NULL;
	      abs_printf ("\t\t(posmemo %u skipped, NULL)\n", trel_state_get_pos(state));
            }
	  else
	    { overview[node_nr] = (int*) abs_calloc (arts_ifd.nr_syntax_nonterminals, sizeof (int),
				   		     "posmemo_dump_table: overview[][]");

              for (rule_nr = 1; rule_nr < arts_ifd.nr_syntax_nonterminals; rule_nr++)
		{ if (posmemo_is_blocked (sext, rule_nr))
		    { overview[node_nr][rule_nr] = -2;
		      /*
		       * If the director sets initialise the posmemos,
		       * it is not known anymore if a blocked rule is
		       * blocked because of that, or because the rule
		       * was tried and actually failed.
		       * So, only print FAILED if this is certain.
		       */
		      abs_printf("%3d     : %s (%d) %s\n",
				trel_state_get_pos(state),
				arts_ifd.nonterm_names[rule_nr].str,
				rule_nr,
				arts_ifd.directors_set_posmemos ?
				    "BLOCKED" : "FAILED");
		    }
                  else if(posmemo_is_unknown (sext, rule_nr))
                    overview[node_nr][rule_nr] = -1;
		  else
		    { PosMemo plijst = posmemo_get_prod_ptr(sext, rule_nr);
                      int nr_ptrs = 0;

                      while (plijst)
			{ abs_printf ("%3d ", trel_state_get_pos(state));	/* Match trellis dump */
                          posmemo_dump_pmprod(plijst);
			  if (plijst->ambig) {
			      PosMemo ambig = plijst->ambig;
			      do {
			        abs_printf ("AMB ");
                                posmemo_dump_pmprod(ambig);
                                nr_ptrs++;
				ambig = ambig->ambig;
			      } while (ambig != NULL);
			  }
                          nr_ptrs++;
                          plijst = plijst->next;
                        };

                      overview[node_nr][rule_nr] = nr_ptrs;
                    };
                };
            };
        };
	state = trel_state_next(trellis, state);
	node_nr++;
    };

  max_node_nr = node_nr;

  /* printed table compression */
  empty_rule = (int*) abs_calloc (arts_ifd.nr_syntax_nonterminals, sizeof(int),
				  "posmemo_dump_table: empty_rule");
  for (rule_nr = 1; rule_nr < arts_ifd.nr_syntax_nonterminals; rule_nr++)
    { empty_rule[rule_nr] = 1;
      node_nr = 0;

      while ((node_nr < max_node_nr) && (empty_rule[rule_nr]))
        { if (overview[node_nr])
	    { switch (overview[node_nr][rule_nr])
		{ case -1: break;
                  case -2: empty_rule[rule_nr] = 0; break;
                  default: empty_rule[rule_nr] = !overview[node_nr][rule_nr];
                }
            };

          node_nr++;
        };
    };

  empty_node = (int*) abs_calloc (trellis_length, sizeof(int), "posmemo_dump_table: empty_node");
  for (node_nr = 0; node_nr < max_node_nr; node_nr++)
    { empty_node[node_nr] = 1;
      rule_nr = 1;

      while ((rule_nr < arts_ifd.nr_syntax_nonterminals) &&
	     (empty_node[node_nr]) && (overview[node_nr]))
	{ switch (overview[node_nr][rule_nr])
	    { case -1: break;
              case -2: empty_node[node_nr] = 0; break;
              default: empty_node[node_nr] = !overview[node_nr][rule_nr];
            };

          rule_nr++;
        };
    };

  abs_printf("nont | (u)nknown/(b)locked_or_failed/nr_posmemos... | nont_name\n");
  /* actually show it: */
  /* first the table */
  for (rule_nr = 1; rule_nr < arts_ifd.nr_syntax_nonterminals; rule_nr++)
    { if (!empty_rule[rule_nr])
	{ abs_printf("%3d|", rule_nr);

          for (node_nr = 0; node_nr < max_node_nr; node_nr++)
	    if (!empty_node[node_nr])
	      { switch (overview[node_nr][rule_nr])
		  { case -1: abs_printf("   u"); break;
                    case -2: abs_printf("   b"); break;
                    default: abs_printf(" %3d", overview[node_nr][rule_nr]);
                  };
              };

          if (arts_ifd.nonterm_names[rule_nr].str)
	    abs_message (" | %s", arts_ifd.nonterm_names[rule_nr].str);
          else abs_message (" | ?");
        };
    };

  /* then a neat line below it */
  abs_printf("---+");
  for (node_nr = 0; node_nr < max_node_nr; node_nr++)
    if (!empty_node[node_nr])
      abs_printf("----");

  /* and of course the numbers */
  abs_printf("\n   |");
  node_nr = 0;
  state = trel_get_initial_state(trellis);
  while (state != NULL)
    { if (!empty_node[node_nr])
      { abs_printf("%3d%c", trel_state_get_pos(state),
			    trel_state_get_lex_state(state));
      }
      state = trel_state_next(trellis, state);
      node_nr++;
    }
  abs_printf("\n");

  /* free the space: */
  for (node_nr = 0; node_nr < max_node_nr; node_nr++)
    if (overview[node_nr])
      abs_free (overview[node_nr], "posmemo_dump_table: overview[][]");
  abs_free (overview, "posmemo_dump_table: overview[]");
  abs_free (empty_rule, "posmemo_dump_table: empty_rule");
}

