/*
   File: lxcn_lexicon_impl.h
   Defines the internal structures to store a lexicon

   Copyright 2009 Radboud University of Nijmegen
 
   This program is free software: you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation, either version 3 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program.  If not, see <http://www.gnu.org/licenses/>.

   CVS ID: "$Id$"
*/
#ifndef IncLexiconImpl
#define IncLexiconImpl

/* Libabase includes */
#include <abase_porting.h>
#include <abase_repr.h>
#include <abase_pool_alloc.h>

/* Local includes */
#include "lxcn_vocabulary.h"
#include "lxcn_fact_table.h"
#include "lxcn_lexicon.h"

#ifdef __cplusplus
extern "C" {
#endif

/*-------------------------------------------------------------------------
// Structure for storing the set affixes
//
// name:    Affix name
// bitset:  The bitset associated with this affix
//	    Note that we read and store the value as a signed 64 bit which
//	    will be the same as the type still used in the runtime system.
//
// The INT and TEXT affixes are saved as rows of INTs/TEXTs
//-----------------------------------------------------------------------*/
typedef struct
{ char *	name;
  Bitset64	bitset;
} SetAffix;

/*----------------------------------------------------------------------------
// Structure for storing the nonterminals and calls
// Nonterminal:
// name:  Nonterminal name
// nont_nr: Nonterminal number given by coder
// arity: Number of parameters
// fpars: The formal parameters of the nonterminal:
//	  Note: INT is coded as -1, TEXT as -2,
//	        a set affix is coded as its index in the set_affixes
// crits: inidicating which fpars are critical parameters (not coded in call)
//
// Call:
// nont_id:	Entry in the nonterminal table
// actuals:	Actual arguments, coded as index in the affix tables
//		Note the formal type can be obtained from the Nonterminal
//--------------------------------------------------------------------------*/
#define FormalINT   (-1)
#define FormalTEXT  (-2)
typedef struct
{ char *	name;
  int		nont_nr;
  int 		arity;
  int *		fpars;
  int		ncrits;
  int *		crits;
} Nonterminal;

typedef struct
{ int		nont_id;
  int *		actuals;
} Call;

struct lexicon_rec
{ int		hybrid_parsing;
  int 		nr_int_affixes;
  int *		int_affixes;
  int		nr_text_affixes;
  char **	text_affixes;
  int		nr_set_affixes;
  SetAffix *	set_affixes;
  int		nr_nonterminals;
  Nonterminal *	nonterminals;
  int		nr_terminals;
  int		nr_calls;
  Call *	calls;
  int		nr_lex_entries;
  int **	all_entries;
  int		nr_vocabularies;
  int *		all_lexeme_markers;
  Vocabulary *  all_vocabularies;
  Vocabulary    all_critical_texts;
  int		nr_facts;
  FactTable *	all_fact_tables;
  Pool 		pool;			/* memory pool */
  /*struct abs_pool *stringpool;*/	/* memory pool */
};

/*------------------------------------------------------------------------------
// Format of lexicon file (as delivered by agfl-lexgen)
//
// 0a) lexicon header
// 0b) int              nr_lexicons
// 0c) char *           lexicon_names[]
// 0d) int              nr_fact_tables
// 0e) char *           fact_table_names[]
// 0f) int              hybrid_parsing_flag
//
// The affix table: the nr of lif affixes is saved to check for lexicon recompilation.
//
// 1a) int		nr_lif_affixes
// 1b) int      	nr_set_affixes
// 1c) setaffix		set_affixes[]
// 2a) int		nr_int_affixes
// 2b) int		int_affixes[]
// 3a) int		nr_text_affixes
// 3b) char *		text_affixes[]
//
// The nonterminal and call table:
//
// 4a) int		nr_nonterminals
// 4b) nonterminal	nonterminals[]
// 4c) int		terminals
// 4d) terminal		terminals[]
// 5a) int		nr_calls
// 5b) call		calls[]
//
// The mapping from list nr to calls
// 6a) int		nr_entries
// 6b) entry		all_entries[] (saved as int *)
//
// The vocabularies
// 7a) int		nr_vocabularies
// 7b) int		all_lexeme_markers
// 7a) int		nr_vocabularies
// 7b) int		all_lexeme_markers[]
// 7c) Vocabulary	all_vocabularies[]
//
// The fact table administration
// 8a) Vocabulary	critical_texts
// 8b) int		nr_fact_tables
// 8c) FactTable	all_fact_tables[]
//----------------------------------------------------------------------------*/

#ifdef __cplusplus
}
#endif
#endif /* IncLxcnLexiconImpl */

