/*
   File: pattern.h
   Regexp routine definitions

   Copyright 2009-2010 Radboud University of Nijmegen

   This program is free software: you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation, either version 3 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program.  If not, see <http://www.gnu.org/licenses/>.

   CVS ID: "$Id$"
*/

#ifndef PATTERN_H
#define PATTERN_H

#include <sys/types.h>

typedef void	RegExp;

/*
//------------------------------------------------------------------------------
// Functionvoid *state;
//	RegExp*
//	new_regexp(const char* regexp_txt, int* result_p)
//
// Description:
//	Compile regular expression regexp_txt. A buffer holding the
//	compiled expression is allocated, even if compilation fails.
//	The location pointed to by result_p is set to indicate the
//	compilation result: code < 0 means regular expressions not
//	supported, code 0 means success, and code > 0 indicates an
//	error value.
//
// Return value:
//	Pointer to compiled expression, even in case of failure.
//------------------------------------------------------------------------------
*/

RegExp*
new_regexp(const char* regexp_txt, int* result_p);

/*
//------------------------------------------------------------------------------
// Function:
//	RegExp *
//	empty_regexp();
//
// Description:
//	Prepare for collecting a regular expression consisting of
//	several alternatives.
//	Conceptually, collects an initial "(".
//
// Return value:
//	Pointer to a expression.
//------------------------------------------------------------------------------
*/
RegExp *
empty_regexp();


/*
//------------------------------------------------------------------------------
// Function:
//	int
//	add_alternative_to_regexp(RegExp *regexp, const char* regexp_txt)
//
// Description:
//	Add an alternative to a regular expression as prepared by
//	a previous call to empty_regexp() and add_alternative_to_regexp().
//	Conceptually, adds an alternative operator and the given regexp,
//	resulting in something like "(foo|bar".
//
// Return value:
//	Error code, 0 if ok.
//------------------------------------------------------------------------------
*/
int
add_alternative_to_regexp(RegExp *regexp, const char* regexp_txt);


/*
//------------------------------------------------------------------------------
// Function:
//	void
//	add_alternative_to_regexp(RegExp *regexp, const char* regexp_txt)
//
// Description:
//	Indicates that all alternatives have been added, and
//	prepares the regular expression for use.
//	Conceptually, adds a ")", resulting in something like "(foo|bar)".
//
// Return value:
//	Error code, 0 if ok.
//------------------------------------------------------------------------------
*/
int 
finalize_regexp(RegExp *regexp);

/*
//------------------------------------------------------------------------------
// Function:
//	void
//	delete_regexp(RegExp* regexp)
//
// Description:
//	Delete compiled regular expression regexp.
//------------------------------------------------------------------------------
*/

void
delete_regexp(RegExp* regexp);

/*
//------------------------------------------------------------------------------
// Function:
//	size_t regexp_error(int error, RegExp* regexp, char* buf, size_t sz)
//
// Description:
//	Convert error code associated with regexp to string,
//	and copy at most sz bytes of this string to buf.
//
// Return value:
//	Length of error message.
//------------------------------------------------------------------------------
*/

size_t
regexp_error(int, RegExp*, char*, size_t);

/*
//------------------------------------------------------------------------------
// Function:
//	const char*
//	match_regexp(const char* str, RegExp* regexp)
//
// Description:
//	Match string str with compiled regular expression regexp.
//	This function will try to match the longest prefix of str.
//
// Return value:
//	A pointer to the first character in str not matched by regexp,
//	or NULL if regexp does not match str.
//------------------------------------------------------------------------------
*/

const char*
match_regexp(const char* str, RegExp* regexp);

/*
//------------------------------------------------------------------------------
// Function:
//	const char*
//	match_regexp_all_alternatives(const char* str, RegExp* regexp, int nalts, int *alts);
//
// Description:
//	Match string str with compiled regular expression regexp.
//	This function will try to match the longest prefix of str, and
//	it records all matches of all top-level alternatives that it finds
//      in alts[], like this:
//      3	end position of the match, or -1 to terminate the list
//      5,3,8   the numbers of the alternatives, starting at 1
//      0       to terminate the list of alternatives
//      -1      to terminate the list of matches.
//
// Return value:
//	A pointer to the first character in str not matched by regexp,
//	or NULL if regexp does not match str.
//------------------------------------------------------------------------------
*/
const char*
match_regexp_all_alternatives(const char* str, RegExp* regexp, int nalts, int *alts);

/*
//------------------------------------------------------------------------------
// Functions:
//    void *prematch_regexp_all_alternatives(const char* str, RegExp* regexp, int nalts, int *alts);
//    int match_regexp_next_alternative(void *state);
//    void postmatch_regexp_all_alternatives(void *state);
//
//
// Description:
//	Match string str with compiled regular expression regexp.
//	This regexp is meant to be constructed by add_alternative_to_regexp()
//	and related functions.
//
//	Call prematch_regexp_all_alternatives() to prepare.
//	int *alts points to an array of size nalts. Untouched if nalts == 0.
//
//	Call match_regexp_next_alternative() with the returned state pointer
//	to get the next position where 1 or more alternative matches end.
//	alts[] is filled with their numbers (starting at 1). A 0 terminates
//	the list (if it fits).
//
//	Call postmatch_regexp_all_alternatives() to clean up.
//
// Return value:
//	prematch_regexp_all_alternatives() returns a state pointer which
//	must be passed to the other functions.
//	match_regexp_next_alternative() returns the length of the match
//	(these values strictly increase).
//------------------------------------------------------------------------------
*/
void *prematch_regexp_all_alternatives(const char* str, RegExp* regexp, int nalts, int *alts);
int match_regexp_next_alternative(void *state);
void postmatch_regexp_all_alternatives(void *state);

#endif /* PATTERN_H */
