/*
   File: acoder.c
   Defines the coder of Agfl assembler
  
   Copyright 2009-2010 Radboud University of Nijmegen

   This program is free software: you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation, either version 3 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program.  If not, see <http://www.gnu.org/licenses/>.

   CVS ID: "$Id$"
*/

#if HAVE_CONFIG_H
#include <config.h>
#endif

/* System includes */
#include <stdio.h>
#include <string.h>
#if HAVE_SYS_STAT_H
#include <sys/stat.h>
#endif

/* Libabase includes */
#include <abase_porting.h>
#include <abase_error.h>
#include <abase_fileutil.h>
#include <abase_memalloc.h>

/* Local includes */
#include "options.h"
#include "lexer.h"
#include "parser.h"
#include "acoder.h"

/*
   AGFL codes in 5 different segments:
   0) absolute data 	(not relocated)
   1) the code		(labels being relocated while loading)
   2) the interface	(loaded into a struct at runtime)
   3) the data		(affix domains, weights &c)
   4) read only text	(accumulated while assembling)
   5) error
*/
char *string_from_segment (segment seg)
{ switch (seg)
    { case abs_segment:		return ("A");
      case code_segment:	return ("C");
      case interface_segment:	return ("I");
      case data_segment:	return ("D");
      case rotext_segment:	return ("R");
      default:			return ("E");
    };
};

/*
   While coding we keep track of the current location
*/
static segment curr_seg;
static int64 curr_lc[MAX_SEGS];

void get_current_location (segment *ret_seg, int64 *ret_value)
{ *ret_seg = curr_seg;
  *ret_value = curr_lc[curr_seg];
};

void set_current_segment (segment seg)
{ curr_seg = seg;
};

/*
   The AOB opens with a standard header for object files, followed
   by the size of each of the non absolute (and non ERROR) segments
*/
static FILE *listing;
static BinFile object;
static void save_segment_sizes ()
{ int ix;
  for (ix = 1; ix < MAX_SEGS - 1; ix++)
     abs_bin_save_int64 (object, curr_lc[ix]); 
};

/*
   An AGFL instruction is saved in the AOB with the following format:
 
   char opcode;
   int  nr_opnds;
   foreach operand
      { char segment;
	int64 value;
      }

   Hence the smallest instructions will take 2 bytes in the AOB
   At runtime, the code will be unpacked into an array of code cells.
   Each code cell is assumed to be large enough to hold an address.
   Although values within segments are coded as ints, they will be
   relocated at runtime as addresses. Address offsets within segments
   are coded as byte addresses.

   Currently we allow for 4 operands (Constant MAX_OPNDS in acoder.h)
*/

/*
   MS: Check the processing here for introduction of Unicode

   Operands in the form of a string are preprocessed to allocate
   a (possibly anonymous) string in the rotext segment.

   The rotext segment is output automagically after the data segment
   through a stack mechanism. This can be optimized using a tree.
   For the moment we refrain from those optimizations.
*/
static int length_from_string (char *s)
{ char *ptr = s;
  int len = 0;
  while (*ptr)
    { if (*ptr == '\\')
	{ ptr++;
	  switch (*ptr)
	    { case 'n':
	      case 'r':
	      case 't':
	      case '"':
	      case '\\': break;
	      case '0': /* Octally coded character */
	      case '1':
	      case '2':
	      case '3': ptr += 2; break;
	      default: len++;		/* Count '\\' */
	    };
	};
      ptr++;
      len++;
    };
  return (len);
};

typedef struct rotext_rec
{ char *text;
  struct rotext_rec *next;
} *rotext_ptr;

static rotext_ptr rotext_fifo;
static rotext_ptr rotext_tail;

static void init_rotext_admin ()
{ rotext_fifo = NULL;
  rotext_tail = NULL;
};

static void push_new_rotext (char *s)
{ /* Allocate new rotext item */
  rotext_ptr rit = (rotext_ptr) abs_malloc (sizeof (struct rotext_rec), "push_new_rotext");
  rit -> text = abs_new_string (s, "push_new_rotext");
  rit -> next = NULL;

  /* Remember it at the tail */
  if (rotext_tail != NULL)
    { rotext_tail -> next = rit;
      rotext_tail = rit;
    }
  else rotext_fifo = rotext_tail = rit;
};

static void preprocess_operand (operand *opnd)
{ if (opnd -> tk == STRING)
    { int len = length_from_string (opnd -> svalue) + 1; /* add \0 */
      opnd -> seg = rotext_segment;
      opnd -> value = curr_lc[rotext_segment];
      curr_lc[rotext_segment] += (int64) len;
      if (pass_two) push_new_rotext (opnd -> svalue);
    };
};

static void preprocess_instruction (instr *ins)
{ int ix;
  for (ix = 0; ix < ins -> nr_opnds; ix++)
     preprocess_operand (&ins -> opnds[ix]);
};

static char pick_char_code (char **ptr)
{ char ch = **ptr;
  if (ch == '\\')
    { *ptr = *ptr + 1;
      switch (**ptr)
	{ case 't': return ('\t');
	  case 'n': return ('\n');
	  case 'r': return ('\r');
	  case '"': return ('"');
	  case '\\': return ('\\');
	  case '0':
	  case '1':
	  case '2':
	  case '3':
	    { /* Octal conversion */
	      int val = (**ptr - '0');
	      *ptr = *ptr + 1;
	      val = val * 8 + (**ptr - '0');
	      *ptr = *ptr + 1;
	      val = val * 8 + (**ptr - '0');
	      return ((char) (val & 0xff));
	    }; break;
	  default:
	    { *ptr = *ptr - 1;
	      return ('\\');
	    };
        };
    };
  return (ch);
};

static void code_rotext_string (char *s)
{ char *ptr;
  for (ptr = s; *ptr; ptr++)
    { char ch = pick_char_code (&ptr);
      abs_bin_save_char (object, ch);
    };
  abs_bin_save_char (object, '\0');
};

static void code_rotext_segment ()
{ rotext_ptr ptr;
  for (ptr = rotext_fifo; ptr != NULL; ptr = ptr -> next)
     code_rotext_string (ptr -> text);
};

/*
   Try and dump the instruction
   Note that we encode tabs in the code generator
*/
static void dump_operand (operand *opnd)
{ fprintf (listing, "%s.%016llx ", string_from_segment (opnd -> seg), opnd -> value);
};

static void dump_instruction (instr *ins)
{ int ix;
  fprintf (listing, "%s.%08llx  ", string_from_segment (curr_seg), curr_lc[curr_seg]);
  fprintf (listing, "%02x.%d  ", ins -> opc, ins -> nr_opnds);
  for (ix = 0; ix < MAX_OPNDS; ix++)
    if (ix < ins -> nr_opnds) dump_operand (&ins -> opnds[ix]);
    else fprintf (listing, "                   ");	/* 19 positions */
  fprintf (listing, "\t");
  fprintf (listing, "%s", get_current_line_buffer ());
};

static void dump_string (char *sval)
{ char *ptr, ch;
  int pos = 0;
  fprintf (listing, "%s.%08llx  ", string_from_segment (curr_seg), curr_lc[curr_seg]);
  fprintf (listing, "      ");
  for (ptr = sval; *ptr; ptr++)
    { /* Had 24 positions and still more to print */
      if (pos == 24)
	{ /* Print the source line */
	  fprintf (listing, "    \t");
	  fprintf (listing, "%s", get_current_line_buffer ());
	  fprintf (listing, "R.              ");
	}
      else if ((pos > 24) && (pos % 24 == 0))
	/* Print the continuation line */
	fprintf (listing, "\nR.              ");      

      /* Pick next character to print */
      ch = pick_char_code (&ptr);
      fprintf (listing, "%02x ", ch & 0xff);
      pos++;
    };

  /* Check if we still have to print the source line */
  if (pos <= 24)
    { while (pos != 24)
	{ fprintf (listing, "   ");
	  pos++;
	};

      /* Print the source line */
      fprintf (listing, "    \t");
      fprintf (listing, "%s", get_current_line_buffer ());
    }
  else fprintf (listing, "\n");
};

static void dump_pseudo_instruction (operand *opnd)
{ int ix;
  fprintf (listing, "%s.%08llx  ", string_from_segment (curr_seg), curr_lc[curr_seg]);
  fprintf (listing, "      ");
  for (ix = 0; ix < MAX_OPNDS; ix++)
    if (!ix) dump_operand (opnd);
    else fprintf (listing, "                   ");
  fprintf (listing, "\t");
  fprintf (listing, "%s", get_current_line_buffer ());
};

static void dump_empty_instruction ()
{ int ix;
  fprintf (listing, "%s.%08llx  ", string_from_segment (curr_seg), curr_lc[curr_seg]);
  fprintf (listing, "      ");
  for (ix = 0; ix < MAX_OPNDS; ix++)
    fprintf (listing, "                   ");
  fprintf (listing, "\t");
  fprintf (listing, "%s", get_current_line_buffer ());
};

/*
   Code the instruction
*/
void code_instruction (instr *ins)
{ /* Check for string operands */
  int ix;
  preprocess_instruction (ins);
  if (!pass_two)
    { curr_lc[curr_seg] += (ins -> nr_opnds + 1) * sizeof (int64);
      return;
    };

  /* Pass 2 actions */
  if (generate_listing) dump_instruction (ins);
  curr_lc[curr_seg] += (ins -> nr_opnds + 1) * sizeof (int64);

  /* Try and write the instruction */
  abs_bin_save_char (object, (char) ((unsigned char) ins -> opc));
  abs_bin_save_int (object, ins -> nr_opnds);
  for (ix = 0; ix < ins -> nr_opnds; ix++)
    { abs_bin_save_char (object, (char) ((unsigned char) ins -> opnds[ix].seg));
      abs_bin_save_int64 (object, ins -> opnds[ix].value);
    };
};

void code_empty_instruction ()
{ /* Empty instructions only need to print themselves in the listing */
  if (generate_listing && pass_two)
    dump_empty_instruction ();
};

void code_string (operand *opnd)
{ /* Pick length and verify we are in the rotext segment */
  int len = length_from_string (opnd -> svalue) + 1; /* add \0 */
  if (curr_seg != rotext_segment)
    abs_bug ("Current segment is not rotext", "code_string");

  if (!pass_two)
    { curr_lc[curr_seg] += len;
      return;
    };

  /* Pass 2 actions */
  if (generate_listing) dump_string (opnd -> svalue);
  opnd -> seg = rotext_segment;
  opnd -> value = curr_lc[rotext_segment];
  curr_lc[rotext_segment] += len;
  push_new_rotext (opnd -> svalue);
};

/*
   For pseudo instructions like .int and .addr,
   we code a single operand as a segment and an int64 in the code.
*/
void code_word (operand *opnd)
{ /* In pass 1 only update the location */
  preprocess_operand (opnd);
  if (!pass_two)
    { curr_lc[curr_seg] += sizeof (int64);
      return;
    };

  /* Pass 2 actions */
  if (generate_listing) dump_pseudo_instruction (opnd);
  curr_lc[curr_seg] += sizeof (int64);

  /* Try and write the operand */
  abs_bin_save_char (object, (char) ((unsigned char) opnd -> seg));
  abs_bin_save_int64 (object, opnd -> value);
};

void maybe_write_shebang_line (BinFile object)
{ if (shebang_text != NULL)
    { FILE *fh = abs_bin_file(object);
      int mask, fd;
      fprintf(fh, "#!%s\n", shebang_text);

#if HAVE_SYS_STAT_H && HAVE_UMASK && HAVE_FCHMOD
      /*
	 Some fiddling to make the file executable.
      */
      mask = umask(0);
      (void)umask(mask);
      fd = fileno(fh);
      /* rwxrwxrwx minus mask */
      fchmod (fd, (S_IRWXU|S_IRWXG|S_IRWXO) & ~mask);
#endif
    };
}

int init_acoder ()
{ /* If starting with pass 2, setup extra admin */
  int ix;
  if (pass_two)
    { object = abs_bin_fopen (object_fname, "w");
      if (object == NULL) return (-1);
      maybe_write_shebang_line (object);
      abs_bin_save_version (object, "object");
      save_segment_sizes ();
      init_rotext_admin ();

      /* Check if we want a listing */
      if (generate_listing)
        { listing = abs_fopen (listing_fname, "w");
	  if (listing == NULL)
	    abs_abort("init_acoder", "Can't open listing file");
	}
    };

  /* (Re) initialize the segment location counters */
  curr_seg = abs_segment;
  for (ix = 0; ix < MAX_SEGS; ix++)
    curr_lc[ix] = int64_const(0);
  return (0);
};

void finish_acoder ()
{ code_rotext_segment ();
  abs_bin_save_eof (object);
  abs_bin_fclose (object);
};
