/*
   File: lxcn_fact_table.h
   Stores a fact table (by which we mean the datastructure through which
   facts can be entered with their appropriate information) for later
   retrieval. Since the number of individual facts in a fact table may
   easily be of the order of one million, the fact table is stored as
   a hash table

   Copyright 2009 Radboud University of Nijmegen
 
   This program is free software: you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation, either version 3 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program.  If not, see <http://www.gnu.org/licenses/>.

   CVS ID: "$Id$"
*/

/* include config.h if autoconfigured */
#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

/* global includes */
#include <stdio.h>
#include <ctype.h>
#include <string.h>

/* Libabase includes */
#include <abase_porting.h>
#include <abase_error.h>
#include <abase_memalloc.h>
#include <abase_fileutil.h>

/* Local includes */
#include "lxcn_fact_table.h"
#include "lxcn_vocabulary.h"
#include "lxcn_lexicon.h"
#include "lxcn_lexicon_impl.h"

/* Note, like in lexical searching we could do a top of allocation here to optimize */
int *lxcn_initialize_search_key ()
{ int *key = (int *) abs_calloc (MAX_CRIT_PARAMS + 1, sizeof (int), "lxcn_initialize_search_key");
  key[0] = 0;	/* should be done by calloc */
  return (key);
}

void lxcn_release_search_key (int *key)
{ abs_free (key, "lxcn_release_search_key");
}

int lxcn_hash_search_key (int *key, int hash_size)
{ int ix;
  unsigned int value = 0;
  for (ix = 1; ix <= key[0]; ix++)
    value = (127 * value + ((unsigned int) key[ix] & 0xffffff)) & 0xffffff;
  return ((int) value % hash_size);
}

int lxcn_lookup_critical_text (Lexicon lexicon, char *text)
{ Vocabulary critical_texts = lexicon -> all_critical_texts;
  int *entry = lxcn_lookup_in_vocabulary (critical_texts, text);
  if (entry == NULL) return (0);
  return (*entry);
}

void lxcn_bin_load_fact_table (Pool p, BinFile bf, FactTable *fct)
{ FactTable table = abs_pool_malloc (p, sizeof (struct fact_table_rec), "lxcn_bin_load_fact_table");
  int ix;

  *fct = table;
  abs_bin_load_int (bf, &table -> hash_size);
  abs_bin_load_int (bf, &table -> nr_crits);
  table -> hash_table = abs_pool_calloc (p, table -> hash_size,
					 sizeof (int *), "lxcn_bin_load_fact_table");
  for (ix = 0; ix < table -> hash_size; ix++)
    { int fsize, iy;
      abs_bin_load_int (bf, &fsize);
      if (fsize == 0)
	{ table -> hash_table[ix] = NULL;
          continue;
	};
      table -> hash_table[ix] = abs_pool_calloc (p, fsize * (table -> nr_crits + 1) + 1,
						 sizeof (int), "lxcn_bin_load_fact_table");
      table -> hash_table[ix][0] = fsize;
      for (iy = 1; iy <= fsize * (table -> nr_crits + 1); iy++)
	abs_bin_load_int (bf, &table -> hash_table[ix][iy]);
    };
}

static int fact_matches (int *facts, int *key, int ix, int nr_crits)
{ int iy;
  for (iy = 0; iy < nr_crits; iy++)
    if (key[iy + 1] != facts[ix + iy])		/* key[0] is length */
      return (0);
  return (1);
}

int lxcn_lookup_in_fact_table (Lexicon lexicon, int fact_nr, int *key, int *entry_nr)
{ FactTable table = lexicon -> all_fact_tables[fact_nr];
  int hash = lxcn_hash_search_key (key, table -> hash_size);
  int nr_crits, nfacts, ix;
  int *facts = table -> hash_table[hash];
  if (facts == NULL) return (0);
  nr_crits = table -> nr_crits;
  nfacts = facts[0];
  for (ix = 1; ix <= nfacts * (nr_crits + 1); ix += (nr_crits + 1))
    if (fact_matches (facts, key, ix, nr_crits))
      { *entry_nr = facts[ix + nr_crits];
	return (1);
      };
  return (0);
}

int lxcn_lookup_triple (Lexicon lex, int *key, int *frequency)
{ int triple_factnr = lex -> nr_facts;
  return (lxcn_lookup_in_fact_table (lex, triple_factnr, key, frequency));
}
