/* 
   File: abase_pool_alloc.c
   Defines managed memory allocation routines

   Copyright 2009 Radboud University of Nijmegen

   This program is free software: you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation, either version 3 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program.  If not, see <http://www.gnu.org/licenses/>.

   CVS ID: "$Id$"
*/

/* standard includes */
#ifdef HAVE_CONFIG_H
#include <config.h>
#endif /* HAVE_CONFIG_H */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <stdarg.h>
#include <assert.h>

/* Local includes */
#include "abase_porting.h"
#include "abase_error.h"
#include "abase_memalloc.h"
#include "abase_pool_alloc.h"

#if WITH_VALGRIND
#include <valgrind.h>
#include <memcheck.h>
#else
#define VALGRIND_CREATE_MEMPOOL(pool, rzB, is_zeroed)
#define VALGRIND_DESTROY_MEMPOOL(pool)
#define VALGRIND_MEMPOOL_ALLOC(pool, addr, size)
#define VALGRIND_MAKE_MEM_NOACCESS(addr, size)
#endif

/*
   Basically the pooled routines have the same interface as
   the nonpooled routines, but the memory is allocaed from
   big chunks, and cannot be individually deallocated.
   Only the pool as a whole can be removed.
*/

struct abs_pool_block {
    char *mem;
    size_t avail;
    struct abs_pool_block *prev;
#if WITH_VALGRIND
    struct abs_pool *pool;
#endif
};

struct abs_pool {
    size_t block_size;
    size_t max_block_size;
    struct abs_pool_block *block;
    size_t total_allocated;
    int nr_allocs;
    int nr_prevs_searched;
    //int bytes_align_wasted;
    int unaligned_allocs_req;
    int unaligned_allocs;
};
static size_t bytes_align_wasted = 0;

static
struct abs_pool_block *
abs_pool_new_block(struct abs_pool *p, size_t size)
{
    struct abs_pool_block *b;

    if (size == 0) {
	size = p->block_size;
	/* abs_message("abs_pool_new_block: size %ld", (long)size); */
    } else {
	/* abs_message("abs_pool_new_block: pre-given size %ld", (long)size); */
    }
    b = abs_malloc(size, "abs_pool_new_block");
    b->mem = (char *)(b + 1);
    b->avail = size - sizeof(*b);
    b->prev = p->block;
    p->block = b;
    p->total_allocated += size;
#if WITH_VALGRIND
    b->pool = p;
    VALGRIND_MAKE_MEM_NOACCESS(b->mem, b->avail);
#endif

    return b;
}

struct abs_pool *
abs_pool_init (size_t start_size, size_t max_size)
{
    struct abs_pool *p;

    if (start_size < sizeof(struct abs_pool_block))
	max_size = start_size = 1024;

    p = abs_malloc(sizeof(struct abs_pool), "abs_pool_init");
    p->block_size = start_size;
    p->max_block_size = max_size;
    p->block = NULL;
    p->total_allocated = 0;
    p->nr_allocs = 0;
    p->nr_prevs_searched = 0;
    //p->bytes_align_wasted = 0;
    p->unaligned_allocs_req = 0;
    p->unaligned_allocs = 0;
    VALGRIND_CREATE_MEMPOOL(p, 0, 0);
    abs_pool_new_block(p, start_size);
    return p;
}

void
abs_pool_free (struct abs_pool *p, const char *place)
{
    if (p != NULL) {
	//abs_pool_stats(p, place);
	struct abs_pool_block *cur = p->block;
	while (cur != NULL) {
	    struct abs_pool_block *next = cur->prev;
	    abs_free(cur, place);
	    cur = next;
	}
	abs_free(p, place);
	VALGRIND_DESTROY_MEMPOOL(p);
    }
}

void
abs_pool_stats (struct abs_pool *p, const char *place)
{
    if (p != NULL) {
	int nr_blocks = 0;
	size_t leftover = 0;	/* in newest block */
	size_t leftovers = 0;	/* in all blocks */
	struct abs_pool_block *cur = p->block;
	int nr_allocs;
	if (cur != NULL) {
	    leftover = cur->avail;
	    while (cur != NULL) {
		nr_blocks++;
		leftovers += cur->avail;
		cur = cur->prev;
	    }
	}
	abs_message("pool has %d blocks, %ld (%ld) bytes left over: %d%% (%s)",
		    nr_blocks, (long)leftovers, (long)leftover,
		    (leftovers * 100) / p->total_allocated,
		    place);
	if (nr_blocks > 1) {
	    abs_message("avg %d/100 bytes per non-head blocks",
		    (leftovers - leftover) * 100 / (nr_blocks - 1));
	} 
	abs_message("%ld system bytes total allocated", (long)p->total_allocated);
	nr_allocs = p->nr_allocs;
	if (nr_allocs == 0)
	    nr_allocs = 1;
	abs_message("%d calls to pool_*_alloc: avg %d/100 bytes (%d/100 inc. leftovers)",
		    p->nr_allocs,
		    (p->total_allocated-leftovers)*100 / nr_allocs,
		    p->total_allocated*100 / nr_allocs);
	abs_message("%d previous blocks searched", p->nr_prevs_searched);
	abs_message("%d bytes wasted on alignment", bytes_align_wasted);
	abs_message("%d unaligned allocs", p->unaligned_allocs_req);
	abs_message("%d unaligned allocs were unaligned", p->unaligned_allocs);
    } else {
	abs_message("pool is NULL at '%s'", place);
    }
}

static
void *
abs_pool_malloc_from_block(struct abs_pool_block *b, size_t size)
{
    void *res = b->mem;
    assert(size <= b->avail);
    b->mem += size;
    b->avail -= size;
    //abs_message("abs_pool_malloc_from_block: %p, %ld bytes", (long)res, size);
    VALGRIND_MEMPOOL_ALLOC(b->pool, res, size);	/* unrounded size would be better */
    return res;
}

#define POOL_ALIGN	(sizeof(void *))
#define POOL_MASK	(POOL_ALIGN-1)

static
unsigned int
abs_pool_align_block(struct abs_pool_block *b)
{
    uintptr_t mem = (uintptr_t) b->mem;
    unsigned int mod = (unsigned int)(mem % POOL_ALIGN);

    if (mod != 0) {
	unsigned int roundup = POOL_ALIGN - mod;

	if (roundup > b->avail) {
	    return roundup;		/* error */
	}

	b->mem += roundup;
	b->avail -= roundup;
	//p->bytes_align_wasted += roundup;
	bytes_align_wasted += roundup;
    }
    return 0;				/* ok */
}

void *
abs_pool_malloc (struct abs_pool *p, size_t size, const char *place)
{
    unsigned int roundup;

    if (p == NULL)
	abs_abort ("abs_pool_malloc", "Called with NULL pool at %s", place);

    if (size == 0)
	return NULL;

    /*
     * If there is insufficient space for the padding, but the
     * requested size is larger than the necessary padding,
     * a new block will be allocated in any case.
     * Otherwise, we increase the size a bit,
     * which then also forces a new block.
     *   size  < roundup < POOL_ALIGN
     *   avail < roundup
     * In both cases, the allocation will be aligned.
     * (Alternatively, set b->avail to 0)
     */
    p->unaligned_allocs_req--;
    roundup = abs_pool_align_block(p->block);
    if (roundup > 0 && size < roundup) {
	assert (p->block->avail < POOL_ALIGN);
	bytes_align_wasted += p->block->avail;
	p->block->avail = 0;
    }
    return abs_pool_malloc_unaligned(p, size, place);
}

void *
abs_pool_malloc_unaligned (struct abs_pool *p, size_t size, const char *place)
{
    struct abs_pool_block *b;

    p->nr_allocs++;
    p->unaligned_allocs_req++;
    b = p->block;
    if (((intptr_t)b->mem) & POOL_MASK)
	p->unaligned_allocs++;

    if (size <= b->avail)
	return abs_pool_malloc_from_block(b, size);

    /* Too large to fit in a single block? */
    if (size + sizeof(struct abs_pool_block) > p->block_size) {
	b = abs_pool_new_block(p, size + sizeof(struct abs_pool_block));
	/*
	 * Move the new and full block away from the front,
	 * if there is a second block anyway.
	 */
	if (b->prev) {
	    struct abs_pool_block *second = b->prev;
	    p->block = second;
	    b->prev = second->prev;
	    second->prev = b;
	}
	return abs_pool_malloc_from_block(b, size);
    }

    /*
     * Is it worth it to look through previous non-full blocks to fill
     * up some extra space? (This is an area which may influence
     * memory efficiency, but at the moment that doesn't seem to be
     * an issue.)
     */
    if (size >= 1024) {
	b = b->prev;
	while (b != NULL) {
	    p->nr_prevs_searched++;
	    if (size <= b->avail)
		return abs_pool_malloc_from_block(b, size);
	    b = b->prev;
	}
    }

    if (p->block_size < p->max_block_size)
	p->block_size *= 2;

    b = abs_pool_new_block(p, 0);
    return abs_pool_malloc_from_block(b, size);

}

void *abs_pool_calloc (struct abs_pool *p, const size_t nr, const size_t size, const char *place)
{ void *result;
  if (size == 1)
    result = abs_pool_malloc_unaligned (p, nr, place);
  else
    result = abs_pool_malloc (p, nr * size, place);
  memset (result, 0, nr * size);
  return result;
};

char *abs_pool_new_string (struct abs_pool *p, const char *old, const char *place)
{ char *result;
  if (old == NULL)
     abs_abort ("abs_pool_new_string", "Called with NULL argument at %s", place);
  result = (char *) abs_pool_malloc_unaligned (p, strlen (old) + 1, place);
  strcpy (result, old);
  return (result);
};

char *abs_pool_new_fmtd_string (struct abs_pool *p, const char *place, const char *format, ...)
{ char buf[MAX_FMT_LEN];
  va_list arg_ptr;
  if (format == NULL)
     abs_abort ("abs_pool_new_fmtd_string", "Called with NULL format at %s", place);
  va_start (arg_ptr, format);
#if HAVE_VSNPRINTF
  vsnprintf (buf, MAX_FMT_LEN, format, arg_ptr);
#else
  vsprintf (buf, format, arg_ptr);
#endif
  va_end (arg_ptr);
  return (abs_pool_new_string (p, buf, place));
};
