/*
   File: dyn_array.c
   Maintains variable sized arrays of integers and texts

   Copyright 2009-2010 Radboud University of Nijmegen

   This program is free software: you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation, either version 3 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program.  If not, see <http://www.gnu.org/licenses/>.

   CVS ID: "$Id$"
*/

/* system includes */
#include <stdio.h>
#include <string.h>

/* libabase includes */
#include <abase_porting.h>
#include <abase_repr.h>
#include <abase_error.h>
#include <abase_fileutil.h>
#include <abase_memalloc.h>

/* local includes */
#include "dyn_array.h"

/* Allocate new int_array */
int_array init_int_array (int room)
{ int_array new_arr = (int_array) abs_malloc (sizeof (struct int_array_rec), "init_int_array");
  int safe_room = (room < 2)?2:room;
  new_arr -> size = 0;
  new_arr -> room = safe_room;
  new_arr -> array = (int *) abs_calloc (safe_room, sizeof (int), "init_int_array");
  return (new_arr);
};

/* Allocate new text_array */
text_array init_text_array (int room)
{ text_array new_arr = (text_array) abs_malloc (sizeof (struct text_array_rec), "init_text_array");
  int safe_room = (room < 2)?2:room;
  new_arr -> size = 0;
  new_arr -> room = safe_room;
  new_arr -> array = (char **) abs_calloc (safe_room, sizeof (char *), "init_text_array");
  return (new_arr);
};

/* Allocate new ptr_array */
ptr_array init_ptr_array (int room)
{ ptr_array new_arr = (ptr_array) abs_malloc (sizeof (struct ptr_array_rec), "init_ptr_array");
  int safe_room = (room < 2)?2:room;
  new_arr -> size = 0;
  new_arr -> room = safe_room;
  new_arr -> array = (void **) abs_calloc (safe_room, sizeof (void *), "init_ptr_array");
  return (new_arr);
};

/* Announce to use 'room' chunks for int_array */
void room_int_array (int_array arr, int room)
{ if (room <= arr -> room) return;
  arr -> array = (int *) abs_realloc ((void *) arr -> array,
				      room * sizeof (int), "room_int_array");
  arr -> room = room;
};

/* Announce to use 'room' chunks for text_array */
void room_text_array (text_array arr, int room)
{ if (room <= arr -> room) return;
  arr -> array = (char **) abs_realloc ((void *) arr -> array,
					room * sizeof (char *), "room_text_array");
  arr -> room = room;
};

/* Announce to use 'room' chunks for ptr_array */
void room_ptr_array (ptr_array arr, int room)
{ if (room <= arr -> room) return;
  arr -> array = (void **) abs_realloc ((void *) arr -> array,
					room * sizeof (void *), "room_ptr_array");
  arr -> room = room;
};

/* Append element to int_array */
void app_int_array (int_array arr, int el)
{ if (arr -> size == arr -> room)
    room_int_array (arr, arr -> size << 1);
  arr -> array[arr -> size] = el;
  arr -> size++;
};

/* Append element to text_array */
void app_text_array (text_array arr, char *el)
{ if (arr -> size == arr -> room)
    room_text_array (arr, arr -> size << 1);
  arr -> array[arr -> size] = el;
  arr -> size++;
};

/* Append element to ptr_array */
void app_ptr_array (ptr_array arr, void *el)
{ if (arr -> size == arr -> room)
    room_ptr_array (arr, arr -> size << 1);
  arr -> array[arr -> size] = el;
  arr -> size++;
};

/* Check if arg is in int_array */
int is_in_int_array (int_array arr, int el)
{ int ix;
  if (arr == int_array_nil)
    abs_bug ("is_in_int_array", "called with null array");
  for (ix = 0; ix < arr -> size; ix++)
    if (arr -> array[ix] == el) return (1);
  return (0);
};

/* Check if arg is on text_array */
#define equal_text(s1,s2) (strcmp((s1),(s2)) == 0)
int is_in_text_array (text_array arr, char *el)
{ int ix;
  if (arr == text_array_nil)
    abs_bug ("is_in_text_array", "called with null array");
  for (ix = 0; ix < arr -> size; ix++)
    if (equal_text (arr -> array[ix], el)) return (1);
  return (0);
};

/* Equality test for int_array's */
int equal_int_array (int_array arr1, int_array arr2)
{ int ix;
  if (arr1 -> size != arr2 -> size) return (0);
  for (ix = 0; ix < arr1 -> size; ix++)
    if (arr1 -> array[ix] != arr2 -> array[ix])
      return (0);
  return (1);
}

/* Uniquely append element to int_array */
void app_uniq_int_array (int_array arr, int el)
{ if (arr == int_array_nil)
    abs_bug ("app_uniq_int_array", "called with null array");
  if (is_in_int_array (arr, el)) return;
  app_int_array (arr, el);
};

/* Uniquely append element to int_array */
void app_uniq_text_array (text_array arr, char *el)
{ if (arr == text_array_nil)
    abs_bug ("app_uniq_text_array", "called with null array");
  if (is_in_text_array (arr, el)) return;
  app_text_array (arr, el);
};

/* Add element to int_array keeping the array sorted and absorbing */
void app_sorted_int_array (int_array arr, int el)
{ int ix, here;
  if (arr == int_array_nil)
    abs_bug ("app_sorted_int_array", "called with null array");

  /* Locate where to insert */
  for (here = 0; (here < arr -> size) && (arr -> array[here] < el); here++) ;
  if ((here < arr -> size) && (arr -> array[here] == el)) return;

  /* Make room for insertion */
  if (arr -> size == arr -> room)
    room_int_array (arr, arr -> size << 1);

  /* Ok, do it */
  for (ix = arr -> size; here < ix; ix--)
     arr -> array[ix] = arr -> array[ix - 1];
  arr -> array[here] = el;
  arr -> size++;
}

/* Binary saving of int arrays */
void bin_save_int_array (BinFile bf, int_array arr)
{ int ix;
  abs_bin_save_int (bf, arr -> size);
  for (ix = 0; ix < arr -> size; ix++)
    abs_bin_save_int (bf, arr -> array [ix]);
}

void bin_save_text_array (BinFile bf, text_array arr)
{ int ix;
  abs_bin_save_int (bf, arr -> size);
  for (ix = 0; ix < arr -> size; ix++)
    abs_bin_save_string (bf, arr -> array [ix]);
}

int bin_cmp_int_array (BinFile bf, int_array arr)
{ int size, ival, ix;
  abs_bin_load_int (bf, &size);
  if (size != arr -> size) return (1);
  for (ix = 0; ix < size; ix++)
    { abs_bin_load_int (bf, &ival);
      if (ival != arr -> array[ix]) return (1);
    };
  return (0);
}

int bin_cmp_text_array (BinFile bf, text_array arr)
{ int size, ix;
  char *sval;
  abs_bin_load_int (bf, &size);
  if (size != arr -> size) return (1);
  for (ix = 0; ix < size; ix++)
    { int status;
      abs_bin_load_string (bf, &sval);
      status = strcmp (sval, arr -> array[ix]);
      abs_free (sval, "bin_cmp_text_array");
      if (status) return (1);
    };
  return (0);
}
