/*
   File: trel_input.h
   Manage input and store in a trellis.

   Copyright 2009 Radboud University of Nijmegen
 
   This program is free software: you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation, either version 3 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program.  If not, see <http://www.gnu.org/licenses/>.

*/
#ifndef IncTtrelInput
#define IncTtrelInput

#include "lxcn_lexicon.h"
#include "pattern.h"		/* XXX copy of application, for now XXX */

#ifdef __cplusplus
extern "C" {
#endif

struct state;
struct transition;
struct trellis;
typedef struct state *State;
typedef struct transition *Transition;
typedef struct trellis *Trellis;

#define LEX_STATE_S		0
#define LEX_STATE_W		1
#define LEX_STATE_I		2
#define LEX_STATE_E		LEX_STATE_S
#define NUM_LEX_STATES		3
#define LEX_STATE_ERROR		-1

Trellis trel_init(int nr_lex_nonts);
void trel_delete(Trellis);
State trel_get_initial_state(Trellis);
int trel_get_length(Trellis);
State trel_may_get_state(Trellis t, int pos, int lex_state);
State trel_get_eos_state(Trellis t);
Transition trel_get_eos_transition(Trellis t);
void trel_free_states_before(Trellis, State s);

typedef void (*trel_free_state_callback)(Trellis, State);
typedef void (*trel_free_trans_callback)(Trellis, Transition);
/* callback for initialising a new StateExtention */
typedef void (*trel_new_state_callback)(Trellis t, State state);

void trel_set_new_state_callback(Trellis t, trel_new_state_callback f);
void trel_set_free_state_callback(Trellis t, trel_free_state_callback f);
void trel_set_free_trans_callback(Trellis t, trel_free_trans_callback f);

void trel_set_input(Trellis t, const char *input, int line, int col);
void trel_set_lexicon(Trellis t, Lexicon l);
void trel_set_white(Trellis t, const char *white);
void trel_set_separators(Trellis t, const char *separators);
void trel_set_match_regex(Trellis t, RegExp *match);
void trel_set_skip_regex(Trellis t, RegExp *skip);

Transition trel_alloc_transition(Trellis t);
void trel_free_transition(Trellis t, Transition trans);

char *trel_trans_get_text(Transition t);
void *trel_trans_get_ext(Transition t);
void trel_trans_set_ext(Transition t, void *ext);
Transition trel_trans_get_next(Transition);
Transition *trel_trans_get_nextptr(Transition);
void trel_trans_set_next(Transition, Transition next);
State trel_trans_get_dest(Transition);
void trel_trans_set_dest(Transition, State);
int trel_trans_get_length(Transition trans);

void *trel_state_get_ext(State s);
void trel_state_set_ext(State s, void *ext);
int trel_state_get_pos(State s);
char trel_state_get_lex_state(State s);
int trel_state_get_line(State s);
int trel_state_get_col(State s);
Transition trel_state_get_trans(State s, int nr);
void trel_state_set_trans(State s, int nr, Transition trans);
int trel_state_is_at_whitespace(Trellis t, State s);

Transition trel_state_scan_whitespace(Trellis t, State s);
Transition trel_state_scan_lexterminal(Trellis t, State s, int nr);
Transition trel_state_scan_terminal(Trellis t, State s);
Transition trel_state_scan_match(Trellis t, State s);
Transition trel_state_scan_skip(Trellis t, State s);
Transition trel_state_scan_any(Trellis t, State s);
Transition trel_state_scan_word(Trellis t, State s);
Transition trel_state_scan_other(Trellis t, State s);
Transition trel_state_scan_eos(Trellis t, State s);
State trel_state_glue(Trellis t, State s);
State trel_state_dissolve(Trellis t, State s);
Transition trel_get_eos(Trellis t);

/*
 * Callbacks for initialising a new Transition Extention.
 *
 * You are free to allocate some extra memory to store application
 * specific data for the transition. You can store a pointer to this
 * extension memory in the transition with trel_trans_set_ext() and get
 * it back with trel_trans_get_ext(). 
 *
 * A few abuses of the callbacks:
 *
 * trel_scanned_lexicon: Returns nonterminal class, or -1 for terminal.
 *                       Also, there is a terminating call with trans==NULL,
 *                       entries==NULL, idx==0, lex_type==0 after all
 *                       different calls for a single state have been done.
 * trel_scanned_match: If called with trans == NULL, it is a pre-call,
 *                     which must return the LexemeType of regexp #id.
 * trel_scanned_skip: ditto.
 */
typedef int (*trel_scanned_lexicon)(Trellis t, State s, Transition trans, int *entries, int idx, int lex_type);
typedef int (*trel_scanned_match)(Trellis t, State s, Transition trans, int alt);
typedef void (*trel_scanned_white)(Trellis t, State s, Transition trans);
typedef void (*trel_scanned_eos)(Trellis t, State s, Transition trans);
typedef void (*trel_scanned_other)(Trellis t, State s, Transition trans);
typedef void (*trel_scanning_completed)(Trellis t, State s);

void trel_set_scanned_lexicon(Trellis t, trel_scanned_lexicon cb);
void trel_set_scanned_match(Trellis t, trel_scanned_match cb);
void trel_set_scanned_skip(Trellis t, trel_scanned_match cb);
void trel_set_scanned_white(Trellis t, trel_scanned_white cb);
void trel_set_scanned_eos(Trellis t, trel_scanned_eos cb);
void trel_set_scanned_other(Trellis t, trel_scanned_other cb);
void trel_set_scanning_completed(Trellis t, trel_scanning_completed cb);

/* Iterating over states */
State trel_state_next(Trellis t, State s);
State trel_state_next_white(Trellis t, State s, int skip);

/* Iterating over transitions */
struct transiter;
typedef struct transiter *TransIter;
TransIter trel_trans_iter(Trellis t, State s);
Transition trel_trans_iter_next(TransIter);
void trel_trans_iter_end(TransIter);


#ifdef __cplusplus
}
#endif
#endif /* IncTtrelInput */
