/* 
   File: abase_mm_alloc.c
   Defines managed memory allocation routines

   Copyright 2009 Radboud University of Nijmegen

   This program is free software: you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation, either version 3 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program.  If not, see <http://www.gnu.org/licenses/>.

   CVS ID: "$Id$"
*/

/* standard includes */
#ifdef HAVE_CONFIG_H
#include <config.h>
#endif /* HAVE_CONFIG_H */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <stdarg.h>

/* Local includes */
#include "abase_porting.h"
#include "abase_error.h"
#include "abase_memalloc.h"
#include "abase_mm_alloc.h"

#define	USE_POOL_FOR_MM	1

#if !USE_POOL_FOR_MM
/*
   Basically the managed routines have the same interface as
   the nonmanaged routines, but their location is remembered
   on a linear list for later removal. We need this kind of
   allocation because affix values dynamically allocated
   during pass one must survive until the positive memos can
   be freed.
*/
typedef struct mm_rec *mm_list;
struct mm_rec
	{ void *entry;
	  mm_list next;
	};
static mm_list mm_anchor;

void abs_mm_init (void)
	{ mm_anchor = NULL;
	}

static void enter_into_mm_list (void *entry, const char *place)
	{ mm_list new_rec = (mm_list) abs_malloc (sizeof (struct mm_rec), place);
	  new_rec -> entry = entry;
	  new_rec -> next = mm_anchor;
	  mm_anchor = new_rec;
	}

void abs_mm_intermediate_free (const char *place)
	{ while (mm_anchor != NULL)
	    { mm_list old = mm_anchor;
	      mm_anchor = mm_anchor -> next;
	      abs_free (old -> entry, place);
	      abs_free (old, place);
	    };
	};

void abs_mm_final (const char *place)
	{ abs_mm_intermediate_free (place);
	};

void *abs_mm_malloc (const size_t size, const char *place)
	{ void *result = abs_malloc (size, place);
	  enter_into_mm_list (result, place);  
	  return (result);
	};

void *abs_mm_calloc (const size_t nr, const size_t size, const char *place)
	{ void *result = abs_calloc (nr, size, place);
	  enter_into_mm_list (result, place);  
	  return (result);
	};

char *abs_mm_new_string (const char *old, const char *place)
	{ char *result;
	  if (old == NULL)
	     abs_abort ("abs_mm_new_string", "Called with NULL argument at %s", place);
	  result = (char *) abs_mm_malloc (strlen (old) + 1, place);
	  strcpy (result, old);
	  return (result);
	};

char *abs_mm_new_fmtd_string (const char *place, const char *format, ...)
	{
#if 0 && HAVE_VASPRINTF
	  char *buf;
	  va_list arg_ptr;
	  char *ret;
	  va_start (arg_ptr, format);
	  vasprintf (&buf, format, arg_ptr);
	  va_end (arg_ptr);
	  ret = abs_mm_new_string (buf, place);

	  free(buf);
	  return ret;
#else /* HAVE_VASPRINTF */
	  char buf[MAX_FMT_LEN];
	  va_list arg_ptr;
	  va_start (arg_ptr, format);
#if HAVE_VSNPRINTF
	  vsnprintf (buf, MAX_FMT_LEN, format, arg_ptr);
#else
	  vsprintf (buf, format, arg_ptr);
#endif
	  va_end (arg_ptr);
	  return (abs_mm_new_string (buf, place));
#endif /* HAVE_VASPRINTF */
	};

#else /* USE_POOL_FOR_MM */

#include "abase_pool_alloc.h"

static Pool mm_pool;

void abs_mm_init (void)
	{ mm_pool = abs_pool_init ( 256 * 1024, 4 * 1024 * 1024 );
	}

void abs_mm_intermediate_free (const char *place)
	{ abs_mm_final (place);
	  abs_mm_init ();
	};

void abs_mm_final (const char *place)
	{ /* abs_pool_stats (mm_pool, place); */
          abs_pool_free (mm_pool, place);
	  mm_pool = NULL;
	};

void *abs_mm_malloc (const size_t size, const char *place)
	{ return abs_pool_malloc (mm_pool, size, place);
	};

void *abs_mm_calloc (const size_t nr, const size_t size, const char *place)
	{ return abs_pool_calloc (mm_pool, nr, size, place);
	};

char *abs_mm_new_string (const char *old, const char *place)
	{ return abs_pool_new_string (mm_pool, old, place);
	};

char *abs_mm_new_fmtd_string (const char *place, const char *format, ...)
	{ char buf[MAX_FMT_LEN];
	  va_list arg_ptr;
	  va_start (arg_ptr, format);
#if HAVE_VSNPRINTF
	  vsnprintf (buf, MAX_FMT_LEN, format, arg_ptr);
#else
	  vsprintf (buf, format, arg_ptr);
#endif
	  va_end (arg_ptr);
	  return (abs_pool_new_string (mm_pool, buf, place));
	};

#endif
