/*
   File: arun_options.c
   Specifies option and argument treatment

   Copyright 2006-2010 Radboud University of Nijmegen
 
   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.
 
   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU Library General Public License for more details.
 
   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.

   CVS ID: "$Id$"
*/

/* standard includes */
#ifdef HAVE_CONFIG_H
#include <config.h>
#endif /* HAVE_CONFIG_H */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <limits.h>
#include <time.h>
#include <ctype.h>

/* libabase includes */
#include <abase_porting.h>
#include <abase_error.h>
#include <abase_version.h>
#include <abase_memalloc.h>

/* Ugly include for max_edit_distance */
#include <lxcn_search.h>

/* libarts includes */
#include "arts_ds.h"
#include "arts_io.h"
#include "arts_minitdb.h"

/* local includes */
#include "arun_options.h"

#define USE_EDIT_DISTANCE 1

static void show_usage ();

/*-----------------------------------------------------------------------------
// Private options to be able to overrule the grammar defaults
//---------------------------------------------------------------------------*/
static char *alphabet_fname = NULL;
static int show_options_option;
static int transduce_option;
static int graph_option;
static int closed_triple_db_option;
static int no_hybrid_execution;
static int paragraph_mode;
static int segment_mode;
static int counters_option;
static struct radixes radix;

static void set_frequency_radixes (char *s, struct radixes *sr);

static void init_options ()
{ show_options_option = 0;
  no_hybrid_execution = 0;
  transduce_option = -1;
  closed_triple_db_option = -1;
  graph_option = -1;
  paragraph_mode = -1;
  segment_mode = -1;
  counters_option = -1;
  { char *name = getenv("USER");
    if (name == NULL) name = getenv("LOGNAME");
    if (name != NULL && strcmp(name, "kees") == 0) arts_ifd.kees_option = 1;

    /* This option must be checked early, before messages are printed */
    if (arts_ifd.kees_option)
      { dup2(fileno(stdout), fileno(stderr));
      }
  }
  radix.lexicon_frequency  = -1;
  radix.tripledb_frequency  = -1;
  radix.penalty  = -1;
  radix.fact_frequency  = -1;
}

/* Change when pragma max_parses is introduced */
static void set_max_number_of_parses (int v)
{ if (v < 1) arts_ifd.max_parses = 1;
  else arts_ifd.max_parses = v;
}

static void set_max_number_of_terminal_parses (int v)
{ if (v < 1) arts_ifd.max_terminal_parses = 1;
  else arts_ifd.max_terminal_parses = v;
}

static void set_max_posmemo_queue_length (int v)
{ if (v < 1) arts_ifd.max_posmemo_queue_length = 1;
  else arts_ifd.max_posmemo_queue_length = v;
}

static void set_max_edit_distance (int v)
{ if (v < 0) max_edit_distance = 0;
  else max_edit_distance = v;
}

static void set_max_parse_time (int v)
{ if (v < 0) arts_ifd.max_parsetime = 0;
  else arts_ifd.max_parsetime = v;
}


static void set_lexicon_frequency_radix  (int v, struct radixes *r)
{ if (v < 0) r->lexicon_frequency = 0;
  else r->lexicon_frequency = v;
}

static void set_tripledb_frequency_radix  (int v, struct radixes *r)
{ if (v < 0) r->tripledb_frequency = 0;
  else r->tripledb_frequency = v;
}

static void set_penalty_radix  (int v, struct radixes *r)
{ if (v < 0) r->penalty = 0;
  else r->penalty = v;
}

static void set_fact_frequency_radix  (int v, struct radixes *r)
{ if (v < 0) r->fact_frequency = 0;
  else r->fact_frequency = v;
}

/*
 * Parse a string like "10,o8,t6".
 *
 */
static void set_max_parses (char *s)
{ while (s && *s)
    { if (isdigit(*s))
	{ int r = atoi(s);
	  set_max_number_of_parses(r);
	  /* set_max_number_of_terminal_parses(r); */
	  set_max_posmemo_queue_length(r);
	}
      else if (*s == 'o')
        { int r = atoi(s + 1);
	  set_max_number_of_parses(r);
	}
      else if (*s == 'q')
        { int r = atoi(s + 1);
	  set_max_posmemo_queue_length(r);
	  /* set_max_number_of_terminal_parses(r); */
	}
      else if (*s == 't')
        { int r = atoi(s + 1);
	  set_max_number_of_terminal_parses(r);
	}
      else
        { abs_error ("unknown option '-P%s'.", s);
	  show_usage();
	  return;
	}
      s = strchr(s, ',');
      if (s == NULL) break;
      s++;
    }
}

/*
 * Parse a string like "10,l8,f6,t4".
 *
 */
static void set_frequency_radixes (char *s, struct radixes *sr)
{ while (s && *s)
    { if (isdigit(*s))
	{ int r = atoi(s);
	  set_lexicon_frequency_radix(r, sr);
	  set_tripledb_frequency_radix(r, sr);
	  set_fact_frequency_radix(r, sr);
	}
      else if (*s == 'l')
        { int r = atoi(s + 1);
	  set_lexicon_frequency_radix(r, sr);
	}
      else if (*s == 'f')
        { int r = atoi(s + 1);
	  set_fact_frequency_radix(r, sr);
	}
      else if (*s == 'p')
        { int r = atoi(s + 1);
	  set_penalty_radix(r, sr);
	}
      else if (*s == 't')
        { int r = atoi(s + 1);
	  set_tripledb_frequency_radix(r, sr);
	}
      else
        { abs_error ("unknown option '-X%s'.", s);
	  show_usage();
	  return;
	}
      s = strchr(s, ',');
      if (s == NULL) break;
      s++;
    }
}

/*------------------------------------------------------------------------------
// Public
//----------------------------------------------------------------------------*/
static void sprint_quoted (char *buf, char *s)
{ char c;
  char *p = buf;

  while ((c = *s++))
    switch (c)
      { case '\n':
	  sprintf (p, "\\n");
	  p += 2;
	  break;
	case '\r':
	  sprintf (p, "\\r");
	  p += 2;
	  break;
	case '\t':
	  sprintf (p, "\\t");
	  p += 2;
	  break;
	default:
	  *p++ = c;
	  break;
      };
  *p = '\0';
}

static char* convert_time_to_text (int time)
{ time_t clock = (time_t) time;
  char *txt_time = ctime (&clock);
  txt_time[24] = '\0';
  return (txt_time);
}

void try_show_options()
{ char buf[256];
  if (!show_options_option) return;
  abs_message ("Grammar last changed on %s", convert_time_to_text (arts_ifd.gra_version));

  if (arts_ifd.white_space_chars != NULL)
    { sprint_quoted (buf, arts_ifd.white_space_chars);
      abs_message ("White space characters are '%s'", buf);
    };

  /* To be activated when max_edit_distance is used */
#if USE_EDIT_DISTANCE
  if (max_edit_distance > 0)
    abs_message ("Maximum edit distance is %d", max_edit_distance);
#endif

  if (arts_ifd.graph_option)
    abs_message ("Displaying lexical graph");

  if (arts_ifd.directors_option)
    abs_message ("Directors available");

  if (arts_ifd.directors_set_posmemos)
    abs_message ("Directors used to initialise positive and negative memo");

  if (arts_ifd.neg_memo_option)
    abs_message ("Negative memoization available");

  if (arts_ifd.max_parses < INT_MAX)
    abs_message ("Maximum number of parses is %d", arts_ifd.max_parses);

  if (arts_ifd.max_posmemo_queue_length < INT_MAX)
    abs_message ("Maximum posmemo queue length is %d", arts_ifd.max_posmemo_queue_length);

  if (arts_ifd.max_terminal_parses < INT_MAX)
    abs_message ("Maximum posmemo queue length for TERMINALs is %d", arts_ifd.max_terminal_parses);

  if (arts_ifd.max_parsetime < INT_MAX)
    abs_message ("Maximum parse time is %d seconds", arts_ifd.max_parsetime);

  if (arts_ifd.label_bracket && !arts_ifd.no_output)
    abs_message ("Labelled bracket output");

  if (arts_ifd.no_output)
    abs_message ("Suppressing output");
  else if (arts_ifd.transduce_option && !arts_ifd.no_output)
    abs_message ("Transducing");

  if (arts_ifd.counters_option)
    abs_message ("Showing counters");

  if (arts_ifd.generate_option)
    abs_message ("This is a sentence generator");

  if (arts_ifd.best_parsings_option)
    abs_message ("Best parsing enabled");

  if (arts_ifd.hybrid_parsing_option)
    abs_message ("Hybrid parsing enabled");

  if (arts_ifd.closed_triple_db_option)
    abs_message ("Closed Triple DB enabled");

  if (arts_ifd.transition_penalties_option)
    abs_message ("Penalties on transitions enabled");

  if (arts_ifd.paragraph_mode)
    abs_message ("Input operates in |paragraph mode|");
  else
    abs_message ("Input operates in |line mode|");

  if (arts_ifd.segment_mode)
    abs_message ("Grammar operates in |segment mode|");
  else
    abs_message ("Grammar operates in |document mode| (full-coverage)");

  if (arts_ifd.input_fname != NULL)
    abs_message ("Input filename: '%s'", arts_ifd.input_fname);

  abs_message("Radixes for converting {lexicon, tripledb, fact} frequencies to bonuses: %d, %d, %d",
	  arts_ifd.radix.lexicon_frequency,
	  arts_ifd.radix.tripledb_frequency,
	  arts_ifd.radix.fact_frequency);
  abs_message("Radix for $PENALTY: %d", arts_ifd.radix.penalty);
}

/*-----------------------------------
// -h shows options
//---------------------------------*/
static void try_show_version ()
{ abs_message ("This is the AGFL runtime environment, version %s", AGFL_VERSION);
}

static void show_usage ()
{ abs_message ("Usage: agfl-run [options...] grammar_name [input_fname]");
  abs_message (
	"OPTION      DEFAULT     PURPOSE\n"
	"-a          off         absorb equivalent parsings\n"
	"-B          off         search parsings for best penalty level\n"

	"-d          off         suppress identical transduction outputs\n"
	"-C          off         use closed triple db\n"
	"-N          off         suppress hybrid parsing\n"
	"-S          off         use ctrl-A as document separator\n"
	"-V                      show version\n"
	"\n"

	/* transduction options */
	"-t          off         transduce\n"
	"-ts         on          standard tree transduction\n"
	"-tb         off         labelled bracket output\n"
	"-ti         off         identity transduction output\n"
	"-tO         off         suppress output\n"
	"\n"

	"-p number   0           input in paragraph mode (0 = line mode)\n"
	"   segment              parse in |segment mode|\n"
	"   document             parse in |document mode|\n"
	"   line                 input in |line mode|\n"
	"   paragraph            input in |paragraph mode|\n"
	/* options with numeric argument */
#if USE_EDIT_DISTANCE
	"-E number   0           maximum edit distance\n"
#endif
	"-l number   0           dis/enable lexical penalties/bonuses on transitions\n" 
	"-m number   unlimited   maximum penalty\n" 
	"-P number   unlimited   maximum number of parses\n"
	"-P 1,q5,t1              -Pq<n>: posmemo queue length (sets -Pt<n> too);\n"
	"                        -Pt<n>: posmemo queue length for TERMINAL rules only\n"
	"-T number   unlimited   maximum total parse time (in seconds)\n"
	"-X number   10          radix for converting frequencies to bonuses\n" 
	"-X l10,f10,t10          (separate radixes for lexicon, facts, triples)\n" 
	"-X p1       1           (separate radix for $PENALTY)\n" 
	"\n"

	/* options with string argument */
	"-A filename off         use alphabet file\n"
	"-i filename input_fname use name to format output of $POS\n"
	"-o filename stdout      write output to file\n"
	"-v option   off         set verbosity. possible values:\n"
	"   options              show parser options\n"
	"   lexer                show all lexer output\n"
	"   parser               show parser stats\n"
	"   research             show research stats\n"
	"   totals               show totals (ie. time)\n"
	"   lpt                  lexer+parser+totals\n"
	"\n"

	/* long name options */
	"Long options:\n"
	"-O trp:filename         add very small extra triple list (< 10 items)\n"
	"-O notrp                disable use of normal triple database\n"
	"-O trp-trans            translate triple-lookups with the alphabet file\n"
	"-O prod                 include 'production' in onderscheidbaarheid\n"
	"-O noautopos            disable automatic position printing at TERMINALs\n"
	"\n"

	/* research options */
	"Research options:       (may be joined with a comma)\n"
	"-rG         off         show lexical graph\n"
	"-rGv        off         show lexical graph verbosely\n"
	"-rc (-rcv)  off         count opcodes and rule calls (verbosely)\n"
	"-rD         off         disable use of director set to block memos\n"
	"-rposmemo   off         print posmemo table\n"
	"-rtrp       off         print triple lookups\n"
	"-rfreemem   off         explicitly free memory at end\n"
	"\n"

	"-h          off         show this help");
    abs_exit (0);
}

#ifdef WIN32
static char *strsep(char **stringp, const char *delim)
{ char *s = *stringp;
  char *tok;

  if (s == NULL) return (NULL);
  for (tok = s;;)
    { const char *spanp = delim;
      int c = *s++;
      int sc;
      do
        { sc = *spanp++;
	  if (sc == c)
            { if (c == 0) s = NULL;
              else s[-1] = 0;
              *stringp = s;
              return (tok);
            }
        }
      while (sc != 0);
    }
  return (NULL);
}
#endif

void check_for_verbose_options (char *arg)
{ for (;;)
    { char *opt;

      opt = strsep(&arg, ",");
      if (!opt) break;
      if (!strcasecmp (opt, "lpt"))
	{ arts_ifd.lexer_stats_option = 1;
  	  arts_ifd.parsing_stats_option = 1;
	  arts_ifd.total_stats_option = 1;
	}
      else if (!strcasecmp (opt, "p") || !strcasecmp (opt, "parser"))
        { arts_ifd.parsing_stats_option = 1;
        }
      else if (!strcasecmp (opt, "research"))
        { arts_ifd.research_stats_option = 1;
        }
      else if (!strcasecmp (opt, "l") || !strcasecmp (opt, "lexer"))
        { arts_ifd.lexer_stats_option = 1;
        }
      else if (!strcasecmp (opt, "t") || !strcasecmp (opt, "totals"))
        { arts_ifd.total_stats_option = 1;
        }
      else if (!strcasecmp (opt, "trp"))
        { arts_ifd.triple_stats_option = 1;
        }
      else if (!strcasecmp (opt, "options"))
        { show_options_option = 1;
        }
      /* For compatibility, leave this here for a little while */
      else if (!strcasecmp (opt, "posmemo"))
        { arts_ifd.show_posmemo_option = 1;
        }
      else
	{ abs_error ("Illegal verbose option '%s'", opt);
          show_usage ();
	}
    }
}

static char *get_optarg (char option, char *ptr, int *ix, int argc, char **argv)
{ /* Check if argument in same command line argument */
  if (*(++ptr))
    return (ptr);

  /* Check if there is a next argument */
  *ix += 1;
  if (*ix < argc)
    return (argv[*ix]);

  /* Complain */
  abs_error ("Missing argument with option '-%c'", option);
  show_usage ();

  /* Unreachable */
  return (NULL);
}

static void scan_options (char *ptr, int *ix, int argc, char **argv) 
{ do
    { switch (*ptr)
	{ /* Concatenable boolean options */
	  case 'a': arts_ifd.absorb_equivalents = 1; break;
	  case 'B': arts_ifd.best_parsings_option = 1; break;
	  case 'C': closed_triple_db_option = 1; break;
	  case 'd': arts_ifd.suppress_identical_transductions = 1; break;
	  case 'N': no_hybrid_execution = 1; break;
	  case 'S': arts_ifd.lcsdoc_sync_option = 1; break;

	  /* Research options, which are prefixed with -r.
	   * They can be combined with a comma.
	   */
	  case 'r': {	/* -rG,Gv,c,D */
		char *arg = get_optarg (*ptr, ptr, ix, argc, argv);
		for (;;)
		  { char *opt;

		    opt = strsep(&arg, ",");
		    if (!opt) break;
		    if (strcmp(opt, "G") == 0) {
			graph_option = 1;
		    } else if (strcmp(opt, "Gv") == 0) {
			graph_option = 2;
		    } else if (strcmp(opt, "c") == 0) {
			counters_option = 1;
		    } else if (strcmp(opt, "vc") == 0) {
			counters_option = 2;
		    } else if (strcmp(opt, "D") == 0) {
			arts_ifd.directors_set_posmemos = 0;
		    } else if (strcmp(opt, "posmemo") == 0) {
			arts_ifd.show_posmemo_option = 1;
		    } else if (strcmp(opt, "trp") == 0) {
			arts_ifd.show_triple_lookups_option++;
		    } else if (strcmp(opt, "freemem") == 0) {
			arts_ifd.free_mem_option = 1;
		    }  else {
			abs_error ("unknown option '-r%s'.", opt);
			show_usage();
			break;
		    }
		  }
		return;
	    }
	  /* Transduction type option */
	  case 't':
		switch(ptr[1])
		  { case '\0': transduce_option = 1; break;
		    case  'b': arts_ifd.label_bracket = 1; break;
		    case  'g': break;
		    case  'i': arts_ifd.identity_transduction = 1; break;
		    case  'O': arts_ifd.no_output = 1; break;
		    case  's': transduce_option = 0; break;
		    case  't': transduce_option = 1; break;
		    default  : abs_error ("unknown option '-t%c'.", ptr[1]);
		               show_usage(); break;
		  }
		return;
	  case 'p':
	    { char *arg = get_optarg (*ptr, ptr, ix, argc, argv);
	      if (arg[0] == '0' || arg[0] == '1')
	         paragraph_mode = (atoi (arg));
	      else if (strcmp(arg, "segment")==0)   segment_mode = 1;
	      else if (strcmp(arg, "document")==0)  segment_mode = 0;
	      else if (strcmp(arg, "paragraph")==0) paragraph_mode = 1;
	      else if (strcmp(arg, "line")==0)      paragraph_mode = 0;
	      else
	       { abs_error ("unknown option '-p %s'.", arg);
		 show_usage();
	       }
	      return;
	    };
	  case 'O': {	/* other long options */
		char *arg = get_optarg (*ptr, ptr, ix, argc, argv);
		for (;;)
		  { char *opt;

		    opt = strsep(&arg, ",");
		    if (!opt) break;
		    if (strncmp(opt, "trp:", 4) == 0) {
			parse_minitdb(opt + 4);
		    } else if (strcmp(opt, "notrp") == 0) {
			arts_ifd.no_tdb_option = 1;
		    } else if (strcmp(opt, "C") == 0) {
			closed_triple_db_option = 1;
		    } else if (strcmp(opt, "noC") == 0) {
			closed_triple_db_option = 0;
		    } else if (strcmp(opt, "prod") == 0) {
			arts_ifd.hash_production = 1;
		    } else if (strcmp(opt, "trp-trans") == 0) {
			arts_ifd.triple_translate = 1;
		    } else if (strcmp(opt, "noautopos") == 0) {
			arts_ifd.print_autopos = 0;
		    } else {
			abs_error ("unknown option '-O%s'.", opt);
			show_usage();
			break;
		    }
		  }
		return;
	    }

	  /* Options with a numeric argument */
	  case 'E':
	    { char *arg = get_optarg (*ptr, ptr, ix, argc, argv);
	      set_max_edit_distance (atoi (arg));
	      return;
	    };
	  case 'l':
	    { char *arg = get_optarg (*ptr, ptr, ix, argc, argv);
	      arts_ifd.transition_penalties_option = (atoi (arg));
	      return;
	    };
	  case 'm':
	    { char *arg = get_optarg (*ptr, ptr, ix, argc, argv);
	      arts_ifd.max_penalty = (atoi (arg));
	      return;
	    };
	  case 'P':
	    { char *arg = get_optarg (*ptr, ptr, ix, argc, argv);
	      set_max_parses (arg);
	      return;
	    };
	  case 'T':
	    { char *arg = get_optarg (*ptr, ptr, ix, argc, argv);
	      set_max_parse_time (atoi (arg));
	      return;
	    };
	  case 'X':
	    { char *arg = get_optarg (*ptr, ptr, ix, argc, argv);
	      set_frequency_radixes (arg, &radix);
	      return;
	    };

	  /* Options with a string argument */
	  case 'A':
	    { char *arg = get_optarg (*ptr, ptr, ix, argc, argv);
	      alphabet_fname = abs_new_string (arg, "scan_options");
	      return;
	    };
	  case 'i':
	    { char *arg = get_optarg (*ptr, ptr, ix, argc, argv);
	      arts_ifd.input_pos_fname = abs_new_string (arg, "scan_options");
	      return;
	    };
	  case 'o':
	    { char *arg = get_optarg (*ptr, ptr, ix, argc, argv);
	      arts_ifd.output_fname = abs_new_string (arg, "scan_options");
	      return;
	    }
	  case 'v':
	    { char *arg = get_optarg (*ptr, ptr, ix, argc, argv);
	      check_for_verbose_options (arg);
	      return;
	    };

	  /* Miscellaneous options */
	  case 'V':
	    try_show_version ();
	    abs_exit (0);
	    /* never returns */
	  case 'h':
	  case '?': show_usage ();
	  default:
	    abs_error ("unknown option '-%c'.", *ptr);
	    show_usage ();
	};
    }
  while (*(++ptr));
}

static char *clean_grammar_name (char *name)
{
  char *newname = abs_new_string (name, "parse_command_line");
  size_t len = strlen(newname);
  if (len > 4) {
      char *ext = newname + len - 4;
      if (strcmp(ext, ".aob") == 0 ||
	  strcmp(ext, ".gra") == 0) {
	  ext[0] = '\0';
      }
  }
  return newname;
}

void parse_command_line (int argc, char **argv)
{ int had_args = 0;
  int ix;
  init_options ();
  for (ix = 1; ix < argc; ix++)
    { char *arg = argv[ix];
      if (arg[0] == '-') scan_options (arg+1, &ix, argc, argv);
      else if (had_args >= 2)
	{ abs_error ("Too many arguments");
	  show_usage ();
	}
      else if (arts_ifd.grammar_name == NULL)
	{ arts_ifd.grammar_name = clean_grammar_name (argv[ix]);
	  had_args++;
	}
      else
	{ if (arts_ifd.input_pos_fname != NULL)
	    { abs_message ("Warning: -i option overridden by argument");
	      abs_free (arts_ifd.input_pos_fname, "parse_command_line");
	    };
	  arts_ifd.input_fname = abs_new_string (argv[ix], "parse_command_line");
	  arts_ifd.input_pos_fname = arts_ifd.input_fname;
	  had_args++;
	};
    };
  
  /* Exit if no source given */
  if (arts_ifd.grammar_name == NULL)
    { abs_error ("No grammar name given");
      show_usage ();
    };
}

/*
   Overrule the options set by the compiler by those on the command line
*/
void overrule_grammar_options ()
{ /* -A alphabet_fname */
  if (alphabet_fname != NULL)
    arts_ifd.alphabet_fname = alphabet_fname;

  /* |graph|, -G */
  if (graph_option != -1)
    arts_ifd.graph_option = graph_option;

  /* |transduce|, -s, -t */
  if (transduce_option != -1)
    arts_ifd.transduce_option = transduce_option; 

  /* |paragraph mode| or |line mode|, -p 1, -p 0 */
  if (paragraph_mode != -1)
    arts_ifd.paragraph_mode = paragraph_mode; 

  /* |segment mode| or |sentence mode|, -p segment, -p sentence */
  if (segment_mode != -1)
    arts_ifd.segment_mode = segment_mode; 

  if (counters_option != -1)
    arts_ifd.counters_option = counters_option; 

  /* |hybrid parsing|, -N */
  if (no_hybrid_execution)
    arts_ifd.hybrid_parsing_option = 0;

  /* |closed triple| */
  if (closed_triple_db_option != -1)
    arts_ifd.closed_triple_db_option = closed_triple_db_option;

  if (!arts_ifd.directors_option)
      arts_ifd.directors_set_posmemos = 0;

  if (arts_ifd.radix_pragmat != NULL)
    set_frequency_radixes(arts_ifd.radix_pragmat, &arts_ifd.radix);

  if (radix.lexicon_frequency != -1)
      arts_ifd.radix.lexicon_frequency = radix.lexicon_frequency;

  if (radix.tripledb_frequency != -1)
      arts_ifd.radix.tripledb_frequency = radix.tripledb_frequency;

  if (radix.penalty != -1)
      arts_ifd.radix.penalty = radix.penalty;

  if (radix.fact_frequency != -1)
      arts_ifd.radix.fact_frequency = radix.fact_frequency;
}
