/*
   File: arts_main.c
   Agfl interpreter driver

   Copyright 2006-2010 Radboud University of Nijmegen
 
   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.
 
   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU Library General Public License for more details.
 
   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.

   CVS ID: "$Id$"
*/

/* Standard includes */
#ifdef HAVE_CONFIG_H
#include <config.h>
#endif /* HAVE_CONFIG_H */

#include <stdio.h>
#include <limits.h>
#include <stdlib.h>
#include <string.h>
#include <signal.h>
#include <sys/stat.h>
#include <ctype.h>
#include <time.h>

/* libabase includes */
#include <abase_porting.h>
#include <abase_error.h>
#include <abase_memalloc.h>
#include <abase_mm_alloc.h>

/* liblexicon includes */
#include <lxcn_lexicon.h>

/* libarts includes */
#include "arts_ds.h"
#include "arts_io.h"
#include "arts_timers.h"
#include "arts_loader.h"
#include "arts_lexer.h"
#include "arts_hybrid.h"
#include "arts_interpreter.h"
#include "arts_minitdb.h"
#include "arts_main.h"

/* Improve location of grammar and dat file */
static int read_lexicon_file (void)
{ char lex_fname[MAXPATHLEN + 1];
  sprintf (lex_fname, "%s.blx", arts_ifd.grammar_name);
  arts_ifd.lexicon = lxcn_read_lexicon (lex_fname);
  return arts_ifd.lexicon ? 0 : -1;
}

static void parse_alphabet_file (char *fname, char **trans_src, char **trans_dst,
				 int **trans_penalties)
{ char src_chars[256];
  char dst_chars[256];
  int penalties[256];
  char buffer[256];
  int lineno = 0;
  int index = 0;
  int ix;

  /* Open the alphabet file */
  FILE *f = fopen (fname, "r");
  if (f == NULL)
     abs_fatal ("cannot open alphabet file '%s'", fname);

  /* Parse the file line for line */
  while (fgets (buffer, 255, f) != NULL)
     { lineno++;
       if (buffer[0] != '#')
	  { char *bptr = buffer;
	    char *eptr;
	    int val;

	    /* Skip white space and convert the first value on the line */    
	    while (isspace ((int) (*bptr))) bptr++;
	    val = strtol (bptr, &eptr, 0);
	    if (bptr == eptr)
	       abs_fatal ("illegal first value on line %d in alphabet file", lineno);
	    if ((val < 0) || (val > 255))
	       abs_fatal ("first value out of range on line %d in alphabet file", lineno);
	    src_chars[index] = (char) val;

	    /* Skip white space and convert the second value on the line */    
	    bptr = eptr;
	    while (isspace ((int) (*bptr))) bptr++;	/* Skip white space */
	    val = strtol (bptr, &eptr, 0);
	    if (bptr == eptr)
	       abs_fatal ("illegal second value on line %d in alphabet file", lineno);
	    if ((val < 0) || (val > 255))
	       abs_fatal ("second value out of range on line %d in alphabet file", lineno);
	    dst_chars[index] = (char) val;

	    /* Skip white space and convert the optional third value on the line */    
	    bptr = eptr;
	    while (isspace ((int) (*bptr))) bptr++;	/* Skip white space */
	    val = strtol (bptr, &eptr, 0);
	    if (bptr[0] != '\0' && bptr[0] != '#' && bptr == eptr)
	       abs_fatal ("illegal third value on line %d in alphabet file: '%s'", lineno, bptr);
	    penalties[index] = val;
	    index++;
          };
     };
  fclose (f);

  /* complete the translation buffers */
  src_chars[index] = '\0';
  dst_chars[index] = '\0';

  /* Check if the user specified a mapping */
  for (ix = 0; src_chars[ix] != 0; ix++)
     { char v2 = src_chars[ix];
       int iy;
       for (iy = ix + 1; src_chars[iy] != 0; iy++)
	 if (src_chars[iy] == v2)
	   abs_fatal ("Multiple mapping in alphabet file for value %d", v2);
     };

  /* Copy the strings and be done */
  *trans_src = abs_new_string (src_chars, "parse_alphabet_file");
  *trans_dst = abs_new_string (dst_chars, "parse_alphabet_file");
  *trans_penalties = abs_calloc (index, sizeof(int), "parse_alphabet_file");
  for (ix = 0; ix < index; ix++)
    (*trans_penalties)[ix] = penalties[ix];
}

static void determine_alphabet_translations ()
{ /* Check for translations */
  if (arts_ifd.alphabet_fname == NULL)			/* |translate off| */
    { arts_ifd.translate_src = "";
      arts_ifd.translate_dst = "";
    }
  else if (*arts_ifd.alphabet_fname == '\0')		/* "" => |translate default| */
    { arts_ifd.translate_src = "ABCDEFGHIJKLMNOPQRSTUVWXYZ";
      arts_ifd.translate_dst = "abcdefghijklmnopqrstuvwxyz";
    }
  else	/* |translate "alphabet_fname"| or command line -A alphabet_fname */
    parse_alphabet_file (arts_ifd.alphabet_fname, &arts_ifd.translate_src, &arts_ifd.translate_dst,
			 &arts_ifd.translate_penalties);
}

static int prelude ()
{ int err = 0;
  /* Make sure we abort with some error message */
  abs_mm_init ();

  /* Read the binary lexicon (always present, even if not really needed) */
  err = read_lexicon_file ();
  if (err != 0) return err;

  /* Make connectivity between trellis, lexicon and code */
  arts_link_lexicon ();

  /* Initialize the rest */
  init_io ();
  determine_alphabet_translations ();
  init_lexer ();
  arts_init_parser ();

  return 0;
}

static void postlude ()
{ arts_end_parser ();
  end_lexer ();
  end_io ();

  if (arts_ifd.free_mem_option)
  { lxcn_free_lexicon (arts_ifd.lexicon);
    abs_mm_final ("postlude");
    free_minitdb();
  }

  /*
   * TODO: should free translation strings and penalties,
   * but they may not have been alloc'ed.
   */
}

/*-----------------------------------------
// Main driver
//---------------------------------------*/
int arts_main ()
{ /* Initialize the lexer part of the datastructure */
  State node = NULL;
  int err;
  err = prelude ();
  if (err != 0) return err;

  /* While we have input, lexicalize and parse it. */
  while (arts_ifd.paragraph_mode ? read_input_block (): read_input_line ())
    { Trellis trellis;
      char* input_text = get_input_text ();
      int input_linenr = get_input_linenumber ();
      int input_colnr = get_input_position ();

      /*------------------
      // Lexicalize input
      //----------------*/
      reset_timers ();

      /* skip empty lines silently */
      if (input_text[0] && strcmp (input_text, "\n") != 0)
        { start_scan_time ();
          trellis = initialize_trellis (input_text, input_linenr, input_colnr);
          stop_scan_time ();

#ifdef COUNTERS
           show_neg_memo_blocks (trellis);
#endif
           if (arts_ifd.max_parses > 0)
	     { int first_parse = 1;
               do
	       { /* Make sure that we have no lingering results */
	         //if (!first_parse) reset_trellis_pos_memos (trellis);
                 node = parse_trellis (trellis, node);
#ifdef COUNTERS
                 show_neg_memo_blocks (trellis);
#endif
		 first_parse = 0;
		 if (arts_ifd.graph_option) print_trellis (trellis);
		 if (node != NULL) trel_free_states_before (trellis, node);
                }
	        while (node);
              };

          /* Clean-up trellis */
          delete_trellis (trellis);
	};

      /* and synchronise with other programs like the LCS */
      maybe_output_sync (0);
      abs_mm_intermediate_free ("main");
      arts_hyb_mark_hybrid_pool_freed ();
    };

  /* finish up and leave */
  postlude ();
  return err;
}

