/*
   File: arts_interpreter.c
   Defines the interpreter for the instructions generated by agfl-coder.

   Copyright 2005-2010 Radboud University of Nijmegen
 
   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.
 
   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU Library General Public License for more details.
 
   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.

   CVS ID: "$Id$"
*/

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif /* HAVE_CONFIG_H */

/* standard includes */
#include <stdio.h>
#include <stdlib.h>
#include <limits.h>
#include <assert.h>
#include <sys/types.h>
#include <string.h>

/* libabase includes */
#include <abase_error.h>
#include <abase_memalloc.h>
#include <abase_mm_alloc.h>

/* liblexicon include */
#include <lxcn_lexicon.h>
#include <lxcn_fact_table.h>

/* libarts includes */
#include "arts_ds.h"
#include "arts_io.h"
#include "arts_lexer.h"
#include "arts_timers.h"
#include "arts_posmemo.h"
#include "arts_hybrid.h"
#include "arts_interpreter.h"
#include "arts_tracer.h"

#ifdef GEN_RTS
#include "rtsgen.h"
#endif /* GEN_RTS */

#ifdef DEBUG_RTS
#define DB_RTS(x) x
#else
#define DB_RTS(x)
#endif /* DEBUG_RTS */

#ifdef DEBUG_HYBRID
#define DB_HYBRID(x) x
#else
#define DB_HYBRID(x)
#endif

#ifdef DEBUG_INTERPRETER
#  undef DEBUG_TRANSLATE
#  define TRACE_MATCH	1
#  undef TRACE_LEX_MATCH
#endif /* DEBUG_INTERPRETER */

#define NEW_TIMEOUT_ON_TERMINALS	1
#define NEW_TIMEOUT_ON_PM_KNOWN		0
#define NEW_TIMEOUT		(NEW_TIMEOUT_ON_TERMINALS || NEW_TIMEOUT_ON_PM_KNOWN)

#define NEST_COLLECTING		1
#define NEST_EXTENDING		2
#define NEST_POSMEMO_CHANGED	4

static void print_pass1_frame(DATA *fp, int recursive);
static void print_pass2_frame(DATA *fp);

/*------------------------------------------------------------------------------
// Penalty management
// Penalties are defined to be 32 bit integers
//----------------------------------------------------------------------------*/
static int penalty_change_would_block (Penalty current, Penalty offset)
{ if (offset == MAX_PENALTY) return (1);
  else if ((offset > 0) && (current > MAX_PENALTY - offset)) return (1);
  else if ((offset < 0) && (current < MIN_PENALTY - offset))
    abs_message ("# Current parse has gathered too many bonuses to be true");
  if (current + offset > arts_ifd.max_penalty) return (1);
  return (0);
}

/*
 * Carefully sum penalties, in cases where the offset may be
 * MIN_PENALTY. Currently this can only realistically happen
 * when playing posmemos, when an unbalanced bracket/brace has been
 * detected.
 */
static Penalty sum_penalties (Penalty current, Penalty offset)
{ if ((offset < 0) && (current < MIN_PENALTY - offset)) return MIN_PENALTY;
  return current + offset;
}

/*------------------------------------------------------------------------------
// Private globals
//----------------------------------------------------------------------------*/
typedef struct
{ int		nr_parses;	/* nr of parses */
  Penalty	penalty;	/* penalty of last parse */
  Position	startpos;	/* start position in current input buffer */
  Position	endpos;		/* end of the text covered by the parse */
} ParseResult;

static Position furthest_pos;
static ParseResult parse_result;

static void reset_parse_result ()
{ parse_result.nr_parses = 0;
  parse_result.penalty = 0;
  parse_result.startpos = 0;
  parse_result.endpos = 0;
}

static void incr_nr_parses ()
{ parse_result.nr_parses++;
}

static int get_nr_parses ()
{ return (parse_result.nr_parses);
}

static Penalty get_penalty ()
{ return (parse_result.penalty);
}

static Position get_parse_startpos ()
{ return (parse_result.startpos);
}

static Position get_parse_endpos ()
{ return (parse_result.endpos);
}

/*------------------------------------------------------------------------------
// Time limit.
//
// The time-out is checked for each succeeding match of a terminal.
// For efficiency reasons, we only check the clock once every
// MAX_TICKS times. High MAX_TICKS increases performance, but
// reduces timer resolution.
//----------------------------------------------------------------------------*/
static int have_time_limit (void)
{ return (arts_ifd.max_parsetime < INT_MAX);
}

static void set_time_limit (void)
{ set_time_out (arts_ifd.max_parsetime);
}

/*------------------------------------------------------------------------------
// Printing
//----------------------------------------------------------------------------*/
static void print_parsed_sentence (void)
{ if (arts_ifd.lexer_stats_option)
    { int input_linenumber = get_input_linenumber ();
      int input_position = get_input_position ();
      char *input_text = get_input_text ();
      int parse_start = (int) get_parse_startpos ();
      int parse_end = (int) get_parse_endpos ();
      char *parse_start_ptr = input_text + parse_start;
      char *parse_end_ptr = input_text + parse_end;

      /* Null check added */
      char *fn = arts_ifd.input_fname;
      if (fn == NULL) fn = "(null)";

      /* Print to stdout? */
      printf ("\n# %s %d %d-%d|", fn, input_linenumber,
	      input_position + parse_start, input_position + parse_end);

      /*
	 Print the parsed input text, replacing any newlines with space.
         This is because it must all be on the one comment line.
      */
      while (parse_start_ptr < parse_end_ptr)
	{ if (*parse_start_ptr == '\n') putchar(' ');
	  else putchar (*parse_start_ptr);
	  parse_start_ptr++;
	}
      putchar('\n');
    };
}

static void start_printing (void)
{ if (arts_ifd.parsing_stats_option)
    { int nr_parses = get_nr_parses ();
      Penalty penalty = get_penalty ();
      Time parse_time = get_parse_time ();

      current_parse_printf ("# parsing %d time %.3f penalty %d posmemo: known %d, blocked %d, unknown %d\n",
			    nr_parses, parse_time, penalty,
			    posmemo_known_count,
			    posmemo_blocked_count,
			    posmemo_unknown_count);
    };
}

static void stop_printing (void)
{ current_parse_add_char ('\n');
  current_parse_finish();
}

static void show_totals (Trellis trellis)
{ int nr_parses = get_nr_parses ();

  if (arts_ifd.total_stats_option)
    { Time scan_time = get_total_scan_time ();
      Time parse_time = get_total_parse_time ();
      Time print_time = get_total_print_time ();
      Time total_time = scan_time + parse_time + print_time;

      abs_printf ("# parsings %d/", nr_parses);
      if (arts_ifd.max_parses == INT_MAX) abs_printf ("unlimited");
      else abs_printf ("%d", arts_ifd.max_parses);
      abs_printf (" scan %.3f parse %.3f print %.3f total %.3f time\n",
		  scan_time, parse_time, print_time, total_time);
    };
  if (arts_ifd.research_stats_option)
    { abs_printf ("# reached input position %d (", furthest_pos);
      print_saved_parse_call_stack();
      abs_printf (")\n");
    };

#ifdef DEBUG_NEGMEMO
  if (arts_ifd.neg_memo_option)
    print_negmemo_table (trellis, 1);
#endif /* DEBUG_NEGMEMO */

  if (arts_ifd.show_posmemo_option)
    posmemo_dump_table (trellis);

#ifdef PROFILING
  if ((prof_count > 0) && reset_profile && !no_output)
    ShowProfile ("sentence");
#endif
}

/*------------------------------------------------------------------------------
// Stack frame layout.
//      The interpreter uses two different stack layouts: one for the first
//      pass, and one for the second pass.
//
//      The first pass frame consists of:
//	  - constants pushed by the CALL instruction (the elements with a
//	    positive offset);
//	  - a fixed number of entries filled in by the CREATE instruction (the
//	    elements with a negative offset);
//	  - zero or more formals;
//	  - zero or more locals;
//	  - zero or more entries for pointers to the sons of this rule.
//      The formals are untouched by the CREATE instruction.
//      The locals and sons are cleared to 0 by the CREATE instruction.
//
//      The second pass frame consists of:
//	  - constants pushed by the PRINT_SON instruction
//	  - parameters for the second pass function.
//
//      Both a stack pointer and a frame pointer are used. The execution model
//      allows to continue execution in another frame.
//
//	Note that the SP grows from high to low memory,
//	always pointing to the first free location
//----------------------------------------------------------------------------*/
#define STACK_SIZE		(128 * 1024)
#define STACK_AVAIL(n)		(sp - (n) - 16 > stack)
static DATA* stack = 0;

#define PASS2_FRAME_HEAD_LEN	(3)     /* length of the second pass frame * head */
#define PASS2_PASS2FP		(3)     /* previous second pass frame pointer */
#define PASS2_FP		(2)     /* previous rule posmemo (was pass1fp) */
#define PASS2_RET_ADDR		(1)     /* return address */

#define FRAME_HEAD_LEN		(2)     /* length of the frame head */
#define SUCC_ADDR_OFF		(2)     /* success continuation address */
#define FAIL_ADDR_OFF		(1)     /* fail continuation address */
#define OLD_FP			(0)     /* pointer to parent frame */
#define NONT_NR_OFF		(-1)
#define NR_FORMALS_OFF		(-2)
#define NR_LOCALS_OFF		(-3)
#define NR_SONS_OFF		(-4)
#define PM_PLAYED		(-5)	/* posmemo played by may_play_pm */
#define ISTATE			(-6)    /* input state at start of rule */
#define SUCCESS_OFF		(-7)    /* nr of successes */
#include "arts_stackframe.h"		/* define public value(s) */
#define NR_SONS_DONE		(-9)    /* nr of sons which have been called */
#define PENORIG			(-10)    /* penalty level at start of rule */
#define VAR_OFFSET		(11)    /* offset for variable frame part */

#define NONT_NR			(fp[NONT_NR_OFF].arg)
#define NR_FORMALS		(fp[NR_FORMALS_OFF].arg)
#define NR_LOCALS		(fp[NR_LOCALS_OFF].arg)
#define NR_SONS			(fp[NR_SONS_OFF].arg)
#define START_INPUT_STATE	(fp[ISTATE].input_state)

#define VARIABLE(nr)		(fp[-(long)(VAR_OFFSET + (nr))])
#define SON_FP(nr)		(fp[-(long)(VAR_OFFSET + NR_FORMALS + NR_LOCALS + (nr))].data)
#define SON_DATA(nr)		(fp[-(long)(VAR_OFFSET + NR_FORMALS + NR_LOCALS + (nr))])

/*------------------------------------------------------------------------------
// Accessing the negative memo values.
// Macro:
//	NEG_MEMO(no)
// Description:
//	Convert memo with number to value of memo.
//----------------------------------------------------------------------------*/
#define NEG_MEMO(ext, no)		((ext)->neg_memos[no])

#define negmemo_is_unknown(ext, nr)	(NEG_MEMO(ext, nr) == NEGMEMO_UNKNOWN)
#define negmemo_is_blocked(ext, nr,pen)	((NEG_MEMO(ext, nr) != NEGMEMO_UNKNOWN)&&(pen >= NEG_MEMO(ext, nr)))
#define negmemo_set_blocked(ext, nr,pen) (NEG_MEMO(ext, nr) = pen)
#define negmemo_set_succeeded(ext, nr)	(NEG_MEMO(ext, nr) = NEGMEMO_SUCCESS)

/*
   The following routine is for debugging and statistical purposes:
*/
void print_negmemo_table (Trellis trellis, int skip_unknown)
{   
    int node_nr;
    int alt_nr;
    int* empty_rule;
    int* empty_node;
    int trellis_length = trel_get_length(trellis) * NUM_LEX_STATES;
    State state;

    int** overview = (int **) abs_calloc (trellis_length, sizeof (int *),
					  "print_negmemo_table: overview[]");

    abs_message ("print_negmemo_table: there are %d negative memos.", arts_ifd.nr_neg_memos);

    /* Build the table: */
    node_nr = 0;
    state = trel_get_initial_state(trellis);
    while (state != NULL)
      { 
	if(!state) overview[node_nr] = NULL;
	else
	  { StateExtension ext = GetStateExtension(state);
	    NegMemo* neg_memo_vec = ext -> neg_memos;

	    if (!neg_memo_vec) overview[node_nr] = NULL;
	    else
	      { overview[node_nr] = (int *) abs_calloc (arts_ifd.nr_neg_memos, sizeof(int),
							"print_negmemo_table: overview[][]");

	        for (alt_nr = 0; alt_nr < arts_ifd.nr_neg_memos; alt_nr++)
	           overview[node_nr][alt_nr] = neg_memo_vec[alt_nr];
	      };
	  };
	state = trel_state_next(trellis, state);
	node_nr++;
      };

    /* printed table compression */
    empty_rule = (int*) abs_calloc (arts_ifd.nr_neg_memos, sizeof(int),
				    "print_negmemo_table: empty_rule");
    for (alt_nr = 0; alt_nr < arts_ifd.nr_neg_memos; alt_nr++)
      { empty_rule[alt_nr] = 1;
	node_nr = 0;

	while ((node_nr < trellis_length) && (empty_rule[alt_nr]))
	  { if (overview[node_nr])
	      { if (overview[node_nr][alt_nr] == NEGMEMO_UNKNOWN)
		  empty_rule[alt_nr] = skip_unknown;
	        else empty_rule[alt_nr] = 0;
	      };
	    node_nr++;
	  };
      };
    empty_node = (int*) abs_calloc (trellis_length, sizeof(int),
				    "print_negmemo_table: empty_node");
    for (node_nr = 0; node_nr < trellis_length; node_nr++)
      { empty_node[node_nr] = 1;
	alt_nr = 0;

	while ((alt_nr < arts_ifd.nr_neg_memos) && (empty_node[node_nr]) && (overview[node_nr]))
	  { switch (overview[node_nr][alt_nr])
	      { case NEGMEMO_UNKNOWN:	empty_node[node_nr] = skip_unknown; break;
	        case NEGMEMO_BLOCKED:	empty_node[node_nr] = 0; break;
	        default: empty_node[node_nr] = !overview[node_nr][alt_nr];
	      };
	    alt_nr++;
	  };
      };

    /* actually print it: */
    /* first the table */
    for (alt_nr = 0; alt_nr < arts_ifd.nr_neg_memos; alt_nr++)
      if (!empty_rule[alt_nr])
	{ abs_printf ("%3d|", alt_nr);
	  for (node_nr = 0; node_nr < trellis_length; node_nr++)
	    if (!empty_node[node_nr])
	      { switch (overview[node_nr][alt_nr])
		  { case NEGMEMO_BLOCKED: abs_printf("          b"); break;
	            case NEGMEMO_UNKNOWN: abs_printf("          u"); break;
	            default: abs_printf(" %10ld", overview[node_nr][alt_nr]);
	          };
	      };
	  abs_printf("\n");
	};

    /* then a neat line below it */
    abs_printf("---+");
    for (node_nr = 0; node_nr < trellis_length; node_nr++)
      if (!empty_node[node_nr]) abs_printf ("-----------");

    /* and of course the numbers */
    abs_printf("\n   |");
    for (node_nr = 0; node_nr < trellis_length; node_nr++)
      if (!empty_node[node_nr]) abs_printf (" %10d", node_nr);
    abs_printf ("\n");

    /* free the space: */
    for (node_nr = 0; node_nr < trellis_length; node_nr++)
      if (overview[node_nr])
	abs_free (overview[node_nr], "print_negmemo_table: overview[][]");

    abs_free (overview, "print_negmemo_table: overview[]");
    abs_free (empty_rule, "print_negmemo_table: empty_rule");
    abs_free (empty_node, "print_negmemo_table: empty_node");
}

static int state_get_endpos (State s, Trellis t)
{ if (s == NULL) return (trel_get_length (t));
  return (trel_state_get_pos (s));
}


/*------------------------------------------------------------------------------
// Maintain a stack of ambiguous nodes during pass2 to keep track of the
// different printings of the parse forest.
//----------------------------------------------------------------------------*/
static PosMemo *ambi_stack;
static int ambi_size;
static int ambi_sptr;
static void init_ambi_stack ()
{ ambi_size = 64;
  ambi_sptr = 0;
  ambi_stack = abs_calloc (ambi_size, sizeof (PosMemo *), "init_ambi_stack");
}

void push_ambi_stack (PosMemo prime)
{ if (ambi_sptr == ambi_size)
    { ambi_size *= 2;
      ambi_stack = abs_realloc (ambi_stack, ambi_size * sizeof (PosMemo *), "push_ambi_stack");
    };
  ambi_stack[ambi_sptr++] = prime;
  ambi_stack[ambi_sptr++] = prime;
}

int try_lookup_sibling_in_ambi_stack (PosMemo pmptr, PosMemo *sibling)
{ PosMemo prime = pmptr -> prime;
  int ix;
  for (ix = 0; ix < ambi_sptr; ix += 2)
    if (ambi_stack[ix] == prime)
      { *sibling = ambi_stack [ix + 1];
	return (1);
      };
  return (0);
}

static int try_update_ambi_stack ()
{ while (ambi_sptr > 0)
    { PosMemo current_top = ambi_stack[ambi_sptr - 1];
      if (current_top -> ambig != NULL)
	{ /* We have another equivalent parse */
	  ambi_stack[ambi_sptr - 1] = current_top -> ambig;
	  return (1);
	};
      ambi_sptr -= 2;
    };
  return (0);
}

static void finish_ambi_stack ()
{ abs_free (ambi_stack, "finish_ambi_stack");
}

/*------------------------------------------------------------------------------
// The type of an execution function should be a function with zero
// parameters returning an execution function (to be able to continue
// with the returned function). Unfortunately the C compiler (and
// indeed the C99 standard) does not permit function types being
// returned in a function type. Hence we must resort to type casting (sigh..)
//
// Forward declare all exec routines
//----------------------------------------------------------------------------*/
typedef void *vptr;
typedef vptr (*exec_proc)(void);
static void exec_illegal ();
static vptr exec_nop (void);
static vptr exec_call_root (void);
static vptr exec_create_root (void);
static vptr exec_create (void);
static vptr exec_destroy (void);
static vptr exec_if_pm_known (void);
static vptr exec_try_block_pm (void);
static vptr exec_make_pm (void);
static vptr exec_play_pm (void);
static vptr exec_may_play_pm (void);
static vptr exec_make_hyb (void);
static vptr exec_jump (void);
static vptr exec_call (void);
static vptr exec_return (void);
static vptr exec_call_rule (void);
static vptr exec_ucall_rule (void);
static vptr exec_mark_collecting (void);
static vptr exec_mark_extending (void);
static vptr exec_umark_collecting (void);
static vptr exec_umark_extending (void);
static vptr exec_if_collecting (void);
static vptr exec_if_extending (void);
static vptr exec_update_nest_pm (void);
static vptr exec_if_nest_pm_chgd (void);
static vptr exec_try_block_nest_pm (void);
static vptr exec_clear (void);
static vptr exec_init_s (void);
#define exec_init_i exec_init_s
#define exec_init_t exec_init_s
static vptr exec_init_v (void);
static vptr exec_adjust_i (void);
static vptr exec_adjust_s (void);
static vptr exec_adjust_t (void);
static vptr exec_adjust_v (void);
static vptr exec_uadjust_s (void);
#define exec_uadjust_i exec_uadjust_s
#define exec_uadjust_t exec_uadjust_s
static vptr exec_uadjust_v (void);
static vptr exec_restrict_i (void);
static vptr exec_restrict_s (void);
static vptr exec_restrict_t (void);
static vptr exec_restrict_v (void);
static vptr exec_urestrict_s (void);
#define exec_urestrict_i exec_urestrict_s
#define exec_urestrict_t exec_urestrict_s
static vptr exec_urestrict_v (void);
static vptr exec_match (void);
#define exec_match_re exec_match
#define exec_skip_re exec_match
#define exec_match_other exec_match
static vptr exec_glue (void);
static vptr exec_uglue (void);
static vptr exec_match_any (void);
static vptr exec_match_sep (void);
static vptr exec_lex_match (void);
static vptr exec_umatch (void);
#define exec_umatch_re exec_umatch
#define exec_umatch_other exec_umatch
static vptr exec_umatch_any (void);
static vptr exec_umatch_sep (void);
static vptr exec_lex_umatch (void);
static vptr exec_init_key (void);
static vptr exec_add_key_i (void);
static vptr exec_add_key_t (void);
static vptr exec_fact_match (void);
static vptr exec_fact_umatch (void);
static vptr exec_release_key (void);
static vptr exec_input_pos (void);
#define exec_input_line exec_input_pos
static vptr exec_uinput_pos (void);
static vptr exec_penalty (void);
static vptr exec_upenalty (void);
static vptr exec_if_pm_added (void);
static vptr exec_success (void);
static vptr exec_usuccess (void);
static vptr exec_end (void);
static vptr exec_pass2 (void);
static vptr exec_pass2_return (void);
static vptr exec_push_f (void);
static vptr exec_push_l (void);
static vptr exec_print_son (void);
static vptr exec_print_abeg (void);
static vptr exec_print_aend (void);
static vptr exec_print_pbeg (void);
static vptr exec_print_psep (void);
static vptr exec_print_pend (void);
static vptr exec_print_val (void);
static vptr exec_print_term_trans (void);
static vptr exec_print_term_tree (void);
static vptr exec_print_lex_trans (void);
static vptr exec_print_lex_tree (void);
static vptr exec_print_fact (void);
static vptr exec_print_re_trans (void);
static vptr exec_print_re_tree (void);
#define exec_print_other_trans exec_print_re_trans
#define exec_print_other_tree exec_print_re_tree
#define exec_print_any_trans exec_print_re_trans
#define exec_print_any_tree exec_print_re_tree
static vptr exec_print_pos_trans (void);
static vptr exec_print_pos_tree (void);
#define exec_print_line_trans	exec_print_pos_trans
#define exec_print_line_tree	exec_print_pos_tree
static vptr exec_print_autopos_trans (void);
static vptr exec_print_pen (void);
static vptr exec_print_pen_inst (void);
static vptr exec_print_sep (void);
static vptr exec_print_indent (void);
static vptr exec_print_nl (void);
static vptr exec_print_n (void);
static vptr exec_print_il (void);
static vptr exec_print_sl (void);
static vptr exec_print_tl_trans (void);
static vptr exec_print_tl_tree (void);
static vptr exec_print_vl (void);
static vptr exec_print_if (void);
static vptr exec_print_sf (void);
static vptr exec_print_tf_trans (void);
static vptr exec_print_tf_tree (void);
static vptr exec_print_vf (void);
static vptr exec_tmemo (void);
static vptr exec_tsmemo (void);
static vptr exec_smemo_s (void);
static vptr exec_trace_ig (void);
static vptr exec_trace_tg (void);
static vptr exec_trace_il (void);
static vptr exec_trace_sl (void);
static vptr exec_trace_tl (void);
static vptr exec_trace_is (void);
static vptr exec_trace_ss (void);
static vptr exec_trace_ts (void);
static vptr exec_position (void);
static vptr exec_hybrid_op (void);
static vptr exec_relator (void);
static vptr exec_enter_text (void);
static vptr exec_enter_trans (void);
static vptr exec_use_son (void);
static vptr exec_use_lex (void);
static vptr exec_enter_int_affix (void);
static vptr exec_enter_text_affix (void);
static vptr exec_enter_set_affix (void);
static vptr exec_done (void);
static vptr exec_choice (void);
static vptr exec_uchoice (void);
static vptr exec_alt_marker (void);
static vptr exec_ualt_marker (void);

/*------------------------------------------------------------------------------
// Include the table for the instruction dispatcher
// If there is any discrepancy between the generated table and
// the execution functions declared above, the compiler should complain
//----------------------------------------------------------------------------*/
#include <opcode.exc>

/*------------------------------------------------------------------------------
// Some instruction may continue their execution in other functions
// Forward declare them as well
//----------------------------------------------------------------------------*/
static vptr pm_generate_production ();
static vptr search_match ();
static vptr match_again ();
static vptr lex_match_again ();
static vptr fact_match_again ();

/*------------------------------------------------------------------------------
// When counting we need to keep track of the number
// of times, an opcode is executed
//----------------------------------------------------------------------------*/
static int opcode_count[nr_of_opcodes];
static void initialize_counters ()
{ int ix;
  for (ix = 0; ix < nr_of_opcodes; ix++)
    opcode_count[ix] = 0;
}
#define opc_count(ix) { if (arts_ifd.counters_option) opcode_count[ix & 0xff]++; }

static void print_counters ()
{ int ix; 
  int sum = 0;
  int v = arts_ifd.counters_option;

  abs_message ("\nNonterminal create/destroy counts:");
  for (ix = 0; ix < arts_ifd.nr_syntax_nonterminals; ix++)
    { if (v > 1)
        abs_message ("%4d %8d %s", ix, arts_ifd.nonterminal_profile_table[ix], 
		   arts_ifd.nonterm_names[ix].str);
      sum += (int) arts_ifd.nonterminal_profile_table[ix];
    };
  abs_message ("Grand total: %d", sum);

  abs_message ("\nOpcode counts:");
  sum = 0;
  for (ix = 0; ix < nr_of_opcodes; ix++)
    { if (v > 1)
        abs_message ("%3d %18s %10d", ix, all_opcode_execs[ix].opcode_name,
	    opcode_count[ix]);
      sum += opcode_count[ix];
    }

  abs_message ("Grand total: %d  key-counts: %d", sum,
	  opcode_count[opc_make_pm] +
	  opcode_count[opc_create]);
  abs_message ("");
  abs_message ("Nr of match agains:       %8d", (int) arts_ifd.nr_match_agains);
  abs_message ("Nr of lex match agains:   %8d", (int) arts_ifd.nr_lex_match_agains);
  abs_message ("Nr of fact match agains:  %8d", (int) arts_ifd.nr_fact_match_agains);
}

/*------------------------------------------------------------------------------
// Define macro's to debug the execution of the interpreter
//----------------------------------------------------------------------------*/
#if defined(DEBUG_INTERPRETER)
#define logexec(x) { \
    abs_message ("logexec(%c): %s pc=%p fp=%p sp=%p state=%p", \
	         'A', all_opcode_execs[x].opcode_name, pc, fp, sp, i_state);\
}

#define logfail(x) { \
    abs_message ("logfail(%c): %s\n", \
	         'A', all_opcode_execs[x].opcode_name); \
}
#else /* NOT DEBUG_INTERPRETER */
#define logexec(x)
#define logfail(x)
#endif /* DEBUG_INTERPRETER */

/*------------------------------------------------------------------------------
// Interpreter registers.
//
//	pc 		program counter; contains address of current instruction.
//
//	next_pc 	new program counter; contains address of next instruction.
//
//	sp 		stack pointer; points to next free position on stack
//
//	fp		frame pointer; points to current stack frame.
//
//	pass2_fp 	Pass 2 frame pointer
//
//	i_state 	state pointer pointing to current input state
//
//	i_transition 	points to current transition to be matched
//			(only used during LEX_MATCH/LEX_UMATCH)
//
//	curr_penalty 	the current penalty level
//
//	pm_penalty_gain set in PLAY_PM, used in pm_generate_production()
//
//	curr_terminal	the current matching terminal
//			set in lEX_{U,}MATCH, used in LEX_MATCH_AGAIN.
//
//	indent 		Current indentation to print the parse tree
//
//	status 		Current status of the abstract machine interpreter
//			(STOPPING or EXECUTING)
//
//	trellis		Current trellis to parse
//
//	pmptr		Current posmemo pointer
//
//	return_to_pass2 Pass 2 administration
//----------------------------------------------------------------------------*/
#define STOPPING 0
#define EXECUTING 1
static CODE *pc;
static CODE *next_pc;
static DATA *sp;
static DATA *fp;
static DATA *pass2_fp;
static PosMemo pmptr;
static int return_to_pass2;
static State i_state;
static Transition i_transition;
static Penalty curr_penalty;
static Penalty pm_penalty_gain;
static Terminal curr_terminal;
static Trellis curr_trellis;
static int indent;
static int status;
static hyb_anchor curr_anchor;

/*------------------------------------------------------------------------------
// Define macros for generative execution
//----------------------------------------------------------------------------*/
#ifdef GEN_RTS
static int chosen_freq;
static int alt_depth;

#ifdef DEBUG_GENERATOR
#define ALT_DEPTH_RESET (alt_depth = 0)
#define ALT_DEPTH_CHECK (assert(alt_depth >= 0))
#define ALT_DEPTH_DECREASE \
{	                                                                 \
    alt_depth--;	                                                  \
    abs_message ("ALT_DEPTH_DECREASE: alt_depth now %d",alt_depth);    \
}
#define ALT_DEPTH_INCREASE \
{	                                                                   \
    alt_depth++;	                                                    \
    abs_message ("ALT_DEPTH_INCREASE: alt_depth now %d",alt_depth);    \
}
#else /* DEBUG_GENERATOR */
#define ALT_DEPTH_RESET (alt_depth = 0)
#define ALT_DEPTH_CHECK (assert(alt_depth >= 0))
#define ALT_DEPTH_DECREASE (alt_depth--)
#define ALT_DEPTH_INCREASE (alt_depth++)
#endif /* DEBUG_GENERATOR */
#endif /* GEN_RTS */

/*------------------------------------------------------------------------------
// Define macros to support decoding and execution
// Note that CONT provides a continuation of the current instruction
//----------------------------------------------------------------------------*/
#define RECALC_NEXT_PC next_pc = pc + ((((pc -> ival) >> 16) & 0xffff) + 1)
#define NEXT return ((vptr) NULL)
#define GOTO(x) { next_pc = x; return ((vptr) NULL); }
#define CONT(x) return ((vptr) x)
#define ITERATE { next_pc = pc; return ((vptr) NULL); }
#define GET_INSTR ((opcode)(pc->ival & 0xFF))

/*------------------------------------------------------------------------------
// Check to see if a pointer points in the agfl interpreter stack
//----------------------------------------------------------------------------*/
int pointsIntoStack (void *p) 
{
    return ((p >= (void *) stack) && (p < (void *) (stack + STACK_SIZE)));
}

/*------------------------------------------------------------------------------
// Macros to push and pop to and from the stack
// Note that the SP grows from high to low memory
// always pointing to the first free location
//----------------------------------------------------------------------------*/
#define PUSH_VALUE(x)	\
{	                \
    sp -> val = x;	\
    sp--;               \
}

#define PUSH_ADDR(x)	\
{	                \
    sp -> code = x;	\
    sp--;	        \
}

#define POP_VALUE(x)    \
{	                \
    sp++;	        \
    x = sp[0].val;	\
}

#define POP_ADDR(x)	\
{	                \
    sp++;               \
    x = sp[0].code;     \
}

/*
 * stack_overflow2 prints a message, and optionally ends the parsing,
 * or does a CONT or GOTO.
 */
static vptr stack_overflow2(char *where, CODE *my_next_pc, vptr cont)
{   abs_printf("Out of stack in '%s'; backtrace:\n", where);
    print_parse_call_stack(NULL);
    abs_printf("\n");
    if (0) {		/* TODO: Perhaps make this some runtime option. */
	exec_end();	/* In case the user ignores the return value */
	NEXT;		/* from stack_overflow*(). */
    } else if (my_next_pc) {
	GOTO(my_next_pc);
    } else {
	CONT(cont);
    }
}

/*
 * stack_overflow will either do a continuation, or optionally end the parsing.
 */
static vptr stack_overflow_cont(char *where,  vptr cont)
{   return stack_overflow2(where, NULL, cont);
}

/*
 * stack_overflow will either do a GOTO, or optionally end the parsing.
 */
static vptr stack_overflow_goto(char *where,  CODE *my_next_pc)
{   return stack_overflow2(where, my_next_pc, NULL);
}

/*
 * stack_overflow will either do a NEXT, or optionally end the parsing.
 */
static vptr stack_overflow(char *where)
{   return stack_overflow2(where, NULL, NULL);
}

/* Define eos transition macro */
#define state_has_eos_transition(t,s) (trel_state_scan_eos(t, s) != NULL)

static State agfl_interpret (Trellis trellis, State start_state)
{   /*------------------------------------------------------------------------------
    // Interpret the compiled grammar.
    //
    // Description:
    //	Set up the interpreters registers and jump to the first instruction.
    //
    // Note:
    //	The interpreter should have been initialized and reset.
    //----------------------------------------------------------------------------*/
    State next_start_state, last_start_state;

    /*----------------------------------
    // Initialize interpreter registers
    //--------------------------------*/
    return_to_pass2 = 0;
    i_transition = NULL;
    curr_penalty = NO_PENALTY;
    curr_trellis = trellis;
    curr_terminal = 0;
    stack = abs_calloc (STACK_SIZE, sizeof(cel), "agfl_interpret: stack");
#if DEBUG_RTS
    memset(stack, 0, STACK_SIZE * sizeof(cel));
#endif
    fp = stack + STACK_SIZE - 1 - FRAME_HEAD_LEN;
    pc = arts_ifd.code;			/* Start of code */
    pass2_fp = 0;
    pmptr = NULL;
    indent = 0;
    sp = fp;

#ifndef GEN_RTS
    if (arts_ifd.segment_mode && (start_state != NULL)) i_state = start_state;
    else i_state = trel_get_initial_state(trellis);
    last_start_state = i_state;
#else /* GEN_RTS */
    chosen_freq = 0;
    alt_depth = 0;
    i_state = NULL;
    choice_admin_init ();
    chosen_freq = 0;
    ALT_DEPTH_RESET;
#endif /* GEN_RTS */
    clear_parse_stack();

    /*
       Here we enter the main dispatcher loop
    */
    status = EXECUTING;
    start_parse_time ();
    while (status == EXECUTING)
      { /* Decode the instruction */
	int opcode = pc -> ival;
	int nr_of_opnds = (opcode >> 16) & 0xffff;
	exec_proc execute;
	next_pc = pc + (nr_of_opnds + 1);
	opcode &= 0xff;
	execute = all_opcode_execs[opcode].opcode_exec;

	/* Update the statistics for this instruction and possibly log it */
	logexec (opcode);
	opc_count (opcode);

	/* Execute the decoded instruction */
	do
	   { vptr result = execute ();
	     execute = (exec_proc) result;
	   }
	while (execute != NULL);
	pc = next_pc;
      };

    /*
       We finished executing in the main loop
       Finalize the current administration and locate the next start state
    */
#ifndef GEN_RTS
    stop_parse_time ();
    DB_RTS (abs_message ("END: current time is %.3fs.", get_parse_time()));
    if (parse_results_get_nr_parses ())
      print_parsed_sentence ();
    parse_results_dump ();
    show_totals (curr_trellis);

    /* Restart interpreter later? Find out where. */
    if (arts_ifd.segment_mode)
      { next_start_state = NULL;	/* For now meaning "unknown yet" */
	/* Try to continue after the first parse, if any */
	if (parse_results_get_nr_parses ())
	  { next_start_state = get_next_state_from_first_parse ();
	  }
	/*
	 * If that wasn't possible, or if the parse was empty,
	 * skip some input.
	 */
	if (!next_start_state || next_start_state == last_start_state)
	  { next_start_state = get_next_whitespace_or_eos (curr_trellis, last_start_state, 1);
	    /* Print what we skipped, if possible. */
            if (next_start_state != NULL)
	      { parse_result.startpos = trel_state_get_pos (last_start_state);
		parse_result.endpos = trel_state_get_pos (next_start_state);
	        print_parsed_sentence ();
		printf ("# (skipped)\n");
	      }
	  }
	/* From here on, next_start_state == NULL means "stop parsing". */
	if (next_start_state && state_has_eos_transition (trellis, next_start_state))
	  { /* we are at the end of the paragraph, so we may stop */
	    next_start_state = NULL;	/* stop parsing */
	  }
      }
    else
      { /* Not in segment-mode, so stop. */
        next_start_state = NULL;	/* stop parsing */
      }


    DB_RTS(abs_message ("# last_start_state=%p, START_INPUT_STATE=%p, next_start_state=%p",
			last_start_state, START_INPUT_STATE, next_start_state));

    parse_results_destroy ();
#else /* GEN_RTS */
#ifdef DEBUG_RTS
    choice_admin_dump_table ();
#endif /* DEBUG_RTS */
    choice_admin_destroy ();
#endif /* GEN_RTS */

    abs_free (stack, "agfl_interpret: stack");
    return (next_start_state);
}

/*------------------------------------------------------------------------------
// Implementation of abstract instructions.
//----------------------------------------------------------------------------*/
static vptr exec_nop ()
{   /* Continue with next instruction */
    NEXT;
}

/*-------------------------------------------------------------------------
// Instruction: CREATE (nont_nr, nr formals, nr locals, nr sons)
// Description: Setup a stack frame 
//	        Note that the frame header has been setup by CALL(_RULE)
//-----------------------------------------------------------------------*/
static vptr exec_create ()
    { int nont_nr = (int) pc[1].arg;
      int nr_formals = (int) pc[2].arg;
      int nr_locals = (int) pc[3].arg;
      int nr_sons = (int) pc[4].arg;
      int total_alloc = nr_formals + nr_locals + nr_sons;

      if (!STACK_AVAIL(total_alloc + VAR_OFFSET))
        return (stack_overflow_cont ("CREATE", exec_end));

      /* Link and allocate frame */
      sp[0].data = fp;                /* store old fp */
      fp = sp;
      sp -= total_alloc + VAR_OFFSET;

      /* Setup variable administration */
      NONT_NR = nont_nr;
      NR_FORMALS = nr_formals;
      NR_LOCALS = nr_locals;
      NR_SONS = nr_sons;

      /* Debug information */
      DB_RTS (abs_message ("CREATE(%d, %d, %d, %d)", nont_nr, nr_formals, nr_locals, nr_sons));
      DB_RTS (abs_message ("stack pointer after entry is %05x", sp));
      arts_ifd.nonterminal_profile_table[nont_nr]++;

      /* Initialize remainder of frame */
      START_INPUT_STATE = i_state;
      fp[PM_PLAYED].pmprod = NULL;
      fp[SUCCESS_OFF].arg = 0;
      fp[NR_SONS_DONE].arg = 0;
      fp[PENORIG].penalty = curr_penalty;
      fp[PMPRODPTR].pmprod = NULL;

      DB_RTS (abs_message ("CREATE (\"%s\"): start_input_state = %p, nont nr = %d, penalty = %d",
			   arts_ifd.nonterm_names[nont_nr].str,
			   i_state, nont_nr, curr_penalty));

      /*
	 Initialise the locals and sons to 0, since otherwise equal_posmemo ()
	 may look at uninitialised values sometimes (and fail to detect
	 identity). This happens because not every alternative uses (sets) the
	 full set of locals and sons so some may remain uninitialised.
	 This is done by the clear instruction starting each alternative.
      */

      /* Continue with next instruction */
      NEXT;
    }

/*-------------------------------------------------------------------------
// Instruction: DESTROY
// Description: destroys the current stack frame except for the frame head
//-----------------------------------------------------------------------*/
static vptr exec_destroy ()
    { /* Pick up fail address from stack and restore penalty, sp and fp */
      DB_RTS (abs_message ("DESTROY stack frame of nonterminal %d, \"%s\"",
	      (int) NONT_NR, arts_ifd.nonterm_names[NONT_NR].str));
      curr_penalty = fp[PENORIG].penalty;
      sp = fp;
      fp = fp[OLD_FP].data;
      NEXT;
    }

/*-------------------------------------------------------------------------
// Instruction: RETURN
// Description: Returns to fail address of caller
//-----------------------------------------------------------------------*/
static vptr exec_return ()
    { int was_rule_call = (sp[SUCC_ADDR_OFF].code != NULL);

      DB_RTS (abs_message ("RETURN: returning from %scall to father through fail address",
			   (was_rule_call)?"rule ":""));
      pop_parse_call ();

      /* Horreur du parse: undo the increment of NR_SONS_DONE by CALL_RULE */
      if (was_rule_call)
        fp[NR_SONS_DONE].arg--;

      /* Pick next pc and clean up return/cont address */
      next_pc = sp[FAIL_ADDR_OFF].code; 
      sp = sp + FRAME_HEAD_LEN;
      NEXT;
    }

/*-------------------------------------------------------------------------
// Instruction: MAKE_PM (transduce_code, print_tree_code, nest_nr)
// Description: Constructs a positive memo
//		Activate hybrid code for this pm if necessary
//		Jump to fail label if the penalty change would block
//-----------------------------------------------------------------------*/
static int hybrid_code_is_null_code (CODE *pc);
static int hybrid_code_is_a_single_son (CODE *pc, hyb_anchor *son);

static vptr exec_make_pm ()
    { /* Administer we succeeded up to this point */
      CODE *routine = (arts_ifd.transduce_option)?pc[1].code:pc[2].code;
      int nest_nr = (int) pc[3].arg;
      Penalty delta_penalty = curr_penalty - fp[PENORIG].penalty;
      DB_RTS (abs_message ("MAKE_PM: curr_penalty = %d", curr_penalty));

      /* If there is hybrid code for this alternative, execute it */
      if (curr_anchor != NULL)
	{ Penalty hybrid_penalty_gain = arts_hyb_try_execute (NONT_NR, NR_FORMALS, curr_anchor);
	  if (hybrid_penalty_gain == MIN_PENALTY) /* Unbalanced parenthesis */
	    delta_penalty = MIN_PENALTY;
	  else if (penalty_change_would_block (curr_penalty, hybrid_penalty_gain))
	    { /*
		 Either we could not find the triples with closed_triple_db in effect or
		 the penalty gain is too much. In this case do not store posmemo
	      */
	      detach_hyb_anchor(curr_anchor, NULL);
	      curr_anchor = NULL;
	      NEXT;
	    }
	  else delta_penalty += hybrid_penalty_gain;
        };

      /*
       * Allocate a posmemo and try to insert it.
       * Ownership of curr_anchor is passed to it, so if the posmemo
       * doesn't survive, then the curr_anchor won't either.
       */
      pmptr = posmemo_add_production (START_INPUT_STATE, (int)NONT_NR,
		delta_penalty, (int)NR_FORMALS, (int)NR_LOCALS,
		(int)fp[NR_SONS_DONE].arg, &(VARIABLE(0).val),
		i_state, routine, curr_anchor);
      DB_RTS(abs_message ("MAKE_PM: pmptr=%p fp=%p", pmptr, fp);)
      if (pmptr != NULL)
        { StateExtension ext = GetStateExtension(START_INPUT_STATE);
	  DB_RTS (abs_message ("MAKE_PM: posmemo changed for rule nr %ld in nest %d",
		  NONT_NR, nest_nr));
	  ext -> nest_markers[nest_nr] |= NEST_POSMEMO_CHANGED;
	  fp[SUCCESS_OFF].arg = 1;

        }

      /* Continue with next instruction */
      curr_anchor = NULL;
      NEXT;
    }

/*-------------------------------------------------------------------------
// Instruction: MAKE_HYB (hybrid_code)
// Description: Constructs the hybrid code and sets curr_anchor accordingly
//-----------------------------------------------------------------------*/
static vptr exec_make_hyb ()
    { if (arts_ifd.hybrid_parsing_option)
	{ CODE *hybrid_code = pc[1].code;
	  if (hybrid_code_is_null_code (hybrid_code))
	    { DB_HYBRID (abs_message ("Skipping hybrid null/nothing code for %s",
				      arts_ifd.nonterm_names[NONT_NR].str));
	      curr_anchor = NULL;
	    }
	  else if (hybrid_code_is_a_single_son (hybrid_code, &curr_anchor))
	    DB_HYBRID (abs_message ("Inheriting hybrid code from single son for %s",
				    arts_ifd.nonterm_names[NONT_NR].str));
	  else
            { /* Push next PC to return at instruction following make_hyb */
	      if (curr_anchor != NULL)
	      { /* doesn't happen */
		abs_message ("curr_anchor not NULL %p when making new one\n", curr_anchor);
		detach_hyb_anchor(curr_anchor, NULL);
	      }
	      curr_anchor = arts_create_hybrid_anchor ();
	      if (!STACK_AVAIL(1))
	        return (stack_overflow_cont ("MAKE_POSMEMO: HYBRID CALL", exec_end));
	      sp -= 1;
	      sp[1].code = next_pc;
	      GOTO (hybrid_code);
	    }
	}

      /* Continue with next instruction */
      NEXT;
    }

static vptr exec_may_play_pm ()
{
    NEXT;
    if (fp[PM_PLAYED].pmprod == NULL) {
        StateExtension ext;
	PosMemo pm;

	/* We can mess with the global i_state since we're at the end of an
	 * alternative.
	 */
        State input_state = START_INPUT_STATE;
        ext = GetStateExtension(input_state);

	pm = posmemo_get_prod_ptr (ext, (int)NONT_NR);
	if (pm != POSMEMO_UNKNOWN && pm != POSMEMO_BLOCKED) {

	    /* TODO: penalty check */
	    if (STACK_AVAIL(2 + NR_FORMALS)) {
		DB_RTS(abs_message ("MAY_PLAY_PM: first production, generating..."));
		fp[PMPRODPTR].pmprod = pm;
		fp[PM_PLAYED].pmprod = pm;
		pm_penalty_gain = posmemo_get_penalty (pm);

		i_state = input_state;
		CONT (pm_generate_production);
		/*
		 * pm_generate_production() requires PMPRODPTR.
		 *
		 * Like play_pm, this may_play_pm instruction will
		 * eventually be re-executed after an ucall_rule
		 * resets the pc.
		 */
	    } else {
		return (stack_overflow_cont ("MAY_PLAY_PM", exec_end));
	    }
	}
    } else {
	/*
	 * play_pm, when it is executed later, will automatically play the
	 * PosMemo following the one we left in PMPRODPTR. But since by that
	 * time it may not be the first any more, erase it.
	 */
	DB_RTS(abs_message ("MAY_PLAY_PM: reset PMPRODPTR"));
	assert (fp[PMPRODPTR].pmprod == NULL ||
		fp[PMPRODPTR].pmprod == fp[PM_PLAYED].pmprod);
	fp[PMPRODPTR].pmprod = NULL;
    }
    NEXT;
}

/*-------------------------------------------------------------------------
// Instruction: PLAY_PM flag
// Description: Inspect the current positive memo;
//		When not blocked nor empty, iterate over the list of
//		all positive memos to generate productions by
//		continuing for every memo with the success continuation
//		in the frame of the father
//
//		The flag is in use for left recursion: it is set when
//		extending left-recursive rules.
//-----------------------------------------------------------------------*/
static vptr exec_play_pm ()
    { /* Pick up fail address and restore input state */
      int extending = (int) pc[1].arg & 1;
      StateExtension ext;
      PosMemo pm;

      i_state = START_INPUT_STATE;
      ext = GetStateExtension(i_state);

#if !NEW_TIMEOUT
      /* Check if we have run out of time */
      if (have_time_limit () && have_time_out ())
	/* Stop executing */
	return (exec_end ());
#endif

      /* Reset the posmemo changed */
      DB_RTS (abs_message ("PLAYing %sposmemos in \"%s\" into father \"%s\"",
		   (extending)?"acceptable ":"",
		   arts_ifd.nonterm_names[NONT_NR].str,
		   arts_ifd.nonterm_names[(fp[OLD_FP].data)[NONT_NR_OFF].arg].str);
	      abs_message ("PLAYing posmemo: start_input_state = %p", i_state));

      /* When if we have a non blocked posmemo, we can generate continuations */
      if (posmemo_is_unblocked (ext, NONT_NR) || fp[PMPRODPTR].pmprod != NULL)
	{ pm_penalty_gain = MAX_PENALTY;
	  DB_RTS (abs_message ("PLAY_PM: \"%s\" is not blocked",
		  arts_ifd.nonterm_names[NONT_NR].str));

	  /* Restore the penalty level */
          curr_penalty = fp[PENORIG].penalty;

	  /*
	     If this is the first time that we generate productions, we fetch the
	     first posmemo; otherwise we pick the next from the list
	  */
	  if (fp[PMPRODPTR].pmprod == NULL)
	    { /* Fill posmemo production pointer with first entry */
	      //fp[PMPRODPTR].pmprod = posmemo_get_prod_ptr (ext, (int)NONT_NR);
	      pm = posmemo_get_prod_ptr (ext, (int)NONT_NR);

	      /* If we get here, there must be a production to try */
	      //assert(fp[PMPRODPTR].pmprod);
	      assert(pm);
	    }
	  else
	    { /* get the next posmemo production pointer */
	      //fp[PMPRODPTR].pmprod = posmemo_get_next_prod (fp[PMPRODPTR].pmprod);
	      pm = posmemo_get_next_prod (fp[PMPRODPTR].pmprod);
	    };

	  /* Fetch the penalty gain */
#if 0
	  if (fp[PMPRODPTR].pmprod != NULL) {
	    pm_penalty_gain = posmemo_get_penalty (fp[PMPRODPTR].pmprod);
	  /* If we would block on the new penalty gain, we must try the next production */
	  while ((fp[PMPRODPTR].pmprod != NULL) &&
		 ((extending == 0) || posmemo_is_acceptable (fp[PMPRODPTR].pmprod)) &&
		  penalty_change_would_block (curr_penalty, pm_penalty_gain))
	    {
	      fp[PMPRODPTR].pmprod = posmemo_get_next_prod (fp[PMPRODPTR].pmprod);
	      if (fp[PMPRODPTR].pmprod != NULL)
	        pm_penalty_gain = posmemo_get_penalty (fp[PMPRODPTR].pmprod);
	    };
#else
	  while (pm != NULL) {
	      pm_penalty_gain = posmemo_get_penalty (pm);

	      if ((extending && !posmemo_is_acceptable (pm)) ||
		      penalty_change_would_block (curr_penalty, pm_penalty_gain) ||
		      (pm == fp[PM_PLAYED].pmprod)) {
		  pm = posmemo_get_next_prod (pm);
	      } else {
		  break;
	      }
	  }
#endif

	  /* If we have a valid production pointer, we generate the production */
	  //if (fp[PMPRODPTR].pmprod != NULL)
	  fp[PMPRODPTR].pmprod = pm;
	  if (pm != NULL)
	    { if (STACK_AVAIL(2 + NR_FORMALS))
		{ DB_RTS(abs_message ("PLAY_PM: productions left, generating..."));
	          CONT (pm_generate_production);
		}
	      else
		/* no return here, we need to do the stuff below */
	        return (stack_overflow_cont ("PLAY_PM", exec_end));
	    }
	  else
	    { DB_RTS(abs_message ("PLAY_PM: no more productions left..."));
	      NEXT;
	    }
	}
      else if (posmemo_is_blocked (ext, (int)NONT_NR))
	{ DB_RTS (abs_message ("PLAY_PM: \"%s\" is blocked",
			       arts_ifd.nonterm_names[NONT_NR].str));
	}
      else
	{ DB_RTS (abs_message ("PLAY_PM: \"%s\" is still unknown",
			       arts_ifd.nonterm_names[NONT_NR].str));
	};
      NEXT;
    }


/*-------------------------------------------------------------------------
// Generate production
// Description: push formals to stack, set input state and penalty level,
// 	        and continue with success continuation.
//-----------------------------------------------------------------------*/
static vptr pm_generate_production (void)
    { int nr_formals = (int) NR_FORMALS;
      PosMemo pmprod = fp[PMPRODPTR].pmprod;
      void *memod_formals = posmemo_get_formal_ptr (pmprod);

      if (!STACK_AVAIL(2 + nr_formals))
        { /* should have been checked in "caller" exec_fail(). */
	  return (stack_overflow_cont ("POSMEMO GENERATE PRODUCTION", exec_end));
        }

      /* Save old PC and FP; picked up by UCALL */
      sp -= 2;
      sp[2].code = pc;
      sp[1].data = fp; 

      /* push result parameters: check for 64 bit longs */
      sp -= nr_formals;
      memcpy (sp + 1, memod_formals, nr_formals * sizeof (Value));

      /* It is no longer possible to fail for a penalty increase */
      i_state = posmemo_get_next_input_state (pmprod, i_state);
      curr_penalty = sum_penalties(curr_penalty, pm_penalty_gain);
      DB_RTS (abs_message("pm_generate_production: pmprod = %p, i_state = %p", pmprod, i_state));

      /*
	 Continue with the continuation which assumes
	 execution in the previous stack frame:
      */
      next_pc = fp[SUCC_ADDR_OFF].code;
      fp = fp[OLD_FP].data;
      NEXT;
    }

/*-------------------------------------------------------------------------
// Instruction: Try block pm
// Description: Sets posmemo to blocked if it is still unknown
//
// This instruction is executed for non leftrecursive rules, after
// testing every alternative for a possible parse.
//-----------------------------------------------------------------------*/
static vptr exec_try_block_pm ()
    { /* Pick up input state ptr */  
      State input_state = START_INPUT_STATE;
      StateExtension ext = GetStateExtension(input_state);

      DB_RTS (abs_message ("TRY_BLOCK_PM for rule %d \"%s\" at pos %d", (int) NONT_NR,
			   arts_ifd.nonterm_names[NONT_NR].str,
			   trel_state_get_pos (input_state)));

      /* If posmemo is still not known, block it */
      if (posmemo_is_unknown (ext, NONT_NR))
	posmemo_set_blocked (ext, NONT_NR);

      /* Continue with next instruction */
      NEXT;
    }

/*-------------------------------------------------------------------------
// Instruction: if pm known
// Description: If posmemo is completely known, jump to play
//-----------------------------------------------------------------------*/
static vptr exec_if_pm_known ()
    { /* Pick up input state ptr and jump address */
      State input_state = START_INPUT_STATE;
      StateExtension ext = GetStateExtension(input_state);
      CODE *jump_addr = pc[1].code;

      DB_RTS (abs_message ("IF_PM_KNOWN for rule %d \"%s\" at pos %u", (int) NONT_NR,
			   arts_ifd.nonterm_names[NONT_NR].str,
			   trel_state_get_pos (input_state)));

#if NEW_TIMEOUT_ON_PM_KNOWN
      /*
       * Checking for time-out on `if_pm_known' instructions is a good
       * location, since this instruction is generated into the entry code
       * of every rule. It is better than checking on `call_rule' instr's
       * (and letting them fail) because here we have the chance to get
       * a result, if already known, by jumping to the `play_pm' instr.
       */
      if (have_time_limit () && have_time_out ())
	/* Stop processing */
	GOTO (jump_addr);
#endif

      /*
	 If the positive memo is known for this input state,
	 we continue through the play continuation
      */
      if (posmemo_is_known (ext, (int)NONT_NR))
	{ /* Posmemo fully known */
	  DB_RTS (abs_message ("IF_PM_KNOWN: posmemo is fully known, jumping to play posmemo"));
	  GOTO (jump_addr);
	};

      /* Continue with next instruction */
      NEXT;
    }

/*-------------------------------------------------------------------------
// Instruction: JUMP
// Description: Jumps to label
//-----------------------------------------------------------------------*/
static vptr exec_jump ()
    { CODE *target = pc[1].code;
      DB_RTS (abs_message ("JUMP: jumping"));
      GOTO (target);
    }

/*---------------------------------------------------------------------------
// Instruction: CALL label
// Description: Simply make a call to a label (for a left recursion nest)
//		The success address is explicitly marked with NULL,
//		so that we can make a distinction between frames
//		that can be continued or those that cannot.
//-------------------------------------------------------------------------*/
static vptr exec_call ()
    { CODE *jmp_point = pc[1].code;
      CODE *fail_addr = next_pc;
      CODE *succ_addr = NULL;

      DB_RTS (abs_message ("CALL from \"%s\"", arts_ifd.nonterm_names[NONT_NR].str));
#if !NEW_TIMEOUT
      if (have_time_limit () && have_time_out ())
	/* Stop processing */
	return (exec_end ());
#endif

      if (!STACK_AVAIL (FRAME_HEAD_LEN))
        return (stack_overflow_goto ("CALL", fail_addr));

      /* Construct precall frame part */
      sp -= FRAME_HEAD_LEN;
      sp[FAIL_ADDR_OFF].code = fail_addr;
      sp[SUCC_ADDR_OFF].code = succ_addr;
      push_parse_call ((int)NONT_NR);

      /* Jump to called address */
      GOTO (jmp_point);
    }

/*-------------------------------------------------------------------------------
// Instruction: CALL_RULE (syntax rule number, start point, fail continuation)
// Description: Call to a rule by building the pre-call stack part and jump
//	        to the starting point.
//
// If the rule is blocked for this position by known posmemo, we fail
// In all other cases we enter the rule to get its formals and continue.
//-----------------------------------------------------------------------------*/
static vptr exec_call_rule ()
    { /* Pick up arguments */
      int rule_nr = (int) pc[1].arg;
      CODE *jmp_point = pc[2].code;
      CODE *fail_addr = pc[3].code;
      CODE *succ_addr = next_pc;
      StateExtension ext = GetStateExtension(i_state);

      DB_RTS (abs_message ("CALLing \"%s\" from \"%s\"",
      		           arts_ifd.nonterm_names[rule_nr].str,
			   arts_ifd.nonterm_names[NONT_NR].str));
#if !NEW_TIMEOUT
      if (have_time_limit () && have_time_out ())
	/* Stop processing */
	return (exec_end ());
#endif

      /* Check the positive memo for this position */
      if (posmemo_is_blocked (ext, rule_nr))
	{ DB_RTS (abs_message ("CALL: \"%s\" is blocked",
			       arts_ifd.nonterm_names[rule_nr].str));
	  GOTO (fail_addr);
	}
#if DEBUG_RTS
      else if (posmemo_is_known (ext, rule_nr))
        DB_RTS (abs_message ("CALL: \"%s\" is known",
			     arts_ifd.nonterm_names[rule_nr].str));
      else
        { /* positive memo is unknown */
	  DB_RTS (abs_message ("CALL: \"%s\" is still unknown",
			       arts_ifd.nonterm_names[rule_nr].str));
	};
#endif

      if (!STACK_AVAIL (FRAME_HEAD_LEN))
        return (stack_overflow_goto ("CALL", fail_addr));

      /* Construct precall frame part */
      sp -= FRAME_HEAD_LEN;
      sp[FAIL_ADDR_OFF].code = fail_addr;
      sp[SUCC_ADDR_OFF].code = succ_addr;
      push_parse_call (rule_nr);

      assert (fp[NR_SONS_DONE].arg >= 0);
      assert (fp[NR_SONS_DONE].arg < NR_SONS);

      DB_RTS (abs_message ("sons done: %ld, son_fp = %p",
			   fp[NR_SONS_DONE].arg, fp + fp[NR_SONS_DONE].arg));
      SON_FP(fp[NR_SONS_DONE].arg) = sp;
      fp[NR_SONS_DONE].arg++;

      /* Jump to called address */
      GOTO (jmp_point);
    }
 
/*---------------------------------------------------------------------------
// Instruction: UCALL_RULE(nr formals)
// Description: clean up previous result parameters from the stack,
//	        restore the frame pointer and jump back to the child rule.
//
// Note that this instruction is actually a undo play pm
//-------------------------------------------------------------------------*/
static vptr exec_ucall_rule ()
    { int nr_formals = (int) pc[1].arg;
      sp += nr_formals;
      next_pc = sp[2].code;
      fp = sp[1].data;
      sp += 2;

      DB_RTS (abs_message ("UCALL_RULE(%d) into rule %d \"%s\"", nr_formals, (int) NONT_NR,
			   arts_ifd.nonterm_names[NONT_NR].str));

      /* Continue with next instruction */
      i_state = START_INPUT_STATE; /* Reset i_state back to what it was
				    * at the start of the rule. This is really
				    * incorrect, it should be what it was
				    * before the call. It just happens to work,
				    * because once we start ucalling, we do it
				    * 1 or more times[1] and then go to the
				    * next alternative, where it should be back
				    * at the start of the rule. The
				    * intermediate states are not used.
				    * [1]: there may be umatches, too; they do
				    * properly restore i_state, so they may
				    * even correct the mistake of ucall!
				    */
      curr_penalty = fp[PENORIG].penalty; /* Same for curr_penalty, except
                                           * that umatch-es may mess up the
					   * penalty (it does a -= ...).
					   */
      NEXT;
    }

/*-------------------------------------------------------------------------
// Instruction: MARK_COLLECTING 
// Description: Mark that we are collecting the kernel of
//	        a leftrecursive nest
//-----------------------------------------------------------------------*/
static vptr exec_mark_collecting ()
    { /* Pick up nest nr and input state ptr */
      int nest_nr = (int) pc[1].arg;
      State input_state = START_INPUT_STATE;
      StateExtension ext = GetStateExtension (input_state);

      DB_RTS (abs_message ("MARK_COLLECTING for nest %d at pos %d",
			   nest_nr, trel_state_get_pos (input_state)));

      /* Do the marking and unmark an extending flag for nest 0 */
      ext -> nest_markers[nest_nr] = NEST_COLLECTING;
      ext -> nest_markers[0] = 0;

      /* Continue with next instruction */
      NEXT;
    }

/*-------------------------------------------------------------------------
// Instruction: IF_COLLECTING
// Description: Jump if we are currently collecting the kernel
//		of a left recursive nest
//-----------------------------------------------------------------------*/
static vptr exec_if_collecting ()
    { /* Pick up nest nr and input state ptr */
      int nest_nr = (int) pc[1].arg;
      CODE *jump_addr = pc[2].code;
      State input_state = START_INPUT_STATE;
      StateExtension ext = GetStateExtension(input_state);
      DB_RTS (abs_message ("IF_COLLECTING for rule %d \"%s\" at state %d", (int) NONT_NR,
			   arts_ifd.nonterm_names[NONT_NR].str,
			   trel_state_get_pos (input_state)));

      /* Check the marking */
      if (ext -> nest_markers[nest_nr] & NEST_COLLECTING)
	GOTO (jump_addr);

      /* Continue with next instruction */
      NEXT;
    }

/*-------------------------------------------------------------------------
// Instruction: UMARK_COLLECTING 
// Description: Unmark that we are collecting the kernel of
//		a left recursive nest
//-----------------------------------------------------------------------*/
static vptr exec_umark_collecting ()
    { /* Pick up nest nr and input state ptr */
      int nest_nr = (int) pc[1].arg;
      State input_state = START_INPUT_STATE;
      StateExtension ext = GetStateExtension(input_state);
      int ix;

      DB_RTS (abs_message ("UMARK_COLLECTING for nest %d at pos %d", 
			   nest_nr, trel_state_get_pos (input_state)));

      /* Undo the marking */
      ext -> nest_markers[nest_nr] &= ~NEST_COLLECTING;

      /*
	 Check if there is still a nest to be unmarked for extension
         if so, remark nest 0 (it may have been reset upon a switch
	 from one nest to the other). If not, reset nest marker 0 in
	 the case that we did not succeed
      */
      for (ix = 1; ix <= arts_ifd.nr_lrec_nests; ix++)
	if (ext -> nest_markers[ix] & NEST_EXTENDING)
	  ext -> nest_markers[0] = NEST_EXTENDING;

      /* Continue with next instruction */
      NEXT;
    }

/*-------------------------------------------------------------------------
// Instruction: MARK_EXTENDING 
// Description: Mark that we are extending the kernel of
//	        a left recursive nest
//-----------------------------------------------------------------------*/
static vptr exec_mark_extending ()
    { /* Pick up nest nr and input state ptr */
      int nest_nr = (int) pc[1].arg;
      State input_state = START_INPUT_STATE;
      StateExtension ext = GetStateExtension (input_state);

      DB_RTS (abs_message ("MARK_EXTENDING for nest %d at pos %d",
			   nest_nr, trel_state_get_pos (input_state)));

      /*
       * Do the marking. Reset NEST_COLLECTING since we don't
       * do both at once. Reset NEST_POSMEMO_CHANGED too since we're
       * only interested in additional posmemos during the extending
       * phase.
       */
      ext -> nest_markers[nest_nr] = NEST_EXTENDING;
      ext -> nest_markers[0] = NEST_EXTENDING;

      /* Continue with next instruction */
      NEXT;
    }

/*-------------------------------------------------------------------------
// Instruction: IF_EXTENDING
// Description: Jump if we are currently extending the kernel
//		of a left recursive nest
//-----------------------------------------------------------------------*/
static vptr exec_if_extending ()
    { /* Pick up nest nr and input state ptr */
      int nest_nr = (int) pc[1].arg;
      CODE *jump_addr = pc[2].code;
      State input_state = START_INPUT_STATE;
      StateExtension ext = GetStateExtension(input_state);
      DB_RTS (abs_message ("IF_EXTENDING for rule %d \"%s\" at state %d", (int) NONT_NR,
			   arts_ifd.nonterm_names[NONT_NR].str,
			   trel_state_get_pos (input_state)));

      /* Check the marking */
      if (ext -> nest_markers[nest_nr] & NEST_EXTENDING)
	GOTO (jump_addr);

      /* Continue with next instruction */
      NEXT;
    }

/*-------------------------------------------------------------------------
// Instruction: UMARK_EXTENDING 
// Description: Unmark that we are collecting the kernel of
//		a left recursive nest
//-----------------------------------------------------------------------*/
static vptr exec_umark_extending ()
    { /* Pick up nest nr and input state ptr */
      int nest_nr = (int) pc[1].arg;
      State input_state = START_INPUT_STATE;
      StateExtension ext = GetStateExtension(input_state);
      int ix;

      DB_RTS (abs_message ("UMARK_EXTENDING for nest %d at pos %d", 
			   nest_nr, trel_state_get_pos (input_state)));

      /* Undo the marking */
      ext -> nest_markers[nest_nr] = 0;

      /*
	 Check if this is the last nest to be unmarked
	 if so, also unmark nest 0 for non left recursive rules
         if not the last nest, remark nest 0 (it may have been reset upon
	 a switch from one nest to the other)
      */
      for (ix = 1; ix <= arts_ifd.nr_lrec_nests; ix++)
	if (ext -> nest_markers[ix] & NEST_EXTENDING)
	  { ext -> nest_markers[0] = NEST_EXTENDING;
	    NEXT;
	  };

      /* Unmark nest 0 */
      ext -> nest_markers[0] = 0;

      /* Continue with next instruction */
      NEXT;
    }

/*-------------------------------------------------------------------------
// Instruction: UPDATE_NEST_PM		nest_nr
// Description: Promote all new posmemos of this nest to acceptable
//		Demote all acceptable posmemos of this nest to unacceptable
//
// When called with nest_nr 0, only the posmemo of the current frame is
// updated.
//-----------------------------------------------------------------------*/
static vptr exec_update_nest_pm ()
    { /* Pick up nest nr */
      int nest_nr = (int) pc[1].arg;
      State input_state = START_INPUT_STATE;	/* Note, uses originator's FP */
      StateExtension ext = GetStateExtension(input_state);
      int ix = 0;

      DB_RTS (abs_message ("UPDATE_NEST_PM for nest %d at state %d", 
			   nest_nr, trel_state_get_pos (input_state)));

      if (nest_nr)
	{ /* Iterate over nest numbers until -1 */
          DATA *nrs = arts_ifd.lrec_nests_table[nest_nr].data;
          while (nrs[ix].ilval != -1)
            { posmemo_update (ext, (int)nrs[ix].ilval);
	      ix++;
	    };
        }
      else posmemo_update (ext, (int) NONT_NR);

      /* Continue with next instruction */
      NEXT;
    }

/*-------------------------------------------------------------------------
// Instruction: TRY_BLOCK_NEST_PM	nestnr
// Description: Iterate over all posmemos of this nest
//		When a posmemo is still unknown, set it to blocked.
//-----------------------------------------------------------------------*/
static vptr exec_try_block_nest_pm ()
    { /* Pick up nest nr */
      int nest_nr = (int) pc[1].arg;
      DATA *nrs = arts_ifd.lrec_nests_table[nest_nr].data;
      State input_state = START_INPUT_STATE;	/* Note, uses originator's FP */
      StateExtension ext = GetStateExtension(input_state);
      int ix = 0;

      DB_RTS (abs_message ("TRY_BLOCK_NEST_PM for nest %d at state %d", 
			   nest_nr, trel_state_get_pos (input_state)));

      /* Iterate over nest numbers until -1 */
      while (nrs[ix].ilval != -1)
        { if (posmemo_is_unknown (ext, (int)nrs[ix].ilval))
            posmemo_set_blocked (ext, (int)nrs[ix].ilval);
	  ix++;
        }

      /* Continue with next instruction */
      NEXT;
    }

/*-------------------------------------------------------------------------
// Instruction: IF_NEST_PM_CHGD		nestnr, label
// Description: Check if we have new posmemos in this nest
//-----------------------------------------------------------------------*/
static vptr exec_if_nest_pm_chgd ()
    { /* Pick up nest nr and input state ptr */
      int nest_nr = (int) pc[1].arg;
      CODE *jump_addr = pc[2].code;
      State input_state = START_INPUT_STATE;
      StateExtension ext = GetStateExtension(input_state);
      DB_RTS (abs_message ("IF_NEST_PM_CHGD for nest %d at state %d", 
			   nest_nr, trel_state_get_pos (input_state)));

      /* Check the marking */
      DB_RTS (abs_message ("NEST MARKER: %d, cleared after jump", ext -> nest_markers[nest_nr]));
      if (ext -> nest_markers[nest_nr] & NEST_POSMEMO_CHANGED)
	{ /* Clear before jump */
	  ext -> nest_markers[nest_nr] &= ~NEST_POSMEMO_CHANGED;
	  GOTO (jump_addr);
	}

      /* Continue with next instruction */
      NEXT;
    }

/*------------------------------------------------------------------------------
// Instructions on variables:
// Note: Instructions that implement predicates are also added here
//----------------------------------------------------------------------------*/

/*------------------------------------------------------------------------------
// Instruction: CLEAR (nr)
// Description: Clear the frame at position 0..nr-1 to initialize
//		all formals and locals to 0.
//----------------------------------------------------------------------------*/
static vptr exec_clear ()
    { ARG nr = pc[1].arg;
      ARG idx;
      for (idx = 0; idx < nr; idx++)
	VARIABLE(idx).ilval = (int64) 0;

      /* Continue with next instruction */
      NEXT;
    }

/*------------------------------------------------------------------------------
// Instruction: INIT_{IST}(position, value)
// Description: initialize INT/SET/TEXT variable at position (the position
//	        is relative to the frame pointer plus fixed part offset)
//	        to the specified value.
//----------------------------------------------------------------------------*/
static vptr exec_init_s ()
    { /* Pick up arguments */
      int pos = (int) pc[1].arg;
      Value val = pc[2].val;

      /* Initialize this variable */
      VARIABLE(pos).val = val;

      /* Continue with next instruction */
      NEXT;
    }

static vptr exec_init_v ()
    { exec_illegal ();

      /* Continue with next instruction */
      NEXT;
    }

static int unify_texts(char **lhsp, char *rhs)
{
    char *lhs = *lhsp;

    if (lhs == TOP_TEXT) {
	*lhsp = rhs;
	return 1;
    }
    if (rhs == TOP_TEXT) {
	return 1;
    }
    if (strcmp(lhs, rhs) == 0) {
	return 1;
    }

    return 0;
}

static int unify_ints(int *lhsp, int rhs)
{
    int lhs = *lhsp;

    if (lhs == TOP_INT) {
	*lhsp = rhs;
	return 1;
    }
    if (rhs == TOP_INT) {
	return 1;
    }
    if (lhs == rhs)
	return 1;

    return 0;
}

/*------------------------------------------------------------------------------
// Instruction: ADJUST_{IST}(target, result param, fail)
// Description: restrict INT/SET/TEXT variable at target (the position
//	        is relative to the frame pointer plus fixed part offset)
//	        to the value after continuing the recent CALL
//
// Note: we use the location of the restriction result value to save the
//       original value. UADJUST will restore from this location
//----------------------------------------------------------------------------*/
static vptr exec_adjust_i ()
    { /* Pick up arguments */
      ARG target = pc[1].arg;
      ARG param = pc[2].arg;
      CODE* fail_addr = pc[3].code;

      /* Pick up restricting and original value */
      Value val = sp[param].val;
      Value orig = VARIABLE(target).val;

      /* Do flat trellis operation */
      DB_RTS(abs_message ("ADJUST_I: adjusting to %d", val.int_par));
      if (unify_ints (&VARIABLE(target).val.int_par, val.int_par) == 0)
	GOTO(fail_addr);

      sp[param].val = orig;			/* save original for restore */

      /* Continue with next instruction */
      NEXT;
    }

static vptr exec_adjust_s ()
    { /* Pick up arguments */
      ARG target = pc[1].arg;
      ARG param = pc[2].arg;
      CODE* fail_addr = pc[3].code;

      /* Pick up restricting and original value */
      Value val = sp[param].val;
      Value orig = VARIABLE(target).val;
      Value result;

      /* Do the trellis operation and intersect the sets */
      result.set_par = orig.set_par & val.set_par;
      DB_RTS (abs_message ("ADJUST_S: %llx <- (orig) %llx & (res from stack) %llx",
		 	   result.set_par, orig.set_par, val.set_par));

      /* If the intersection of the sets is empty --> fail */
      if (result.set_par == 0)
	GOTO (fail_addr);

      /* save original value and set the variable to the intersection */
      sp[param].val = orig;
      VARIABLE(target).val = result;

      /* Continue with next instruction */
      NEXT;
    }

static vptr exec_adjust_t ()
    { /* Pick up arguments */
      ARG target = pc[1].arg;		/* local stack variable */
      ARG param = pc[2].arg;		/* parameter from previous call */
      CODE* fail_addr = pc[3].code;

      /* Pick up restricting and original value */
      Value val = sp[param].val;
      Value orig = VARIABLE(target).val;

      /* Do flat trellis operation */
      DB_RTS (abs_message ("ADJUST_T: adjusting %s to %s",
	      VARIABLE(target).val.text_par, val.text_par));
      if (unify_texts(&VARIABLE(target).val.text_par, val.text_par) == 0)
	GOTO(fail_addr);			/* consistent substitution fails */
      sp[param].val = orig;			/* Save orig pointer for restore */

      /* Continue with next instruction */
      NEXT;
    }

static vptr exec_adjust_v ()
    { /* Unimplemented */
      exec_illegal ();

      /* Continue with next instruction */
      NEXT;
    }

static vptr exec_uadjust_s ()
    { /* Pick up arguments */
      ARG target = pc[1].arg;
      ARG param = pc[2].arg;
      Value val = sp[param].val;
      DB_RTS (abs_message ("UADJUST: unadjusting"));
	
      VARIABLE(target).val = val;

      /* Continue with next instruction */
      NEXT;
    }

static vptr exec_uadjust_v ()
    { /* Unimplemented */
      exec_illegal ();

      /* Continue with next instruction */
      NEXT;
    }

/*------------------------------------------------------------------------------
// Instruction: RESTRICT_{IST}(target, value, fail)
// Description: restrict INT/SET/TEXT variable at target (the position
//	        is relative to the frame pointer plus fixed part offset)
//	        by the value. Upon success, the original value is pushed
//		for later restore by URESTRICT_{IST}
//----------------------------------------------------------------------------*/
static vptr exec_restrict_i ()
    { /* Pick up arguments */
      ARG target = pc[1].arg;
      Value val = pc[2].val;
      CODE* fail_addr = pc[3].code;

      /* Pick up original value and do flat trellis operation, if stack allows */
      Value orig = VARIABLE(target).val;

      if (!STACK_AVAIL(1))
	return (stack_overflow_goto ("RESTRICT_I", fail_addr));

      /* Do flat trellis operation */
      if (unify_ints (&VARIABLE(target).val.int_par, val.int_par) == 0)
	GOTO(fail_addr);
      PUSH_VALUE (orig);			/* Save for restore */

      /* Continue with next instruction */
      NEXT;
    }

static vptr exec_restrict_s ()
    { /* Pick up arguments */
      ARG target = pc[1].arg;
      Value val = pc[2].val;
      CODE* fail_addr = pc[3].code;

      /* Pick up original value */
      Value orig = VARIABLE(target).val;
      Value result;

      /* Do the trellis operation and intersect the sets */
      result.set_par = orig.set_par & val.set_par;
      if (result.set_par == 0)			/* intersection of the sets is empty -- fail */
	GOTO (fail_addr);

      if (!STACK_AVAIL(1))
	return (stack_overflow_goto ("RESTRICT_S", fail_addr));

      /* push original value and assign intersection to variable */
      PUSH_VALUE (orig);
      VARIABLE(target).val = result;

      /* Continue with next instruction */
      NEXT;
    }

static vptr exec_restrict_t ()
    { /* Pick up arguments */
      ARG target = pc[1].arg;
      Value val = pc[2].val;
      CODE* fail_addr = pc[3].code;

      /* Pick up original value and do flat trellis operation */
      Value orig = VARIABLE(target).val;

      if (!STACK_AVAIL(1))
	return (stack_overflow_goto ("RESTRICT_T", fail_addr));

      DB_RTS (abs_message ("exec_restrict_t: target %d \"%s\" :: \"%s\"",
			   target,
			   VARIABLE(target).val.text_par, val.text_par));
      if (unify_texts(&VARIABLE(target).val.text_par, val.text_par) == 0)
	GOTO(fail_addr);			/* consistent substitution fails */
      
      PUSH_VALUE (orig);			/* Push orig pointer for restore */

      /* Continue with next instruction */
      NEXT;
    }

static vptr exec_restrict_v ()
    { /* Unimplemented */
      exec_illegal ();

      /* Continue with next instruction */
      NEXT;
    }

static vptr exec_urestrict_s ()
    { /* Pick up argument */
      ARG target = pc[1].arg;
	
      POP_VALUE (VARIABLE(target).val);

      /* Continue with next instruction */
      NEXT;
    }

static vptr exec_urestrict_v ()
    { /* Unimplemented */
      exec_illegal ();

      /* Continue with next instruction */
      NEXT;
    }

/*------------------------------------------------------------------------------
// Instruction: INPUT_POS (position, target, fail)
// Description: Allocate and sprintf "Fname, line %d, column %d" where the
//		values are taken from the current input trellis state
//		and save this string at position in the current frame
//		(relative to the frame pointer plus fixed part offset)
//
//		If an affix target is given (i.e.) the position is >= 0,
//		restrict this target with the sprinted value.
//		Upon success, the original value is pushed for later restore
//----------------------------------------------------------------------------*/
static vptr exec_input_pos ()
    { /* Pick up arguments */
      opcode instr = GET_INSTR;
      int position = (int) pc[1].arg;
      int target = (int) pc[2].arg;
      CODE* fail_addr = pc[3].code;
      char *dstr;
      int pos;

      /* Pick up original value */
      DB_RTS(abs_message ("INPUT_POS (%d, %d, failaddr)", position, target));

      /* Calculate value and store in current frame */
      if (i_state == NULL)
	abs_abort ("INPUT_POS", "Cannot take position from null input state");
#if VERBOSE_DOLLAR_POS	/* could make this a runtime option */
      if (arts_ifd.input_pos_fname == NULL)
	dstr = abs_mm_new_fmtd_string ("$INPUT_POS", "line %d, column %d",
				       trel_state_get_line(i_state), 
				       trel_state_get_col(i_state));
      else dstr = abs_mm_new_fmtd_string ("$INPUT_POS", "File '%s', line %d, column %d",
					  arts_ifd.input_pos_fname,
				  	  trel_state_get_line(i_state), 
					  trel_state_get_col(i_state));
#else
      if (instr == opc_input_pos) {
	  pos = trel_state_get_pos(i_state);
      } else if (instr == opc_input_line) {
	  pos = trel_state_get_line(i_state);
      /* } else if (instr == opc_input_col) {
	  pos = trel_state_get_col(i_state); */
      } else
	  pos = -1;
      dstr = abs_mm_new_fmtd_string ("$INPUT_POS", "%d", pos);
#endif
      VARIABLE(position).str = dstr;

      /* If we have a target affix, do the corresponding trellis operation */
      if (target >= 0)
	{ /* Do flat trellis operation */
	  Value orig = VARIABLE(target).val;
	  if (!STACK_AVAIL(1))
	    return (stack_overflow_goto ("$POS", fail_addr));

	  if (unify_texts(&VARIABLE(target).val.text_par, dstr) == 0)
	    GOTO(fail_addr);			/* consistent substitution fails */
	  PUSH_VALUE (orig);			/* Push orig pointer for restore */
	};

      /* Continue with next instruction */
      NEXT;
    }

static vptr exec_uinput_pos ()
    { /* Pick up arguments */
      int target = (int) pc[2].arg;

      /* Restore the original target affix, if present */
      DB_RTS(abs_message ("UINPUT_POS (%d, %d)", (int) pc[1].arg, target));
      if (target >= 0)
	POP_VALUE (VARIABLE(target).val);

      /* Continue with next instruction */
      NEXT;
    }

/*--------------------------------------------------------------------------
// Optimization instructions
//------------------------------------------------------------------------*/

/*--------------------------------------------------------------------------
// Instruction: TMEMO(nr, fail addr)
// Description: Test memo nr, if blocked for current or lower penalty level
//	        then goto fail addr, otherwise continue.
// Note: TMEMO is only concerned with negative memo
//       nr is the alt nr in the grammar
//------------------------------------------------------------------------*/
static vptr exec_tmemo ()
    { /* Pick up arguments */
      int memo_nr = (int) pc[1].arg;
      CODE* fail_addr = pc[2].code;
      StateExtension ext = GetStateExtension(START_INPUT_STATE);
      DB_RTS(abs_message ("TMEMO(%d, fail_addr)", memo_nr));


      if (
#if ENABLE_INTERACTIVE_TRACER
	  tracer_start_alt(memo_nr, fp, curr_trellis, i_state) ||
#endif /* ENABLE_INTERACTIVE_TRACER */
	      negmemo_is_blocked (ext, memo_nr, curr_penalty))
	{ /* blocked for this penalty level -> fail */
	  logfail(opc_tmemo);
	  GOTO (fail_addr);
	};

      /* Continue with next instruction */
      NEXT;
    }

/*--------------------------------------------------------------------------
// Instruction: TSMEMO(memo nr, fail addr)
// Description: Test and set memo: same as TMEMO, but if unknown set memo
//	      nr to blocked and continue.
// Note: TSMEMO is only concerned with negative memo
//       memo nr is the alt nr in the grammar
//------------------------------------------------------------------------*/
static vptr exec_tsmemo ()
    { /* Pick up arguments and log instructions */
      int memo_nr = (int) pc[1].arg;
      CODE* fail_addr = pc[2].code;
      StateExtension ext = GetStateExtension(START_INPUT_STATE);
      DB_RTS (abs_message ("TSMEMO(%d) at pos %d", memo_nr, trel_state_get_pos (i_state)));

      /*
	 If we know that we are blocked for this alternative at
	 this position, we immediately fail.
      */
      if (
#if ENABLE_INTERACTIVE_TRACER
	  tracer_start_alt(memo_nr, fp, curr_trellis, i_state) ||
#endif /* ENABLE_INTERACTIVE_TRACER */
	      negmemo_is_blocked (ext, memo_nr, curr_penalty))
	{ /* negmemo is blocked -> fail */
	  DB_RTS (abs_message ("Current negmemo is %d against penalty %d, failing",
			       NEG_MEMO (ext, memo_nr), curr_penalty));
	  logfail (opc_tsmemo);
	  GOTO (fail_addr);
	};

      /*
	 If we are extending a nest, we know we are going to pass
         this point again, so do not set the negative memo to blocked
      */
      if (!(ext -> nest_markers[0]) && negmemo_is_unknown (ext, memo_nr))
	negmemo_set_blocked (ext, memo_nr, curr_penalty);

      /* Continue with next instruction */
      NEXT;
    }

/*---------------------------------------------------------------------------
// Instruction: SMEMO_S(memo nr)
// Description: Set memo nr to succeeded for current penalty level.
// Note: SMEMO_S is only concerned with negative memo
//       memo nr is the alt nr in the grammar
//-------------------------------------------------------------------------*/
static vptr exec_smemo_s ()
    { int memo_nr = (int) pc[1].arg;
      StateExtension ext;
      DB_RTS(abs_message ("SMEMO_S(%d) at pos %d", memo_nr, trel_state_get_pos (i_state)));

#if ENABLE_INTERACTIVE_TRACER
      tracer_end_alt(memo_nr, fp, curr_trellis, i_state);
#endif /* ENABLE_INTERACTIVE_TRACER */
      /*
	 If we are not extending a nest and arrive here
	 while the negmemo is still unknown, something is wrong
	*/
      ext = GetStateExtension(START_INPUT_STATE);
      if (!(ext -> nest_markers[0]) && negmemo_is_unknown (ext, memo_nr))
	abs_abort ("SMEMO_S", "Negative memo is unknown when it should not be");
	
      negmemo_set_succeeded (ext, memo_nr);

      /* Continue with next instruction */
      NEXT;
    }

/*------------------------------------------------------------------------------
// Input matching instructions:
// Instruction: MATCH    (id, local, target, fail address)
//		MATCH_RE (id, local, target, fail address)
// 		SKIP_RE  (id, local, target, fail address)
//		MATCH_OTHER (0, local, target, fail address)
//
// Matching of grammar terminals, regexps matches and skips
// is performed by the same code. In the initialization phase
// of the interpreter, the terminal type has been encoded in
// the first argument of the instruction.
//
// The local is used to hold the pointer to the actual transition in the
// trellis matching the terminal type and id. The target is the position
// in the current frame of a text affix whose value is adjusted by the
// matched text.
//----------------------------------------------------------------------------*/
static vptr exec_match ()
    { Terminal terminal = (Terminal) pc[1].arg;		/* get token to be matched */
      DB_RTS (abs_message ("MATCH/SKIP at pos %d", trel_state_get_pos (i_state)));

#ifndef GEN_RTS
#ifdef TRACE_MATCH
      if (TERM_IS_MATCH (terminal))
	abs_message ("MATCH MATCH_RE pos %d: %d == \"%s\"", trel_state_get_pos (i_state),
	             DECODE_REGEXP_NUMBER(terminal),
	             arts_ifd.match_regexp_names[DECODE_REGEXP_NUMBER(terminal)].str);
      else if (TERM_IS_SKIP (terminal))
	abs_message ("MATCH SKIP_RE pos %d: %d == \"%s\"", trel_state_get_pos (i_state),
	             DECODE_REGEXP_NUMBER(terminal),
	             arts_ifd.skip_regexp_names[DECODE_REGEXP_NUMBER(terminal)].str);
      else if (TERM_IS_OTHER (terminal))
	abs_message ("MATCH OTHER pos %d", trel_state_get_pos (i_state));
      else /* TERM_IS_TERM */
	abs_message ("MATCH pos %d: %d == \"%s\"", trel_state_get_pos (i_state),
	             DECODE_TERM_NUMBER(terminal),
	             arts_ifd.term_names[DECODE_TERM_NUMBER(terminal)].str);
#endif /* TRACE_MATCH */
      if (TERM_IS_MATCH (terminal))
	i_transition = trel_state_scan_match (curr_trellis, i_state);
      else if (TERM_IS_SKIP (terminal))
	i_transition = trel_state_scan_skip (curr_trellis, i_state);
      else if (TERM_IS_OTHER (terminal))
	i_transition = trel_state_scan_other (curr_trellis, i_state);
      else /* TERM_IS_TERM */
        { if (DECODE_TERM_NUMBER(terminal) == arts_ifd.eos_terminal)
	    i_transition = trel_state_scan_eos (curr_trellis, i_state);
	  else
	    i_transition = trel_state_scan_terminal (curr_trellis, i_state);
        }
#if TRACE_MATCH
      abs_message("exec_match: i_transition = %p terminal = %x", i_transition, terminal);
#endif
      CONT (search_match);

#else /* GEN_RTS */
      gen_output_add_token (term_names[terminal]);

      /* Continue with the next instruction */
      NEXT;
#endif /* GEN_RTS */
    };

static vptr search_match ()
    { /*
        We get here by continuing from MATCH[_RE, _OTHER], or by
        failing on the penalty change from MATCH_AGAIN
        At this point i_transition is always valid or NULL
      */
      Terminal terminal = (Terminal) pc[1].arg;		/* get token to be matched */

#if NEW_TIMEOUT_ON_TERMINALS
      if (have_time_limit () && have_time_out ())
        { CODE *fail_addr = pc[3].code; /* Stop processing */
	  GOTO (fail_addr);
	}
#endif

      while (i_transition != NULL)
	{ /* do we have another entry? */
          TransitionExtension ext = GetTransitionExtension(i_transition);
#if TRACE_MATCH
	  abs_message ("state_has_eos_transition == %d",
		       state_has_eos_transition (curr_trellis, i_state));
	  abs_message ("ext->t == %x, t == %x", ext->terminal, terminal);
#endif /* TRACE_MATCH */
	  if (ext -> terminal != terminal)
	    /* This entry doesn't match, try next one */
	    i_transition = trel_trans_get_next(i_transition);
	  else break;
	};

      /*
	 We have either a matching terminal (re, other) or the end of the list
	 Proceed at match again
      */
      CONT (match_again);
    };

static vptr match_again ()
    { /*
	We get here by continuing from MATCH[_RE, _OTHER], or UMATCH[_RE, _OTHER]
	At this point i_transition is always valid or NULL
      */
      CODE *fail_addr = pc[3].code;
      Penalty pen_gain;
#if NEW_TIMEOUT_ON_TERMINALS
      if (have_time_limit () && have_time_out ())
        { /* Stop processing */
	  GOTO (fail_addr);
        }
#endif
      /* Debug info */
      arts_ifd.nr_match_agains++;

      if (!STACK_AVAIL(3 + 1))
        { return (stack_overflow_goto ("MATCH", fail_addr));
        }

      if (i_transition != NULL)
	{ ARG target = pc[2].arg;		/* Position of text affix */
          TransitionExtension ext;

	  /* If the penalty gain would cause us to block, take the next transition */
          ext = GetTransitionExtension(i_transition);
	  if (arts_ifd.transition_penalties_option) {
	      pen_gain = ext->penalty;
	  } else {
	      pen_gain = 0;
	  }

	  if (penalty_change_would_block (curr_penalty, pen_gain))
	    { i_transition = trel_trans_get_next(i_transition);
	      CONT (search_match);
	    };

	  /* adjust optional text affix; if absent, offset = -1 */
	  if (target >= 0)
	    { /* Do flat trellis operation */
	      Value orig = VARIABLE(target).val;
	      char *dstr = trel_trans_get_text(i_transition);	/* Matched text in trellis */
	      if (unify_texts(&VARIABLE(target).val.text_par, dstr) == 0) {
	          i_transition = trel_trans_get_next(i_transition);
		  CONT (search_match);		/* Try next transition or fail */
	      }
	      /* Here we are committed to the match; can't fail any more */
	      PUSH_VALUE (orig);		/* Push orig pointer for restore */
	      DB_RTS(abs_message("PUSH_VALUE org for match_again; after: sp=%p", sp));
	    };

	  ext->flags |= UsedBit;
	  curr_penalty += pen_gain;

	  /* Push our current program location, transition and input position */
	  sp -= 3;
	  sp[3].code = pc;
	  sp[2].input_transition = i_transition;
	  sp[1].input_state = i_state;

#if TRACE_MATCH
	  abs_message ("saved state = %p", i_state);
#endif /* TRACE_MATCH */

	  /* save trellis transition as a son in local stack frame */
	  SON_DATA(fp[NR_SONS_DONE].arg).input_transition = i_transition;
	  fp[NR_SONS_DONE].arg++;
	  i_state = trel_trans_get_dest (i_transition);
#if TRACE_MATCH
	  abs_message ("at son position %ld", fp[NR_SONS_DONE].arg);
	  abs_message ("new state = %p pos=%u", i_state, i_state ? trel_state_get_pos(i_state) : -1);
#endif /* TRACE_MATCH */

	  /*
	     Record how far we got maximally into the input;
	     this is useful for debugging grammars.
	  */
	  if (i_state && (trel_state_get_pos (i_state) > furthest_pos))
	    { furthest_pos = trel_state_get_pos (i_state);
	      save_parse_call_stack();
	    };

	  /* Continue with the instruction following the MATCH{_RE,_OTHER} */
	  NEXT;
	};

#if TRACE_MATCH
      abs_message ("MATCH{_RE,_OTHER}: no more transitions");
#endif /* TRACE_MATCH */

      /* If all transitions are tried, nothing is left, so fail */
      GOTO(fail_addr);
    }

static vptr exec_umatch ()
    { /* note: unlike with LEX_UMATCH, the Terminal is not in pc[1] */
      ARG target = pc[1].arg;
      Terminal terminal;
      TransitionExtension ext;
      CODE *saved_pc;
      int greedy;

#ifndef GEN_RTS
      i_state = sp[1].input_state;	/* restore previous input position */
#if TRACE_MATCH
      abs_message ("restored state = %p", i_state);
#endif /* TRACE_MATCH */

      /* Restore the transition and the old pc */
      i_transition = sp[2].input_transition;
      saved_pc = sp[3].code;
      sp += 3;

      /* Absent target is coded with -1 */
      if (target >= 0) {
	  DB_RTS(abs_message("POP_VALUE for exec_umatch; before sp=%p", sp));
	  POP_VALUE (VARIABLE(target).val);
      }

      ext = GetTransitionExtension(i_transition);
      terminal = ext -> terminal;
      if (arts_ifd.transition_penalties_option)
	curr_penalty -= ext->penalty;

      fp[NR_SONS_DONE].arg--;
      /* not needed since taken from stack:
       * i_transition = SON_DATA(fp[NR_SONS_DONE].arg).input_transition;
       */
      assert(SON_DATA(fp[NR_SONS_DONE].arg).input_transition == i_transition);
      /* SON_DATA(fp[NR_SONS_DONE].arg).input_transition = NULL; cleanup; not strictly needed */

      /*
       * Make $MATCH and $SKIP always greedy by doing only one
       * transition, which is always the longest
       * (because it is first; true for regexes only).
       * It should be ok for strings matching different regexes, since
       * they have different 'terminal' numbers and the code searches
       * for them by number.
       * TODO: This is too greedy, for rules such as $MATCH(...) + ...
       *       However, you can't just look in the follower set of the
       *       $MATCH to see if a + may occur, since that will also make
       *       the cases where it doesn't un-greedy, and that would be
       *       too illogical.
      greedy = 0;
       */
      greedy = (TERM_IS_MATCH(terminal) || TERM_IS_SKIP(terminal));
      if (greedy) {
	  i_transition = NULL;
      } else {
	  /* this transition already done */
	  i_transition = trel_trans_get_next(i_transition);
		
	  while (i_transition != NULL &&
		  (ext = GetTransitionExtension(i_transition)) &&
		  ext -> terminal != terminal)
	    /* This entry doesn't match, try next one */
	    i_transition = trel_trans_get_next(i_transition);

	  if (i_transition != NULL)
	    { pc = saved_pc;
	      RECALC_NEXT_PC;
	      CONT (match_again);
	    };
      }

#else /* GEN_RTS */
      gen_output_remove_token();
#endif /* GEN_RTS */

      /* Continue with the next instruction */
      NEXT;
    }

static vptr exec_match_any ()
{
    /* Loop over all entries in the trellis and match the shortest one.
     * Among the shortest ones, choose the lowest penalty.
     * There is at most only a single match, so no multiple parses
     * can result.
     * Don't match $end of sentence$ or a fact.
     */
    Transition shortest = NULL;
    TransitionExtension ext;
    Penalty penalty_gain;
    int only_lexicon_words = (int)pc[1].arg;

#if NEW_TIMEOUT_ON_TERMINALS
      if (have_time_limit () && have_time_out ())
        { /* Stop processing */
	  CODE *fail_addr = pc[3].code;
	  GOTO(fail_addr);
	}
#endif
    /* XXX doesn't know about penalties, so can't scan for the lowest */
    if (only_lexicon_words)
       shortest = trel_state_scan_word(curr_trellis, i_state);
    else
       shortest = trel_state_scan_any(curr_trellis, i_state);
    if (shortest != NULL) {
	ext = GetTransitionExtension(shortest);
	penalty_gain = ext->penalty;
    }

    if (shortest != NULL && STACK_AVAIL(3) &&
	!penalty_change_would_block (curr_penalty, penalty_gain)) {
	ARG target = pc[2].arg;		/* Position of text affix */
	char *dstr = trel_trans_get_text(shortest);/* Matched text in trellis */

	/* adjust optional text affix; if absent, offset = -1 */
	if (target >= 0)
	  { /* Do flat trellis operation before affecting stack and state */
	    Value orig = VARIABLE(target).val;
	    if (unify_texts(&VARIABLE(target).val.text_par, dstr) == 0)
	      { CODE *fail_addr = pc[3].code;
		GOTO (fail_addr);
	      }
	    /* Here we are committed to the match; can't fail anymore */
	    PUSH_VALUE (orig);		/* Push orig pointer for restore */
	  };

	/* succeed */
        curr_penalty += penalty_gain;
	/*
	 * Push our current pc, transition, input position.
	 * print_re and umatch_any need sp[2].
	 * sp[3] we do only to do the same as exec_match_re().
	 */
	sp -= 3;
	sp[3].code = pc;
	sp[2].input_transition = shortest;
	sp[1].input_state = i_state;

	/* save trellis transition as a son in local stack frame */
	SON_DATA(fp[NR_SONS_DONE].arg).input_transition = shortest;
	fp[NR_SONS_DONE].arg++;
	i_state = trel_trans_get_dest (shortest);
	ext->flags |= UsedBit;

    } else {
	/* fail */
	CODE *fail_addr = pc[3].code;
	GOTO(fail_addr);
    }

      /* Continue with the next instruction */
    NEXT;
}

static vptr exec_umatch_any ()
{
    Transition t;

    ARG target = pc[1].arg;		/* Position of text affix */

    i_state = sp[1].input_state;	/* restore previous input position */
    t = sp[2].input_transition;
    sp += 3;

    /* Absent target is coded with -1 */
    if (target >= 0)
	POP_VALUE (VARIABLE(target).val);

    fp[NR_SONS_DONE].arg--;
    /* not needed since taken from stack:
     * t = SON_DATA(fp[NR_SONS_DONE].arg).input_transition;
     */
    assert(SON_DATA(fp[NR_SONS_DONE].arg).input_transition == t);

    curr_penalty -= GetTransitionExtension(t)->penalty;	/* undo penalty */
    NEXT;
}

static vptr exec_match_sep ()
{
    ARG target = pc[1].arg;		/* Position of text affix; unused */

#if NEW_TIMEOUT_ON_TERMINALS
      if (have_time_limit () && have_time_out ())
        { /* Stop processing */
	  CODE *fail_addr = pc[2].code;
	  GOTO(fail_addr);
	}
#endif

    if (trel_state_is_at_whitespace(curr_trellis, i_state)) {
	/* success, don't need to do anything */
	(void)target; /* suppress "unused" warning */
#if 0
	if (target >= 0)
	  PUSH_VALUE (orig);		/* Push orig pointer for restore */
#endif
    } else {
	/* fail */
	CODE *fail_addr = pc[2].code;
	GOTO(fail_addr);
    }

      /* Continue with the next instruction */
    NEXT;
}

static vptr exec_umatch_sep ()
{
    ARG target = pc[1].arg;		/* Position of text affix; unused */

    (void)target; /* suppress "unused" warning */
#if 0
    /* Absent target is coded with -1 */
    if (target >= 0)
	POP_VALUE (VARIABLE(target).val);
#endif

    NEXT;
}

#ifndef GEN_RTS
static vptr exec_lex_match ()
    { int nont_nr, nont_class;
      curr_terminal = (Terminal) pc[1].arg;   /* get lexicon nont to be matched */

#if NEW_TIMEOUT_ON_TERMINALS
      if (have_time_limit () && have_time_out ())
        { /* Stop processing */
	  CODE *fail_addr = pc[2].code;
	  GOTO(fail_addr);
	}
#endif
      if (!STACK_AVAIL(2))
        { CODE *fail_addr = pc[2].code; /* cf. lex_match_again() */
	  return (stack_overflow_goto ("LEX_MATCH", fail_addr));
        }
      sp -= 2;
      sp[1].input_state = i_state;        /* save current input position */
      sp[2].code = pc;                    /* save current program counter */

      nont_nr = DECODE_NONT_NUMBER (curr_terminal);
      nont_class = (int)arts_ifd.lex_nont_nrs_table[nont_nr].ilval;
#ifdef DEBUG_RTS
      if ((nont_class < 0) || (nont_class >= arts_ifd.nr_lexicon_nonterminals))
	abs_abort ("exec_lex_match", "bad nont nr %d", nont_nr);
#endif

      i_transition = trel_state_scan_lexterminal(curr_trellis, i_state, nont_class);
      CONT (lex_match_again);
    }

/*
 * Glue input characters. Go to a different input state which indicates
 * that there is no need to skip whitespace before the next terminal.
 * This basically always succeeds, but we need to store the old state.
 * (The stack may be full; this is the only thing that can fail)
 */
static vptr exec_glue ()
{
    State new_state;

#if TRACE_MATCH
    abs_message("exec_skip_white: i_transition = %p", i_transition);
#endif
    new_state = trel_state_glue(curr_trellis, i_state);

    if (new_state != NULL && STACK_AVAIL(1)) {
	/* succeed */
	/*
	 * Push our current input position.
	 */

	sp--;
	sp[1].input_state = i_state;

	i_state = new_state;
    } else {
	/* fail */
	CODE *fail_addr = pc[1].code;
	GOTO(fail_addr);
    }

      /* Continue with the next instruction */
    NEXT;
}

static vptr exec_uglue ()
{
    i_state = sp[1].input_state;	/* restore previous input state */
    sp++;
#if TRACE_MATCH
    abs_message("exec_uglue: back to state %p", i_state);
#endif

    NEXT;
}


/*
   When continuing in lex_match_again, i_transition should point to
   the next transition to try from this i_state and pc pointing to the lex_match
*/
static vptr lex_match_again ()
    { int arity = DECODE_NONT_ARITY (curr_terminal);
      CODE *fail_addr = pc[2].code;
      Penalty pen_gain;
      TransitionExtension ext;

      /* Debug info */
      arts_ifd.nr_lex_match_agains++;

#if NEW_TIMEOUT_ON_TERMINALS
      if (have_time_limit () && have_time_out ())
        { /* Stop processing */
	  goto timeout;
	}
#endif
      /* Iterate over the transitions */
      while (i_transition != NULL)
	{ ext = GetTransitionExtension(i_transition);
	  if (arts_ifd.transition_penalties_option) pen_gain = ext->penalty;
          else pen_gain = 0;

          if (!penalty_change_would_block (curr_penalty, pen_gain))
	    { Value *val_ptr;
	      int i;

	      /* Check if we have stack space; if not, no chance to continue */
	      if (!STACK_AVAIL (arity))
	        { stack_overflow ("LEX_MATCH again");
		  break;
		};

	      /* Yep, we may continue */
	      assert(ext -> terminal == curr_terminal);
	      curr_penalty += pen_gain;

	      /* restrict formals to lexicon entry parameters */
	      val_ptr = ext->params;
	      for (i = arity - 1; i >= 0; --i)
		{ DB_RTS (abs_message ("pushing value 0x%lx", val_ptr[i].set_par));
		  PUSH_VALUE(val_ptr[i]);
		};

	      /* Save the transition as a son */
	      SON_DATA(fp[NR_SONS_DONE].arg).input_transition = i_transition;
	      fp[NR_SONS_DONE].arg++;

	      i_state = trel_trans_get_dest(i_transition);
	      ext->flags |= UsedBit;

	      /*
		 Record how far we got maximally into the input;
		 this is useful for debugging grammars.
	      */
	      if (trel_state_get_pos (i_state) > furthest_pos)
		{ furthest_pos = trel_state_get_pos (i_state);
	  	  save_parse_call_stack ();
		};

	      /* Continue with the next instruction */
	      NEXT;
	    };

	  /* We only come here when the penalty change would block */
          i_transition = trel_trans_get_next(i_transition);
	};

timeout:
      /* Restore input position and stack, then fail */
      i_state = sp[1].input_state;	/* restore input position */
      sp += 2;				/* compensate exec_lex_match() */
      GOTO (fail_addr);			/* and fail */
    }

static vptr exec_lex_umatch ()
    { int arity;

      curr_terminal = (Terminal) pc[1].arg;	    	/* get lexicon nont to be matched */
      fp[NR_SONS_DONE].arg--;
      i_transition = SON_DATA(fp[NR_SONS_DONE].arg).input_transition;

      if (arts_ifd.transition_penalties_option)
	{ /* restore penalty level */
	  curr_penalty -= GetTransitionExtension(i_transition)->penalty;
	};

      arity = DECODE_NONT_ARITY (curr_terminal);
      sp += arity;				/* remove previous results from the stack */
      pc = sp[2].code;				/* restore old pc to lex_match */
      i_transition = trel_trans_get_next(i_transition);
      RECALC_NEXT_PC;
      CONT (lex_match_again);			/* try to match next entry */
    }

#else /* GEN_RTS */

static vptr exec_lex_match ()
    { exec_illegal ();
      NEXT;
    }

static vptr exec_lex_umatch ()
    { exec_illegal ();
      NEXT;
    }
#endif /* GEN_RTS */

/*------------------------------------------------------------------------------
// Fact matching instructions:
// Instruction: INIT_KEY    (local)
//		ADD_KEY_I   (local2, local, fail address)
// 		ADD_KEY_T   (local2, local, fail address)
//		FACT_MATCH  
//		FACT_UMATCH
//		RELEASE_KEY (local)
//
// The local is used to hold the key allocated 
//----------------------------------------------------------------------------*/
#ifndef GEN_RTS
static vptr exec_init_key ()
    { int position = (int) pc[1].arg;
      DB_RTS(abs_message ("INIT_KEY at pos(%d)", position));
      VARIABLE(position).iptr = lxcn_initialize_search_key ();
      NEXT;
    }

static vptr exec_add_key_i ()
    { int ipos = (int) pc[1].arg;
      int kpos = (int) pc[2].arg;
      CODE* fail_addr = pc[3].code;
      int ival;
      int *key;

      /* We have a critical int, make sure it has a value */
      DB_RTS(abs_message ("ADD_KEY_I at pos(%d) to key(%d)", ipos, kpos));
      ival = VARIABLE(ipos).val.int_par;
      if (ival == TOP_INT) GOTO (fail_addr);

      /* Add critical int to key */
      key = VARIABLE(kpos).iptr;
      key[key[0] + 1] = ival;
      key[0]++;
      NEXT;
    }

static vptr exec_add_key_t ()
    { int tpos = (int) pc[1].arg;
      int kpos = (int) pc[2].arg;
      CODE* fail_addr = pc[3].code;
      char *tval;
      int ival;
      int *key;

      /* We have a critical text, make sure it has a value and occurs as critical text */
      DB_RTS(abs_message ("ADD_KEY_T at pos(%d) to key(%d)", tpos, kpos));
      tval = VARIABLE(tpos).val.text_par;
      if (tval == TOP_TEXT) GOTO(fail_addr);
      ival = lxcn_lookup_critical_text (arts_ifd.lexicon, tval);
      if (!ival) GOTO(fail_addr);

      /* Add critical text index to key */
      key = VARIABLE(kpos).iptr;
      key[key[0] + 1] = ival;
      key[0]++;
      NEXT;
    }

static vptr exec_fact_match ()
    { Lexicon lexicon = arts_ifd.lexicon;
      int fact_nr = (int) pc[1].arg - 1;
      int kpos = (int) pc[2].arg;
      CODE* fail_addr = pc[3].code;
      int *key = VARIABLE(kpos).iptr;
      int entry_list_nr, nr_entries;
      int *entries;
      Transition *where;
      int found;
      int new_entry;
      StateExtension sext = GetStateExtension(i_state);

      /* Do the fact lookup; if not matched, fail */
      DB_RTS(abs_message ("FACT_MATCH(%d) with key(%lu)", fact_nr + 1, kpos));
      if (!lxcn_lookup_in_fact_table (lexicon, fact_nr, key, &entry_list_nr))
        GOTO (fail_addr);
      nr_entries = lxcn_get_entries_from_nr (lexicon, entry_list_nr, &entries);

      /*
       * Put all possible transitions for this fact in the trellis,
       * at the head of the list, if not already there.
       *
       * (Advancing "where" is an optimisation in case the desired
       *  sequence of transitions is already present: it avoids rotating
       *  the whole list all the way, and back into its original order)
       *
       * Then fact_match_again() can easily traverse the list.
       *
       * It is important to re-use identical facts, so that equal 
       * PosMemos can properly be determined (they have equal sons).
       *
       * Keeping facts per-trellis-position is sufficient but maybe a 
       * somewhat less-specific position will be okay as well, 
       * enabling still more sharing, as long as lookup is efficient.
       */

      where = &sext->facts;
      for (new_entry = 0; new_entry < nr_entries; new_entry++) {
	  found = init_or_find_fact_transition (curr_trellis, entries, new_entry, i_state, where);

	  /* was: where = &(*where)->next; */
	  where = trel_trans_get_nextptr(*where);
      }

      /* Push pc, pointer to lexicon entries, and nr of entries */
      if (!STACK_AVAIL(3))
        return (stack_overflow_goto ("FACT_MATCH", fail_addr));
      sp -= 3;
      sp[3].code = pc;
      sp[2].input_transition = sext->facts;
      sp[1].ival = nr_entries;
      CONT (fact_match_again);
    };

static vptr fact_match_again ()
    { CODE *fail_addr = pc[3].code;

      /* Debug info */
      arts_ifd.nr_fact_match_agains++;

      while (1)
        { Transition trans;
	  Penalty pen_gain;
	  int arity;
	  TransitionExtension ext;	/* for fact transition */

	  /* Iterate over the entries, from back to front */
	  int new_entry = sp[1].ival - 1;
	  if (new_entry < 0) break;
	  sp[1].ival = new_entry;

	  /* We have a valid entry nr, get fact transition for it */
	  trans =  sp[2].input_transition;
	  ext = GetTransitionExtension(trans);	/* fact transition */
	  sp[2].input_transition = trel_trans_get_next(trans);
	  arity = DECODE_NONT_ARITY (ext -> terminal);

	  if (arts_ifd.transition_penalties_option) pen_gain = ext->penalty;
          else pen_gain = 0;

          if (!penalty_change_would_block (curr_penalty, pen_gain))
	    { Value *val_ptr;
	      int ix;

	      /* Check if we have stack space; if not, no chance to continue */
	      if (!STACK_AVAIL (arity))
	        { stack_overflow ("FACT_MATCH again");
		  break;
		};

	      /* Yep, we may continue */
	      curr_penalty += pen_gain;
	      ext->flags |= UsedBit;

	      /* prepare to adjust arguments to lexicon entry parameters */
	      val_ptr = ext->params;
	      for (ix = arity - 1; 0 <= ix; ix--)
		{ DB_RTS (abs_message ("pushing value 0x%llx", val_ptr[ix].set_par));
		  PUSH_VALUE(val_ptr[ix]);
		};

	      /* Save the transition as a son */
	      SON_DATA(fp[NR_SONS_DONE].arg).input_transition = trans;
	      fp[NR_SONS_DONE].arg++;

	      /* Continue with the next instruction */
	      NEXT;
	    };
	};

      sp += 3;
      GOTO (fail_addr);			/* and fail */
    };

static vptr exec_fact_umatch ()
    { TransitionExtension ext;
      Transition trans;
      int arity;
      DB_RTS(abs_message ("FACT_UMATCH"));

      fp[NR_SONS_DONE].arg--;
      trans = SON_DATA(fp[NR_SONS_DONE].arg).input_transition;
      ext = GetTransitionExtension(trans);
      arity = DECODE_NONT_ARITY (ext -> terminal);
      if (arts_ifd.transition_penalties_option)
	/* restore penalty level */
	curr_penalty -= ext->penalty;
	
      sp += arity;				/* remove previous results from the stack */

      /* Restore old pc to fact_match instruction and iterate */
      pc = sp[3].code;				/* restore old pc */
      RECALC_NEXT_PC;
      CONT (fact_match_again);			/* try to match next entry */
    };

static vptr exec_release_key ()
    { int position = (int) pc[1].arg;
      DB_RTS(abs_message ("RELEASE_KEY at pos(%d)", position));
      lxcn_release_search_key (VARIABLE(position).iptr);
      NEXT;
    };
#else /* GEN_RTS */
static vptr exec_fact_match ()
    { exec_illegal ();
      NEXT;
    }

static vptr exec_fact_umatch ()
    { exec_illegal ();
      NEXT;
    }

#endif /* GEN_RTS */

/*------------------------------------------------------------------------------
// Additional flow control instructions (penalty):
//----------------------------------------------------------------------------*/
static vptr exec_penalty ()
    { /* Pick up argument and fail address */
      Penalty penalty_gain = (Penalty) bonus_from_frequency((int)pc[1].arg,
	                                              arts_ifd.radix.penalty);
      CODE *fail_addr = pc[2].code;
      DB_RTS (abs_message ("PENALTY(%d): old penalty %d", penalty_gain, curr_penalty));

      /* Check if new penalty allows continuation */
      if (penalty_change_would_block (curr_penalty, penalty_gain))
	GOTO (fail_addr);
      curr_penalty += penalty_gain;

      /* Continue with the next instruction */
      NEXT;
    }

static vptr exec_upenalty ()
    { Penalty penalty_gain = (Penalty) pc[1].arg;

      /* Readjust curr_penalty */
      curr_penalty -= penalty_gain;
      DB_RTS(abs_message ("UPENALTY(%d): back to penalty %d", penalty_gain, curr_penalty));

      /* Continue with the next instruction */
      NEXT;
    }

/*------------------------------------------------------------------------------
// Instructions for commit semantics
//----------------------------------------------------------------------------*/
static vptr exec_if_pm_added ()
    { if (fp[SUCCESS_OFF].arg)	/* Did we make a posmemo until now? */
	{ DB_RTS (abs_message ("IF_PM_ADDED: skipping further alternatives fp=%p", fp));
	  GOTO (pc[1].code);		/* skip other alternatives */
	};

      /* Continue with the next instruction */
      DB_RTS (abs_message ("IF_PM_ADDED: trying next alternative fp=%p", fp));
      NEXT;				/* try other alts */
    }

/*------------------------------------------------------------------------------
// Instructions for the start rule:
//----------------------------------------------------------------------------*/
static vptr exec_success ()
    {
      /*
       * In non-segment mode, require EOS after a parse, except
       * when we had a time out.
       */
      if (!arts_ifd.segment_mode &&
	  !state_has_eos_transition(curr_trellis, i_state) &&
	  !(have_time_limit () && have_time_out ()))
        { CODE *fail_addr = pc[1].code;
	  DB_RTS(abs_message ("SUCCESS: no EOS found, failing after all..."));
	  GOTO (fail_addr);
	}

#ifndef GEN_RTS
      DB_RTS(abs_message ("SUCCESS: i_state = %p", i_state));
      stop_parse_time();

#else /* GEN_RTS */
      incr_nr_parses();
#endif /* GEN_RTS */

      /* Continue with the next instruction */
      NEXT;
    }

static vptr exec_usuccess ()
    { DB_RTS(abs_message ("USUCCESS: "));

#ifndef GEN_RTS
      start_parse_time();
#endif /* GEN_RTS */

      /* Continue with the next instruction */
      NEXT;
    }

static vptr exec_end ()
    { status = STOPPING;
      return (NULL);
    }

/*------------------------------------------------------------------------------
// Transduction
//----------------------------------------------------------------------------*/
static vptr exec_pass2 ()
    {
#ifndef GEN_RTS
      DB_RTS (abs_message ("PASS2: %s", (return_to_pass2)?"returning":"entering"));
      if (return_to_pass2)
	{ stop_print_time ();
	  stop_printing ();
	};

      while (!return_to_pass2 || try_update_ambi_stack ())
	{ /*
	     We have to execute PASS2 with the first parse OR
	     we have to reexecute PASS2 with another parse
	  */
	  int startpos = state_get_endpos (START_INPUT_STATE, curr_trellis);
	  int endpos = state_get_endpos (i_state, curr_trellis);
	  int length = endpos - startpos;
	  incr_nr_parses ();

	  if (current_parse_is_acceptable (curr_penalty, length))
	    { /* This parse can be added */
	      DB_RTS (abs_message ("PASS2: found parse len %d, penalty %d, accepted",
				   length, curr_penalty));
	      DB_RTS (abs_message ("START_INPUT_STATE = %p, end state = %p",
				   START_INPUT_STATE, i_state));
	      parse_result.penalty = curr_penalty;
	      parse_result.startpos = startpos;
	      parse_result.endpos = endpos;
	      current_parse_set_prio_and_add (curr_penalty, i_state, length);
	      start_print_time ();
	      start_printing ();

	      if (arts_ifd.no_output) break;
	      if (arts_ifd.identity_transduction)
	      {
		current_parse_add_nstring(length, get_input_text() + startpos);
		break;
	      } else
	      {
#define PRINT_TREE 0
#if PRINT_TREE
		posmemo_print_tree(SON_FP(0)[PMPRODPTR].pmprod);
#else

		if (!STACK_AVAIL(PASS2_FRAME_HEAD_LEN))
		    return (stack_overflow_cont ("PASS2", exec_end));

		/* Setup PASS2 frame */
		sp -= PASS2_FRAME_HEAD_LEN;
		sp[PASS2_PASS2FP].data = 0;

		/* Return to this instruction, to check for more output */
		sp[PASS2_FP].pmprod = pmptr;	/* MS: waar is die gezet */
		sp[PASS2_RET_ADDR].code = pc;
#endif
		return_to_pass2 = 1;

		/* Set pmptr to 0; PRINT_SON (0) in root print/trans tree will detect it */
		pmptr = NULL;
#if !PRINT_TREE
		pass2_fp = sp;
		next_pc = (arts_ifd.transduce_option)?pc[1].code:pc[2].code;
#endif
		NEXT; 
	      }
	    }
	  else
	    { /* force the ambi check */
	      DB_RTS (abs_message ("PASS2: found parse len %d, penalty %d, refused",
				   length, curr_penalty));
	      return_to_pass2 = 1;
	    }
	};

      /* We left the PASS2 loop, so we continue with next instruction */
      return_to_pass2 = 0;
      NEXT; 

#else /* GEN_RTS */
      gen_output_show();
      NEXT;
#endif /* GEN_RTS */
    }

/*-----------------------------------------------------------
// RETRN
//---------------------------------------------------------*/
static vptr exec_pass2_return ()
    { DATA* new_pass2_fp;
      int nr_formals;

      next_pc = pass2_fp[PASS2_RET_ADDR].code;

      /* get nr_formals (of myself, the son) */
      if (pmptr != NULL) nr_formals = pmptr -> nr_formals;
      else nr_formals = 0;

      /* Restore the posmemo of the father */
      pmptr = pass2_fp[PASS2_FP].pmprod;
      DB_RTS (
	abs_printf ("RETRN: nr_formals = %d ", nr_formals);
	if (pmptr)
	  { int nont_nr = pmptr -> nont_nr;
      	    char *name;
      	    if ((0 <= nont_nr) && (nont_nr < arts_ifd.nr_syntax_nonterminals))
      	      name = arts_ifd.nonterm_names[nont_nr].str;
      	    else name = "(outofrange)";
      	    abs_printf ("to rule %d \"%s\"", nont_nr, name);
	  }
	abs_printf("\n");
      );

      /* Restore the stack frame */
      new_pass2_fp = pass2_fp[PASS2_PASS2FP].data;
      sp += PASS2_FRAME_HEAD_LEN + nr_formals;
      pass2_fp = new_pass2_fp;

      /* print_pass2_frame(pass2_fp); */
      NEXT;
    }

/*---------------------------------------------------------------------
// PUSH_F,L
//
// Formals are taken from the stack, when being printed or passed on.
// Since they may have been restricted somewhere up in the parse tree,
// their values must be percolated down during the second pass. So the
// calling convention to print a Son is to first push the values of
// the formals of the son and then call the son's second pass code.
//-------------------------------------------------------------------*/
static vptr exec_push_f ()
    { int formal = (int) pc[1].arg;
      Value formal_value = pass2_fp[PASS2_FRAME_HEAD_LEN + formal + 1].val;
      DB_RTS (abs_message ("PUSH_F: formal == %d, value == %llx, sp = %p",
			   formal, formal_value.set_par, sp));

      if (!STACK_AVAIL(1))
	return (stack_overflow_cont("PUSH_F", exec_end));
      PUSH_VALUE(formal_value);

      /* Continue with the next instruction */
      NEXT;
    }

static vptr exec_push_l ()
    { int local = (int) pc[1].arg; 

      if (!STACK_AVAIL(1))
	return (stack_overflow_cont ("PUSH_L", exec_end));
      PUSH_VALUE (posmemo_get_local (pmptr, local));
      DB_RTS(abs_message ("PUSH_L: local == %d, value == %llx, sp = %p",
			  local, VARIABLE(local).val.set_par, sp));

      /* Continue with the next instruction */
      NEXT;
    }

/*-----------------------------------------------------------
// PRINT_SON
//---------------------------------------------------------*/
static vptr exec_print_son ()
    { int son_nr = (int) pc[1].arg;
      DB_RTS (abs_message ("PRINT_SON: curr pmptr = %p, sonnr = %d, nr_sons = %d, stack size = %d",
			   pmptr, son_nr, pmptr ? pmptr->nr_sons : 1, sp - stack));

      if (!STACK_AVAIL(PASS2_FRAME_HEAD_LEN))
	return (stack_overflow_cont("PRINT_SON", exec_end));

      /* create a new pass2 stack frame */
      sp -= PASS2_FRAME_HEAD_LEN;
      sp[PASS2_PASS2FP].data = pass2_fp;

      /* to remember our current posmemo pointer */
      sp[PASS2_FP].pmprod = pmptr;
      sp[PASS2_RET_ADDR].code = next_pc;
      pass2_fp = sp;

      /* pick up posmemo for either the son or the root rule */
      if (pmptr != NULL) pmptr = posmemo_get_son (pmptr,son_nr);
      else pmptr = SON_FP(0)[PMPRODPTR].pmprod;	/* Root rule */

      /* Check for ambiguity: we may pass this point more than once */
      if (pmptr -> prime != NULL)
	{ PosMemo sibling;
	  assert (pmptr -> prime == pmptr);
	  if (try_lookup_sibling_in_ambi_stack (pmptr, &sibling)) pmptr = sibling;
	  else push_ambi_stack (pmptr -> prime);
	};

      /* continue with the transduction of the son */
      DB_RTS(abs_message ("Using %p as PMPROD to PRINT SON (rule %d \"%s\")", pmptr, pmptr->nont_nr,
			  arts_ifd.nonterm_names[pmptr->nont_nr].str));
      next_pc = posmemo_get_pass2 (pmptr); 
      NEXT;
    }

#define INDENT	2

/*-----------------------------------------------------------
// Print the begin and end of an alternative
// Only used in tree output
//
// if labelled bracket output wanted, print open and closed
// curly braces; otherwise use indentation
//---------------------------------------------------------*/
static vptr exec_print_abeg ()
    { char* alt_text = pc[1].val.text_par;
      DB_RTS(abs_message ("PRINT_ABEG: alt_text = %s", alt_text));

      if (arts_ifd.label_bracket)
	{ if (alt_text && arts_ifd.parsing_stats_option)
	    current_parse_printf ("{%s:", alt_text);
	  else current_parse_add_char('{');
	}
      else
	{ if (alt_text && arts_ifd.parsing_stats_option)
	  {
#if DEBUG_POSMEMO
	    current_parse_printf (" <alt %s> PM:%p penalty=%d\n", alt_text, pmptr, pmptr->penalty);
#else
	    current_parse_printf (" <alt %s>\n", alt_text);
#endif
	  }
	  else current_parse_add_char('\n');

	  indent += INDENT;
	};

      /* Continue with next instruction */
      NEXT;
    }

/*----------------------------------------------------------
// PRINT_AEND: print end alternative.
// Only used in tree output
//--------------------------------------------------------*/
static vptr exec_print_aend ()
    { DB_RTS (abs_message ("PRINT_AEND: "));

      if (arts_ifd.label_bracket) current_parse_add_char('}');
      else indent -= INDENT;

      /* Continue with next instruction */
      NEXT;
    }

//---------------------------------------------------------
// PRINT_PBEG, PRINT_PSEP and PRINT_PEND serve to print
// the affix list with a production i.e. they print a
// '(', ',' and ')' (a 'pack')
// Only used in tree output
//-------------------------------------------------------*/
static vptr exec_print_pbeg ()
    { DB_RTS (abs_message ("PRINT_PBEG: "));

      current_parse_add_char('(');
	
      /* Continue with next instruction */
      NEXT;
    }

static vptr exec_print_psep ()
    { DB_RTS (abs_message ("PRINT_PSEP: "));

      current_parse_add_char(',');
      if (!arts_ifd.label_bracket) current_parse_add_char(' ');
	
      /* Continue with next instruction */
      NEXT;
    }

static vptr exec_print_pend ()
    { DB_RTS (abs_message ("PRINT_PEND: "));
      current_parse_add_char (')');
	
      /* Continue with next instruction */
      NEXT;
    }

/*--------------------------------------------------
// PRINT_VAL: Print a text value
//------------------------------------------------*/
static vptr exec_print_val ()
    { char *text = pc[1].val.text_par;
      DB_RTS (abs_message ("PRINT_PVAL: text = '%s'", text));

      current_parse_add_string(text);

      /* Continue with next instruction */
      NEXT;
    }

/*-----------------------------------------------------------
// PRINT_TERM: print the terminal belonging to a transition
//---------------------------------------------------------*/
static vptr exec_print_term_trans ()
    { /* MS: Hier zou je een typecheck kunnen doen... */
      Transition p_transition = posmemo_get_son_transition (pmptr,
							    (int)pc[1].arg);
      char *text = trel_trans_get_text(p_transition);
      DB_RTS(abs_message ("PRINT_TERM_TRANS: text = \"%s\"", text));

      if (arts_ifd.transduce_option)
	current_parse_add_lexeme (text);

      /* Continue with next instruction */
      NEXT;
    }

static vptr exec_print_term_tree ()
    { /* MS: Hier zou je een typecheck kunnen doen... */
      Transition p_transition = posmemo_get_son_transition (pmptr,
							    (int)pc[1].arg);
      char *text = trel_trans_get_text(p_transition);
      TransitionExtension ext;
      LexemeType lextype;

      DB_RTS(abs_message ("PRINT_TERM_TREE: text = \"%s\"", text));

      if (!arts_ifd.label_bracket) current_parse_add_space(indent);

      ext = GetTransitionExtension(p_transition);
      lextype = ext->lextype;

#if DEBUG_POSMEMO
      current_parse_printf ("TR:%p ", p_transition);
#endif
      current_parse_add_char ('\"');
      if (lextype == Suffix || lextype == Infix) current_parse_add_char ('-');
      current_parse_add_lexeme (text);
      if (lextype == Prefix || lextype == Infix) current_parse_add_char ('-');
      current_parse_add_char ('\"');

      if (!arts_ifd.label_bracket && arts_ifd.transition_penalties_option)
	{ current_parse_add_string (" [");
	  current_parse_add_int(GetTransitionExtension(p_transition)->penalty);
	  current_parse_add_string ("]");
	}

      if (!arts_ifd.label_bracket) current_parse_add_char ('\n');

      /* Continue with next instruction */
      NEXT;
    }

/*-------------------------------------------------------------
// PRINT_LEX: print the text belonging to a lexicon transition
//-----------------------------------------------------------*/
static vptr exec_print_lex_trans ()
    { /* Pick corresponding transition from saved one in stack frame */
      /* MS: Hier zou je een typecheck kunnen doen... */
      Transition p_transition = posmemo_get_son_transition(pmptr,(int)pc[1].arg);
      char *text = trel_trans_get_text(p_transition);
      DB_RTS(abs_message ("PRINT_LEX_TRANS: text = \"%s\"", text));

      current_parse_add_lexeme (text);

      /* Continue with next instruction */
      NEXT;
    }

static vptr exec_print_lex_tree ()
    { /* Pick corresponding transition from saved one in stack frame */
      /* MS: Hier zou je een typecheck kunnen doen... */
      Transition p_transition = posmemo_get_son_transition(pmptr,(int)pc[1].arg);
      char *text = trel_trans_get_text(p_transition);
      DB_RTS(abs_message ("PRINT_LEX_TREE: text = \"%s\"", text));

	{ if (!arts_ifd.label_bracket)
	    current_parse_add_space(1);

	  current_parse_add_char('\"');
	  current_parse_add_lexeme(text);
	  current_parse_add_char('\"');

	  if (!arts_ifd.label_bracket && arts_ifd.transition_penalties_option)
	    { current_parse_add_string (" [");
	      current_parse_add_int(GetTransitionExtension(p_transition)->penalty);
	      current_parse_add_string ("]");
	    }

	  if (!arts_ifd.label_bracket)
	    current_parse_add_char('\n');
	};

      /* Continue with next instruction */
      NEXT;
    }

/*-------------------------------------------------------------
// PRINT_FACT
//-----------------------------------------------------------*/
static vptr exec_print_fact ()
    { Transition p_transition = posmemo_get_son_transition(pmptr,(int)pc[1].arg);
      DB_RTS(abs_message ("PRINT_FACT"));

      if (!arts_ifd.label_bracket && !arts_ifd.transduce_option)
	{ if (arts_ifd.transition_penalties_option)
	    { current_parse_add_string (" [");
	      current_parse_add_int(GetTransitionExtension(p_transition)->penalty);
	      current_parse_add_string ("]");
	    };
#if DEBUG_POSMEMO
	  current_parse_printf("{%p}", GetTransitionExtension(p_transition));
#endif
	  current_parse_add_string ("\n");
    	}

      /* Continue with next instruction */
      NEXT;
    }

/*-------------------------------------------------------------
// PRINT_RE: print the text belonging to a regular expression
// PRINT_OTHER: print the text not matched by anything
// PRINT_ANY: print the text of any transition
// PRINT_WORD: same as PRINT_ANY, since MATCH_WORD is also MATCH_ANY
// For the time being we use the same code.
//-----------------------------------------------------------*/
static vptr exec_print_re_trans ()
    { /* Pick up argument */
      int var = (int) pc[1].arg;
      Transition p_transition;
      char *text;

      /* Pick transition from saved location in frame */
      p_transition = posmemo_get_son_transition (pmptr,var);

      text = trel_trans_get_text(p_transition);
      DB_RTS(abs_message ("PRINT_RE(or OTHER or ANY): text = %s", text));

      current_parse_add_string (text);

      /* Continue with next instruction */
      NEXT;
    }

static vptr exec_print_re_tree ()
    { /* Pick up argument */
      int var = (int) pc[1].arg;
      Transition p_transition;
      char *text;

      /* Pick transition from saved location in frame */
      p_transition = posmemo_get_son_transition (pmptr,var);

      text = trel_trans_get_text(p_transition);
      DB_RTS(abs_message ("PRINT_RE(or OTHER or ANY): text = %s", text));

      if (!arts_ifd.label_bracket)
	current_parse_add_space (indent);

      current_parse_add_char ('\"');
      current_parse_add_string (text);

      if (!arts_ifd.label_bracket)
	{ current_parse_add_string ("\"");
	  if (arts_ifd.transition_penalties_option)
	    { current_parse_add_string (" [");
	      current_parse_add_int(GetTransitionExtension(p_transition)->penalty);
	      current_parse_add_string ("]");
	    }
	  current_parse_add_string ("\n");
	}
      else current_parse_add_char ('\"');

      /* Continue with next instruction */
      NEXT;
    }

/*---------------------------------------------------------------
// PRINT_POS: print a position
//-------------------------------------------------------------*/
static vptr exec_print_pos_trans ()
    { /* Pick up argument */
      int local = (int) pc[1].arg;
      Value val;

      /* If transducing, print the saved position */
      DB_RTS (abs_message ("PRINT_POS: loc = %d", local));
      /* Pick up text from posmemo and print it */
      val = posmemo_get_variable (pmptr, local);
      if (val.text_par == NULL)
        abs_abort ("PRINT_POS", "chaos awaits the pointer in posmemo");
      else current_parse_add_string (val.text_par);

      /* Continue with next instruction */
      NEXT;
    }

static vptr exec_print_pos_tree ()
    { opcode instr = GET_INSTR;
      current_parse_add_space (indent);
      if (instr == opc_print_pos_tree) 
	current_parse_add_string ("$POS\n");
      else if (instr == opc_print_line_tree) 
	current_parse_add_string ("$LINE\n");
      else
	current_parse_add_string ("$POSLINE\n");

      /* Continue with next instruction */
      NEXT;
    }

static vptr exec_print_autopos_trans ()
    { /* Pick up argument */
      int spos, epos;

      if (arts_ifd.print_autopos) {
	  spos = trel_state_get_pos(pmptr -> this_state);
	  epos = trel_state_get_pos(pmptr -> next_state);
	  current_parse_printf("%d-%d:", spos, epos);
      }

      /* Continue with next instruction */
      NEXT;
    }

/*---------------------------------------------------------------
// PRINT_PEN: print a penalty
//-------------------------------------------------------------*/
static vptr exec_print_pen ()
    { /* Pick up penalty */
      Penalty pen = (Penalty) pc[1].arg;
      DB_RTS (abs_message ("PRINT_PEN: pen = %d", pen));

      if (!arts_ifd.label_bracket)
	{ current_parse_add_space(indent);
	  if (pen == 1) current_parse_add_string ("$PENALTY");
	  else
	    { current_parse_add_string ("$PENALTY(");
	      current_parse_add_int (pen);
	      current_parse_add_char (')');
	    };
	  current_parse_add_char ('\n');
	};

      /* Continue with next instruction */
      NEXT;
    }

/*---------------------------------------------------------------
// PRINT_PEN_INST: print the penalty incurred by this rule
//-------------------------------------------------------------*/
static vptr exec_print_pen_inst ()
    { /* Pick up penalty */
      Penalty pen = (pmptr) ? pmptr->penalty : -999;
      DB_RTS (abs_message ("PRINT_PEN_INST: pen = %d", pen));

      if (!arts_ifd.label_bracket)
	current_parse_add_int (pen);

      /* Continue with next instruction */
      NEXT;
    }


/*---------------------------------------------------------------
// PRINT_SEP: print a separator
//-------------------------------------------------------------*/
static vptr exec_print_sep ()
    { DB_RTS (abs_message ("PRINT_SEP"));
      may_be_current_parse_add_space ();

      /* Continue with next instruction */
      NEXT;
    }

/*---------------------------------------------------------------
// PRINT_INDENT: print an indentation
//-------------------------------------------------------------*/
static vptr exec_print_indent ()
    { DB_RTS (abs_message ("PRINT_INDENT"));
      current_parse_add_space (indent);

      /* Continue with next instruction */
      NEXT;
    }

/*---------------------------------------------------------------
// PRINT_NL: print a newline character
//-------------------------------------------------------------*/
static vptr exec_print_nl ()
    { DB_RTS (abs_message ("PRINT_NL"));
      current_parse_add_char ('\n');

      /* Continue with next instruction */
      NEXT;
    }

/*---------------------------------------------------------------
// PRINT_N: print nonterminal name
//-------------------------------------------------------------*/
static vptr exec_print_n ()
    { /* Pick up argument */
      char* text = pc[1].val.text_par;
      DB_RTS(abs_message ("PRINT_N: nont = %s", text));

      if (!arts_ifd.label_bracket)
	current_parse_add_space (indent);
      current_parse_add_string (text);

      /* Continue with next instruction */
      NEXT;
    }

static vptr exec_print_if ()
    { /* Pick up argument */
      int formal = (int) pc[1].arg;
      Value val = pass2_fp[PASS2_FRAME_HEAD_LEN + formal + 1].val;

      DB_RTS(abs_message ("PRINT_IF: int formal %d has value %d", formal, val.int_par));
      print_integer_affix (val.int_par, 1);

      /* Continue with next instruction */
      NEXT;
    }

static vptr exec_print_il ()
    { /* Pick up argument */
      int local = (int) pc[1].arg;
      Value val = posmemo_get_local (pmptr,local);

      DB_RTS(abs_message ("PRINT_IL: int local %d has value %d", local, val.int_par));
      print_integer_affix (val.int_par, 1);

      /* Continue with next instruction */
      NEXT;
    }

static vptr exec_print_sf ()
    { /* Pick up arguments */
      int formal = (int) pc[1].arg;
      int domain = (int) pc[2].arg;
      Value val = pass2_fp[PASS2_FRAME_HEAD_LEN + formal + 1].val;

      DB_RTS (abs_message ("PRINT_SF: set formal = 0x%llx", val.set_par));
      print_set_affix (val.set_par, domain, 1);

      /* Continue with next instruction */
      NEXT;
    }

static vptr exec_print_sl ()
    { /* Pick up arguments */
      int local =  (int) pc[1].arg;
      int domain = (int) pc[2].arg;
      Value val = posmemo_get_local(pmptr,local);

      DB_RTS (abs_message ("PRINT_SL: set local = 0x%llx", val.set_par));
      print_set_affix (val.set_par, domain, 1);

      /* Continue with next instruction */
      NEXT;
    }

static vptr exec_print_tf_trans ()
    { int formal = (int) pc[1].arg;
      Value val = pass2_fp[PASS2_FRAME_HEAD_LEN + formal + 1].val;

      DB_RTS(abs_message ("PRINT_TF: text formal = '%s'", val.text_par));
      /* quote the string in tree transduction */
      print_text_affix(val.text_par, 1, 0);

      /* Continue with next instruction */
      NEXT;
    }

static vptr exec_print_tf_tree ()
    { int formal = (int) pc[1].arg;
      Value val = pass2_fp[PASS2_FRAME_HEAD_LEN + formal + 1].val;

      DB_RTS(abs_message ("PRINT_TF: text formal = '%s'", val.text_par));
      /* quote the string in tree transduction */
      print_text_affix(val.text_par, 1, 1);

      /* Continue with next instruction */
      NEXT;
    }

static vptr exec_print_tl_trans ()
    { int local = (int) pc[1].arg;
      Value val = posmemo_get_local(pmptr,local);

      DB_RTS(abs_message ("PRINT_TL_TRANS: text local = '%s'", val.text_par));
      /* quote the string in tree transduction */
      print_text_affix(val.text_par, 1, 0);

      /* Continue with next instruction */
      NEXT;
    }

static vptr exec_print_tl_tree ()
    { int local = (int) pc[1].arg;
      Value val = posmemo_get_local(pmptr,local);

      DB_RTS(abs_message ("PRINT_TL_TREE: text local = '%s'", val.text_par));
      /* quote the string in tree transduction */
      print_text_affix(val.text_par, 1, 1);

      /* Continue with next instruction */
      NEXT;
    }

static vptr exec_print_vf ()
    { exec_illegal ();
      NEXT;
    }

static vptr exec_print_vl ()
    { exec_illegal ();
      NEXT;
    }


/*------------------------------------------------------------------------------
// Generative grammar instructions:
//----------------------------------------------------------------------------*/
static vptr exec_choice ()
#ifndef GEN_RTS
    { exec_illegal ();
      NEXT;
    }
#else /* GEN_RTS */
#define NR_TRIES_PER_ALTGROUP 1
    { int choice_nr = (int) pc[1].arg;
      int nr_alts = (int) pc[2].arg;

      logexec(choice);
      opc_count(choice);
      ALT_DEPTH_CHECK;

      if (get_nr_parses())
	{ /* a sentence is generated, we can stop */
	  choice_admin_remove (alt_depth, choice_nr);
	  chosen_freq = -2;
	}
      else if (choice_admin_get_nr_tries (alt_depth, choice_nr) >=
      	 (NR_TRIES_PER_ALTGROUP * nr_alts))
	{ /* nr_alts == nr of tries, which all failed, so fail */
	  choice_admin_remove(alt_depth, choice_nr);
	  chosen_freq = -3;
	}
      else
	{ /* failed to generate a sentence, but not everything is tried */
	  /* TODO: check STACK_AVAIL(1)... */
	  PUSH_ADDR(pc);
	  chosen_freq = rand() % nr_alts;
/*        chosen_freq = choice_admin_get_nr_tries(alt_depth, choice_nr); */
	  choice_admin_incr_nr_tries(alt_depth, choice_nr);
	};
    }
#endif /* GEN_RTS */

static vptr exec_uchoice ()
#ifndef GEN_RTS
    { exec_illegal ();
      NEXT;
    }
#else /* GEN_RTS */
    { int choice_nr = (int) pc[1].arg;
      if (((chosen_freq == -2) || (chosen_freq == -3)) &&
	   !choice_admin_get_nr_tries(alt_depth, choice_nr))
	{ /* Step to next instruction */
	  NEXT;
	}
      else
	{ CODE* choice_addr;
	  POP_ADDR(choice_addr);
	  assert(choice_addr);
	  GOTO (choice_addr);
	};
    }
#endif /* GEN_RTS */

static vptr exec_alt_marker ()
#ifndef GEN_RTS
    { exec_illegal ();
      NEXT;
    }
#else /* GEN_RTS */
    { int my_freq = 1;
      CODE* next_alt = pc[1].code;

      ALT_DEPTH_CHECK;
      if ((chosen_freq < my_freq) && (chosen_freq >= 0))
	{ /* we're gonna generate for this alternative */
	  chosen_freq = -1;
	  ALT_DEPTH_INCREASE;
	}
      else if (chosen_freq >= 0)
	{ /* this alternative is not chosen, goto next one */
	  chosen_freq -= my_freq;
	}

      /* else we're done, so skip this alternative */
      NEXT;
    }
#endif /* GEN_RTS */

static vptr exec_ualt_marker ()
#ifndef GEN_RTS
    { exec_illegal ();
      NEXT;
    }
#else /* GEN_RTS */
    { ALT_DEPTH_CHECK;
      ALT_DEPTH_DECREASE;
      NEXT;
    }
#endif /* GEN_RTS */

/*-------------------------------------------------------------------------
// Instruction: CREATE_ROOT (nr formals, nr locals, nr sons)
// Description: Setup an initial stack frame 
// Note that the frame head of this stack frame is never to be used
//-----------------------------------------------------------------------*/
static vptr exec_create_root ()
    { int nr_formals = (int) pc[1].arg;
      int nr_locals = (int) pc[2].arg;
      int nr_sons = (int) pc[3].arg;
      int i;

      sp[0].data = fp;	        /* store old fp */
      fp = sp;

      sp -= nr_formals + nr_locals + nr_sons + VAR_OFFSET;
      NR_FORMALS = nr_formals;
      NR_LOCALS = nr_locals;
      NR_SONS = nr_sons;
      DB_RTS (NONT_NR = 0);	/* for in FAIL. */

      START_INPUT_STATE = i_state;
      fp[PM_PLAYED].pmprod = NULL;
      fp[SUCCESS_OFF].arg = 0;
      fp[NR_SONS_DONE].arg = 0;
      fp[PENORIG].penalty = curr_penalty;

      /*
	 Initialise the locals and sons to 0, since otherwise equal_posmemo()
	 may look at uninitialised values sometimes (and fail to detect
	 identity). This happens because not every alternative uses (sets) the
	 full set of locals and sons so some may remain uninitialised.
      */
      for (i = 0; i < nr_locals + nr_sons; i++)
	VARIABLE(nr_formals + i).arg = 0;

      DB_RTS(abs_message ("CREATE_ROOT: start_input_state = %p, sp = %p, stack bottom = %p",
			  i_state, sp, stack));
      fp[PMPRODPTR].pmprod = NULL;

      /* Continue with next instruction */
      NEXT;
    }

static vptr exec_call_root ()
    { int rule_nr = (int) pc[1].arg;
      CODE *jmp_point = pc[2].code;
      CODE *fail_addr = pc[3].code;

      DB_RTS(abs_message ("CALL_ROOT: current time is %.3fs.", get_parse_time()));

      /* Set up precall stack frame */
      sp -= FRAME_HEAD_LEN;
      sp[FAIL_ADDR_OFF].code = fail_addr;
      sp[SUCC_ADDR_OFF].code = next_pc;
      push_parse_call (rule_nr);

      SON_FP(fp[NR_SONS_DONE].arg) = sp;
      fp[NR_SONS_DONE].arg++;
      DB_RTS(abs_message ("Root call: Starting parse, penalty = %d", curr_penalty));

      /* Jump to next instruction */
      GOTO (jmp_point);
    }

/*-------------------------------------------------------------------------
// Tracing instructions
//-----------------------------------------------------------------------*/
/* trace integer global: old trace, from agfl-coder -T option */
static vptr exec_trace_ig ()
    { int trace_int = *(pc[1].iptr);

      abs_message ("Trace: '%d' at pos %d", trace_int, trel_state_get_pos  (i_state));
      /* Continue with next instruction */
      NEXT;
    }

/* trace text global: $TRACE("text") */
static vptr exec_trace_tg ()
    { char *trace_text = pc[1].str;

      abs_message ("Trace: '%s' at pos %d", trace_text, trel_state_get_pos (i_state));
      /* Continue with next instruction */
      NEXT;
    }

/* trace integer local: $TRACE(INT) */
static vptr exec_trace_il ()
    { int local = (int) pc[1].arg;
      DATA data = VARIABLE(local);

      abs_printf ("Trace: '");
      print_integer_affix(data.val.int_par, 0);
      abs_message ("' at pos %d", trel_state_get_pos (i_state));
      NEXT;
    }

/* trace set local: $TRACE(SOMESET) */
static vptr exec_trace_sl ()
    { State input_state = i_state;
      int local =  (int) pc[1].arg;
      int domain = (int) pc[2].arg;
      DATA data = VARIABLE(local);

      abs_printf ("Trace: '");
      print_set_affix (data.val.set_par, domain, 0);
      abs_message ("' at pos %d", trel_state_get_pos (input_state));

      /* Continue with next instruction */
      NEXT;
    }

/* trace text local: $TRACE(TEXT) */
static vptr exec_trace_tl ()
    { int local = (int) pc[1].arg;
      DATA data = VARIABLE(local);

      abs_printf ("Trace: ");
      print_text_affix(data.val.text_par, 0, 1);
      abs_message (" at pos %d", trel_state_get_pos (i_state));
      /* Continue with next instruction */
      NEXT;
    }

/* trace integer from stack */
static vptr exec_trace_is ()
    { int local = (int) pc[1].arg;
      DATA data = sp[local];

      abs_printf ("Trace: '");
      print_integer_affix(data.val.int_par, 0);
      abs_message ("' at pos %d", trel_state_get_pos (i_state));
      NEXT;
    }

/* trace set from stack */
static vptr exec_trace_ss ()
    { State input_state = i_state;
      int local =  (int) pc[1].arg;
      int domain = (int) pc[2].arg;
      DATA data = sp[local];

      abs_printf ("Trace: '");
      print_set_affix (data.val.set_par, domain, 0);
      abs_message ("' at pos %d", trel_state_get_pos (input_state));

      /* Continue with next instruction */
      NEXT;
    }

/* trace text from stack */
static vptr exec_trace_ts ()
    { int local = (int) pc[1].arg;
      DATA data = sp[local];

      abs_printf ("Trace: ");
      print_text_affix(data.val.text_par, 0, 1);
      abs_message (" at pos %d", trel_state_get_pos (i_state));
      /* Continue with next instruction */
      NEXT;
    }

static vptr exec_position ()
    { int position = (int) pc[1].arg;
      
      if (arts_ifd.alternatives_profile_table == NULL)
        { int ix;
	  arts_ifd.alternatives_profile_table = (int64 *)
	    abs_calloc (arts_ifd.nr_positions, sizeof (int64), "exec_position");
          for (ix = 0; ix < arts_ifd.nr_positions; ix++)
            arts_ifd.alternatives_profile_table[ix] = 0;
	}

      arts_ifd.alternatives_profile_table[position]++;

      /* Continue with next instruction */
      NEXT;
    }

/*------------------------------------------------------------------------------
// Hybrid instructions
//----------------------------------------------------------------------------*/
static int hybrid_code_is_null_code (CODE *pc)
{ if (pc == NULL) return (1);
  return (GET_INSTR == opc_done);
}

static int hybrid_code_is_a_single_son (CODE *pc, hyb_anchor *son)
{ PosMemo son_pmptr;
  DATA *son_fp;
  int nr;
  if (GET_INSTR != opc_use_son) return (0);
  pc++;
  nr = (int) pc -> arg;
  pc++;
  if (GET_INSTR != opc_done) return (0);
  son_fp = SON_FP(nr);
  son_pmptr = son_fp[PMPRODPTR].pmprod;
  *son = son_pmptr -> hybrid;
  return (1);
}

static vptr exec_hybrid_op ()
    { int ival = (int) pc[1].arg;
      arts_hyb_enter_operator (curr_anchor, ival);
      NEXT;
    }

static vptr exec_relator ()
    { int val = (int) pc[1].arg;
      char *rel_nm = pc[2].str;
      arts_hyb_enter_relator (curr_anchor, (val)?HYB_ENTER_REL_RIGHT:HYB_ENTER_REL_LEFT, rel_nm);
      NEXT;
    }

static vptr exec_enter_text ()
    { char *text = pc[1].str;
      arts_hyb_enter_string (curr_anchor, text);
      NEXT;
    }

static vptr exec_enter_trans ()
    { Transition p_transition = SON_DATA(pc[1].arg).input_transition;
      char *text = trel_trans_get_text(p_transition);
      arts_hyb_enter_string (curr_anchor, text);
      NEXT;
    }

static vptr exec_use_son ()
    { int arg = (int) pc[1].arg;
      DATA *son_fp = SON_FP(arg);
      PosMemo son_pmptr = son_fp[PMPRODPTR].pmprod;
      if (son_pmptr -> hybrid != NULL)
        arts_hyb_enter_son (curr_anchor, son_pmptr -> hybrid);
      NEXT;
    }

static vptr exec_use_lex ()
    { Transition p_transition = SON_DATA(pc[1].arg).input_transition;
      char *text = trel_trans_get_text(p_transition);
      arts_hyb_enter_string (curr_anchor, text);

      NEXT;
    }

static vptr exec_enter_int_affix ()
    { int arg = (int) pc[1].arg;
      Value val = VARIABLE(arg).val;
      int ival = val.int_par;
      char conv_buf[32];
      char *text;
      if (ival == TOP_INT) text = "INT";
      else
        { sprintf (conv_buf, "%d", ival);
          text = abs_mm_new_string (conv_buf, "exec_enter_int_affix"); /* memory leak */
	};
      arts_hyb_enter_string (curr_anchor, text);
      NEXT;
    }

static vptr exec_enter_text_affix ()
    { int arg = (int) pc[1].arg;
      Value val = VARIABLE(arg).val;
      char *text = val.text_par;
      if (text == TOP_TEXT) text = "TEXT";
      arts_hyb_enter_string (curr_anchor, text);
      NEXT;
    }

static vptr exec_enter_set_affix ()
    { Value val = VARIABLE(pc[1].arg).val;
      ARG domain = pc[2].arg;
      char *text = conv_mm_set_affix (val.set_par, (int)domain); /* memory leak */
      /* if (val.set_par & (val.set_par-1)) { abs_message("# Warning: more than one value in set: %s", text); } */
      arts_hyb_enter_string (curr_anchor, text);
      NEXT;
    }

static vptr exec_done ()
    { next_pc = sp[1].code;
      sp += 1;
      arts_hyb_optimize (curr_anchor);
#ifdef DEBUG_HYBRID
      abs_message ("Built hybrid code for %s", arts_ifd.nonterm_names[NONT_NR].str);
      arts_hyb_dump (curr_anchor);
      abs_message ("----");
#endif
      NEXT;
    }

/*------------------------------------------------------------------------------
// Illegal instructions
//----------------------------------------------------------------------------*/
static void exec_illegal ()
    { abs_bug ("agfl_interpret", "rts: illegal instruction");
    };

/* Write parse results in the .prd file */
void write_profile_table()
{ int position;

  if (arts_ifd.alternatives_profile_table == NULL)
	return;
  for (position = 0; position < arts_ifd.nr_positions; position++)
    profile_printf ("%d:%ld\n", position, arts_ifd.alternatives_profile_table[position]);
}


/*------------------------------------------------------------------------------
// Dumping information
//----------------------------------------------------------------------------*/
static char *get_nont_name (int nont_nr)
{ if ((0 <= nont_nr) && (nont_nr < arts_ifd.nr_syntax_nonterminals))
    return (arts_ifd.nonterm_names[nont_nr].str);
  else return ("<outofrange>");
}

static void print_pass1_frame(DATA *fp, int recursive)
{
    DATA *old_fp;
    long nont_nr, nr_formals, nr_locals, nr_sons;
    int i, ix;

    abs_printf("Pass 1 frame at 0x%08lx:\n", fp);
    if (!fp)
	return;

    nont_nr    = (long)fp[NONT_NR_OFF].arg;
    old_fp     =       fp[OLD_FP].data;
    nr_formals = (long)fp[NR_FORMALS_OFF].arg;
    nr_locals  = (long)fp[NR_LOCALS_OFF].arg;
    nr_sons    = (long)fp[NR_SONS_OFF].arg;

    abs_printf("   SUCC_ADDR: 0x%08lx\n", fp[SUCC_ADDR_OFF].code);
    abs_printf("   FAIL_ADDR: 0x%08lx\n", fp[FAIL_ADDR_OFF].code);
    abs_printf("      OLD_FP: 0x%08lx\n", old_fp);
    abs_printf("     NONT_NR: %ld \"%s\"\n", nont_nr, get_nont_name(nont_nr));
    abs_printf("  NR_FORMALS: %ld\n", nr_formals);
    abs_printf("   NR_LOCALS: %ld\n", nr_locals);
    abs_printf("     NR_SONS: %ld\n", nr_sons);
    abs_printf("   PM_PLAYED: 0x%08lx\n", fp[PM_PLAYED].pmprod);
    abs_printf("      ISTATE: 0x%08lx\n", fp[ISTATE].input_state);
    abs_printf("   PMPRODPTR: 0x%08lx\n", fp[PMPRODPTR].pmprod);
    abs_printf("     SUCCESS: %ld\n", (long)fp[SUCCESS_OFF].arg);
    abs_printf("NR_SONS_DONE: %ld\n", (long)fp[NR_SONS_DONE].arg);
    abs_printf("     PENORIG: %ld\n", (long)fp[PENORIG].penalty);

    abs_printf("Formals:   ");
    ix = -VAR_OFFSET;
    for (i = 0; i < nr_formals; i++, ix--)
	abs_printf(" %l8x", fp[ix].arg);
    abs_printf("\nLocals:    ");
    for (i = 0; i < nr_locals; i++, ix--)
	abs_printf(" %l8x", fp[ix].arg);
    abs_printf("\nSons:      ");
    for (i = 0; i < nr_sons; i++, ix--) {
	char c = pointsIntoStack(fp[ix].code) ? 'S' : 'T';
	abs_printf(" %c:%l8x", c, fp[ix].arg);
    }
    abs_printf("\n");

    if (recursive && old_fp && old_fp != fp) {
	abs_printf("going to parent:\n");
	print_pass1_frame(old_fp, recursive - 1);
      }
}

static void print_pass2_frame (DATA *fp)
{
    PosMemo pmptr;

    abs_message ("Pass 2 frame at 0x%p:", fp);
    if (!fp) return;
    abs_message ("      PASS2_PASS2FP: 0x%p (parent's pass2_fp)", fp[PASS2_PASS2FP].data);
    pmptr = fp[PASS2_FP].pmprod;
    abs_printf ("      PASS2_FP: 0x%p (parent's posmemo)", pmptr);
    if (pmptr)
      { int nont_nr = pmptr -> nont_nr;
      	char *name = get_nont_name (nont_nr);
      	abs_printf (" rule %d \"%s\"", nont_nr, name);
      }
    abs_message ("\n      PASS2_RET_ADDR: 0x%p", fp[PASS2_RET_ADDR].code);
}

/*------------------------------------------------------------------------------
// Parsing interface
//----------------------------------------------------------------------------*/
static void reset_parser (Trellis trellis)
{
    reset_parse_result ();
    parse_results_destroy ();
    parse_results_init ();
#ifdef GEN_RTS
    parse_results_start_new ();
#endif /* GEN_RTS */
}

State parse_trellis (Trellis trellis, State start_node)
{ reset_parser (trellis);

  if (have_time_limit ()) {
#ifdef DEBUG_TIME_OUT
	abs_message ("setting timeout to %ld secs", arts_ifd.max_parsetime);
#endif /* DEBUG_TIME_OUT */

	set_time_limit();
    }

#ifdef DEBUG_NEGMEMO
  if (neg_memo_option || directors_option)
    print_negmemo_table(trellis, 0);
#endif /* DEBUG_NEGMEMO */

  return (agfl_interpret (trellis, start_node));
}

/*------------------------------------------------------------------------------
// Prelude and postlude of module
//----------------------------------------------------------------------------*/
void arts_init_parser ()
{ init_ambi_stack ();
  posmemo_init ();

  if (arts_ifd.counters_option)
    initialize_counters ();
}

void arts_end_parser ()
{ if (arts_ifd.free_mem_option)
    { posmemo_done ();
      finish_ambi_stack ();
    };

  if (arts_ifd.counters_option)
    print_counters ();

  /* write_profile_table (); */
}
