/*
   File: arts_hybrid.c
   Support for hybrid parsing

   Copyright 2010 Radboud University of Nijmegen
 
   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.
 
   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU Library General Public License for more details.
 
   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.

   CVS ID: "$Id$
*/

/* standard includes */
#ifdef HAVE_CONFIG_H
#include <config.h>
#endif /* HAVE_CONFIG_H */
#include <stdio.h>
#include <string.h>
#include <ctype.h>

/* libabase includes */
#include <abase_error.h>
#include <abase_memalloc.h>
#include <abase_mm_alloc.h>

/* liblexicon includes */
#include <lxcn_lexicon.h>
#include <lxcn_fact_table.h>

/* libarts includes */
#include "arts_ds.h"
#include "arts_io.h"
#include "arts_hybrid.h"
#include "arts_lexer.h"
#include "arts_minitdb.h"
#include "arts_escape.h"

#define DEBUG_HYBRID	0
#define LOOK_AT_HYBRID	1
#if DEBUG_HYBRID
#define DB_FL(x)	x
#define DB_HASH(x)	x
#else
#define DB_FL(x)
#define DB_HASH(x)
#endif /* DEBUG_HYBRID */

int statistic_hybrid_tried;
int statistic_hybrid_whitelist_success;
int statistic_hybrid_blacklist_success;
int statistic_hybrid_fail;
int statistic_term_tried;
int statistic_triples_tried;

struct modifier_list_rec;
struct term_cache_rec;

static void detach_modifier_list (struct modifier_list_rec *rec);
static void detach_hyb_instr (hyb_code instr);
static void detach_term_cache (struct term_cache_rec *cache);

#if LOOK_AT_HYBRID
typedef uintptr_t hash_t;
typedef struct {
    hash_t work;
    hash_t head;
    hash_t modifiers;
    hash_t relator;
    int num_separators;
    int num_modifiers;
} hash_ctx;
static void hash_hyb_code(hash_ctx *hash, hyb_code a);
#endif /* LOOK_AT_HYBRID */
static void log_hyb_code (hyb_code a);

#define detach_term(t)	detach_string_list(t)

static int hybrid_inited;
static int wildcard_idx;

void hybrid_init ()
{ if (!hybrid_inited)
    { wildcard_idx = lxcn_lookup_critical_text (arts_ifd.lexicon, "*");
#if DEBUG_HYBRID
      abs_message ("wildcard_idx = %d", wildcard_idx);
#endif /* DEBUG_HYBRID */
      hybrid_inited = 1;
    }
}

#if LOOK_AT_HYBRID
static void hash_enter(hash_ctx *hash, hash_t value);

static void hash_init(hash_ctx *hash)
{ hash->work = 0;
  hash->head = 0;
  hash->modifiers = 0;
  hash->relator = 0;
  hash->num_separators = 0;
  hash->num_modifiers = 0;
}

/*
 * The hash must be independent of the order of relator/modifiers and
 * alternations.
 * When the first relator is encountered, the hash value so far must
 * be that of the head, and is set aside.
 * The same when an alternation occurs; the collected value must be
 * folded into either head or modifier-hash.
 * For each relator + modifier code sequences, their respective hash
 * values are summed (not xored, this may result in hash values with
 * lots of zeros for similar code sequences, and even numbers of identical
 * sequences cancel out).
 * At the end, the hash of the final part (also relator+modifiers) is
 * collected as well, then the aggregate of the rel/mod hashes is
 * finally entered into the result.
 */
static void hash_reset_for_relator(hash_ctx *hash)
{ if (hash->num_modifiers == 0)
  { hash->head += hash->work;
  } else
  { hash->modifiers += hash->work;
  }
  DB_HASH(printf(" %llx <REL ", (long long)hash->work);)
  hash->work = 0;
  hash->num_modifiers++;
  hash->num_separators = 0;
}

/*
 * Save the hash of the relator, before the separator,
 * so that when we come to the next alternative we can restore it.
 * (That seems easier than to keep it separate; then we'd need to
 * incorporate it separately and explicitly later)
 */
static void hash_record_separator(hash_ctx *hash)
{ if (hash->num_modifiers > 0)
  { if (hash->num_separators == 0)
    { hash->relator = hash->work;
    }
    DB_HASH(printf("%llx , ", (long long)hash->work);)
    hash->num_separators = 1;
  }
}

static void hash_reset_for_alternation(hash_ctx *hash)
{ if (hash->num_modifiers == 0)
  { hash->head += hash->work;
  } else
  { hash->modifiers += hash->work;
  }
  DB_HASH(printf("%llx | %llx ", (long long)hash->work, (long long)hash->relator);)
  hash->work = hash->relator;
  /* keep num_separators as it was; no need to save relator hash value again */
}

static hash_t hash_collect(hash_ctx *hash)
{
  hash_reset_for_alternation(hash);
  if (hash->num_modifiers > 0)
  { /*hash_reset_for_relator(hash); * work all done in rese...alternation() */
    hash->work = hash->head;
    hash_enter(hash, hash->modifiers);
    hash->modifiers = 0;
    hash->num_modifiers = 0;
  } else
  { hash->work = hash->head;
  }
  DB_HASH(printf(" ==> %llx\n", (long long)hash->work);)
  return hash->work;
}

static void hash_enter(hash_ctx *hash, hash_t value)
{
    hash_t high_bits = hash->work >> (CHAR_BIT*sizeof(hash_t) - 7);
    hash->work = (hash->work << 7) ^ high_bits ^ value;
    DB_HASH(printf("+ %llx -> %llx ", (long long)value, (long long)hash->work);)
}

static void hash_string(hash_ctx *hash, char *a)
{ while (*a) hash_enter(hash, (unsigned char)*a++);
}

#endif /* LOOK_AT_HYBRID */

static void log_string(char *a)
{ char *escd = arts_dupstr_escaped(a);
  abs_printf("%s", escd);
  abs_free(escd, "log_string");
}

/*
 * [A-Z]+[a-z_]*
 */
static void check_relator_wellformedness(char *relator)
{ char *r = relator;

  /* if (*r == '\0') abs_message("Warning: malformed empty relator ''"); */
  while (isupper(*r)) r++;
  while (islower(*r) || *r == '_') r++;
  if (*r != '\0') abs_message("# Warning: malformed relator '%s'", relator);
}

char *skip_pos_prefixes (char *text)
{ char *ptr;
  char *last = text;
  for (ptr = text; *ptr; ptr++)
    if (*ptr == ':') last = ptr + 1;
  return (last);
}

static string_list string_list_freelist;
DB_FL(static int string_list_freelist_count;)
DB_FL(static int string_list_alloc_count;)

static void detach_string_list (string_list t)
{ if (t != null_term)
    { detach_string_list (t -> next);
      t -> next = string_list_freelist;
      string_list_freelist = t;
    };
}

static string_list new_string_list ()
{ string_list t;
  if (string_list_freelist != null_term)
    { t = string_list_freelist;
      string_list_freelist = t -> next;
      DB_FL(string_list_freelist_count++;)
    }
  else
    { t = abs_mm_malloc (sizeof (struct string_list_rec), "new_string_list");
      DB_FL(string_list_alloc_count++;)
    }
  return (t);
}

#if LOOK_AT_HYBRID
static void hash_string_list(hash_ctx *hash, string_list a)
{ if (a == NULL) return;
  hash_string (hash, a -> text);
  hash_string_list(hash, a -> next);
}
#endif /* LOOK_AT_HYBRID */

static void log_string_list(string_list a)
{ if (a == NULL) return;
  log_string (a -> text);
  log_string_list(a -> next);
}

/*
 * Stripping pos_prefixes here is too early; it works if somebody
 * just writes
 * 	... / "V:" bla
 * but it fails for things like
 * 	$POS + word + $POS / $POS.1 + "-" + $POS.2 + ":" + word
 * since it works on each part separately.
 * Therefore the stripping has to be done later.
 *
 * An optimisation could be to detect a colon here, and if present,
 * throw away any parts before it, keeping the colon for later (in case
 * it gets appended to something else). (Just detach_string_list() them)
 */
static void add_to_term (term *head, char *text)
{ string_list new_part = new_string_list();
  new_part -> text = text;
  new_part -> index = 0;
  new_part -> next = NULL;
  if ((*head) == null_term) *head = new_part;
  else
    { string_list ptr = *head;
      while (ptr -> next != NULL) ptr = ptr -> next;
      ptr -> next = new_part;
    };
}

static void add_separator_to_term (term *head)
{ string_list sep_part;
  string_list ptr;
  char *text;
  if ((*head) == null_term) return;	/* No leading separators */

  /* Locate last part */
  ptr = *head;
  while (ptr -> next != NULL) ptr = ptr -> next;

  /* Locate last character of last part */
  for (text = ptr -> text; *text; text++) ;
  if (text[-1] == ' ') return;

  /* Definitely a separator to add */
  sep_part = new_string_list();
  sep_part -> text = " ";
  sep_part -> index = 0;
  sep_part -> next = NULL;
  ptr -> next = sep_part;
}

#if LOOK_AT_HYBRID
static void print_term_user (int idx, term t)
{ term ptr;
  if (idx == wildcard_idx)
    { abs_printf("*");
    }
  else
    { abs_printf("\"");
      for (ptr = t; ptr != null_term; ptr = ptr -> next)
        { abs_printf ("%s", ptr -> text);
        };
      abs_printf("\"");
    }
}
#endif /* LOOK_AT_HYBRID */

#if DEBUG_HYBRID
static void print_term (term t)
{ term ptr;
  int first = 1;
  for (ptr = t; ptr != null_term; ptr = ptr -> next)
    { if (first) first = 0;
      else abs_printf (" + ");
      abs_printf ("'%s'", ptr -> text);
    };
}
#endif /* DEBUG_HYBRID */

void print_critical_term (int idx, term t)
{ char buffer[MAXINPUT];
  char *dptr = buffer;
  string_list ptr;

  if (idx == wildcard_idx)
    { abs_printf("*");
      return;
    }
  
  if (t == null_term) return;
  if (t -> next == NULL)
    { abs_printf("\"%s\"", skip_pos_prefixes(t -> text));
      return;
    }
  for (ptr = t; ptr != NULL; ptr = ptr -> next)
    { char *sptr;
      for (sptr = ptr -> text; *sptr; sptr++, dptr++)
        { *dptr = *sptr;
	  if (*dptr == ':') dptr = buffer - 1;	/* eqv. skip_pos_prefixes(buffer) */
	}
    };

  /* Trailing spaces are silently eaten */
  while (dptr > buffer && dptr[-1] == ' ') dptr--;
  *dptr = '\0';

  abs_printf("\"%s\"", skip_pos_prefixes(buffer));
}

static int lookup_critical_term (term t)
{ Lexicon lexicon = arts_ifd.lexicon;
  char buffer[MAXINPUT];
  char *dptr = buffer;
  string_list ptr;

  if (t == null_term) return (0);
  if (t -> index != 0) return (t -> index);
  statistic_term_tried++;
   /*
    * Small optimization; with the scanning for spaces, it
    * becomes less useful, if at all...
    */
  if (t -> next == NULL &&
	  !arts_ifd.triple_translate &&
	  strchr(t -> text, ' ') == NULL)
    { t -> index = lxcn_lookup_critical_text (lexicon, skip_pos_prefixes (t -> text));
      return (t -> index);
    };

  for (ptr = t; ptr != NULL; ptr = ptr -> next)
    { char *sptr;
      for (sptr = ptr -> text; *sptr; sptr++, dptr++)
        { *dptr = *sptr;
	  if (*dptr == ':') dptr = buffer - 1;	/* eqv. skip_pos_prefixes(buffer) */
	}
    };

  /* Trailing spaces are silently eaten */
  while (dptr > buffer && dptr[-1] == ' ') dptr--;
  *dptr = '\0';

  /* Optionally map everything to lowercase
   * (really: use alphabet file)
   */
  if (arts_ifd.triple_translate) {
      dptr = buffer;
      while (*dptr) {
	  *dptr = lxcn_translate(*dptr);
	  dptr++;
      }
  }

  t -> index = lxcn_lookup_critical_text (lexicon, buffer);
  return (t -> index);
}

static term_list term_list_freelist;
DB_FL(static int term_list_freelist_count;)
DB_FL(static int term_list_alloc_count;)

static term_list new_term_list ()
{ term_list list;
  if (term_list_freelist != null_term_list)
  { list = term_list_freelist;
    term_list_freelist = list -> rest;
    DB_FL(term_list_freelist_count++;)
  } else {
    list = (term_list) abs_mm_malloc (sizeof (struct term_list_rec), "new_term_list");
    DB_FL(term_list_alloc_count++;)
  }
  list -> refcount = 1;
  return list;
}

static void attach_term_list (struct term_list_rec *rec)
{ if (rec != null_term_list)
    rec -> refcount++;
}

static void detach_term_list (struct term_list_rec *rec)
{ if (rec != null_term_list)
  { rec -> refcount--;
    if (rec -> refcount <= 0)
    { detach_term (rec -> first);
      detach_term_list (rec -> rest);
      rec -> rest = term_list_freelist;
      term_list_freelist = rec;
    }
  }
}

#if LOOK_AT_HYBRID
/* The hash must be independent of the order of the terms */
static void hash_term_list(hash_ctx *hash, term_list a)
{ hash_t term_list_hash = 0;

  if (a == NULL) return;

  while (a != NULL)
  { hash_ctx localhash;
    hash_init(&localhash);
    hash_string_list(&localhash, a -> first);
    term_list_hash += localhash.work;
    a = a -> rest;
  }
  hash_enter(hash, 't');
  hash_enter(hash, term_list_hash);
}
#endif /* LOOK_AT_HYBRID */

static void log_term_list(term_list a)
{ if (a == NULL) return;
  abs_printf("\"");
  log_string_list(a -> first);
  while (a -> rest != NULL)
  { abs_printf("\"|\"");
    a = a -> rest;
    log_string_list(a -> first);
  }
  abs_printf("\"");
}

static int term_list_length(term_list a)
{ int length = 0;
  while (a != NULL)
  { length++;
    a = a -> rest;
  }
  return length;
}

/*
   Term lists for heads and modifiers may persist because of caching.
   We therefore have to allocate them from the recoverable malloc space
*/
static void add_string_to_terms (term_list *elems, char *text)
{ if (*elems == null_term_list)
    { *elems = new_term_list ();
      (*elems) -> first = null_term;
      (*elems) -> rest = null_term_list;
    }
  add_to_term (&(*elems) -> first, text);
}

static void add_separator_to_terms (term_list *elems)
{ if (*elems == null_term_list) return;
  add_separator_to_term (&(*elems) -> first);
}

static void push_terms (term_list *elems)
{ term_list nt;
  if (*elems == null_term_list)
    { /* Strange situation: [ | */
      *elems = new_term_list ();
      (*elems) -> first = null_term;
      (*elems) -> rest = null_term_list;
    };
  nt = new_term_list ();
  nt -> first = null_term;
  nt -> rest = *elems;	/* move counted reference */
  *elems = nt;
}

#if DEBUG_HYBRID
static void print_term_list (term_list tl)
{ term_list tl_ptr;
  int first = 1;
  for (tl_ptr = tl; tl_ptr != null_term_list; tl_ptr = tl_ptr -> rest)
    { if (first) first = 0;
      else abs_printf (" | ");
      print_term (tl_ptr -> first);
    };
}
#endif /* DEBUG_HYBRID */

/* Define the structure of the instructions for the hybrid machine */
struct hyb_anchor_rec
{ /* May be more info */
  hyb_code first;
  union {		/* last is used only while building the linked list. */
    hyb_code last;	/* Deliberately destroy the pointer to prevent later use */
#if LOOK_AT_HYBRID
    hash_t hash;	/* Once the hash is calculated, the hybrid code must not change in any way */
#endif /* LOOK_AT_HYBRID */
  };
  PosMemo owner;
};

typedef struct term_cache_rec *term_cache;
struct term_cache_rec
{ Penalty gain; 	/* or Penalty vanwege ds beter een int */
  term_list heads;
  hyb_code skip;
  int rec;
};
#define empty_cache ((term_cache) NULL)
#if LOOK_AT_HYBRID
static void hash_term_cache(hash_ctx *hash, term_cache a);
#endif /* LOOK_AT_HYBRID */
static void log_term_cache(term_cache a, int opcode);

struct hyb_instr_rec
{ int opcode;
  union { char *arg;
          term_cache cache;
          hyb_anchor son;
        } uni;
  hyb_code next;
  hyb_code prev;
};

hyb_anchor hyb_anchor_freelist;
DB_FL(static int hyb_anchor_freelist_count;)
DB_FL(static int hyb_anchor_alloc_count;)

hyb_anchor arts_create_hybrid_anchor ()
{ hyb_anchor new;
  if (hyb_anchor_freelist != NULL)
  { new = hyb_anchor_freelist;
    hyb_anchor_freelist = (hyb_anchor) (void *)(new -> first);
    DB_FL(hyb_anchor_freelist_count++;)
  } else
  { new = (hyb_anchor) abs_mm_malloc (sizeof (struct hyb_anchor_rec),
					       "arts_create_hyb_anchor");
    DB_FL(hyb_anchor_alloc_count++;)
  }
  new -> first = hyb_null_code;
  new -> last = hyb_null_code;
  new -> owner = NULL;
  return (new);
}

/*
 * A hyb_anchor is linked back to the PosMemo which created it and owns
 * it (the backlink is established after running it, in exec_make_pm()).
 *
 * A hyb_anchor can be attached to multiple PosMemo-s, due to the
 * single_son optimisation (hybrid_code_is_a_single_son()).
 *
 * The owning PosMemo remains the deepest nested son, which will
 * also be the last of these PosMemo-s to be deleted.
 */
void detach_hyb_anchor (hyb_anchor anchor, PosMemo owner)
{ if (anchor != NULL)
  { if ((anchor -> owner == owner))
    { detach_hyb_instr (anchor -> first);
      anchor -> owner = NULL;
      anchor -> first = (void *)hyb_anchor_freelist;
      hyb_anchor_freelist = anchor;
    }
  }
}

#if LOOK_AT_HYBRID
int eqv_hyb_anchor (hyb_anchor a, hyb_anchor b)
{ if (a == b) return 1;
  if (a == NULL || b == NULL) return 0;
  if (a -> hash != b -> hash) return 0;
  return 1;	/* Trust the hash not to have collisions */
}

hash_t hash_hyb_anchor(hyb_anchor a)
{ hash_ctx hash;
  if (a == NULL) return 0;
  hash_init(&hash);
  hash_hyb_code(&hash, a -> first);
  a -> hash = hash_collect(&hash);
  return a -> hash;
}

#endif /* LOOK_AT_HYBRID */

/*
 * hash_hyb_anchor() should hash the things that are shown by
 * log_hyb_anchor(), so that the latter gives a good representation of
 * the things that are taken into account by the former.
 */

void log_hyb_anchor(hyb_anchor a)
{ if (a == NULL) return;
#if DEBUG_HYBRID
  if (arts_ifd.hash_production) abs_printf("H:%llx ", (long long)a -> hash);
#endif /* DEBUG_HYBRID */
  log_hyb_code(a -> first);
}
static void add_hybrid_instruction (hyb_anchor anchor, hyb_code instr)
{ if (anchor -> first == hyb_null_code)
    { /* First instruction for this anchor */
      anchor -> first = instr;
      anchor -> last = instr;
    }
  else
    { /* Append after last instruction */
      anchor -> last -> next = instr;
      instr -> prev = anchor -> last;
      anchor -> last = instr;
    };
}

static hyb_code hyb_instr_freelist;
DB_FL(static int hyb_instr_freelist_count;)
DB_FL(static int hyb_instr_alloc_count;)

static hyb_code get_hyb_instr ()
{ hyb_code instr;
  if (hyb_instr_freelist != NULL)
  { instr = hyb_instr_freelist;
    hyb_instr_freelist = hyb_instr_freelist -> next;
    DB_FL(hyb_instr_freelist_count++;)
  } else
  { instr = (hyb_code) abs_mm_malloc (sizeof (struct hyb_instr_rec),
                                             "get_hyb_instr");
    DB_FL(hyb_instr_alloc_count++;)
  }
  return instr;
}

static void detach_hyb_instr (hyb_code instr)
{ if (instr != NULL)
    { detach_hyb_instr (instr -> next);
      switch (instr -> opcode)
      { case HYB_USE_SON:
	  /*
	   * Don't detach_hyb_anchor (instr -> uni.son, NULL);
	   * because the son belongs to another PosMemo.
	   */
	  break;
        case HYB_SQR_OPEN:
        case HYB_SQR_CLOSE:	/* should be NULL anyway */
        case HYB_CURL_OPEN:
        case HYB_CURL_CLOSE:	/* should be NULL anyway */
	  detach_term_cache (instr -> uni.cache);
	  break;
	case HYB_ENTER_STRING:
	case HYB_ENTER_REL_LEFT:
	case HYB_ENTER_REL_RIGHT:
	  /*
	   * The string pointed to is unmanaged, usually from the acode
	   * or the trellis.
	   */
	case HYB_BAR:
	case HYB_SEPARATOR:
	  /* These two have no argument at all */
	  break;
	default:
	  abs_message("detach_hyb_instr: opcode %d\n", instr -> opcode);
	  return;
      }
      instr -> opcode = 100 + instr -> opcode;
      instr -> next = hyb_instr_freelist;
      hyb_instr_freelist = instr;
    }
}

#if LOOK_AT_HYBRID

static void hash_hyb_use_son(hash_ctx *hash, hyb_anchor a)
{ if (a == NULL) return;
  hash_hyb_code(hash, a -> first);
}

static void hash_hyb_code(hash_ctx *hash, hyb_code a)
{ if (a == NULL) return;
  
  switch (a -> opcode)
  { case HYB_USE_SON:
        hash_hyb_use_son(hash, a -> uni.son);
	hash_hyb_code(hash, a -> next);
	break;
    case HYB_SQR_OPEN:
    case HYB_CURL_OPEN:
	hash_term_cache(hash, a -> uni.cache);
	hash_hyb_code(hash, a -> uni.cache -> skip);
	/*hash_hyb_code (hash, a -> next);*/
	break;
    case HYB_SQR_CLOSE:		/* should be NULL anyway */
    case HYB_CURL_CLOSE:	/* should be NULL anyway */
	hash_hyb_code (hash, a -> next);
	break;
    case HYB_ENTER_STRING:
	hash_string(hash, a -> uni.arg);
	hash_hyb_code(hash, a -> next);
	break;
    case HYB_ENTER_REL_LEFT:
    case HYB_ENTER_REL_RIGHT:
	hash_reset_for_relator(hash);
	hash_enter(hash, a -> opcode);
	hash_string(hash, a -> uni.arg);
	hash_hyb_code(hash, a -> next);
	break;
    case HYB_BAR:
	hash_reset_for_alternation(hash);
	hash_hyb_code (hash, a -> next);
	break;
    case HYB_SEPARATOR:		/* TODO: consecutive separators must be idempotent */
	hash_enter(hash, a -> opcode);
	hash_record_separator(hash);
	hash_hyb_code (hash, a -> next);
	break;
    default:
	abs_message("hash_hyb_instr: opcode %d\n", a -> opcode);
	hash_hyb_code (hash, a -> next);
	break;
  }
}
#endif /* LOOK_AT_HYBRID */

static void log_hyb_use_son(hyb_anchor a)
{ PosMemo pa;

  if (a == NULL) return;

  pa = a -> owner;

  /*abs_printf("USE_SON_%d(", pa -> nont_nr);*/
  log_hyb_anchor(a);
  /*abs_printf(" %d)", pa -> nont_nr);*/
}

static void log_hyb_code(hyb_code a)
{ if (a == NULL) return;
  
  switch (a -> opcode)
  { case HYB_USE_SON:
        log_hyb_use_son(a -> uni.son);
	log_hyb_code(a -> next);
	break;
    case HYB_SQR_OPEN:
    case HYB_CURL_OPEN:
	if (a -> uni.cache && a -> uni.cache -> skip)
	{ log_term_cache(a -> uni.cache, a -> opcode);
#if DEBUG_HYBRID
	  abs_printf("=%c", a->opcode == HYB_SQR_OPEN? '[' : '{');
	  log_hyb_code(a -> next);
#else
	  log_hyb_code(a -> uni.cache -> skip -> next);
#endif /* DEBUG_HYBRID */
	} else
	{ abs_printf("%c", a->opcode == HYB_SQR_OPEN? '[' : '{');
	}
	break;
    case HYB_SQR_CLOSE:	
    case HYB_CURL_CLOSE:
	abs_printf("%c", a->opcode == HYB_SQR_CLOSE? ']' : '}');
	log_hyb_code (a -> next);
	break;
    case HYB_ENTER_STRING:
	log_string(a->uni.arg);
	log_hyb_code(a -> next);
	break;
    case HYB_ENTER_REL_LEFT:
    case HYB_ENTER_REL_RIGHT:
	abs_printf("%c", a->opcode == HYB_ENTER_REL_LEFT? '<' : '>');
	log_string(a -> uni.arg);
	log_hyb_code(a -> next);
	break;
    case HYB_BAR:
	abs_printf("|");
	log_hyb_code (a -> next);
	break;
    case HYB_SEPARATOR:
	abs_printf(" ");
	log_hyb_code (a -> next);
	break;
    default:
	abs_printf("log_hyb_instr: opcode %d\n", a -> opcode);
	log_hyb_code (a -> next);
	break;
  }
}

static term_cache term_cache_freelist;
DB_FL(static int term_cache_freelist_count;)
DB_FL(static int term_cache_alloc_count;)

static term_cache new_term_cache ()
{ term_cache cache;
  if (term_cache_freelist != NULL)
  { cache = term_cache_freelist;
    term_cache_freelist = (term_cache) cache -> heads;
    DB_FL(term_cache_freelist_count++;)
  } else
  { cache = (term_cache) abs_mm_malloc (sizeof (struct term_cache_rec), "new_term_cache");
    DB_FL(term_cache_alloc_count++;)
  }
  return (cache);
}

static void detach_term_cache (term_cache cache)
{ if (cache != NULL)
    { detach_term_list (cache -> heads);
      /*
       * Don't detach_hyb_instr (cache -> skip);
       * because the skip points to a further instruction in the stream
       * up to which this cache represents the results (typically
       * the HYB_*_CLOSE instr)
       */
      cache -> heads = (void *)term_cache_freelist;
      term_cache_freelist = cache;
    }
}

#if LOOK_AT_HYBRID
/*
 * For each of the cached strings, add STRING BAR to the hash.
 * pretend that the instructions HYB_ENTER_STRING <string>
 * separated by HYB_BAR are present.
 */
static void hash_term_cache(hash_ctx *hash, term_cache a)
{ term_list tl;
  if (a == NULL) return;
  
  tl = a -> heads;
  while (tl)
  { term_list next = tl -> rest;
    /* emulate the hash of HYB_ENTER_STRING, hash_string() */
    hash_string_list(hash, tl->first);
    tl = next;
    /* emulate the hash of HYB_BAR */
    if (tl)
    { hash_reset_for_alternation(hash);
    }
  }
}
#endif /* LOOK_AT_HYBRID */

static void log_term_cache(term_cache a, int opcode)
{ int length;
  char open, close;
  char *quote;

  if (a == NULL) return;
  if (opcode == HYB_CURL_OPEN) {
    open  = '{';
    close = '}';
  } else {
    open  = '[';
    close = ']';
  }

  abs_printf("%c", open);
#if DEBUG_HYBRID
  abs_printf("$PENALTY(%d)", a -> gain);
#endif /* DEBUG_HYBRID */

  length = term_list_length(a -> heads);

  if (length > 0) {
    if (length > 1) quote = "";
    else            quote = "\"";

    abs_printf("%s", quote);
    log_string_list(a -> heads -> first);
    abs_printf("%s", quote);
  }
  abs_printf("%c ", close);
}

void arts_hyb_enter_operator (hyb_anchor anchor, int ival)
{ hyb_code instr = get_hyb_instr ();
  instr -> opcode = ival;
  instr -> uni.arg = NULL;
  instr -> next = hyb_null_code;
  instr -> prev = hyb_null_code;
  add_hybrid_instruction (anchor, instr);
}

void arts_hyb_enter_relator (hyb_anchor anchor, int ival, char *rel_nm)
{ hyb_code instr = get_hyb_instr ();
  instr -> opcode = ival;
  instr -> uni.arg = rel_nm;
  instr -> next = hyb_null_code;
  instr -> prev = hyb_null_code;
  add_hybrid_instruction (anchor, instr);
}

void arts_hyb_enter_string (hyb_anchor anchor, char *s)
{ hyb_code instr = get_hyb_instr ();
  instr -> opcode = HYB_ENTER_STRING;
  instr -> uni.arg = s;
  instr -> next = hyb_null_code;
  instr -> prev = hyb_null_code;
  add_hybrid_instruction (anchor, instr);
}

void arts_hyb_enter_son (hyb_anchor anchor, hyb_anchor son_anchor)
{ hyb_code instr = get_hyb_instr ();
  instr -> opcode = HYB_USE_SON;
  instr -> uni.son = son_anchor;
  instr -> next = hyb_null_code;
  instr -> prev = hyb_null_code;
  add_hybrid_instruction (anchor, instr);
}

static void arts_dump_hybrid_instruction (hyb_code instr)
{ switch (instr -> opcode)
    { case HYB_SQR_OPEN: 	abs_message ("["); break;
      case HYB_SQR_CLOSE: 	abs_message ("]"); break;
      case HYB_CURL_OPEN: 	abs_message ("{"); break;
      case HYB_CURL_CLOSE: 	abs_message ("}"); break;
      case HYB_BAR: 		abs_message ("|"); break;
      case HYB_SEPARATOR:	abs_message (", "); break;
      case HYB_ENTER_STRING: 	abs_message ("Text: '%s'", instr -> uni.arg); break;
      case HYB_ENTER_REL_LEFT:  abs_message ("<%s", instr -> uni.arg); break;
      case HYB_ENTER_REL_RIGHT: abs_message (">%s", instr -> uni.arg); break;
      case HYB_USE_SON: 	abs_message ("<SON>"); break;
      default: break;
    };
}

void arts_hyb_dump (hyb_anchor anchor)
{ hyb_code instr;
  if (anchor == hyb_null_anchor)
    abs_message ("NULL");

  for (instr = anchor -> first; instr != hyb_null_code; instr = instr -> next)
    arts_dump_hybrid_instruction (instr);
}

/*
   After constructing the hybrid code, an optimization run is passed over
   an anchor to optimize parts of the construction and most notably allow
   relators with attached affixes
*/
static void try_optimize_enter_relator (hyb_code instr, hyb_anchor anchor)
{ hyb_code last_ex;
  char *new_relator;
  for (;;) /* in case more HYB_ENTER_STRING instances follow */
    { last_ex = instr -> next;
      if (last_ex == hyb_null_code) return;
      if (last_ex -> opcode != HYB_ENTER_STRING) return;
      new_relator = abs_mm_new_fmtd_string ("try_optimize_enter_relator", "%s%s",
				        instr -> uni.arg, last_ex -> uni.arg);
      instr -> uni.arg = new_relator;	/* memory leak */
      instr -> next = last_ex -> next;
      if (last_ex -> next == hyb_null_code) anchor -> last = instr;
      else last_ex -> next -> prev = instr;
    }
}

static void try_optimize_instr (hyb_code instr, hyb_anchor anchor)
{ switch (instr -> opcode)
    { case HYB_ENTER_REL_LEFT:
      case HYB_ENTER_REL_RIGHT:
	  try_optimize_enter_relator (instr, anchor);
	  check_relator_wellformedness (instr -> uni.arg);
	  break;
      default: break;
    };
}

void arts_hyb_optimize (hyb_anchor anchor)
{ hyb_code instr;

  for (instr = anchor -> first; instr != hyb_null_code; instr = instr -> next)
    try_optimize_instr (instr, anchor);
}

typedef struct eval_stack_rec *eval_stack;

#define STATE_HEADS 0
#define STATE_MODS 1
struct eval_stack_rec
{ hyb_code open_ptr;
  int state;
  term_list heads;
  modifier_list mods;
  eval_stack father;
};

#define empty_eval_stack   ((eval_stack) NULL)

/* Als werkt eval_stack_recs op een freelist bijhouden: ze zijn recyclebaar */
static eval_stack eval_stack_freelist;
DB_FL(static int eval_stack_freelist_count;)
DB_FL(static int eval_stack_alloc_count;)

static void push_eval_stack (eval_stack *father, hyb_code open_ptr)
{ eval_stack new_stack;
  if (eval_stack_freelist != empty_eval_stack)
  { new_stack = eval_stack_freelist;
    eval_stack_freelist = eval_stack_freelist -> father;
    DB_FL(eval_stack_freelist_count++;)
  } else 
  { new_stack = abs_mm_malloc (sizeof (struct eval_stack_rec), "push_eval_stack");
    DB_FL(eval_stack_alloc_count++;)
  }
  new_stack -> open_ptr = open_ptr;
  new_stack -> state = STATE_HEADS;
  new_stack -> heads = null_term_list;
  new_stack -> mods = null_modifier_list;
  new_stack -> father = *father;
  *father = new_stack;
}

static void detach_eval_stack_rec (struct eval_stack_rec *rec)
{ detach_modifier_list (rec -> mods);
  rec -> mods = null_modifier_list;
  detach_term_list (rec -> heads);
  rec -> heads = null_term_list;
  rec -> father = eval_stack_freelist;
  eval_stack_freelist = rec;
}

static void try_push_term_list (eval_stack stack)
{ if (stack == empty_eval_stack) return;
  if (stack -> state == STATE_HEADS) push_terms (&stack -> heads);
  else push_terms (&stack -> mods -> elems);
}

static void try_push_string (eval_stack stack, char *text)
{ if (stack == empty_eval_stack) return;
  if (stack -> state == STATE_HEADS) add_string_to_terms (&stack -> heads, text);
  else add_string_to_terms (&stack -> mods -> elems, text);
}

static void try_push_separator (eval_stack stack)
{ if (stack == empty_eval_stack) return;
  if (stack -> state == STATE_HEADS) add_separator_to_terms (&stack -> heads);
  else add_separator_to_terms (&stack -> mods -> elems);
}

static modifier_list modifier_list_freelist;
DB_FL(static int modifier_list_freelist_count;)
DB_FL(static int modifier_list_alloc_count;)

static void push_relator (eval_stack stack, int dir, char *rel)
{ modifier_list nml;
  if (stack == empty_eval_stack) return;
  if (stack -> state == STATE_HEADS) stack -> state = STATE_MODS;
  if (modifier_list_freelist != null_modifier_list)
  { nml = modifier_list_freelist;
    modifier_list_freelist = nml -> next;
    DB_FL(modifier_list_freelist_count++;)
  } else
  { nml = (modifier_list) abs_mm_malloc (sizeof (struct modifier_list_rec), "push_relator");
    DB_FL(modifier_list_alloc_count++;)
  }
  nml -> dir = dir;
  nml -> relator = rel;
  nml -> elems = null_term_list;
  nml -> next = stack -> mods;
  stack -> mods = nml;
  /*
   * Thanks to try_optimize_enter_relator(), the concatenation of
   * relator and affix(es) was already performed, so we could
   * already check there.
  check_relator_wellformedness (rel);
   */
}

static void detach_modifier_list (struct modifier_list_rec *rec)
{
  if (rec != null_modifier_list)
  { detach_modifier_list (rec -> next);
    rec -> next = null_modifier_list;
    detach_term_list (rec -> elems);
    rec -> elems = NULL;
    rec -> next = modifier_list_freelist;
    modifier_list_freelist = rec;
  }
}

#if DEBUG_HYBRID
static void print_relator_list (modifier_list mods)
{ modifier_list mods_ptr;
  for (mods_ptr = mods; mods_ptr != null_modifier_list; mods_ptr = mods_ptr -> next)
    { abs_printf ("%c%s [ ", (mods_ptr -> dir)?'>':'<', mods_ptr -> relator);
      print_term_list (mods_ptr -> elems);
      abs_printf ("] ");
    };
}
#endif /* DEBUG_HYBRID */

static void try_cache_heads (eval_stack old, hyb_code instr, Penalty gain)
{ /* Go backwards through the current anchor to see if we can match */
  hyb_code ptr;
  for (ptr = instr -> prev; ptr != hyb_null_code; ptr = ptr -> prev)
    if (ptr == old -> open_ptr)
      { /* We have a matching open and close pair */
        if (ptr -> uni.cache != empty_cache)
	  { if (ptr -> uni.cache -> rec) return;	/* Recovery cache */
	    else abs_abort ("try_cache_heads", "Term cache not empty when it should be");
	  };
        ptr -> uni.cache = new_term_cache ();
	ptr -> uni.cache -> gain = gain;
	if (instr -> opcode == HYB_SQR_CLOSE)
	  { ptr -> uni.cache -> heads = old -> heads;
	    attach_term_list (ptr -> uni.cache -> heads);
	  }
	else ptr -> uni.cache -> heads = null_term_list;
	ptr -> uni.cache -> skip = instr;
	ptr -> uni.cache -> rec = 0;
#if DEBUG_HYBRID
	abs_printf (" cached with penalty %d", gain);
#endif /* DEBUG_HYBRID */
	return;
      };
}

static void propagate_term (term_list *elems, term first)
{ term ptr;
  for (ptr = first; ptr != null_term; ptr = ptr -> next)
    add_string_to_terms (elems, ptr -> text);
}

static void propagate_heads (eval_stack stack, term_list heads)
{ term_list ptr;
  if (stack == empty_eval_stack) return;
  for (ptr = heads; ptr != null_term_list; ptr = ptr -> rest)
    { if (stack -> state == STATE_HEADS)
	{ propagate_term (&stack -> heads, ptr -> first);
	  if (ptr -> rest != null_term_list)
	    push_terms (&stack -> heads);
	}
      else
	{ propagate_term (&stack -> mods -> elems, ptr -> first);
	  if (ptr -> rest != null_term_list)
	    push_terms (&stack -> mods -> elems);
	};
    };
}

/*
   Windows already defines max and min as standard macros
*/
#ifndef WIN32
static int min(int a, int b)
{
    return a < b ? a : b;
}

static int max(int a, int b)
{
    return a > b ? a : b;
}
#endif

void print_triple_statistics()
{
    if (!arts_ifd.triple_stats_option) return;

    abs_message("# triples conceptually tried: %4d", statistic_hybrid_tried);
    abs_message("# triples             failed: %4d", statistic_hybrid_fail);
    abs_message("# terms     really looked up: %4d", statistic_term_tried);
    abs_message("# triples   really looked up: %4d", statistic_triples_tried);
    abs_message("# triples  whitelist success: %4d", statistic_hybrid_whitelist_success);
    abs_message("# triples  blacklist success: %4d", statistic_hybrid_blacklist_success);
}

static void print_triple(term_list head, modifier_list mptr, term_list elems)
{
  abs_printf("[ ");
  print_critical_term (-1, head -> first);
  abs_printf(", %c%s, ", (mptr -> dir)?'>':'<',mptr -> relator);
  print_critical_term (-1, elems -> first);
  abs_printf(" ] ");
}

static void unfound_triples(char *message, term_list head, modifier_list mods)
{
  modifier_list mptr = mods;
  int print = arts_ifd.show_triple_lookups_option > 1;

  if (print)
    abs_printf("Triple lookup: %s; ", message);

  for ( ; mptr != null_modifier_list; mptr = mptr -> next)
    { term_list elems = mptr -> elems;
      for ( ; elems != null_term_list; elems = elems -> rest)
	{ statistic_hybrid_fail++;
	  statistic_hybrid_tried++;
          if (print) print_triple(head, mptr, elems);
        }
    }
  if (print)
    abs_printf("\n");
}

static void unfound_triples_rel(char *message, term_list head, modifier_list mptr)
{
  int print = arts_ifd.show_triple_lookups_option > 1;

  if (print)
    abs_printf("Triple lookup: %s; ", message);

    { term_list elems = mptr -> elems;
      for ( ; elems != null_term_list; elems = elems -> rest)
	{ statistic_hybrid_fail++;
	  statistic_hybrid_tried++;
          if (print) print_triple(head, mptr, elems);
        }
    }

  if (print)
    abs_printf("\n");
}

void unfound_triple_mod(char *message, term_list head, modifier_list mptr, term_list elem)
{
  int print = arts_ifd.show_triple_lookups_option > 1;

  statistic_hybrid_fail++;
  statistic_hybrid_tried++;

  if (print)
    { abs_printf("Triple lookup: %s; ", message);
      print_triple(head, mptr, elem);
      abs_printf("\n");
    }
}

static int lookup_triples_tdb (term_list heads, modifier_list mods)
{ Lexicon lexicon = arts_ifd.lexicon;
  term_list hptr;
  int sum = 0;
  int wildcards_used = 0;
  for (hptr = heads; hptr != null_term_list; hptr = hptr -> rest)
    { modifier_list mptr = mods;
      int hd_idx = lookup_critical_term (hptr -> first);
      if (!hd_idx)
        { hd_idx = wildcard_idx;
	  wildcards_used++;
	}
#if DEBUG_HYBRID
      abs_printf ("Triple left part lookup: \"");
      print_term (hptr -> first);
      abs_message ("\" => idx %d", hd_idx);
#endif /* DEBUG_HYBRID */
      if (arts_ifd.triple_stats_option || arts_ifd.show_triple_lookups_option > 1)
	{ if (!hd_idx)
	    unfound_triples("no head", hptr, mods);
	  else if (hd_idx == wildcard_idx)
	    unfound_triples("no head => *", hptr, mods);
        }
      if (!hd_idx && arts_ifd.closed_triple_db_option) return (MAX_PENALTY);
      if (!hd_idx) continue;	/* head is not found in triple set */
      for ( ; mptr != null_modifier_list; mptr = mptr -> next)
	{ term_list elems = mptr -> elems;
	  int dir = mptr -> dir;
	  int rel_idx = lxcn_lookup_critical_text (lexicon, mptr -> relator);

#if DEBUG_HYBRID
	  abs_message ("Triple relator lookup: \"%c%s\" => rel_idx %d",
		       (dir)?'>':'<',mptr -> relator, rel_idx);
#endif /* DEBUG_HYBRID */
          if (arts_ifd.triple_stats_option || arts_ifd.show_triple_lookups_option > 1)
	    { if (!rel_idx)
	        unfound_triples_rel("no relator", hptr, mptr);
            }
          if (!rel_idx && arts_ifd.closed_triple_db_option) return (MAX_PENALTY);
	  if (!rel_idx) continue;
	  for ( ; elems != null_term_list; elems = elems -> rest)
	    { int el_idx = lookup_critical_term (elems -> first);
	      int frequency = 0;
	      int key[4];
	      int result;
              if (!el_idx && !wildcards_used)
                { el_idx = wildcard_idx;
	          wildcards_used++;
	        }
#if DEBUG_HYBRID
	      abs_printf ("Triple right part lookup: \"");
	      print_term (elems -> first);
	      abs_message ("\" => idx %d", el_idx);
#endif /* DEBUG_HYBRID */
              if (arts_ifd.triple_stats_option || arts_ifd.show_triple_lookups_option > 1)
	        { if (!el_idx)
	            unfound_triple_mod("no modifier", hptr, mptr, elems);
	          else if (el_idx == wildcard_idx)
	            unfound_triple_mod("no modifier => *", hptr, mptr, elems);
                }
              if (!el_idx && arts_ifd.closed_triple_db_option) return (MAX_PENALTY);
	      if (!el_idx) continue;
	      key[0] = 3;
	      key[2] = rel_idx;
	      if (dir)
		{ key[1] = el_idx;
		  key[3] = hd_idx;
		}
	      else
		{ key[1] = hd_idx;
		  key[3] = el_idx;
		};


	      result = lxcn_lookup_triple (lexicon, key, &frequency);
	      statistic_hybrid_tried++;
	      statistic_triples_tried++;
	      if (arts_ifd.show_triple_lookups_option)
		{ abs_printf("Triple lookup: [ ");
                  print_critical_term (hd_idx, hptr -> first);
		  abs_printf(", %c%s, ", (dir)?'>':'<',mptr -> relator);
                  print_critical_term (el_idx, elems -> first);
		  abs_printf(" ] => ");
		  if (result) abs_printf("f=%d\n", frequency);
		  else abs_printf("not found\n");
		}
	      if (result)
	        { if (frequency >= 0) statistic_hybrid_whitelist_success++;
		  else                statistic_hybrid_blacklist_success++;
		}
	      else
		  statistic_hybrid_fail++;

	      if (!wildcards_used && wildcard_idx && frequency >= 0)
		{ int save;
		  int frequency1 = 0, frequency2 = 0;
		  int result1, result2;
#if DEBUG_HYBRID
		  int frequency0 = frequency;
		  abs_message ("Try wildcards");
#endif /* DEBUG_HYBRID */
		  save = key[1];
		  key[1] = wildcard_idx;
		  result1 = lxcn_lookup_triple (lexicon, key, &frequency1);
		  key[1] = save;
		  statistic_triples_tried++;

	          if (arts_ifd.show_triple_lookups_option)
		    { abs_printf("Triple lookup: [ *");
		      abs_printf(", %c%s, ", (dir)?'>':'<',mptr -> relator);
                      print_critical_term (el_idx, elems -> first);
		      abs_printf(" ] => ");
		      if (result1) abs_printf("f=%d\n", frequency);
		      else abs_printf("not found\n");
		    }

		  save = key[3];
		  key[3] = wildcard_idx;
		  result2 = lxcn_lookup_triple (lexicon, key, &frequency2);
		  key[3] = save;
		  statistic_triples_tried++;

	          if (arts_ifd.show_triple_lookups_option)
		    { abs_printf("Triple lookup: [ ");
                      print_critical_term (hd_idx, hptr -> first);
		      abs_printf(", %c%s, ", (dir)?'>':'<',mptr -> relator);
		      abs_printf("* ] => ");
		      if (result2) abs_printf("f=%d\n", frequency);
		      else abs_printf("not found\n");
		    }


		  /* Failed lookups result in a neutral frequency: 0.
		   * Prefer negative frequencies over positive.
		   * If still undecided, take the largest magnitude.
		   */
		  if (frequency < 0 || frequency1 < 0 || frequency2 < 0)
		    { frequency = min(frequency, min(frequency1, frequency2));
#if DEBUG_HYBRID
		      abs_message ("Prefer negative: %d, %d and %d -> %d", frequency0, frequency1, frequency2, frequency); 
#endif /* DEBUG_HYBRID */
		    } else
		    { frequency = max(frequency, max(frequency1, frequency2));
#if DEBUG_HYBRID
		      abs_message ("Prefer largest: %d, %d and %d -> %d", frequency0, frequency1, frequency2, frequency); 
#endif /* DEBUG_HYBRID */
		    }
		    result = result || result1 || result2;
		  }
#if DEBUG_HYBRID
	      if (!result) abs_message (" => fail");
	      else abs_message (" => frequency %d", frequency);
#endif /* DEBUG_HYBRID */
	      if (!result && arts_ifd.closed_triple_db_option) return (MAX_PENALTY);
	      if (!result) continue;
	      sum -= bonus_from_frequency (frequency, arts_ifd.radix.tripledb_frequency);
	    };
	};
    };
  return (sum);
}

static int lookup_triples (term_list heads, modifier_list mods)
{
    int sum = 0;

    if (!arts_ifd.no_tdb_option) sum += lookup_triples_tdb (heads, mods);
    sum += minitdb_lookup_triples (heads, mods);

    return sum;
}

static Penalty pop_eval_stack (int nont_nr, int nr_formals, eval_stack *stack, hyb_code instr)
{ eval_stack old;
  modifier_list top_mod;
  int summed_bonus = 0;
  int sqr = (instr -> opcode == HYB_SQR_CLOSE);
  if (*stack == empty_eval_stack) return (0);
  old = *stack;
  if (((old -> open_ptr -> opcode == HYB_SQR_OPEN) && !sqr) ||
      ((old -> open_ptr -> opcode == HYB_CURL_OPEN) && sqr))
    { abs_message ("# Wrongly balanced %s while evaluating transduction of nonterminal %s/%d",
		   (sqr)?"{ ]":"[ }",
		   arts_ifd.nonterm_names[nont_nr].str, nr_formals);
      /* Try recover from unbalancing by marking the term cache */
      detach_term_cache(old -> open_ptr -> uni.cache);
      old -> open_ptr -> uni.cache = new_term_cache ();
      old -> open_ptr -> uni.cache -> heads = null_term_list;
      old -> open_ptr -> uni.cache -> skip = instr;
      old -> open_ptr -> uni.cache -> rec = 1;

      /* State a terrible bonus for this transduction */
      return (MIN_PENALTY);
    };
  top_mod = old -> mods;
  if (top_mod != null_modifier_list)
    { summed_bonus = lookup_triples (old -> heads, top_mod);
#if DEBUG_HYBRID
      abs_printf ("Cross product of [ ");
      print_term_list (old -> heads);
      abs_printf (" ] with ");
      print_relator_list (old -> mods);
      abs_printf (" yielded %d", summed_bonus);
#endif /* DEBUG_HYBRID */
    }
#if DEBUG_HYBRID
  else
    { abs_printf ("No cross product, just heads [ ");
      print_term_list (old -> heads);
      abs_printf (" ]");
    }
#endif /* DEBUG_HYBRID */
  try_cache_heads (old, instr, summed_bonus); /* Summed bonus/frequency */
#if DEBUG_HYBRID
  abs_printf ("\n");
#endif /* DEBUG_HYBRID */
  if (instr -> opcode == HYB_SQR_CLOSE)
    propagate_heads (old -> father, old -> heads);
  *stack = old -> father;
  detach_eval_stack_rec (old);
  return (summed_bonus);
}

/*
   When we find a MIN_PENALTY, the hybrid parsing was unbalanced.
   When we find a MAX_PENALTY, the lookup had failed with closed_triples enabled
   If neither is the case, we may add the penalties
*/
inline
static Penalty sum_penalties (Penalty one, Penalty two)
{ if ((one == MIN_PENALTY) || (two == MIN_PENALTY)) return (MIN_PENALTY);
  if ((one == MAX_PENALTY) || (two == MAX_PENALTY)) return (MAX_PENALTY);
  return (one + two);
}

static Penalty hybrid_execute_anchor (int nont_nr, int nrf, hyb_anchor anchor, eval_stack *stack);
static Penalty hybrid_execute_instr  (int nont_nr, int nrf, hyb_code instr,    eval_stack *stack)
{ switch (instr -> opcode)
    { case HYB_SQR_OPEN:	push_eval_stack (stack, instr); break;
      case HYB_SQR_CLOSE:	return (pop_eval_stack (nont_nr, nrf, stack, instr));
      case HYB_CURL_OPEN:	push_eval_stack (stack, instr); break;
      case HYB_CURL_CLOSE:	return (pop_eval_stack (nont_nr, nrf, stack, instr));
      case HYB_BAR:		try_push_term_list (*stack); break;
      case HYB_SEPARATOR:	try_push_separator (*stack); break;
      case HYB_ENTER_STRING:	try_push_string (*stack, instr -> uni.arg); break;
      case HYB_ENTER_REL_LEFT:	push_relator (*stack, 0, instr -> uni.arg); break;
      case HYB_ENTER_REL_RIGHT:	push_relator (*stack, 1, instr -> uni.arg); break;
      case HYB_USE_SON:
	/* Do not let penalties propagate upwards, only the terms */
	(void) hybrid_execute_anchor (nont_nr, nrf, instr -> uni.son, stack);
	break;
      default:
	abs_abort ("hybrid_execute_instr", "No such hybrid opcode %d", instr -> opcode);
    };
  return ((Penalty) 0);
}

static int instr_is_cached (hyb_code *instr, eval_stack stack, Penalty *gain)
{ hyb_code ti = *instr;
  if ((ti -> opcode != HYB_SQR_OPEN) && (ti -> opcode != HYB_CURL_OPEN)) return (0);
  if (ti -> uni.cache == empty_cache) return (0);
  propagate_heads (stack, ti -> uni.cache -> heads);
  *instr = ti -> uni.cache -> skip;
  *gain = sum_penalties (*gain, ti -> uni.cache -> gain);
  return (1);
}

static Penalty hybrid_execute_anchor (int nont_nr, int nrf, hyb_anchor anchor, eval_stack *stack)
{ hyb_code instr;
  Penalty gain = (Penalty) 0;
  for (instr = anchor -> first; instr != hyb_null_code; instr = instr -> next)
    if (!instr_is_cached (&instr, *stack, &gain))
      gain = sum_penalties (gain, hybrid_execute_instr (nont_nr, nrf, instr, stack));
  return (gain);
}

Penalty arts_hyb_try_execute (int nont_nr, int nrf, hyb_anchor anchor)
{ eval_stack stack = empty_eval_stack;

  /* Check if this hybrid code belongs to this posmemo and is not an inherited one */
  if (anchor -> owner != NULL) return ((Penalty) 0);
  hybrid_init ();
  return (hybrid_execute_anchor (nont_nr, nrf, anchor, &stack));
}

void arts_hyb_try_set_owner (hyb_anchor anchor, PosMemo pmptr)
{ if (anchor -> owner == NULL)
    anchor -> owner = pmptr;
}

void arts_hyb_mark_hybrid_pool_freed ()
{
    DB_FL(abs_printf("string_list_freelist_count   =%10d ", string_list_freelist_count);)
    DB_FL(abs_message("string_list_alloc_count   =%10d", string_list_alloc_count);)
    DB_FL(abs_printf("term_list_freelist_count     =%10d ", term_list_freelist_count);)
    DB_FL(abs_message("term_list_alloc_count     =%10d", term_list_alloc_count);)
    DB_FL(abs_printf("hyb_anchor_freelist_count    =%10d ", hyb_anchor_freelist_count);)
    DB_FL(abs_message("hyb_anchor_alloc_count    =%10d", hyb_anchor_alloc_count);)
    DB_FL(abs_printf("hyb_instr_freelist_count     =%10d ", hyb_instr_freelist_count);)
    DB_FL(abs_message("hyb_instr_alloc_count     =%10d", hyb_instr_alloc_count);)
    DB_FL(abs_printf("term_cache_freelist_count    =%10d ", term_cache_freelist_count);)
    DB_FL(abs_message("term_cache_alloc_count    =%10d", term_cache_alloc_count);)
    DB_FL(abs_printf("eval_stack_freelist_count    =%10d ", eval_stack_freelist_count);)
    DB_FL(abs_message("eval_stack_alloc_count    =%10d", eval_stack_alloc_count);)
    DB_FL(abs_printf("modifier_list_freelist_count =%10d ", modifier_list_freelist_count);)
    DB_FL(abs_message("modifier_list_alloc_count =%10d", modifier_list_alloc_count);)
    eval_stack_freelist = empty_eval_stack;
    modifier_list_freelist = null_modifier_list;
    term_list_freelist = null_term_list;
    string_list_freelist = null_term;
    hyb_anchor_freelist = NULL;
    hyb_instr_freelist = NULL;
    term_cache_freelist = NULL;
    
    print_triple_statistics();
}
