/*
   File: driver.c
   Compiler driver for agfl with new codegenerator
   Implements make-like functionality for grammar and lexicon.
  
   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.
  
   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU Library General Public License for more details.
  
   You should have received a copy of the GNU Library General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.

   Copyright 2001-2008, Radboud University, Nijmegen
  
   CVS ID: "$Id: driver.c,v 1.2 2008/07/18 18:32:46 marcs Exp $"
*/
#ifdef HAVE_CONFIG_H
#include <config.h>
#endif /* HAVE_CONFIG_H */

#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <stdarg.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <assert.h>

/* libabase includes */
#include <abase_version.h>
#include <abase_porting.h>
#include <abase_error.h>
#include <abase_process.h>
#include <abase_fileutil.h>

/*------------------------------------------------------------------------------
// Alternatively, use the new code generator to generate machine independent
// assembler, assemble it using agfl-as and then use the agfl-run interpreter
//----------------------------------------------------------------------------*/

/*-------------------------------------------------------------------------
// Update later:
// Developers may use agfl-maint instead of agfl-coder. It has more options
// available and does not pass optimization options to agfl by default.
-------------------------------------------------------------------------*/
#if MAINTAINER_MODE
#  define DIRECTORS_FLAG	0
#  define LEFTFAC_FLAG		0
#  define NEGMEMO_FLAG		0
#  define POSMEMO_FLAG		0
#else
#  define DIRECTORS_FLAG	1
#  define LEFTFAC_FLAG		0
#  define NEGMEMO_FLAG		1
#  define POSMEMO_FLAG		1
#endif

static void show_version()
{ abs_message ("Agfl version %s, Copyright 2008, Radboud University of Nijmegen.", AGFL_VERSION);
}

static char agfl_help_mesg[] =
  "Usage: agfl [ -a ] [ -c ] [ -f ] [ -G ] [ -h ] [ -m ] [ -p ] [ -v ] [ -w ] <grammar>\n"
  "\n"
  "Options are:\n"
  "    -a           analyze additional properties\n"
  "    -c           show counters after parsing\n"
  "    -f           force new generation\n"
  "    -G           generate generative grammar\n"
  "    -h           show this help, and exit\n"
  "    -m           prefix nonterminals with module name\n"
  "    -p           profile grammar\n"
  "    -V           show version\n"
  "    -v           be verbose\n"
  "    -w           suppress compiler warnings\n"
  "\n"
  "Arguments are:\n"
  "    <grammar>    name of grammar file  (.gra)\n";

#if MAINTAINER_MODE
static char agflmaint_help_mesg[] =
  "Additional options for debug version:\n"
  "    -e           link Electric Fence\n"
  "    -g           debug\n"
  "    -l           write log file\n"
  "    -s           strip executable\n"
  "    -S           leave generated assembler file\n"
  "    -t           show compile time\n"
  "    -A           suppress well-formedness analysis\n"
  "    -D           use director set lookahead\n"
  "    -F           left-factorize grammar\n"
  "    -P           use positive memoizing\n"
  "    -N           use negative memoizing\n"
  "    -_           toggle underscore flag\n";
#endif

static void show_help ()
{ show_version();
  abs_message (agfl_help_mesg);

#if MAINTAINER_MODE
  abs_message (agflmaint_help_mesg);
#endif
}

typedef struct
{ char *grammar_file;
  int additional_analysis;            /* -a */
  int counters;                       /* -c */
  int force;                          /* -f */
  int debug;                          /* -g */
  int generative_grammar;             /* -G */
  int write_log;                      /* -l */
  int profile;                        /* -p */
  int strip;                          /* -s */
  int keep_assembler_file;            /* -S */
  int show_time;                      /* -t */
  int verbose;                        /* -v */
  int suppress_warnings;              /* -w */
  int suppress_analysis;              /* -A */
  int directors;                      /* -D */
  int left_factorize;                 /* -F */
  int positive_memo;                  /* -P */
  int negative_memo;                  /* -N */
  int module_names;                   /* -m */
} Options;

void init_options (Options* options)
{ options -> grammar_file = NULL;
  options -> additional_analysis = 0;
  options -> counters = 0;
  options -> force = 0;
  options -> debug = 0;
  options -> generative_grammar = 0;
  options -> write_log = 0;
  options -> profile = 0;
  options -> keep_assembler_file = 0;
  options -> show_time = 0;
  options -> verbose = 0;
  options -> suppress_warnings = 0;
  options -> suppress_analysis = 0;
  options -> directors = DIRECTORS_FLAG;
  options -> left_factorize = LEFTFAC_FLAG;
  options -> positive_memo = POSMEMO_FLAG;
  options -> negative_memo = NEGMEMO_FLAG;
  options -> module_names = 0;
}

void process_options (int argc, char** argv, Options* options)
{ char *opt;
  char c;

  /*
     Check whether options specified
  */
  if (argc == 1)
    { show_version ();
      abs_exit (1);
    };

  /*
     Process options
  */
  while ((opt = *++argv))
    { if ((opt[0] == '-') && (opt[1] != '\0'))
	{ while((c = *++opt))
	    { switch (c)
		{ case 'a':
                    options -> additional_analysis++;
                    break;
                  case 'c':
                    options -> counters++;
                    break;
                  case 'f':
                    options -> force++;
                    break;
                  case 'G':
		    abs_error ("This version of AGFL does not support building generating grammars.");
		    abs_exit (1);
                    options -> generative_grammar = 1;
                    options -> directors = 0;
                    options -> positive_memo = 0;
                    options -> negative_memo = 0;
                    break;
                  case 'h':
                    show_help(stdout);
                    exit (1);
                  case 'm':
                    options -> module_names = 1;
                    break;
                  case 'p':
                    options -> profile++;
                    break;
                  case 'V':
                    show_version();
                    exit (0);
                  case 'v':
                    options -> verbose++;
                    break;
                  case 'w':
                    options -> suppress_warnings++;
                    break;
#if MAINTAINER_MODE
                  case 'g':
                    options -> debug++;
                    break;
                  case 'l':
                    options -> write_log++;
                    break;
                  case 's':
                    options -> strip++;
                    break;
                  case 'S':
                    options -> keep_assembler_file++;
                    break;
                  case 't':
                    options -> show_time++;
                    break;
                  case 'A':
                    options -> suppress_analysis++;
                    break;
                  case 'D':
                    options -> directors++;
                    break;
                  case 'F':
                    options -> left_factorize++;
                    break;
                  case 'P':
                    options -> positive_memo++;
                    break;
                  case 'N':
                    options -> negative_memo++;
                    break;
#endif /* MAINTAINER_MODE */
                  default:
                    abs_error ("unknown option `-%c'", c);
                    abs_error ("use `agfl -h' for help");
                    abs_exit (1);
                };
            };
        }
      else
        { if (options -> grammar_file == NULL)
            options -> grammar_file = opt;
          else
	    { abs_error ("multiple input files specified"
                         " (`%s' and `%s')", options -> grammar_file, opt);
              abs_exit (1);
            };
        }
    };

  /*
     Check whether options valid
  */
  if (options -> grammar_file == NULL)
    { abs_error ("no input file specified");
      abs_exit (1);
    };

  if (options -> verbose >= 2)
    { abs_printf ("  %s ",
#if MAINTAINER_MODE
	    "agflmaint");
#else
	    "agfl");
#endif /* MAINTAINER_MODE */
      
      if (options -> additional_analysis) abs_printf (" -a");
      if (options -> counters           ) abs_printf (" -c");
      if (options -> force              ) abs_printf (" -f");
      if (options -> debug              ) abs_printf (" -g");
      if (options -> generative_grammar ) abs_printf (" -G");
      if (options -> write_log          ) abs_printf (" -l");
      if (options -> profile            ) abs_printf (" -p");
      if (options -> strip              ) abs_printf (" -s");
      if (options -> keep_assembler_file) abs_printf (" -S");
      if (options -> show_time          ) abs_printf (" -t");
      if (options -> verbose > 0        ) abs_printf (" -v");
      if (options -> verbose > 1        ) abs_printf (" -v");
      if (options -> suppress_warnings  ) abs_printf (" -w");
      if (options -> suppress_analysis  ) abs_printf (" -A");
      if (options -> directors          ) abs_printf (" -D");
      if (options -> left_factorize     ) abs_printf (" -F");
      if (options -> positive_memo      ) abs_printf (" -P");
      if (options -> negative_memo      ) abs_printf (" -N");
      if (options -> module_names       ) abs_printf (" -m");
      if (options -> grammar_file       ) abs_printf (" %s", options -> grammar_file);
      abs_message ("");
    };
}

void strip_extension (char *path, char *ext)
{ int len1 = (int) strlen (path);
  int len2 = (int) strlen (ext);

  if ((len1 > len2) && strcmp (path + len1 - len2, ext) == 0)
    path[len1 - len2] = '\0';
}

void get_last_dir (char* name, char *path)
{ char *last_dir;
  char *p;

  last_dir = path;
  for (p = path; p[0] != '\0'; p++)
    if (((p[0] == '/') || (p[0] == DIRSEP)) && (p[1] != '\0'))
      last_dir = p;
  strcpy(name, last_dir);
}

#ifdef WIN32
#define DIR_SEP "\\"
#else
#define DIR_SEP "/"
#endif
/*
   This is very dubious code, doing something special in the case
   that the argument is a directory. What was the original intention?
*/
void get_base_name (char* base, char *name, char *ext)
{ char path[MAXPATHLEN];

  strcpy (base, name);
  strip_extension (base, ext);
#ifdef WIN32
  /* Unsafe copy: no snprintf */
  sprintf (path, "%s%s", base, ext);
#else
  snprintf (path, MAXPATHLEN, "%s%s", base, ext);
#endif

  if (!abs_file_exists (path))
    { if (abs_is_directory (base))
        { char name[MAXPATHLEN];

          get_last_dir(name, base);
          strcat(base, DIR_SEP);
          strcat(base, name);
#ifdef WIN32
          /* Unsafe copy */
          sprintf (path, "%s%s", base, ext);
#else
          snprintf (path, MAXPATHLEN, "%s%s", base, ext);
#endif
        };

      if (!abs_file_exists (path))
        { abs_error ("cannot locate input file `%s'", path);
          abs_exit (1);
        };
    };

  if (!abs_is_normal_file (path))
    { abs_error ("`%s' is not a regular file", path);
      abs_exit (1);
    };

  if (strlen (base) + 4 > MAXPATHLEN)
    { abs_error ("`%s' is too long", base);
      abs_exit (1);
    };
}

int have_lexicon (char* base)
{ return (abs_file_ext_exists (base, "lif"));
}

/*
   Check whether file with path1 is newer than file with path2.
   Return false if file with path1 does not exist.
   Return true if file with path2 does not exist.
*/
int file_is_newer (char* path1, char* path2)
{ time_t time1 = 0;
  time_t time2 = 0;
  if (abs_file_mtime (path1, &time1)) return (0);
  (void) abs_file_mtime (path2, &time2);

  return (time1 > time2);
}

void check_ext_file_exists (char *name, char *ext)
{ if (!abs_file_ext_exists (name, ext))
    { abs_error ("cannot locate input file `%s.%s'", name, ext);
      exit(1);
    };
}

void delete_file (const char* path, int verbose)
{ if (verbose >= 2)
    abs_message ("delete file '%s'", path);

  /* Note remove is an existing C function, both in Unix as Win32 */
  remove (path);
}

void try_dump_argv (char *argv[], Options *opt)
{ char **argptr;
  if (!opt -> verbose) return;
  for (argptr = argv; (*argptr) != NULL; argptr++)
    { if (argptr != argv) abs_printf (" ");
      abs_printf ("%s", *argptr);
    };
  abs_message ("");
}

void compile_grammar (char *base, Options *opt)
{ char *coder_cmdline[32];
  char asm_file[MAXPATHLEN];
  int nrargs = 0;
  check_ext_file_exists (base, "gra");

  /* Compose the command line */
  coder_cmdline[nrargs++] = "agfl-coder";
  if (opt -> additional_analysis) coder_cmdline[nrargs++] = "-a";
  if (opt -> counters)            coder_cmdline[nrargs++] = "-c";
  if (opt -> debug)               coder_cmdline[nrargs++] = "-g";
  if (opt -> generative_grammar)  coder_cmdline[nrargs++] = "-G";
  if (opt -> write_log)           coder_cmdline[nrargs++] = "-l";
  if (opt -> module_names)        coder_cmdline[nrargs++] = "-m";
  if (opt -> profile)             coder_cmdline[nrargs++] = "-p";
  if (opt -> show_time)           coder_cmdline[nrargs++] = "-t";
  if (opt -> verbose)             coder_cmdline[nrargs++] = "-v";
  if (opt -> verbose > 1)         coder_cmdline[nrargs++] = "-v";
  if (opt -> suppress_warnings)   coder_cmdline[nrargs++] = "-w";
  if (opt -> suppress_analysis)   coder_cmdline[nrargs++] = "-A";
  if (opt -> directors)           coder_cmdline[nrargs++] = "-D";
  if (opt -> left_factorize)      coder_cmdline[nrargs++] = "-F";
  if (opt -> positive_memo)       coder_cmdline[nrargs++] = "-P";
  if (opt -> negative_memo)       coder_cmdline[nrargs++] = "-N";
  coder_cmdline[nrargs++] = base;
  coder_cmdline[nrargs] = NULL;

  /* Try and execute agfl-coder */
  try_dump_argv (coder_cmdline, opt);
  if (abs_execute (coder_cmdline) == 0)
    return;

  /* If we reached this point, something went wrong, delete .s file */
  sprintf (asm_file, "%s.s", base);
  delete_file (asm_file, opt -> verbose);
  abs_exit (1);
}

void assemble_code (char *base, Options *opt)
{ char *as_cmdline[16];
  char asm_file[MAXPATHLEN];
  char obj_file[MAXPATHLEN];
  int nrargs = 0;
#ifdef WIN32
  sprintf (asm_file, "%s.s", base);
  sprintf (obj_file, "%s.aob", base);
#else
  snprintf (asm_file, MAXPATHLEN, "%s.s", base);
  snprintf (obj_file, MAXPATHLEN, "%s.aob", base);
#endif
  check_ext_file_exists (base, "s");
  
  /* Compose the assembler line */
  as_cmdline[nrargs++] = "agfl-as";
  as_cmdline[nrargs++] = asm_file;
  as_cmdline[nrargs++] = "-o";
  as_cmdline[nrargs++] = obj_file;
  as_cmdline[nrargs] = NULL;

  /* Try and execute agfl-as */
  try_dump_argv (as_cmdline, opt);
  if (abs_execute (as_cmdline))
    { /* Assembly went wrong, delete object */
      delete_file (obj_file, opt -> verbose);
      abs_exit (1);
    };

  /* Keep the assembler file, if requested */
  if (!opt -> keep_assembler_file)
    delete_file (asm_file, opt -> verbose);
}

void compile_lexica (char* base, Options *opt)
{ char blx_file[MAXPATHLEN];
  char *lexgen_cmdline[10];
  int nrargs = 0;

  /* Compose lexgen command line */
  if (!have_lexicon (base)) return;
  lexgen_cmdline[nrargs++] = "agfl-lexgen";
  if (opt -> verbose)     lexgen_cmdline[nrargs++] = "-v";
  if (opt -> verbose > 1) lexgen_cmdline[nrargs++] = "-v";
  if (opt -> force)       lexgen_cmdline[nrargs++] = "-f";
  lexgen_cmdline[nrargs++] = base;
  lexgen_cmdline[nrargs] = NULL;

  /* Try and execute lexgen */
  try_dump_argv (lexgen_cmdline, opt);
  if (abs_execute (lexgen_cmdline) == 0)
    return;

  /* Some error occurred, delete blx (blf) file */
#ifdef WIN32
  sprintf (blx_file, "%s%s", base, ".blx");
#else
  snprintf (blx_file, MAXPATHLEN, "%s%s", base, ".blx");
#endif
  delete_file (blx_file, opt -> verbose);
  abs_exit (1);
}

int main (int argc, char* argv[])
{ Options options;
  char base_name[MAXPATHLEN];

  /* Parse options */
  init_options (&options);
  process_options (argc, argv, &options);

  /* Pick the right grammar and compile it */
  get_base_name (base_name, options.grammar_file, ".gra");
  compile_grammar (base_name, &options);
  assemble_code (base_name, &options);

  /* lexicon generator decides if it needs to make a lexicon..: */
  compile_lexica (base_name, &options);
  return (0);
}
