/*
   File: abase_process.c
   Executes a command line as subprocess

   Copyright 2008 Radboud University of Nijmegen
 
   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.
 
   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU Library General Public License for more details.
 
   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.

   CVS ID: "$Id: abase_process.c,v 1.2 2008/05/14 15:42:31 marcs Exp $"
*/

/* include <config.h> if autoconfigured */
#ifdef HAVE_CONFIG_H
#include <config.h>
#endif /* HAVE_CONFIG_H */

/* global includes */
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#ifndef WIN32
#include <unistd.h>
#include <sys/types.h>
#ifdef HAVE_SYS_WAIT_H
#include <sys/wait.h>
#endif /* HAVE_SYS_WAIT_H */
#include <errno.h>
#endif /* WIN32 */

/* Local includes */
#include "abase_porting.h"
#include "abase_error.h"
#include "abase_process.h"
#include "abase_memalloc.h"

#ifdef WIN32
static char *flatten_cmdline (char *argv [])
{ char buf[MAXPATHLEN];
  char **argptr;
  char *dptr;
  int len = 0;

  for (argptr = argv, dptr = buf; (*argptr) != NULL; argptr++)
    { int alen = strlen (*argptr);
      if (len + alen + 1 > MAXPATHLEN)
	abs_fatal ("Too long command line");
      if (argptr != argv)
        *dptr++ = ' ';
      strcpy (dptr, *argptr);
      dptr += alen;
    };
  *dptr = '\0';
  return (abs_new_string (buf, "flatten_cmdline"));
}
#endif

int abs_execute (char *argv[])
#ifdef WIN32
{ STARTUPINFO si;
  PROCESS_INFORMATION pi;
  char *cmdline = flatten_cmdline (argv);
  memset ((void *) &si, 0, sizeof (si));
  si.cb = sizeof (si);
  memset ((void *) &pi, 0, sizeof (pi));

  if (!CreateProcess (NULL, cmdline, NULL, NULL, FALSE, 0, NULL, NULL, &si, &pi))
    abs_fatal ("could not create process: error %d", GetLastError ());
  else
    { /* Parent code */
      DWORD result = WaitForSingleObject (pi.hProcess, INFINITE);
      CloseHandle (pi.hProcess);
      CloseHandle (pi.hThread);
      if (result == WAIT_OBJECT_0) return (0);
      return (1);
    };

  /* Default: unsuccessfull */
  return (0);
}
#else
{ pid_t pid;
  int status;

  /* Fork off process */
  pid = fork ();
  if (pid < 0)
    { /* could not fork, bad if so */
      abs_fatal ("could not fork: %s", strerror (errno));
    }
  else if (pid == 0)
    { /* child code */
      execvp (argv[0], argv);

      /* if we reach this point, we could not exec */
      abs_fatal ("could not exec %s: %s", argv[0], strerror (errno));
    };

  /* parent code: wait for child to finish */
  waitpid (pid, &status, 0);
  return (status);
}
#endif /* WIN32 */
