/* 
   File: abase_mm_alloc.c
   Defines managed memory allocation routines

   Copyright 2006 Radboud University of Nijmegen
 
   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.
 
   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU Library General Public License for more details.
 
   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.

   CVS ID: "$Id: abase_mm_alloc.c,v 1.2 2006/04/05 13:21:31 marcs Exp $"
*/

/* standard includes */
#ifdef HAVE_CONFIG_H
#include <config.h>
#endif /* HAVE_CONFIG_H */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <stdarg.h>

/* Local includes */
#include "abase_porting.h"
#include "abase_error.h"
#include "abase_memalloc.h"
#include "abase_mm_alloc.h"

/*
   Basically the managed routines have the same interface as
   the nonmanaged routines, but their location is remembered
   on a linear list for later removal. We need this kind of
   allocation because affix values dynamically allocated
   during pass one must survive until the positive memos can
   be freed.
*/
typedef struct mm_rec *mm_list;
struct mm_rec
	{ void *entry;
	  mm_list next;
	};
static mm_list mm_anchor;

void abs_mm_init (void)
	{ mm_anchor = NULL;
	}

static void enter_into_mm_list (void *entry, const char *place)
	{ mm_list new_rec = (mm_list) abs_malloc (sizeof (struct mm_rec), place);
	  new_rec -> entry = entry;
	  new_rec -> next = mm_anchor;
	  mm_anchor = new_rec;
	}

void abs_mm_free (const char *place)
	{ while (mm_anchor != NULL)
	    { mm_list old = mm_anchor;
	      mm_anchor = mm_anchor -> next;
	      abs_free (old -> entry, place);
	      abs_free (old, place);
	    };
	};

void *abs_mm_malloc (const size_t size, const char *place)
	{ void *result = abs_malloc (size, place);
	  enter_into_mm_list (result, place);  
	  return (result);
	};

void *abs_mm_calloc (const size_t nr, const size_t size, const char *place)
	{ void *result = abs_calloc (nr, size, place);
	  enter_into_mm_list (result, place);  
	  return (result);
	};

char *abs_mm_new_string (const char *old, const char *place)
	{ char *result;
	  if (old == NULL)
	     abs_abort ("abs_mm_new_string", "Called with NULL argument at %s", place);
	  result = (char *) abs_mm_malloc (strlen (old) + 1, place);
	  strcpy (result, old);
	  return (result);
	};

char *abs_mm_new_fmtd_string (const char *place, const char *format, ...)
	{ char buf[MAX_FMT_LEN];
	  va_list arg_ptr;
	  va_start (arg_ptr, format);
	  vsprintf (buf, format, arg_ptr);
	  va_end (arg_ptr);
	  return (abs_mm_new_string (buf, place));
	};
