/*
   File: entries.c
   Maintains the administration of all parts of speech i.e. all
   combinations of a nonterminal id x actual parameters.

   Copyright 2007 Radboud University of Nijmegen
 
   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.
 
   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU Library General Public License for more details.
 
   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.

   CVS ID: "$Id: entries.c,v 1.5 2007/10/31 14:57:50 marcs Exp $"
*/

/* system includes */
#include <stdio.h>
#include <string.h>

/* libabase includes */
#include <abase_repr.h>
#include <abase_error.h>
#include <abase_memalloc.h>
#include <abase_fileutil.h>

/* liblexicon includes */
#include <lxcn_input.h>      /* For the moment */
#include <lxcn_vocabulary.h>

/* local includes */
#include "options.h"
#include "dyn_array.h"
#include "affix_values.h"
#include "nonterminals.h"
#include "entries.h"
#include "globals.h"

/* An entry is the combination of a call and a frequency */
typedef struct entry_rec
{ int call_id;
  int frequency;
} *entry;

/* Note: entry list 0 is reserved */
#define DEFAULT_ENTRIES 16384
static struct entry_rec *all_entries;
static int my_nr_of_entries, my_room_of_entries;
static ptr_array all_entry_lists;

static int_array create_or_locate_entry_list (int *info_ptr)
{ int_array new_el;
  int new_idx;
  if (*info_ptr) return ((int_array) all_entry_lists -> array[*info_ptr]);
  new_el = new_int_array ();
  new_idx = all_entry_lists -> size;
  app_ptr_array (all_entry_lists, (void *) new_el);
  *info_ptr = new_idx;
  return (new_el);
}

static void create_new_entry (int call_id, int frequency)
{ entry new_entry;
  if (my_nr_of_entries == my_room_of_entries)
    { my_room_of_entries *= 2;
      all_entries =
	(entry) abs_realloc ((void *) all_entries, my_room_of_entries * sizeof (struct entry_rec),
			     "create_new_entry");
    };
  new_entry = &all_entries[my_nr_of_entries];
  my_nr_of_entries++;
  new_entry -> call_id = call_id;
  new_entry -> frequency = frequency;
}

void register_new_entry (int *info_ptr, int call_id, int frequency)
{ int_array el = create_or_locate_entry_list (info_ptr); 
  entry this_entry;
  int ix;
  for (ix = 0; ix < el -> size; ix++)
    { this_entry = &all_entries[el -> array[ix]];
      if (this_entry -> call_id == call_id)
	{ /* Found entry: add frequency */
	  this_entry -> frequency += frequency;
	  return;
	};
    };
  app_int_array (el, my_nr_of_entries);
  create_new_entry (call_id, frequency);
}

int nr_of_entries ()
{ return (my_nr_of_entries);
}

int nr_of_entry_lists ()
{ return (all_entry_lists -> size);
}

/*
  Generate info file
*/
void generate_entries_info (FILE *info)
{ int ix;
  fprintf (info, "Entry lists:\n");
  for (ix = 1; ix < all_entry_lists -> size; ix++)
    { int_array el = (int_array) all_entry_lists -> array[ix];
      int iy;
      fprintf (info, "%8d:", ix);
      for (iy = 0; iy < el -> size; iy++)
        { entry this_entry = &all_entries[el -> array[iy]];
	  fprintf (info, " (%d,%d)", this_entry -> call_id, this_entry -> frequency);
	};
      fprintf (info, "\n");
    };
}

/*
  Generate dump file
*/
static void dump_string (FILE *dump, char *key)
{ char *kptr = key;
  fputc ('\"', dump);
  for (; *kptr; kptr++)
    { switch (*kptr)
	{ case '\t': fprintf (dump, "\\t"); break;
	  case '\n': fprintf (dump, "\\n"); break;
	  case '\r': fprintf (dump, "\\r"); break;
	  case '\"': fprintf (dump, "\\\""); break;
	  default: fputc (*kptr, dump);
	};
    };
  fputc ('\"', dump);
}

static void dump_entry (FILE *dump, char *key, int entry_nr)
{ entry this_entry = &all_entries[entry_nr];
  dump_string (dump, key);
  fprintf (dump, "\t");
  dump_call (dump, this_entry -> call_id);
  if (this_entry -> frequency > 1)
    fprintf (dump, "\t%d", this_entry -> frequency);
  fprintf (dump, "\n");
}

static void dump_vocabulary_entry (void *arg, char *key, int info)
{ FILE *dump = (FILE *) arg;
  int_array el = (int_array) all_entry_lists -> array[info];
  int ix;
  for (ix = 0; ix < el -> size; ix++)
    dump_entry (dump, key, el -> array[ix]);
}

void dump_vocabulary_entries (FILE *dump)
{ int ix;
  for (ix = 0; ix < all_lexeme_markers -> size; ix++)
    { int marker = all_lexeme_markers -> array[ix];
      Vocabulary voc = (Vocabulary) all_vocabularies -> array[ix];
      void *arg = (void *) dump;
      fprintf (dump, "Vocabulary with marker %d\n", marker);
      lxcn_iterate_over_vocabulary (voc, dump_vocabulary_entry, arg);
    };
}

/*
   Binary saving of entries and vocabularies
*/
static void my_bin_save_entries (BinFile bf, int_array el)
{ int ix;
  abs_bin_save_int (bf, el -> size);
  for (ix = 0; ix < el -> size; ix++)
    { entry this_entry = &all_entries[el -> array[ix]];
      abs_bin_save_int (bf, this_entry -> call_id);
      abs_bin_save_int (bf, this_entry -> frequency);
    };
}

void bin_save_vocabularies (BinFile bf)
{ int ix;
  abs_bin_save_int (bf, all_lexeme_markers -> size);
  for (ix = 0; ix < all_lexeme_markers -> size; ix++)
    { int marker = all_lexeme_markers -> array[ix];
      Vocabulary voc = (Vocabulary) all_vocabularies -> array[ix];
      abs_bin_save_int (bf, marker);
      lxcn_bin_save_trie (bf, voc);
    };
}

void bin_save_entries (BinFile bf)
{ int ix;
  abs_bin_save_int (bf, all_entry_lists -> size);
  for (ix = 1; ix < all_entry_lists -> size; ix++)
    my_bin_save_entries (bf, (int_array) all_entry_lists -> array[ix]);
}

/*
   Initialization stuff
*/
void init_entries ()
{ all_entries = (struct entry_rec *) abs_calloc (DEFAULT_ENTRIES, sizeof (struct entry_rec),
						 "init_entries");
  my_nr_of_entries = 0;
  my_room_of_entries = DEFAULT_ENTRIES;
  all_entry_lists = init_ptr_array (DEFAULT_ENTRIES);
  app_ptr_array (all_entry_lists, (void *) NULL);	/* Entry 0 is reserved */
}
