/*
   File: arun_options.c
   Specifies option and argument treatment

   Copyright 2006 Radboud University of Nijmegen
 
   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.
 
   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU Library General Public License for more details.
 
   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.

   CVS ID: "$Id: arun_options.c,v 1.6 2007/10/31 14:57:24 marcs Exp $"
*/

/* standard includes */
#ifdef HAVE_CONFIG_H
#include <config.h>
#endif /* HAVE_CONFIG_H */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <limits.h>
#include <time.h>

/* libabase includes */
#include <abase_error.h>
#include <abase_version.h>
#include <abase_memalloc.h>

/* Ugly include for max_edit_distance */
#include <lexicon_search.h>

/* libarts includes */
#include "arts_ds.h"
#include "arts_io.h"

/* local includes */
#include "arun_options.h"

/*-----------------------------------------------------------------------------
// Private options to be able to overrule the grammar defaults
//---------------------------------------------------------------------------*/
static char *alphabet_fname = NULL;
static int show_options_option;
static int transduce_option;
static int graph_option;
/* max_edit_distance;   // defined in lexicon_search.h */

static void init_options ()
{ show_options_option = 0;
  transduce_option = -1;
  graph_option = -1;
}

/* Change when pragma max_parses is introduced */
static void set_max_number_of_parses (int v)
{ if (v < 0) arts_ifd.max_parses = 0;
  else arts_ifd.max_parses = v;
}

static void set_max_edit_distance (int v)
{ if (v < 0) max_edit_distance = 0;
  else max_edit_distance = v;
}

static void set_max_parse_time (int v)
{ if (v < 0) arts_ifd.max_parsetime = 0;
  else arts_ifd.max_parsetime = v;
}

/*------------------------------------------------------------------------------
// Public
//----------------------------------------------------------------------------*/
static void sprint_quoted (char *buf, char *s)
{ char c;
  char *p = buf;

  while ((c = *s++))
    switch (c)
      { case '\n':
	  sprintf (p, "\\n");
	  p += 2;
	  break;
	case '\r':
	  sprintf (p, "\\r");
	  p += 2;
	  break;
	case '\t':
	  sprintf (p, "\\t");
	  p += 2;
	  break;
	default:
	  *p++ = c;
	  break;
      };
  *p = '\0';
}

static char* convert_time_to_text(long time)
{ time_t clock = (time_t) time;
  char *txt_time = ctime (&clock);
  txt_time[24] = '\0';
  return (txt_time);
}

void try_show_options()
{ char buf[256];
  if (!show_options_option) return;
  abs_message ("Grammar last changed on %s", convert_time_to_text (arts_ifd.gra_version));

  if (arts_ifd.word_terminator_chars != NULL)
    { sprint_quoted (buf, arts_ifd.word_terminator_chars);
      abs_message ("Word terminators are '%s'", buf);
    };

  if (arts_ifd.invisible_chars != NULL)
    { sprint_quoted (buf, arts_ifd.invisible_chars);
      abs_message ("Invisible characters are '%s'", buf);
    };

  /* To be activated when max_edit_distance is used */
#if USE_EDIT_DISTANCE
  if (max_edit_distance > 0)
    abs_message ("Maximum edit distance is %d", max_edit_distance);
#endif

  if (arts_ifd.graph_option)
    abs_message ("Displaying lexical graph");

  if (arts_ifd.directors_option)
    abs_message ("Directors available");

  if (arts_ifd.pos_memo_option)
    abs_message ("Positive memoization available");

  if (arts_ifd.neg_memo_option)
    abs_message ("Negative memoization available");

  if (arts_ifd.max_parses < INT_MAX)
    abs_message ("Maximum number of parses is %d", arts_ifd.max_parses);

  if (arts_ifd.max_parsetime < INT_MAX)
    abs_message ("Maximum parse time is %d seconds", arts_ifd.max_parsetime);

  if (arts_ifd.label_bracket && !arts_ifd.no_output)
    abs_message ("Labelled bracket output");

  if (arts_ifd.no_output)
    abs_message ("Suppressing output");
  else if (arts_ifd.transduce_option && !arts_ifd.no_output)
    abs_message ("Transducing");

  if (arts_ifd.counters_option)
    abs_message ("Showing counters\n");

  if (arts_ifd.generate_option)
    abs_message ("This is a sentence generator");

  if (arts_ifd.best_parsings_option)
    abs_message ("Best parsing enabled");

  if (arts_ifd.hybrid_parsing_option)
    abs_message ("Hybrid parsing enabled");

  if (arts_ifd.input_fname != NULL)
    abs_message ("Input filename: '%s'\n", arts_ifd.input_fname);
}

/*-----------------------------------
// -h shows options
//---------------------------------*/
static void try_show_version ()
{ abs_message ("This is the AGFL runtime environment, version %s", AGFL_VERSION);
}

static void show_usage ()
{ abs_message ("Usage: agfl-run [options...] grammar_name [input_fname]");
  abs_message (
	"OPTION      DEFAULT     PURPOSE\n"
	"-P number   infinite    maximum number of parses\n"
	"-T number   infinite    maximum total parse time (in seconds)\n"
#if USE_EDIT_DISTANCE
	"-E number   0           maximum edit distance\n"
#endif
	"-B          off         search parsings for best penalty level\n"
	"-b          off         labelled bracket output\n"
	"-H          off         enables hybrid parsing\n"
	"-O          off         suppress output\n"
	"-i name     input_fname use name to format output of $POS\n"
	"-o file     stdout      write output to file\n"
	"-t          off         transduce\n"
	"-s          on          standard transduction\n"
	"-A file     off         use alphabet file\n"
	"-G          off         show lexical graph\n"
	"-V                      show version\n"
	"-S          off         enable I/O synchronisation\n"
	"-v          off         set verbosity. possible values:\n"
	"            options     show parser options\n"
	"            lexer       show all lexer output\n"
	"            parser      show parser stats\n"
	"            research    show research stats\n"
	"            totals      show totals (ie. time)\n"
	"            lpt         lexer+parser+totals\n"
	"-c          off         count opcodes and rule calls\n"
	"-h          off         show this help");
    abs_exit (0);
}

void check_for_verbose_options (char *arg)
{ if (!strcasecmp (arg, "lpt"))
    { arts_ifd.lexer_stats_option = 1;
      arts_ifd.parsing_stats_option = 1;
      arts_ifd.total_stats_option = 1;
      return;
    };
    
  if (!strcasecmp (arg, "parser"))
    { arts_ifd.parsing_stats_option = 1;
      return;
    };

  if (!strcasecmp (arg, "research"))
    { arts_ifd.research_stats_option = 1;
      return;
    };

  if (!strcasecmp (arg, "lexer"))
    { arts_ifd.lexer_stats_option = 1;
      return;
    };

  if (!strcasecmp (arg, "totals"))
    { arts_ifd.total_stats_option = 1;
      return;
    };

  if (!strcasecmp (arg, "options"))
    { show_options_option = 1;
      return;
    };

  abs_error ("Illegal verbose option '%s'", arg);
  show_usage ();
}

static char *get_optarg (char option, char *ptr, int *ix, int argc, char **argv)
{ /* Check if argument in same command line argument */
  if (*(++ptr))
    return (ptr);

  /* Check if there is a next argument */
  *ix += 1;
  if (*ix < argc)
    return (argv[*ix]);

  /* Complain */
  abs_error ("Missing argument with option '-%c'", option);
  show_usage ();

  /* Unreachable */
  return (NULL);
}

static void scan_options (char *ptr, int *ix, int argc, char **argv) 
{ do
    { switch (*ptr)
	{ /* Concatenable boolean options */
	  case 'b': arts_ifd.label_bracket = 1; break;
	  case 'c': arts_ifd.counters_option = 1; break;
	  case 'G': graph_option = 1; break;
	  case 'B': arts_ifd.best_parsings_option = 1; break;
	  case 'H': arts_ifd.hybrid_parsing_option = 1; break;
	  case 'O': arts_ifd.no_output = 1; break;
	  case 'S': arts_ifd.lcsdoc_sync_option = 1; break;
	  case 's': transduce_option = 0; break;
	  case 't': transduce_option = 1; break;

	  /* Options with a numeric argument */
	  case 'E':
	    { char *arg = get_optarg (*ptr, ptr, ix, argc, argv);
	      set_max_edit_distance (atoi (arg));
	      return;
	    };
	  case 'P':
	    { char *arg = get_optarg (*ptr, ptr, ix, argc, argv);
	      set_max_number_of_parses (atoi (arg));
	      return;
	    };
	  case 'T':
	    { char *arg = get_optarg (*ptr, ptr, ix, argc, argv);
	      set_max_parse_time (atoi (arg));
	      return;
	    };

	  /* Options with a string argument */
	  case 'i':
	    { char *arg = get_optarg (*ptr, ptr, ix, argc, argv);
	      arts_ifd.input_pos_fname = abs_new_string (arg, "scan_options");
	      return;
	    };
	  case 'A':
	    { char *arg = get_optarg (*ptr, ptr, ix, argc, argv);
	      alphabet_fname = abs_new_string (arg, "scan_options");
	      return;
	    };
	  case 'o':
	    { char *arg = get_optarg (*ptr, ptr, ix, argc, argv);
	      arts_ifd.output_fname = abs_new_string (arg, "scan_options");
	      return;
	    }
	  case 'v':
	    { char *arg = get_optarg (*ptr, ptr, ix, argc, argv);
	      check_for_verbose_options (arg);
	      return;
	    };

	  /* Miscellaneous options */
	  case 'V':
	    try_show_version ();
	    abs_exit (0);
	    /* never returns */
	  case 'h':
	  case '?': show_usage ();
	  default:
	    abs_error ("unknown option '-%c', use 'agfl-run -h' for help", *ptr);
	    show_usage ();
	};
    }
  while (*(++ptr));
}

void parse_command_line (int argc, char **argv)
{ int had_args = 0;
  int ix;
  init_options ();
  for (ix = 1; ix < argc; ix++)
    { char *arg = argv[ix];
      if (arg[0] == '-') scan_options (arg+1, &ix, argc, argv);
      else if (had_args >= 2)
	{ abs_error ("Too many arguments");
	  show_usage ();
	}
      else if (arts_ifd.grammar_name == NULL)
	{ arts_ifd.grammar_name = abs_new_string (argv[ix], "parse_command_line");
	  had_args++;
	}
      else
	{ if (arts_ifd.input_pos_fname != NULL)
	    { abs_message ("Warning: -i option overridden by argument");
	      abs_free (arts_ifd.input_pos_fname, "parse_command_line");
	    };
	  arts_ifd.input_fname = abs_new_string (argv[ix], "parse_command_line");
	  arts_ifd.input_pos_fname = arts_ifd.input_fname;
	  had_args++;
	};
    };
  
  /* Exit if no source given */
  if (arts_ifd.grammar_name == NULL)
    { abs_error ("No grammar name given");
      show_usage ();
    };
}

/*
   Overrule the options set by the compiler by those on the command line
*/
void overrule_grammar_options ()
{ /* -A alphabet_fname */
  if (alphabet_fname != NULL)
    arts_ifd.alphabet_fname = alphabet_fname;

  /* |graph|, -G */
  if (graph_option != -1)
    arts_ifd.graph_option = graph_option;

  /* |transduce|, -s, -t */
  if (transduce_option != -1)
    arts_ifd.transduce_option = transduce_option; 
}
