/*
   File: lxcn_input.c
   Defines the various lexeme types and holds the character tables while lexing

   Copyright 2005 Radboud University of Nijmegen
 
   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.
 
   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU Library General Public License for more details.
 
   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.

   CVS ID: "$Id: lxcn_input.c,v 1.2 2007/10/31 14:58:06 marcs Exp $
*/
#ifdef HAVE_CONFIG_H
#include <config.h>
#endif /* HAVE_CONFIG_H */

/* standard includes */
#include <stdio.h>
#include <ctype.h>
#include <string.h>

/* abase includes */
#include <abase_error.h>

/* local includes */
#include "lxcn_input.h"

/*------------------------------------------------------------------------------
// Global data
//----------------------------------------------------------------------------*/
static char blank_table[256];
static char terminator_table[256];
static char invisible_table[256];
static char translate_table[256];

/*------------------------------------------------------------------------------
// Table functions
//----------------------------------------------------------------------------*/
static void set_table (char *table, char *str)
{ int ix;
  char c;
  if (str == NULL) abs_abort ("set_table", "called with NULL string");

  for (ix = 0; ix < 256; ix++) table[ix] = '\0';
  while ((c = *str++)) table[(int) ((unsigned char) c)] = c;
}

static void set_trans_table (char* table, char* src, char* dst)
{ int ix;
  char c;
  if (src == NULL || dst == NULL)
    abs_abort ("set_trans_table", "called with NULL string");

  for (ix = 0; ix < 256; ix++) table[ix] = (char) ix;
  while ((c = *src++)) table[(int) ((unsigned char) c)] = *dst++;
}

/*------------------------------------------------------------------------------
// Token types and markers
//----------------------------------------------------------------------------*/
int is_an_old_lex_marker (char marker, LexemeType *lex_type)
{ switch (marker)
    { case PrefixMark:	   *lex_type = Prefix; break;
      case SuffixMark:	   *lex_type = Suffix; break; 
      case InfixMark:	   *lex_type = Infix; break; 
      case MultiTokenMark: *lex_type = MultiToken; break;
      default: return (0);
    };
  return (1);
}

char lxcn_get_lex_mark (LexemeType lex_type)
{ switch (lex_type)
    { case Prefix:	 return (PrefixMark); break;
      case Infix:	 return (InfixMark); break;
      case Suffix:	 return (SuffixMark); break;
      case MultiToken:	 return (MultiTokenMark); break;
      default:		 return (EmptyMark);
    };
}

static int has_lex_mark (char *lexeme, int *hyph_beg, int *hyph_end)
{ *hyph_beg = 0;
  *hyph_end = 0;
  switch (lexeme[0])
    { case EmptyMark: return (1);
      case PrefixMark: *hyph_end = 1; return (1);
      case InfixMark:  *hyph_beg = 1; *hyph_end = 1; return (1);
      case SuffixMark: *hyph_beg = 1;
      case MultiTokenMark:   return (1);
      default: return (0);
    };
}

void lxcn_print_lexeme (char *lexeme)
{ char *ptr = lexeme;
  int hyph_beg;
  int hyph_end;
  if (has_lex_mark (lexeme, &hyph_beg, &hyph_end)) ptr++;
  if (hyph_beg) abs_printf ("-");
  while (*ptr)
    { switch (*ptr)
	{ case '\n': abs_printf ("\\n"); break;
	  case '\r': abs_printf ("\\r"); break;
	  case '\t': abs_printf ("\\t"); break;
	  case '\\': abs_printf ("\\\\"); break;
	  case '-': abs_printf ("\\-"); break;
	  default:
	    if (('\0' < *ptr) && (*ptr < ' ')) abs_printf ("\\%03o", (unsigned) (*ptr));
	    else abs_printf ("%c", *ptr);
	};
      ptr++;
    };
  if (hyph_end) abs_printf ("-");
}

int lxcn_is_eos (char c)
{ return (c == EosMark);
}

int lxcn_is_blank (char c)
{ return (blank_table[(int) ((unsigned char) c)] != '\0');
}

int lxcn_is_terminator (char c)
{ return (terminator_table[(int) ((unsigned char) c)] != '\0');
}

int lxcn_is_invisible (char c)
{ return (invisible_table[(int) ((unsigned char) c)] != '\0');
}

char lxcn_translate (char c)
{ return (translate_table[(int) ((unsigned char) c)]);
}

void lxcn_init_char_tables (char *blanks, char *terminators, char *invisibles, 
			    char *translate_src, char *translate_dst)
{ set_table (blank_table, blanks);
  set_table (terminator_table, terminators);
  set_table (invisible_table, invisibles);
  set_trans_table (translate_table, translate_src, translate_dst);
  terminator_table[EosMark] = 1;
}
