/*
   File: arts_interpreter.c
   Defines the interpreter for the instructions generated by agfl-coder.

   Copyright 2005-2006 Radboud University of Nijmegen
 
   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.
 
   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU Library General Public License for more details.
 
   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.

   CVS ID: "$Id: arts_interpreter.c,v 1.14 2007/11/21 15:38:37 marcs Exp $"
*/

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif /* HAVE_CONFIG_H */

/* standard includes */
#include <stdio.h>
#include <stdlib.h>
#include <limits.h>
#include <assert.h>
#include <sys/types.h>
#include <string.h>

/* libabase includes */
#include <abase_error.h>
#include <abase_memalloc.h>
#include <abase_mm_alloc.h>

/* liblexicon include */
#include <lexicon.h>

/* libarts includes */
#include "arts_ds.h"
#include "arts_io.h"
#include "arts_lexer.h"
#include "arts_timers.h"
#include "arts_posmemo.h"
#include "arts_interpreter.h"

#ifdef GEN_RTS
#include "rtsgen.h"
#endif /* GEN_RTS */

#ifdef DEBUG_RTS
#define DB_RTS(x) x
#else
#define DB_RTS(x)
#endif /* DEBUG_RTS */

#ifdef DEBUG_INTERPRETER
#  undef DEBUG_TRANSLATE
#  define TRACE_MATCH
#  undef TRACE_LEX_MATCH
#endif /* DEBUG_INTERPRETER */

/*------------------------------------------------------------------------------
// Penalty management
// Penalties are defined to be 32 bit integers
//----------------------------------------------------------------------------*/
static int penalty_change_would_block (Penalty current, Penalty offset)
{ if (offset == MAX_PENALTY) return (1);
  else if ((offset > 0) && (current > MAX_PENALTY - offset)) return (1);
  else if ((offset < 0) && (current < MIN_PENALTY - offset))
    abs_abort ("penalty_change_would_block",
	       "Current parse has gathered too much bonusses to be true");
  return (0);
}

/*------------------------------------------------------------------------------
// Private globals
//----------------------------------------------------------------------------*/
typedef struct
{ int		nr_parses;	/* nr of parses */
  Penalty	penalty;	/* penalty of last parse */
  Position	startpos;	/* start position in current input buffer */
  Position	endpos;		/* end of the text covered by the parse */
} ParseResult;

static Position furthest_pos;
static ParseResult parse_result;

static void reset_parse_result ()
{ parse_result.nr_parses = 0;
  parse_result.penalty = 0;
  parse_result.startpos = 0;
  parse_result.endpos = 0;
}

static void incr_nr_parses ()
{ parse_result.nr_parses++;
}

static int get_nr_parses ()
{ return (parse_result.nr_parses);
}

static Penalty get_penalty ()
{ return (parse_result.penalty);
}

static Position get_parse_startpos ()
{ return (parse_result.startpos);
}

static Position get_parse_endpos ()
{ return (parse_result.endpos);
}

/*------------------------------------------------------------------------------
// Parsing call stack, for diagnostics.
//----------------------------------------------------------------------------*/
#define PARSE_CALL_STACKSIZE	1024
static int parse_call_stack[PARSE_CALL_STACKSIZE];
static int parse_call_stack_pointer;
static int saved_parse_call_stack[PARSE_CALL_STACKSIZE];
static int saved_parse_call_stack_pointer;

static void print_parse_call_stack();
static void print_saved_parse_call_stack();

static void push_parse_call(int ruleno)
{ if (parse_call_stack_pointer < PARSE_CALL_STACKSIZE-1)
    parse_call_stack[parse_call_stack_pointer++] = ruleno;
}

static void pop_parse_call()
{ if (parse_call_stack_pointer > 0)
    parse_call_stack_pointer--;
}

static void save_parse_call_stack()
{ int i;
  saved_parse_call_stack_pointer = parse_call_stack_pointer;
  for (i = 0; i < parse_call_stack_pointer; i++)
    saved_parse_call_stack[i] = parse_call_stack[i];
}

static void print_parse_call_stack()
{ int i;

  for (i = 0; i < parse_call_stack_pointer; i++)
    abs_printf(">%s ", arts_ifd.nonterm_names[parse_call_stack[i]]);
}

static void print_saved_parse_call_stack()
{ int i;

  for (i = 0; i < saved_parse_call_stack_pointer; i++)
    abs_printf(">%s ", arts_ifd.nonterm_names[saved_parse_call_stack[i]]);
}

/*------------------------------------------------------------------------------
// Time limit.
//
// The time-out is checked for each succeeding match of a terminal.
// For efficiency reasons, we only check the clock once every
// MAX_TICKS times. High MAX_TICKS increases performance, but
// reduces timer resolution.
//----------------------------------------------------------------------------*/
static int have_time_limit (void)
{ return (arts_ifd.max_parsetime < INT_MAX);
}

static void set_time_limit (void)
{ set_time_out (arts_ifd.max_parsetime);
}

/*------------------------------------------------------------------------------
// Printing
//----------------------------------------------------------------------------*/
static void print_parsed_sentence (void)
{ if (arts_ifd.lexer_stats_option)
    { int input_linenumber = get_input_linenumber ();
      int input_position = get_input_position ();
      char *input_text = get_input_text ();
      int parse_start = (int) get_parse_startpos ();
      int parse_end = (int) get_parse_endpos ();
      char *parse_start_ptr = input_text + parse_start;
      char *parse_end_ptr = input_text + parse_end;

      /* Print to stdout? */
      printf ("\n# %s %d %d-%d|", arts_ifd.input_fname, input_linenumber,
	      input_position + parse_start, input_position + parse_end);

      /* Print the parsed input text, replacing any newlines with space.
       * This is because it must all be on the one comment line. */
      while (parse_start_ptr < parse_end_ptr)
	{ if (*parse_start_ptr == '\n') putchar(' ');
	  else putchar(*parse_start_ptr);
	  parse_start_ptr++;
	}
      putchar('\n');
    };
}

static void start_printing (void)
{ if (arts_ifd.parsing_stats_option)
    { int nr_parses = get_nr_parses ();
      Penalty penalty = get_penalty ();
      Time parse_time = get_parse_time ();

      current_parse_printf ("# parsing %d", nr_parses);
      current_parse_printf (" time %.3f", parse_time);
      current_parse_printf (" penalty %d\n", penalty);
    };
}

static void stop_printing (void)
{ if (!arts_ifd.no_output) 
    current_parse_add_char ('\n');
}

static void show_totals (Trellis *trellis)
{ int nr_parses = get_nr_parses ();

  if (arts_ifd.total_stats_option)
    { Time scan_time = get_total_scan_time ();
      Time parse_time = get_total_parse_time ();
      Time print_time = get_total_print_time ();
      Time total_time = scan_time + parse_time + print_time;

      abs_printf ("# parsings %d/", nr_parses);
      if (arts_ifd.max_parses == INT_MAX) abs_printf ("unlimited");
      else abs_printf ("%d", arts_ifd.max_parses);
      abs_printf (" scan %.3f parse %.3f print %.3f total %.3f time\n",
		  scan_time, parse_time, print_time, total_time);
    };
  if (arts_ifd.research_stats_option)
    { abs_printf ("# reached input position %d (", furthest_pos);
      print_saved_parse_call_stack();
      abs_printf (")\n");
    };

#ifdef DEBUG_NEGMEMO
  if (arts_ifd.neg_memo_option)
    print_negmemo_table (trellis, 1);
#endif /* DEBUG_NEGMEMO */

#if defined(DEBUG_POSMEMO)
  if (arts_ifd.pos_memo_option)
    { StateNode** state_row = GET_TRELLIS_STATE_ROW(trellis);
      StateNode* state = *state_row;

      /* XXX abstract this knowledge */
      int root_rule_nr = arts_ifd.nr_lexicon_nonterminals;
      PosMemo root_rule_at_start = posmemo_get_prod_ptr (state, root_rule_nr);

      /* XXX hack! sometimes == 1, that's probably a bug... */
      if ((intptr_t) root_rule_at_start > 0x10000)
	posmemo_rdump_pmprod (root_rule_at_start, 0);
      else
	posmemo_dump_table (trellis);
    };
#endif /* DEBUG_POSMEMO */

#ifdef PROFILING
  if ((prof_count > 0) && reset_profile && !no_output)
    ShowProfile ("sentence");
#endif
}

/*------------------------------------------------------------------------------
// Stack frame layout.
//      The interpreter uses two different stack layouts: one for the first
//      pass, and one for the second pass.
//
//      The first pass frame consists of:
//	  - constants pushed by the CALL instruction (the elements with a
//	    positive offset);
//	  - a fixed number of entries filled in by the CREATE instruction (the
//	    elements with a negative offset);
//	  - zero or more formals;
//	  - zero or more locals;
//	  - zero or more entries for pointers to the sons of this rule.
//      The formals are untouched by the CREATE instruction.
//      The locals and sons are cleared to 0 by the CREATE instruction.
//
//      The second pass frame consists of:
//	  - constants pushed by the PRINT_SON instruction
//	  - parameters for the second pass function.
//
//      Both a stack pointer and a frame pointer are used. The execution model
//      allows to continue execution in another frame.
//
//	Note that the SP grows from high to low memory,
//	always pointing to the first free location
//----------------------------------------------------------------------------*/
#define STACK_SIZE		(128 * 1024)
static DATA* stack = 0;

#define PASS2_FRAME_HEAD_LEN	(3)     /* length of the second pass frame * head */
#define PASS2_PASS2FP		(3)     /* previous second pass frame pointer */
#define PASS2_FP		(2)     /* previous rule frame pointer */
#define PASS2_RET_ADDR		(1)     /* return address */

#define FRAME_HEAD_LEN		(3)     /* length of the frame head */
#define NONT_NR_OFF		(3)
#define SUCC_ADDR_OFF		(2)     /* success continuation address */
#define FAIL_ADDR_OFF		(1)     /* fail continuation address */
#define OLD_FP			(0)     /* pointer to parent frame */
#define NR_FORMALS_OFF		(-1)
#define NR_LOCALS_OFF		(-2)
#define NR_SONS_OFF		(-3)
#define PASS2_CODE		(-4)    /* second pass code pointer */
#define ISTATE			(-5)    /* input state at start of rule */
#define NR_SUCCS_OFF		(-6)    /* nr of successes */
#define PMPRODPTR		(-7)    /* pointer to current positive memo production */
#define FAIL_OFF		(-8)	/* points to FAIL instruction */
#define NR_SONS_DONE		(-9)    /* nr of sons which have been called */
#define PENORIG			(-10)   /* penalty level at start of rule */
#define VAR_OFFSET		(11)    /* offset for variable frame part */

#define NR_FORMALS		(fp[NR_FORMALS_OFF].arg)
#define NR_LOCALS		(fp[NR_LOCALS_OFF].arg)
#define VARIABLE(nr)		(fp[-(long)(VAR_OFFSET + (nr))])

#define NR_SONS			(fp[NR_SONS_OFF].arg)
#define NONT_NR			(fp[NONT_NR_OFF].arg)
#define SON_FP(nr)		(fp[-(long)(VAR_OFFSET + NR_FORMALS + NR_LOCALS + (nr))].data)

#define START_INPUT_STATE	(fp[ISTATE].input_state)

/*------------------------------------------------------------------------------
// Accessing the negative memo values.
// Macro:
//	NEG_MEMO(no)
// Description:
//	Convert memo with number to value of memo.
//----------------------------------------------------------------------------*/
#define NEG_MEMO(no)		(((fp[ISTATE].input_state)->neg_memos)[no])

#define negmemo_is_unknown(nr)		(NEG_MEMO(nr) == NEGMEMO_UNKNOWN)
#define negmemo_is_blocked(nr,pen)	((NEG_MEMO(nr) != NEGMEMO_UNKNOWN)&&(pen >= NEG_MEMO(nr)))
#define negmemo_set_blocked(nr,pen)	(NEG_MEMO(nr) = pen)
#define negmemo_set_succeeded(nr)	(NEG_MEMO(nr) = NEGMEMO_SUCCESS)

static int memo_enabled (long rule_nr)
{ return (arts_ifd.memo_enable_table[rule_nr] != 0);
}

/*
   The following routine is for debugging and statistical purposes:
*/
void print_negmemo_table (Trellis *trellis, int skip_unknown)
{   StateNode** state_row = GET_TRELLIS_STATE_ROW(trellis);
    int node_nr;
    int alt_nr;
    int* empty_rule;
    int* empty_node;

    long** overview = (long**) abs_calloc (trellis -> length, sizeof (long*),
					   "print_negmemo_table: overview[]");

    abs_message ("print_negmemo_table: there are %d negative memos.", arts_ifd.nr_neg_memos);

    /* Build the table: */
    for (node_nr = 0; node_nr < trellis -> length; node_nr++)
      { StateNode* state = *state_row++;

	if(!state) overview[node_nr] = NULL;
	else
	  { NegMemo* neg_memo_vec = state -> neg_memos;

	    if (!neg_memo_vec) overview[node_nr] = NULL;
	    else
	      { overview[node_nr] = (long*) abs_calloc (arts_ifd.nr_neg_memos, sizeof(long),
							"print_negmemo_table: overview[][]");

	        for (alt_nr = 0; alt_nr < arts_ifd.nr_neg_memos; alt_nr++)
	           overview[node_nr][alt_nr] = neg_memo_vec[alt_nr];
	      };
	  };
      };

    /* printed table compression */
    empty_rule = (int*) abs_calloc (arts_ifd.nr_neg_memos, sizeof(int),
				    "print_negmemo_table: empty_rule");
    for (alt_nr = 0; alt_nr < arts_ifd.nr_neg_memos; alt_nr++)
      { empty_rule[alt_nr] = 1;
	node_nr = 0;

	while ((node_nr < trellis -> length) && (empty_rule[alt_nr]))
	  { if (overview[node_nr])
	      { if (overview[node_nr][alt_nr] == NEGMEMO_UNKNOWN)
		  empty_rule[alt_nr] = skip_unknown;
	        else empty_rule[alt_nr] = 0;
	      };
	    node_nr++;
	  };
      };
    empty_node = (int*) abs_calloc (trellis -> length, sizeof(int),
				    "print_negmemo_table: empty_node");
    for (node_nr = 0; node_nr < trellis->length; node_nr++)
      { empty_node[node_nr] = 1;
	alt_nr = 0;

	while ((alt_nr < arts_ifd.nr_neg_memos) && (empty_node[node_nr]) && (overview[node_nr]))
	  { switch (overview[node_nr][alt_nr])
	      { case NEGMEMO_UNKNOWN:	empty_node[node_nr] = skip_unknown; break;
	        case NEGMEMO_BLOCKED:	empty_node[node_nr] = 0; break;
	        default: empty_node[node_nr] = !overview[node_nr][alt_nr];
	      };
	    alt_nr++;
	  };
      };

    /* actually print it: */
    /* first the table */
    for (alt_nr = 0; alt_nr < arts_ifd.nr_neg_memos; alt_nr++)
      if (!empty_rule[alt_nr])
	{ abs_printf ("%3d|", alt_nr);
	  for (node_nr = 0; node_nr < trellis -> length; node_nr++)
	    if (!empty_node[node_nr])
	      { switch (overview[node_nr][alt_nr])
		  { case NEGMEMO_BLOCKED: abs_printf("          b"); break;
	            case NEGMEMO_UNKNOWN: abs_printf("          u"); break;
	            default: abs_printf(" %10ld", overview[node_nr][alt_nr]);
	          };
	      };
	  abs_printf("\n");
	};

    /* then a neat line below it */
    abs_printf("---+");
    for (node_nr = 0; node_nr < trellis -> length; node_nr++)
      if (!empty_node[node_nr]) abs_printf ("-----------");

    /* and of course the numbers */
    abs_printf("\n   |");
    for (node_nr = 0; node_nr < trellis -> length; node_nr++)
      if (!empty_node[node_nr]) abs_printf (" %10d", node_nr);
    abs_printf ("\n");

    /* free the space: */
    for (node_nr = 0; node_nr < trellis -> length; node_nr++)
      if (overview[node_nr])
	abs_free (overview[node_nr], "print_negmemo_table: overview[][]");

    abs_free (overview, "print_negmemo_table: overview[]");
    abs_free (empty_rule, "print_negmemo_table: empty_rule");
    abs_free (empty_node, "print_negmemo_table: empty_node");
}

static int state_get_endpos (StateNode *s, Trellis *t)
{ if (s == NULL)
    s = t -> states_row[t->length - 1];
  return (STATE_POS(s));
}


/*------------------------------------------------------------------------------
// Maintain a stack of ambiguous node during pass2 to keep track of the
// different printings of the parse forest.
//----------------------------------------------------------------------------*/
static PosMemo *ambi_stack;
static int ambi_size;
static int ambi_sptr;
static void init_ambi_stack ()
{ ambi_size = 64;
  ambi_sptr = 0;
  ambi_stack = abs_calloc (ambi_size, sizeof (PosMemo *), "init_ambi_stack");
}

static void push_ambi_stack (PosMemo prime)
{ if (ambi_sptr == ambi_size)
    { ambi_size *= 2;
      ambi_stack = abs_realloc (ambi_stack, ambi_size * sizeof (PosMemo *), "push_ambi_stack");
    };
  ambi_stack[ambi_sptr++] = prime;
  ambi_stack[ambi_sptr++] = prime;
}

static int try_lookup_sibling_in_ambi_stack (PosMemo pmptr, PosMemo *sibling)
{ PosMemo prime = pmptr -> prime;
  int ix;
  for (ix = 0; ix < ambi_sptr; ix += 2)
    if (ambi_stack[ix] == prime)
      { *sibling = ambi_stack [ix + 1];
	return (1);
      };
  return (0);
}

static int try_update_ambi_stack ()
{ while (ambi_sptr > 0)
    { PosMemo current_top = ambi_stack[ambi_sptr - 1];
      if (current_top -> equiv != NULL)
	{ /* We have another equivalent parse */
	  ambi_stack[ambi_sptr - 1] = current_top -> equiv;
	  return (1);
	};
      ambi_sptr -= 2;
    };
  return (0);
}

static void finish_ambi_stack ()
{ abs_free (ambi_stack, "finish_ambi_stack");
}

/*------------------------------------------------------------------------------
// Check to see if a pointer points in the agfl interpreter stack
//----------------------------------------------------------------------------*/
int pointsIntoStack (void *p) 
{
    return ((p >= (void *) stack) && (p < (void *) (stack + STACK_SIZE)));
}

/*------------------------------------------------------------------------------
// Macros to push and pop to and from the stack
// Note that the SP grows from high to low memory
// always pointing to the first free location
//----------------------------------------------------------------------------*/
#define PUSH_VALUE(x)	\
{	                \
    sp -> val = x;	\
    sp--;               \
    assert(sp > 0);	\
}

#define PUSH_ADDR(x)	\
{	                \
    sp -> code = x;	\
    sp--;	        \
    assert(sp > 0);	\
}

#define POP_VALUE(x)    \
{	                \
    sp++;	        \
    x = sp[0].val;	\
}

#define POP_ADDR(x)	\
{	                \
    sp++;               \
    x = sp[0].code;     \
}

/*------------------------------------------------------------------------------
// The type of an execution function should be a function with zero
// parameters returning an execution function (to be able to continue
// with the returned function). Unfortunately the C compiler (and
// indeed the C99 standard) does not permit function types being
// returned in a function type. Hence we must resort to type casting (sigh..)
//
// Forward declare all exec routines
//----------------------------------------------------------------------------*/
typedef void *vptr;
typedef vptr (*exec_proc)(void);
static void exec_illegal ();
static vptr exec_nop (void);
static vptr exec_create (void);
static vptr exec_cont (void);
static vptr exec_fail (void);
static vptr exec_call (void);
static vptr exec_ucall (void);
static vptr exec_clear (void);
static vptr exec_init_s (void);
#define exec_init_i exec_init_s
#define exec_init_t exec_init_s
static vptr exec_init_v (void);
static vptr exec_adjust_i (void);
static vptr exec_adjust_s (void);
static vptr exec_adjust_t (void);
static vptr exec_adjust_v (void);
static vptr exec_uadjust_s (void);
#define exec_uadjust_i exec_uadjust_s
#define exec_uadjust_t exec_uadjust_s
static vptr exec_uadjust_v (void);
static vptr exec_restrict_i (void);
static vptr exec_restrict_s (void);
static vptr exec_restrict_t (void);
static vptr exec_restrict_v (void);
static vptr exec_urestrict_s (void);
#define exec_urestrict_i exec_urestrict_s
#define exec_urestrict_t exec_urestrict_s
static vptr exec_urestrict_v (void);
static vptr exec_match (void);
#define exec_match_re exec_match
#define exec_skip_re exec_match
#define exec_match_other exec_match
static vptr exec_lex_match (void);
static vptr exec_umatch (void);
#define exec_umatch_re exec_umatch
#define exec_umatch_other exec_umatch
static vptr exec_lex_umatch (void);
static vptr exec_input_pos (void);
static vptr exec_uinput_pos (void);
static vptr exec_penalty (void);
static vptr exec_upenalty (void);
static vptr exec_commit (void);
static vptr exec_success (void);
static vptr exec_usuccess (void);
static vptr exec_end (void);
static vptr exec_pass2 (void);
static vptr exec_retrn (void);
static vptr exec_push_f (void);
static vptr exec_push_l (void);
static vptr exec_print_son (void);
static vptr exec_print_abeg (void);
static vptr exec_print_aend (void);
static vptr exec_print_pbeg (void);
static vptr exec_print_psep (void);
static vptr exec_print_pend (void);
static vptr exec_print_val (void);
static vptr exec_print_term (void);
static vptr exec_print_lex (void);
static vptr exec_print_re (void);
#define exec_print_other exec_print_re
static vptr exec_print_pos (void);
static vptr exec_print_pen (void);
static vptr exec_print_n (void);
static vptr exec_print_il (void);
static vptr exec_print_sl (void);
static vptr exec_print_tl (void);
static vptr exec_print_vl (void);
static vptr exec_print_if (void);
static vptr exec_print_sf (void);
static vptr exec_print_tf (void);
static vptr exec_print_vf (void);
static vptr exec_tmemo (void);
static vptr exec_tsmemo (void);
static vptr exec_smemo_s (void);
static vptr exec_root_call (void);
static vptr exec_root_create (void);
static vptr exec_mark_lrec (void);
static vptr exec_umark_lrec (void);
static vptr exec_trace (void);
static vptr exec_position (void);
static vptr exec_choice (void);
static vptr exec_uchoice (void);
static vptr exec_alt_marker (void);
static vptr exec_ualt_marker (void);

/*------------------------------------------------------------------------------
// Include the table for the instruction dispatcher
// If there is any discrepancy between the generated table and
// the execution functions declared above, the compiler should complain
//----------------------------------------------------------------------------*/
#include <opcode.exc>


/*------------------------------------------------------------------------------
// Some instruction may continue their execution in other functions
// Forward declare them as well
//----------------------------------------------------------------------------*/
static vptr pm_generate_production ();
static vptr search_match ();
static vptr match_again ();
static vptr lex_match_again ();

/*------------------------------------------------------------------------------
// When counting we need to keep track of the number
// of times, an opcode is executed
//----------------------------------------------------------------------------*/
static int opcode_count[nr_of_opcodes];
static void initialize_counters ()
{ int ix;
  for (ix = 0; ix < nr_of_opcodes; ix++)
    opcode_count[ix] = 0;
}
#define opc_count(ix) { if (arts_ifd.counters_option) opcode_count[ix & 0xff]++; }

/*------------------------------------------------------------------------------
// Define macro's to debug the execution of the interpreter
//----------------------------------------------------------------------------*/
#if defined(DEBUG_INTERPRETER)
#define logexec(x) { \
    abs_message ("logexec(%c): %s pc=%p fp=%p sp=%p state=%p", \
	         'A', all_opcode_execs[x].opcode_name, pc, fp, sp, i_state);\
}

#define logfail(x) { \
    abs_message ("logfail(%c): %s\n", \
	         'A', all_opcode_execs[x].opcode_name); \
}
#else /* NOT DEBUG_INTERPRETER */
#define logexec(x)
#define logfail(x)
#endif /* DEBUG_INTERPRETER */

/*------------------------------------------------------------------------------
// Interpreter registers.
//
//	pc 		program counter; contains address of current instruction.
//
//	next_pc 	new program counter; contains address of next instruction.
//
//	sp 		stack pointer; points to next free position on stack
//
//	fp		frame pointer; points to current stack frame.
//
//	pass2_fp 	Pass 2 frame pointer
//
//	i_state 	state pointer pointing to current input state
//
//	i_transition 	points to current transition to be matched
//			(only used during LEX_MATCH/LEX_UMATCH)
//
//	curr_penalty 	the current penalty level
//
//	curr_terminal	the current matching terminal
//
//	indent 		Current indentation to print the parse tree
//
//	status 		Current status of the abstract machine interpreter
//
//	trellis		Current trellis to parse
//
//	?pmptr		Current posmemo pointer
//
//	?return_to_pass2 Pass 2 administration
//----------------------------------------------------------------------------*/
#define STOPPING 0
#define EXECUTING 1
static CODE *pc;
static CODE *next_pc;
static DATA *sp;
static DATA *fp;
static DATA *pass2_fp;
static PosMemo pmptr;
static int return_to_pass2;
static StateIndicator i_state;
static Transition *i_transition;
static Penalty curr_penalty;
static ARG curr_terminal;
static Trellis *curr_trellis;
static int indent;
static int status;

/*------------------------------------------------------------------------------
// Define macros for generative execution
//----------------------------------------------------------------------------*/
#ifdef GEN_RTS
static int chosen_freq;
static int alt_depth;

#ifdef DEBUG_GENERATOR
#define ALT_DEPTH_RESET (alt_depth = 0)
#define ALT_DEPTH_CHECK (assert(alt_depth >= 0))
#define ALT_DEPTH_DECREASE \
{	                                                                 \
    alt_depth--;	                                                  \
    abs_message ("ALT_DEPTH_DECREASE: alt_depth now %d",alt_depth);    \
}
#define ALT_DEPTH_INCREASE \
{	                                                                   \
    alt_depth++;	                                                    \
    abs_message ("ALT_DEPTH_INCREASE: alt_depth now %d",alt_depth);    \
}
#else /* DEBUG_GENERATOR */
#define ALT_DEPTH_RESET (alt_depth = 0)
#define ALT_DEPTH_CHECK (assert(alt_depth >= 0))
#define ALT_DEPTH_DECREASE (alt_depth--)
#define ALT_DEPTH_INCREASE (alt_depth++)
#endif /* DEBUG_GENERATOR */
#endif /* GEN_RTS */

/*------------------------------------------------------------------------------
// Define macros to support decoding and execution
// Note that CONT provides a continuation of the current instruction
//----------------------------------------------------------------------------*/
#define RECALC_NEXT_PC next_pc = pc + ((((pc -> ival) >> 16) & 0xffff) + 1)
#define NEXT return ((vptr) NULL)
#define GOTO(x) { next_pc = x; return ((vptr) NULL); }
#define CONT(x) return ((vptr) x)

static StateIndicator agfl_interpret (Trellis *trellis, StateIndicator start_state)
{   /*------------------------------------------------------------------------------
    // Interpret the compiled grammar.
    //
    // Description:
    //	Set up the interpreters registers and jump to the first instruction.
    //
    // Note:
    //	The interpreter should have been initialized and reset.
    //----------------------------------------------------------------------------*/
    StateIndicator next_start_state;

    /*----------------------------------
    // Initialize interpreter registers
    //--------------------------------*/
    return_to_pass2 = 0;
    i_transition = NULL;
    curr_penalty = NO_PENALTY;
    curr_trellis = trellis;
    curr_terminal = 0;
    stack = abs_calloc (STACK_SIZE, sizeof(cel), "agfl_interpret: stack");
    fp = stack + STACK_SIZE - 1 - FRAME_HEAD_LEN;
    pc = arts_ifd.code;			/* Start of code */
    pass2_fp = 0;
    pmptr = NULL;
    indent = 0;
    sp = fp;

#ifndef GEN_RTS
    if (arts_ifd.segment_mode && (start_state != NULL)) i_state = start_state;
    else i_state = GET_FIRST_STATE_INDICATOR(trellis);
#else /* GEN_RTS */
    chosen_freq = 0;
    alt_depth = 0;
    i_state = NULL;
    choice_admin_init ();
    chosen_freq = 0;
    ALT_DEPTH_RESET;
#endif /* GEN_RTS */

    /*
       Here we enter the main dispatcher loop
    */
    status = EXECUTING;
    start_parse_time ();
    while (status == EXECUTING)
      { /* Decode the instruction */
	int opcode = pc -> ival;
	int nr_of_opnds = (opcode >> 16) & 0xffff;
	exec_proc execute;
	next_pc = pc + (nr_of_opnds + 1);
	opcode &= 0xff;
	execute = all_opcode_execs[opcode].opcode_exec;

	/* Update the statistics for this instruction and possibly log it */
	logexec (opcode);
	opc_count (opcode);

	/* Execute the decoded instruction */
	do
	   { vptr result = execute ();
	     execute = (exec_proc) result;
	   }
	while (execute != NULL);
	pc = next_pc;
      };

    /*
       We finished executing in the main loop
       Finalize the current administration and locate the next start state
    */
#ifndef GEN_RTS
    stop_parse_time ();
    DB_RTS (abs_message ("END: current time is %.3fs.", get_parse_time()));
    print_parsed_sentence ();
    parse_results_dump ();
    show_totals (curr_trellis);

    /* exit interpreter */
    if (!arts_ifd.segment_mode)
      { /* Not in paragraph-mode, so stop. */
	next_start_state = NULL;
      }
    else
      { /* Paragraph mode, check if we reached the time limit */
	if (have_time_limit() && have_time_out())
	  { /* we have reached the time limit, see what we can do now...: */
	    if (parse_results_get_nr_parses ())
    	      { /* we found at least one parse, so start after that one */
	        next_start_state = get_statenode_from_first_parse ();
	        if (next_start_state && state_has_eos_transition (next_start_state))
	          /* we are at the end of the paragraph, so we may stop */
	          next_start_state = NULL;
	      }
    	    else
	      /* we didn't even find 1 single parsing, bail out */
	      next_start_state = get_shortest_transition (curr_trellis, START_INPUT_STATE);
	  }
	else
	  { /* We found a good segment, so start after it the next time: */
	    next_start_state = get_statenode_from_first_parse ();
	    if (!next_start_state)
	      next_start_state = get_shortest_transition (curr_trellis, START_INPUT_STATE);
	    else if (state_has_eos_transition (next_start_state))
	      /* we are at the end of the sentence, so we can stop! */
	      next_start_state = NULL;
	  };
      };

    parse_results_destroy ();
#else /* GEN_RTS */
#ifdef DEBUG_RTS
    choice_admin_dump_table ();
#endif /* DEBUG_RTS */
    choice_admin_destroy ();
#endif /* GEN_RTS */

    abs_free (stack, "agfl_interpret: stack");
    return (next_start_state);
}

/*------------------------------------------------------------------------------
// Implementation of abstract instructions.
//----------------------------------------------------------------------------*/
static vptr exec_nop ()
{   /* Continue with next instruction */
    NEXT;
}

static vptr exec_cont ()
    { DB_RTS (abs_message ("CONT: curr_penalty = %d", curr_penalty));
      if (!memo_enabled (NONT_NR))
	{ unsigned ix = NR_FORMALS;
	  CODE* routine = (arts_ifd.transduce_option)?pc[1].code:pc[2].code;

	  fp[NR_SUCCS_OFF].arg++;               /* for commit: parsing found */

	  sp -= 2;
	  sp[2].code = next_pc;
	  sp[1].data = fp;

	  /* push result parameters */
	  while (ix > 0)
	    { ix--;
	      PUSH_VALUE (VARIABLE(ix).val);
	      DB_RTS (abs_message ("pushing value 0x%lx", VARIABLE(ix).val.set_par));
	    }

	  fp[PASS2_CODE].code = routine;
	  DB_RTS(abs_message ("2nd pass routine @%p", routine));

	  /*
	     continue with the continuation which assumes
	     execution in the previous stack frame:
	  */
	  next_pc = fp[SUCC_ADDR_OFF].code;
	  fp = fp[OLD_FP].data;
	  DB_RTS (abs_message ("CONTinuing into \"%s\"",
			       arts_ifd.nonterm_names[NONT_NR]));
	}
      else /* The usual case for posmemo code */
	{ /* Select whether to print the parse tree or to transduce */
	  CODE* routine = (arts_ifd.transduce_option)?pc[1].code:pc[2].code;
	  fp[NR_SUCCS_OFF].arg++;               /* for commit: parsing found */

	  /* Administer we succeeded up to this point */
	  posmemo_add_production (START_INPUT_STATE, NONT_NR,
				  curr_penalty - fp[PENORIG].penalty,
	                          NR_FORMALS, NR_LOCALS, fp[NR_SONS_DONE].arg,
				  &(VARIABLE(0).val), i_state, routine);
	};

      /* Continue with next instruction */
      NEXT;
    }

/*-------------------------------------------------------------------------
// Instruction: CREATE (nr formals, nr locals, nr sons, fail)
// Description: Setup a stack frame 
//	        Note that the frame header has already been setup by CALL
//-----------------------------------------------------------------------*/
static vptr exec_create ()
    { ARG nr_formals = pc[1].arg;
      ARG nr_locals = pc[2].arg;
      ARG nr_sons = pc[3].arg;
      CODE *fail_addr = pc[4].code;
      int total_alloc = nr_formals + nr_locals + nr_sons;

      /* Link and allocate frame */
      sp[0].data = fp;                /* store old fp */
      fp = sp;
      sp -= total_alloc + VAR_OFFSET;

      /* Setup variable administration */
      NR_FORMALS = nr_formals;
      NR_LOCALS = nr_locals;
      NR_SONS = nr_sons;

      DB_RTS (abs_message ("CREATE(%ld, %ld, %ld, FAIL)", NR_FORMALS, NR_LOCALS, NR_SONS));

      /* Initialize remainder of frame */
      START_INPUT_STATE = i_state;
      fp[FAIL_OFF].code = fail_addr;
      fp[NR_SUCCS_OFF].arg = 0;
      fp[NR_SONS_DONE].arg = 0;
      fp[PENORIG].penalty = curr_penalty;

      DB_RTS (abs_message ("CREATE: start_input_state = %p, nont nr = %ld, penalty = %d",
			   i_state, NONT_NR, curr_penalty));

      /*
	 Initialise the locals and sons to 0, since otherwise equal_posmemo()
	 may look at uninitialised values sometimes (and fail to detect
	 identity). This happens because not every alternative uses (sets) the
	 full set of locals and sons so some may remain uninitialised.
      */
      if (memo_enabled (NONT_NR))
	fp[PMPRODPTR].pmprod = posmemo_get_prod_ptr (i_state, NONT_NR);

      /*
	 If the positive memo is known for this input state and not blocked,
	 we continue through the fail continuation
      */
      if (memo_enabled (NONT_NR))
	{ if (posmemo_is_known (i_state, NONT_NR) && posmemo_is_unblocked (i_state, NONT_NR))
	    { // CODE* failcont = posmemo_get_failcont (fp[PMPRODPTR].pmprod);
	      DB_RTS (abs_message ("CREATE: known & unblocked"));
	      fp[PMPRODPTR].pmprod = NULL;
	      GOTO (fail_addr);
	    }
	  else fp[PMPRODPTR].pmprod = NULL;
	};

      /* Continue with next instruction */
      NEXT;
    }

/*-------------------------------------------------------------------------
// Instruction: FAIL
// Description: current rule is done; fail if there are no successful
//	        productions, otherwise generate those productions
//	        by continuing this instruction in pm_generate_production.
//-----------------------------------------------------------------------*/
static vptr exec_fail ()
    { /* Check if we have run out of time */
      if (have_time_limit () && have_time_out ())
	/* Stop executing */
	return (exec_end ());

      i_state = START_INPUT_STATE;

      DB_RTS (abs_message ("FAIL in \"%s\"...", arts_ifd.nonterm_names[NONT_NR]);
	      abs_message ("  into \"%s\"",
			   arts_ifd.nonterm_names[(fp[OLD_FP].data)[NONT_NR_OFF].arg]);
	      abs_message ("FAIL: start_input_state = %p", i_state));
      if (memo_enabled (NONT_NR))
	{ if (!posmemo_is_blocked (i_state, NONT_NR))
	    { Penalty pen_gain = MAX_PENALTY;
	      DB_RTS (abs_message ("FAIL: \"%s\" is not blocked",
				   arts_ifd.nonterm_names[NONT_NR]));

	      /* Check if this is the first time that we generate productions */
	      if (fp[PMPRODPTR].pmprod == NULL)
		{ /* Fill posmemo production pointer with first entry */
		  fp[PMPRODPTR].pmprod = posmemo_get_prod_ptr (i_state, NONT_NR);

		  /* If we get here, there must be a production to try */
		  assert(fp[PMPRODPTR].pmprod);
		}
	      else
		{ /* Restore the penalty level and get the next posmemo production pointer */
		  curr_penalty -= posmemo_get_penalty (fp[PMPRODPTR].pmprod);
		  fp[PMPRODPTR].pmprod = posmemo_get_next_prod (fp[PMPRODPTR].pmprod);
		};

	      /* Fetch the penalty gain */
	      if (fp[PMPRODPTR].pmprod != NULL)
		pen_gain = posmemo_get_penalty (fp[PMPRODPTR].pmprod);

	      /* If we would block on the new penalty gain, we must try the next production */
	      while ((fp[PMPRODPTR].pmprod != NULL) &&
		     penalty_change_would_block (curr_penalty, pen_gain))
		{ fp[PMPRODPTR].pmprod = posmemo_get_next_prod (fp[PMPRODPTR].pmprod);
		  if (fp[PMPRODPTR].pmprod != NULL)
		    pen_gain = posmemo_get_penalty (fp[PMPRODPTR].pmprod);
		};

	      /* If we have a valid production pointer, we generate the production */
	      if (fp[PMPRODPTR].pmprod != NULL)
		{ DB_RTS(abs_message ("FAIL: productions left, generating..."));
	          CONT (pm_generate_production);
		}
	      else DB_RTS(abs_message ("FAIL: no more productions left, failing..."));
	    }
	  else DB_RTS (abs_message ("FAIL: \"%s\" is blocked", arts_ifd.nonterm_names[NONT_NR]));
	};

      /* should not be needed, but just to be sure: */
      curr_penalty = fp[PENORIG].penalty;

      /* Pick up fail address from stack and restore sp and fp */
      next_pc = fp[FAIL_ADDR_OFF].code; 
      sp = fp + FRAME_HEAD_LEN;
      pop_parse_call();
      fp = fp[OLD_FP].data;
      fp[NR_SONS_DONE].arg--;
      NEXT;
    }

/*-------------------------------------------------------------------------
// Generate (more) productions
// Description: push formals to stack, set input state and penalty level,
// 	        and continue with success continuation.
//-----------------------------------------------------------------------*/
static vptr pm_generate_production ()
    { unsigned nr_formals = NR_FORMALS;
      void *memod_formals = posmemo_get_formal_ptr (fp[PMPRODPTR].pmprod);

      /* Save old PC and FP */
      sp -= 2;
      sp[2].code = pc;
      sp[1].data = fp; 

      /* push result parameters: check for 64 bit longs */
      sp -= nr_formals;
      memcpy (sp + 1, memod_formals, nr_formals * sizeof (Value));

      /* It is no longer possible to fail for a penalty increase */
      i_state = posmemo_get_input_state (fp[PMPRODPTR].pmprod);
      curr_penalty += posmemo_get_penalty (fp[PMPRODPTR].pmprod);
      DB_RTS (abs_message("pm_generate_production: i_state = %p", i_state));

      /*
	 Continue with the continuation which assumes
	 execution in the previous stack frame:
      */
      next_pc = fp[SUCC_ADDR_OFF].code;
      fp = fp[OLD_FP].data;
      NEXT;
    }

/*---------------------------------------------------------------------------
// Instruction: CALL (syntax rule number, start point, fail continuation)
// Description: Call to a rule by building the pre-call stack part and jump
//	        to the starting point.
//
// With positive memoization extra checks are done: we check if we already
// know the productions and if so, jump to the appropriate FAIL instruction
// to produce the results. If blocked (no productions), we just fail.
// If the positive memoization is still unknown, we enter the rule.
//-------------------------------------------------------------------------*/
static vptr exec_call ()
    { /* Pick up arguments */
      ARG rule_nr = pc[1].arg;
      CODE *jmp_point = pc[2].code;
      CODE *fail_addr = pc[3].code;
      CODE *succ_addr = next_pc;

      DB_RTS (abs_message ("CALLing \"%s\" from \"%s\"",
      		           arts_ifd.nonterm_names[rule_nr], arts_ifd.nonterm_names[NONT_NR]));
      if (have_time_limit() && have_time_out())
	/* Stop processing */
	return (exec_end ());

      /* Check if this rule is blocked for this position for left recursion */
      if (i_state -> lrec_markers[rule_nr])
	{ DB_RTS (abs_message ("CALL: \"%s\" is blocked for left recursion",
			       arts_ifd.nonterm_names[rule_nr]));
	  GOTO (fail_addr);
	}

      /* Check the positive memo for this position */
      else if (memo_enabled (rule_nr))
	{ if (posmemo_is_blocked (i_state, rule_nr))
	    { DB_RTS (abs_message ("CALL: \"%s\" is blocked", arts_ifd.nonterm_names[rule_nr]));
	      GOTO (fail_addr);
	    }
	  else if (posmemo_is_known (i_state, rule_nr))
	    DB_RTS (abs_message ("CALL: \"%s\" is known", arts_ifd.nonterm_names[rule_nr]));
	  else
	    { /* positive memo is unknown */
	      posmemo_set_blocked (i_state, rule_nr);
	      DB_RTS (abs_message ("CALL: \"%s\" is unknown, changing to blocked",
				   arts_ifd.nonterm_names[rule_nr]));
	    };
	};

      /* Construct precall frame part */
      sp -= FRAME_HEAD_LEN;
      sp[FAIL_ADDR_OFF].code = fail_addr;
      sp[SUCC_ADDR_OFF].code = succ_addr;
      sp[NONT_NR_OFF].arg = rule_nr;

      push_parse_call (rule_nr);

      assert (fp[NR_SONS_DONE].arg >= 0);
      assert (fp[NR_SONS_DONE].arg < NR_SONS);

      DB_RTS (abs_message ("sons done: %ld, son_fp = %p",
			   fp[NR_SONS_DONE].arg, fp + fp[NR_SONS_DONE].arg));
      SON_FP(fp[NR_SONS_DONE].arg) = sp;
      fp[NR_SONS_DONE].arg++;

      /* Jump to called address */
      GOTO (jmp_point);
    }
 
/*---------------------------------------------------------------------------
// Instruction: UCALL(nr formals)
// Description: clean up previous result parameters from the stack and 
//	        jump to the continuation of the child rule.
//-------------------------------------------------------------------------*/
static vptr exec_ucall ()
    { ARG nr_formals = pc[1].arg;
      sp += nr_formals;
      next_pc = sp[2].code;
      fp = sp[1].data;
      sp += 2;

      DB_RTS (abs_message ("UCALL(%ld) into rule %ld", nr_formals, NONT_NR));

      /* Continue with next instruction */
      NEXT;
    }

/*------------------------------------------------------------------------------
// Instructions on variables:
// Note: Instructions that implement predicates are also added here
//----------------------------------------------------------------------------*/

/*------------------------------------------------------------------------------
// Instruction: CLEAR (nr)
// Description: Clear the frame at position 0..nr-1 to initialize
//		all formals and locals to 0.
//----------------------------------------------------------------------------*/
static vptr exec_clear ()
    { ARG nr = pc[1].arg;
      ARG idx;
      for (idx = 0; idx < nr; idx++)
	VARIABLE(idx).lval = 0L;

      /* Continue with next instruction */
      NEXT;
    }

/*------------------------------------------------------------------------------
// Instruction: INIT_{IST}(position, value)
// Description: initialize INT/SET/TEXT variable at position (the position
//	        is relative to the frame pointer plus fixed part offset)
//	        to the specified value.
//----------------------------------------------------------------------------*/
static vptr exec_init_s ()
    { /* Pick up arguments */
      ARG pos = pc[1].arg;
      Value val = pc[2].val;

      /* Initialize this variable */
      VARIABLE(pos).val = val;

      /* Continue with next instruction */
      NEXT;
    }

static vptr exec_init_v ()
    { exec_illegal ();

      /* Continue with next instruction */
      NEXT;
    }

/*------------------------------------------------------------------------------
// Instruction: ADJUST_{IST}(target, result param, fail)
// Description: restrict INT/SET/TEXT variable at target (the position
//	        is relative to the frame pointer plus fixed part offset)
//	        to the value after continuing the recent CALL
//
// Note: we use the location of the restriction result value to save the
//       original value. UADJUST will restore from this location
//----------------------------------------------------------------------------*/
static vptr exec_adjust_i ()
    { /* Pick up arguments */
      ARG target = pc[1].arg;
      ARG param = pc[2].arg;
      CODE* fail_addr = pc[3].code;

      /* Pick up restricting and original value */
      Value val = sp[param].val;
      Value orig = VARIABLE(target).val;

      /* Do flat trellis operation */
      DB_RTS(abs_message ("ADJUST_I: adjusting to %d", val.int_par));
      if (val.int_par == TOP_INT)		/* No restriction but save original for restore */
	sp[param].val = orig;
      else if (orig.int_par == TOP_INT)		/* orig was unknown, assign new and save old */
	{ sp[param].val = orig;
	  VARIABLE(target).val = val;
	}
      else if (orig.int_par != val.int_par)	/* consistent substitution fails */
	GOTO(fail_addr);

      /* Continue with next instruction */
      NEXT;
    }

static vptr exec_adjust_s ()
    { /* Pick up arguments */
      ARG target = pc[1].arg;
      ARG param = pc[2].arg;
      CODE* fail_addr = pc[3].code;

      /* Pick up restricting and original value */
      Value val = sp[param].val;
      Value orig = VARIABLE(target).val;
      Value result;

      /* Do the trellis operation and intersect the sets */
      result.set_par = orig.set_par & val.set_par;
      DB_RTS (abs_message ("ADJUST_S: %ld <- (orig) %ld & (res) %ld",
		 	   result.set_par, orig.set_par, val.set_par));

      /* If the intersection of the sets is empty --> fail */
      if (result.set_par == 0)
	GOTO (fail_addr);

      /* save original value and set the variable to the intersection */
      sp[param].val = orig;
      VARIABLE(target).val = result;

      /* Continue with next instruction */
      NEXT;
    }

static vptr exec_adjust_t ()
    { /* Pick up arguments */
      ARG target = pc[1].arg;
      ARG param = pc[2].arg;
      CODE* fail_addr = pc[3].code;

      /* Pick up restricting and original value */
      Value val = sp[param].val;
      Value orig = VARIABLE(target).val;

      /* Do flat trellis operation */
      DB_RTS (abs_message ("ADJUST_T: adjusting to %s", val.text_par));
      if (val.text_par == TOP_TEXT)		/* No restriction but save original for restore */
	sp[param].val = orig;
      else if (orig.text_par == TOP_TEXT)	/* orig was unknown, assign new and save old */
	{ sp[param].val = orig;
	  VARIABLE(target).val = val;
	}
      else if (strcmp (orig.text_par, val.text_par) != 0)
	{ GOTO(fail_addr);			/* consistent substitution fails */
	}
      else sp[param].val = orig;		/* Save orig pointer for restore */

      /* Continue with next instruction */
      NEXT;
    }

static vptr exec_adjust_v ()
    { /* Unimplemented */
      exec_illegal ();

      /* Continue with next instruction */
      NEXT;
    }

static vptr exec_uadjust_s ()
    { /* Pick up arguments */
      ARG target = pc[1].arg;
      ARG param = pc[2].arg;
      Value val = sp[param].val;
	
      VARIABLE(target).val = val;

      /* Continue with next instruction */
      NEXT;
    }

static vptr exec_uadjust_v ()
    { /* Unimplemented */
      exec_illegal ();

      /* Continue with next instruction */
      NEXT;
    }

/*------------------------------------------------------------------------------
// Instruction: RESTRICT_{IST}(target, value, fail)
// Description: restrict INT/SET/TEXT variable at target (the position
//	        is relative to the frame pointer plus fixed part offset)
//	        by the value. Upon success, the original value is pushed
//		for later restore by URESTRICT_{IST}
//----------------------------------------------------------------------------*/
static vptr exec_restrict_i ()
    { /* Pick up arguments */
      ARG target = pc[1].arg;
      Value val = pc[2].val;
      CODE* fail_addr = pc[3].code;

      /* Pick up original value and do flat trellis operation */
      Value orig = VARIABLE(target).val;
      if (val.int_par == TOP_INT)		/* No restriction but push original for restore */
	{ PUSH_VALUE (orig);
	}
      else if (orig.int_par == TOP_INT)		/* orig was unknown, assign new and push old */
	{ PUSH_VALUE (orig);
	  VARIABLE(target).val = val;
	}
      else if (orig.int_par != val.int_par)	/* consistent substitution fails */
	{ GOTO (fail_addr);
	}
      else PUSH_VALUE (orig);			/* Save for restore */

      /* Continue with next instruction */
      NEXT;
    }

static vptr exec_restrict_s ()
    { /* Pick up arguments */
      ARG target = pc[1].arg;
      Value val = pc[2].val;
      CODE* fail_addr = pc[3].code;

      /* Pick up original value */
      Value orig = VARIABLE(target).val;
      Value result;

      /* Do the trellis operation and intersect the sets */
      result.set_par = orig.set_par & val.set_par;
      if (result.set_par == 0)			/* intersection of the sets is empty -- fail */
	GOTO (fail_addr);

      /* push original value and assign intersection to variable */
      PUSH_VALUE (orig);
      VARIABLE(target).val = result;

      /* Continue with next instruction */
      NEXT;
    }

static vptr exec_restrict_t ()
    { /* Pick up arguments */
      ARG target = pc[1].arg;
      Value val = pc[2].val;
      CODE* fail_addr = pc[3].code;

      /* Pick up original value and do flat trellis operation */
      Value orig = VARIABLE(target).val;

      if (val.text_par == TOP_TEXT)		/* No restriction but push original for restore */
	{ PUSH_VALUE(orig);
	}
      else if (orig.text_par == TOP_TEXT)	/* orig was unknown, assign new and push old */
	{ PUSH_VALUE(orig);
	  VARIABLE(target).val = val;
	}
      else if (strcmp (orig.text_par, val.text_par) != 0)
	{ GOTO (fail_addr);			/* consistent substitution fails */
	}
      else PUSH_VALUE (orig);			/* Push orig pointer for restore */

      /* Continue with next instruction */
      NEXT;
    }

static vptr exec_restrict_v ()
    { /* Unimplemented */
      exec_illegal ();

      /* Continue with next instruction */
      NEXT;
    }

static vptr exec_urestrict_s ()
    { /* Pick up argument */
      ARG target = pc[1].arg;
	
      POP_VALUE (VARIABLE(target).val);

      /* Continue with next instruction */
      NEXT;
    }

static vptr exec_urestrict_v ()
    { /* Unimplemented */
      exec_illegal ();

      /* Continue with next instruction */
      NEXT;
    }

/*------------------------------------------------------------------------------
// Instruction: INPUT_POS (position, target, fail)
// Description: Allocate and sprintf "Fname, line %d, column %d" where the
//		values are taken from the current input trellis state
//		and save this string at position in the current frame
//		(relative to the frame pointer plus fixed part offset)
//		Without posmemo, UINPUT_POS frees this string.
//		With posmemo, the string is freed later
//
//		If a affix target is given (i.e.) the position is >= 0,
//		restrict this target with the sprinted value.
//		Upon success, the original value is pushed for later restore
//----------------------------------------------------------------------------*/
static vptr exec_input_pos ()
    { /* Pick up arguments */
      ARG position = pc[1].arg;
      ARG target = pc[2].arg;
      CODE* fail_addr = pc[3].code;
      char *dstr;

      /* Pick up original value */
      DB_RTS(abs_message ("INPUT_POS (%ld, %ld, failaddr)", position, target));

      /* Calculate value and store in current frame */
      if (i_state == NULL)
	abs_abort ("INPUT_POS", "Cannot take position from null input state");
      if (arts_ifd.input_pos_fname == NULL)
	dstr = abs_mm_new_fmtd_string ("$INPUT_POS", "line %d, column %d",
				       i_state -> linenr, i_state -> colnr);
      else dstr = abs_mm_new_fmtd_string ("$INPUT_POS", "File '%s', line %d, column %d",
					  arts_ifd.input_pos_fname,
					  i_state -> linenr, i_state -> colnr);
      VARIABLE(position).str = dstr;

      /* If we have a target affix, do the corresponding trellis operation */
      if (target >= 0)
	{ /* Do flat trellis operation */
	  Value orig = VARIABLE(target).val;
	  if (orig.text_par == TOP_TEXT)	/* orig was unknown, assign new and push old */
	    { PUSH_VALUE(orig);
	      VARIABLE(target).str = dstr;
	    }
	  else if (strcmp (orig.text_par, dstr) != 0)
	    { GOTO (fail_addr);			/* consistent substitution fails */
	    }
	  else PUSH_VALUE (orig);		/* Push orig pointer for restore */
	};

      /* Continue with next instruction */
      NEXT;
    }

static vptr exec_uinput_pos ()
    { /* Pick up arguments */
      ARG target = pc[2].arg;

      /* Restore the original target affix, if present */
      DB_RTS(abs_message ("UINPUT_POS (%ld, %ld)", pc[1].arg, target));
      if (target >= 0)
	POP_VALUE (VARIABLE(target).val);

      /* Continue with next instruction */
      NEXT;
    }

/*--------------------------------------------------------------------------
// Optimization instructions
//------------------------------------------------------------------------*/

/*--------------------------------------------------------------------------
// Instruction: TMEMO(nr, fail addr)
// Description: Test memo nr, if blocked for current or lower penalty level
//	      then goto fail addr, otherwise continue.
// Note: TMEMO is only concerned with negative memo
//       nr is the alt nr in the grammar
//------------------------------------------------------------------------*/
static vptr exec_tmemo ()
    { /* Pick up arguments */
      ARG memo_nr = pc[1].arg;
      CODE* fail_addr = pc[2].code;

      if (negmemo_is_blocked (memo_nr, curr_penalty))
	{ /* blocked for this penalty level -> fail */
	  logfail(opc_tmemo);
	  GOTO (fail_addr);
	};

      /* Continue with next instruction */
      NEXT;
    }

/*--------------------------------------------------------------------------
// Instruction: TSMEMO(memo nr, fail addr)
// Description: Test and set memo: same as TMEMO, but if unknown set memo
//	      nr to blocked and continue.
// Note: TSMEMO is only concerned with negative memo
//       memo nr is the alt nr in the grammar
//------------------------------------------------------------------------*/
static vptr exec_tsmemo ()
    { /* Pick up arguments and log instructions */
      ARG memo_nr = pc[1].arg;
      CODE* fail_addr = pc[2].code;
      DB_RTS (abs_message ("TSMEMO(%lu) at pos %u", memo_nr, STATE_POS(i_state)));

      /*
	 If we know that we are blocked for this alternative at
	 this position, we immediately fail.
      */
      if (negmemo_is_blocked (memo_nr, curr_penalty))
	{ /* negmemo is blocked -> fail */
	  DB_RTS (abs_message ("Current negmemo is %d against penalty %d, failing",
			       NEG_MEMO (memo_nr), curr_penalty));
	  logfail (opc_tsmemo);
	  GOTO (fail_addr);
	};

      /*
	 If we are marked for left recursion, we know we are going to pass
	   this point again, so do not set the negative memo to blocked
      */
      if (!(START_INPUT_STATE -> lrec_markers[NONT_NR]) && negmemo_is_unknown (memo_nr))
	negmemo_set_blocked (memo_nr, curr_penalty);

      /* Continue with next instruction */
      NEXT;
    }

/*---------------------------------------------------------------------------
// Instruction: SMEMO_S(memo nr)
// Description: Set memo nr to succeeded for current penalty level.
// Note: SMEMO_S is only concerned with negative memo
//       memo nr is the alt nr in the grammar
//-------------------------------------------------------------------------*/
static vptr exec_smemo_s ()
    { ARG memo_nr = pc[1].arg;
      DB_RTS(abs_message ("SMEMO_S(%lu) at pos %u", memo_nr, STATE_POS(i_state)));

      /*
	 If we are not marked for left recursion and arrive here
	 while the negmemo is still unknown, something is wrong
	*/
      if (!(START_INPUT_STATE -> lrec_markers[NONT_NR]) && negmemo_is_unknown (memo_nr))
	abs_abort ("SMEMO_S", "Negative memo is unknown when it should not be");
	
      negmemo_set_succeeded (memo_nr);

      /* Continue with next instruction */
      NEXT;
    }

/*------------------------------------------------------------------------------
// Input matching instructions:
// Instruction: MATCH    (id, local, target, fail address)
//		MATCH_RE (id, local, target, fail address)
// 		SKIP_RE  (id, local, target, fail address)
//		MATCH_OTHER (0, local, target, fail address)
//
// Matching of grammar terminals, regexps matches and skips
// is performed by the same code. In the initialization phase
// of the interpreter, the terminal type has been encoded in
// the first argument of the instruction.
//
// The local is used to hold the pointer to the actual transition in the
// trellis matching the terminal type and id. The target is the position
// in the current frame of a text affix whose value is adjusted by the
// matched text.
//----------------------------------------------------------------------------*/
static vptr exec_match ()
    { Terminal terminal = pc[1].arg;		/* get token to be matched */

#ifndef GEN_RTS
      i_transition = GET_STATE_TRANSLIST (curr_trellis, i_state,
					  DECODE_TERM_OR_RE_CLASS(terminal));
#ifdef TRACE_MATCH
      if (TERM_IS_MATCH (terminal))
	abs_message ("MATCH MATCH_RE pos %u: %ld == \"%s\"",
	              STATE_POS(i_state), DECODE_REGEXP_NUMBER(terminal),
	              arts_ifd.match_regexp_names[DECODE_REGEXP_NUMBER(terminal)]);
      else if (TERM_IS_SKIP (terminal))
	abs_message ("MATCH SKIP_RE pos %u: %ld == \"%s\"",
	              STATE_POS(i_state), DECODE_REGEXP_NUMBER(terminal),
	              arts_ifd.skip_regexp_names[DECODE_REGEXP_NUMBER(terminal)]);
      else if (TERM_IS_OTHER (terminal))
	abs_message ("MATCH OTHER pos %u", STATE_POS(i_state));
      else /* TERM_IS_TERM */
	abs_message ("MATCH pos%u: %ld == \"%s\"",
	              STATE_POS(i_state), DECODE_TERM_NUMBER(terminal),
	              arts_ifd.term_names[DECODE_TERM_NUMBER(terminal)]);
#endif /* TRACE_MATCH */
      CONT (search_match);

#else /* GEN_RTS */
      gen_output_add_token (term_names[terminal]);

      /* Continue with the next instruction */
      NEXT;
#endif /* GEN_RTS */
    };

static vptr search_match ()
    { /*
        We get here by continuing from MATCH[_RE, _OTHER], or by
        failing on the penalty change from MATCH_AGAIN
        At this point i_transition is always valid or NULL
      */
      Terminal terminal = pc[1].arg;		/* get token to be matched */
      while (i_transition != NULL)
	{ /* do we have another entry? */
#ifdef TRACE_MATCH
	  abs_message ("state_has_eos_transition == %d", state_has_eos_transition(i_state));
	  abs_message ("c->t == %lu, t == %lu", i_transition -> terminal, terminal);
#endif /* TRACE_MATCH */
	  if (i_transition -> terminal != terminal)
	    /* This entry doesn't match, try next one */
	    i_transition = i_transition -> next;
	  else break;
	};

      /*
	 We have either a matching terminal (re, other) or the end of the list
	 Proceed at match again
      */
      CONT (match_again);
    };

static vptr match_again ()
    { /*
	We get here by continuing from MATCH[_RE, _OTHER], or UMATCH[_RE, _OTHER]
	At this point i_transition is always valid or NULL
      */
      CODE *fail_addr = pc[4].code;
      Penalty pen_gain;
      if (i_transition == NULL) pen_gain = MAX_PENALTY;
      else if (arts_ifd.hybrid_parsing_option) pen_gain = TRANSITION_PENALTY(i_transition);
      else pen_gain = 0;

      if (i_transition != NULL)
	{ ARG position = pc[2].arg;		/* Position to save transition */
	  ARG target = pc[3].arg;		/* Position of text affix */
	  char *dstr = i_transition -> text;	/* Matched text in trellis */

	  /* If the penalty gain would cause us to block, take the next transition */
	  if (penalty_change_would_block (curr_penalty, pen_gain))
	    { i_transition = i_transition -> next;
	      CONT (search_match);
	    };

	  curr_penalty += pen_gain;

	  /* Push our current program location, transition and input position */
	  /* save input position on stack */
	  sp -= 3;
	  sp[3].code = pc;
	  sp[2].input_transition = i_transition;
	  sp[1].input_state = i_state;

#ifdef TRACE_MATCH
	  abs_message ("saved state = %p", i_state);
	  abs_message ("at local %ld = %p", position, &VARIABLE(position));
#endif /* TRACE_MATCH */

	  /* save trellis transition in local stack frame */
	  VARIABLE(position).input_transition = i_transition;
	  i_state = TRANSITION_DEST_STATE_INDICATOR (i_transition, curr_trellis);
#ifdef TRACE_MATCH
	  abs_message ("new state = %p pos=%u", i_state, i_state ? STATE_POS(i_state) : -1);
#endif /* TRACE_MATCH */

	  /* adjust optional text affix; if absent, offset = -1 */
	  if (target >= 0)
	    { /* Do flat trellis operation */
	      Value orig = VARIABLE(target).val;
       	      if (orig.text_par == TOP_TEXT)	/* orig was unknown, assign new and push old */
	        { PUSH_VALUE(orig);
      	          VARIABLE(target).str = dstr;
      	        }
	      else if (strcmp (orig.text_par, dstr) != 0)
	        { /* consistent substitution fails */
      	          PUSH_VALUE (orig);
		  CONT (exec_umatch);		/* Proceed to unmatch */
      	  	}
	      else PUSH_VALUE (orig);		/* Push orig pointer for restore */
	    };
	  
	  /*
	     Record how far we got maximally into the input;
	     this is useful for debugging grammars.
	  */
	  if (i_state && STATE_POS(i_state) > furthest_pos)
	    { furthest_pos = STATE_POS(i_state);
	      save_parse_call_stack();
	    };

	  /* Continue with the instruction following the MATCH{_RE,_OTHER} */
	  NEXT;
	};

#ifdef TRACE_MATCH
      abs_message ("MATCH{_RE,_OTHER}: no more transitions");
#endif /* TRACE_MATCH */

      /* If all transitions are tried, nothing is left, so fail */
      GOTO(fail_addr);
    }

static vptr exec_umatch ()
    { /* note: unlike with LEX_UMATCH, the Terminal is not in pc[1] */
      ARG target = pc[1].arg;
      Terminal terminal;

      /* Absent target is coded with -1 */
      if (target >= 0)
	POP_VALUE (VARIABLE(target).val);

#ifndef GEN_RTS
      i_state = sp[1].input_state;	/* restore previous input position */
      sp += 1;
#ifdef TRACE_MATCH
      abs_message ("restored state = %p", i_state);
#endif /* TRACE_MATCH */

      /* Restore the transition and optionally the penalty */
      i_transition = sp[1].input_transition;
      terminal = i_transition -> terminal;
      if (arts_ifd.hybrid_parsing_option)
	curr_penalty -= TRANSITION_PENALTY(i_transition);
      sp += 2;

      /* this transition already done */
      i_transition = i_transition -> next;
	    
      while ((i_transition != NULL) && (i_transition -> terminal != terminal))
	/* This entry doesn't match, try next one */
	i_transition = i_transition -> next;

      if (i_transition != NULL)
	{ pc = sp[0].code;
	  RECALC_NEXT_PC;
	  CONT (match_again);
	};

#else /* GEN_RTS */
      gen_output_remove_token();
#endif /* GEN_RTS */

      /* Continue with the next instruction */
      NEXT;
    }

#ifndef GEN_RTS
static vptr exec_lex_match ()
    { curr_terminal = pc[1].arg;   /* get lexicon nont to be matched */

      sp -= 2;
      sp[1].input_state = i_state;        /* save current input position */
      sp[2].code = pc;                    /* save current program counter */

      i_transition = GET_STATE_TRANSLIST (curr_trellis, i_state,
					  DECODE_NONT_CLASS (curr_terminal));
      CONT (lex_match_again);
    }

static vptr lex_match_again ()
    { Penalty pen_gain;

      if (i_transition == NULL) pen_gain = MAX_PENALTY;
      else if (arts_ifd.hybrid_parsing_option) pen_gain = TRANSITION_PENALTY (i_transition);
      else pen_gain = 0;

      if (penalty_change_would_block (curr_penalty, pen_gain))
	{ CODE *fail_addr = pc[2].code;
	  i_state = sp[1].input_state;	/* restore input position */
	  sp += 2;
	  GOTO (fail_addr);			/* and fail */
	}
      else
	{ /* we can go on */
	  int i, arity;
	  Value *val_ptr;
	  assert(i_transition -> terminal == curr_terminal);
	  curr_penalty += pen_gain;

	  /* restrict formals to lexicon entry parameters */
	  val_ptr = TRANSITION_PARAMS (i_transition);
	  arity = DECODE_NONT_ARITY (curr_terminal);
	  for (i = arity - 1; i >= 0; --i)
	    { DB_RTS (abs_message ("pushing value 0x%lx", val_ptr[i].set_par));
	      PUSH_VALUE(val_ptr[i]);
	    };

	  SON_FP(fp[NR_SONS_DONE].arg) = (DATA*) i_transition;
	  fp[NR_SONS_DONE].arg++;

	  i_state = TRANSITION_DEST_STATE_INDICATOR(i_transition, curr_trellis);

	  /*
	     Record how far we got maximally into the input;
	     this is useful for debugging grammars.
	  */
	  if (STATE_POS(i_state) > furthest_pos)
	    { furthest_pos = STATE_POS(i_state);
      	      save_parse_call_stack();
	    };

	  /* Continue with the next instruction */
	  NEXT;
	};
    }

static vptr exec_lex_umatch ()
    { int arity;

      curr_terminal = pc[1].arg;	    	/* get lexicon nont to be matched */
      fp[NR_SONS_DONE].arg--;
      i_transition = (Transition *) SON_FP(fp[NR_SONS_DONE].arg);

      arity = DECODE_NONT_ARITY (curr_terminal);
      if (arts_ifd.hybrid_parsing_option)
	{ /* restore penalty level */
	  curr_penalty -= TRANSITION_PENALTY(i_transition);
	};

      sp += arity;				/* remove previous results from the stack */
      pc = sp[2].code;				/* restore old pc */
      i_transition = i_transition -> next;
      RECALC_NEXT_PC;
      CONT (lex_match_again);			/* try to match next entry */
    }

#else /* GEN_RTS */

static vptr exec_match ()
    { exec_illegal ();
      NEXT;
    }

static vptr exec_lex_umatch ()
    { exec_illegal ();
      NEXT;
    }
#endif /* GEN_RTS */

/*------------------------------------------------------------------------------
// Additional flow control instructions (penalty/commit):
//----------------------------------------------------------------------------*/
static vptr exec_penalty ()
    { /* Pick up argument and fail address */
      Penalty penalty_gain = pc[1].penalty;
      CODE* fail_addr = pc[2].code;
      DB_RTS (abs_message ("PENALTY(%d): old penalty %d", penalty_gain, curr_penalty));

      /* Check if new penalty allows continuation */
      if (penalty_change_would_block (curr_penalty, penalty_gain))
	GOTO (fail_addr);
      curr_penalty += penalty_gain;

      /* Continue with the next instruction */
      NEXT;
    }

static vptr exec_upenalty ()
    { Penalty penalty_gain = pc[1].penalty;

      /* Readjust curr_penalty */
      curr_penalty -= penalty_gain;
      DB_RTS(abs_message ("UPENALTY(%d): back to penalty %d", penalty_gain, curr_penalty));

      /* Continue with the next instruction */
      NEXT;
    }

static vptr exec_commit ()
    { if (fp[NR_SUCCS_OFF].arg)	/* continuation taken (SUCCESS)? */
	{ DB_RTS (abs_message ("COMMIT: skipping further alternatives"));
	  GOTO(pc[1].code);		/* skip other alternatives */
	};

      /* Continue with the next instruction */
      DB_RTS (abs_message ("COMMIT: trying next alternative"));
      NEXT;				/* try other alts for lower penlevel */
    }


/*------------------------------------------------------------------------------
// Instructions for the start rule:
//----------------------------------------------------------------------------*/
static vptr exec_success ()
    {
#ifndef GEN_RTS
      fp[NR_SUCCS_OFF].arg++;
      DB_RTS(abs_message ("SUCCESS: i_state = %p", i_state));
      stop_parse_time();

#else /* GEN_RTS */
      incr_nr_parses();
#endif /* GEN_RTS */

      /* Continue with the next instruction */
      NEXT;
    }

static vptr exec_usuccess ()
    { DB_RTS(abs_message ("USUCCESS: "));

#ifndef GEN_RTS
      start_parse_time();
#endif /* GEN_RTS */

      /* Continue with the next instruction */
      NEXT;
    }

static vptr exec_end ()
    { status = STOPPING;
      return (NULL);
    }

/*------------------------------------------------------------------------------
// Transduction
//----------------------------------------------------------------------------*/
static vptr exec_pass2 ()
    {
#ifndef GEN_RTS
      DB_RTS (abs_message ("PASS2: %s", (return_to_pass2)?": returning to":"entering"));
      if (return_to_pass2)
	{ stop_print_time ();
	  stop_printing ();
	};

      while (!return_to_pass2 || try_update_ambi_stack ())
	{ /*
	     We have to execute PASS2 with the first parse OR
	     we have to reexecute PASS2 with another parse
	  */
	  unsigned startpos = state_get_endpos (START_INPUT_STATE, curr_trellis);
	  unsigned endpos = state_get_endpos (i_state, curr_trellis);
	  unsigned length = endpos - startpos;
	  incr_nr_parses ();

	  if (current_parse_is_acceptable (curr_penalty, length))
	    { /* This parse can be added */
	      DB_RTS (abs_message ("PASS2: found parse len %d, penalty %d, accepted",
				   length, curr_penalty));
	      DB_RTS (abs_message ("START_INPUT_STATE = %p, end state = %p",
				   START_INPUT_STATE, i_state));
	      parse_result.penalty = curr_penalty;
	      parse_result.startpos = startpos;
	      parse_result.endpos = endpos;
	      current_parse_set_prio_and_add (curr_penalty, i_state, length);
	      start_print_time ();
	      start_printing ();

	      /* Setup PASS2 frame */
	      sp -= PASS2_FRAME_HEAD_LEN;
	      sp[PASS2_PASS2FP].data = 0;

	      /* Return to this instruction, to check for more output */
	      sp[PASS2_FP].pmprod = pmptr;
	      sp[PASS2_RET_ADDR].code = pc;
	      return_to_pass2 = 1;

	      /* Set pmptr to 0; PRINT_SON (0) in root print/trans tree will detect it */
	      pmptr = NULL;
	      pass2_fp = sp;
	      next_pc = (arts_ifd.transduce_option)?pc[1].code:pc[2].code;
	      NEXT; 
	    }
	  else
	    { /* force the ambi check */
	      DB_RTS (abs_message ("PASS2: found parse len %d, penalty %d, refused",
				   length, curr_penalty));
	      return_to_pass2 = 1;
	    }
	};

      /* We left the PASS2 loop, so we continue with next instruction */
      return_to_pass2 = 0;
      NEXT; 

#else /* GEN_RTS */
      gen_output_show();
      NEXT;
#endif /* GEN_RTS */
    }

/*-----------------------------------------------------------
// RETRN
//---------------------------------------------------------*/
static vptr exec_retrn ()
    { DATA* new_pass2_fp;
      unsigned nr_formals;

      next_pc = pass2_fp[PASS2_RET_ADDR].code;

      /* get nr_formals (of myself, the son) */
      if (pmptr != NULL) nr_formals = pmptr -> nr_formals;
      else nr_formals = 0;

      /* Restore the posmemo of the father */
      pmptr = pass2_fp[PASS2_FP].pmprod;
      DB_RTS (
	abs_printf ("RETRN: nr_formals = %u ", nr_formals);
	if (pmptr)
	  { int nont_nr = pmptr -> nont_nr;
      	    char *name;
      	    if ((0 <= nont_nr) && (nont_nr < arts_ifd.nr_syntax_nonterminals))
      	      name = arts_ifd.nonterm_names[nont_nr];
      	    else name = "(outofrange)";
      	    abs_printf ("to rule %d \"%s\"", nont_nr, name);
	  }
	abs_printf("\n");
      );

      /* Restore the stack frame */
      new_pass2_fp = pass2_fp[PASS2_PASS2FP].data;
      sp += PASS2_FRAME_HEAD_LEN + nr_formals;
      pass2_fp = new_pass2_fp;
      NEXT;
    }

static vptr exec_push_f ()
    { ARG formal = pc[1].arg;

      /* Hoe kom je hieraan? */
      Value formal_value = pass2_fp[PASS2_FRAME_HEAD_LEN + formal + 1].val;
      DB_RTS (abs_message ("PUSH_F: formal == %ld, value == %08x, sp = %p",
			   formal, formal_value.set_par, sp));

      PUSH_VALUE(formal_value);

      /* Continue with the next instruction */
      NEXT;
    }

static vptr exec_push_l ()
    { ARG local = pc[1].arg; 

      PUSH_VALUE (posmemo_get_local (pmptr,local));
      DB_RTS(abs_message ("PUSH_L: local == %ld, value == %08x, sp = %p",
			  local, VARIABLE(local).val.set_par, sp));

      /* Continue with the next instruction */
      NEXT;
    }

/*-----------------------------------------------------------
// PRINT_SON
//---------------------------------------------------------*/
static vptr exec_print_son ()
    { ARG son_nr = pc[1].arg;
      DB_RTS (abs_message ("PRINT_SON: curr pmptr = %p, sonnr = %ld, nr_sons = %d, stack size = %d",
			   pmptr, son_nr, pmptr ? pmptr->nr_sons : 1, sp - stack));

      /* create a new pass2 stack frame */
      sp -= PASS2_FRAME_HEAD_LEN;
      sp[PASS2_PASS2FP].data = pass2_fp;

      /* to remember our current posmemo pointer */
      sp[PASS2_FP].pmprod = pmptr;
      sp[PASS2_RET_ADDR].code = next_pc;
      pass2_fp = sp;

      /* pick up posmemo for either the son or the root rule */
      if (pmptr != NULL) pmptr = posmemo_get_son (pmptr,son_nr);
      else pmptr = SON_FP(0)[PMPRODPTR].pmprod;	/* Root rule */

      /* Check for ambiguity: we may pass this point more than once */
      if (pmptr -> prime != NULL)
	{ PosMemo sibling;
	  assert (pmptr -> prime == pmptr);
	  if (try_lookup_sibling_in_ambi_stack (pmptr, &sibling)) pmptr = sibling;
	  else push_ambi_stack (pmptr -> prime);
	};

      /* continue with the transduction of the son */
      DB_RTS(abs_message ("Using %p as PMPROD to PRINT SON (rule %d \"%s\")",
			  pmptr, pmptr->nont_nr, arts_ifd.nonterm_names[pmptr->nont_nr]));
      next_pc = posmemo_get_pass2 (pmptr); 
      NEXT;
    }

/*-----------------------------------------------------------
// Print the begin and end of an alternative
// Only used in tree output
//
// if labelled bracket output wanted, print open and closed
// curly braces; otherwise use indentation
//---------------------------------------------------------*/
static vptr exec_print_abeg ()
    { char* alt_text = pc[1].val.text_par;
      DB_RTS(abs_message ("PRINT_ABEG: alt_text = %s", alt_text));

      if (arts_ifd.label_bracket)
	{ if (alt_text && arts_ifd.parsing_stats_option)
	    current_parse_printf ("{%s:", alt_text);
	  else current_parse_add_char('{');
	}
      else
	{ if (alt_text && arts_ifd.parsing_stats_option)
	    current_parse_printf (" <alt %s>\n", alt_text);
	  else current_parse_add_char('\n');

	  indent += 2;
	};

      /* Continue with next instruction */
      NEXT;
    }

/*----------------------------------------------------------
// PRINT_AEND: print end alternative.
//--------------------------------------------------------*/
static vptr exec_print_aend ()
    { DB_RTS (abs_message ("PRINT_AEND: "));

      if (arts_ifd.label_bracket) current_parse_add_char('}');
      else indent -= 2;

      /* Continue with next instruction */
      NEXT;
    }

//---------------------------------------------------------
// PRINT_PBEG, PRINT_PSEP and PRINT_PEND serve to print
// the affix list with a production i.e. they print a
// '(', ',' and ')'
//-------------------------------------------------------*/
static vptr exec_print_pbeg ()
    { DB_RTS (abs_message ("PRINT_PBEG: "));

      current_parse_add_char('(');
	
      /* Continue with next instruction */
      NEXT;
    }

static vptr exec_print_psep ()
    { DB_RTS (abs_message ("PRINT_PSEP: "));

      current_parse_add_char(',');
      if (!arts_ifd.label_bracket) current_parse_add_char(' ');
	
      /* Continue with next instruction */
      NEXT;
    }

static vptr exec_print_pend ()
    { DB_RTS (abs_message ("PRINT_PEND: "));
      current_parse_add_char (')');
	
      /* Continue with next instruction */
      NEXT;
    }

/*--------------------------------------------------
// Print a text value
//------------------------------------------------*/
static vptr exec_print_val ()
    { char* text = pc[1].val.text_par;
      DB_RTS (abs_message ("PRINT_PVAL: text = %s", text));

      if (arts_ifd.transduce_option) current_parse_add_string(text);
      else abs_bug ("agfl_interpret", "unexpected PRINT_VAL in tree");

      /* Continue with next instruction */
      NEXT;
    }

/*-----------------------------------------------------------
// PRINT_TERM: print the terminal belonging to a transition
//---------------------------------------------------------*/
static vptr exec_print_term ()
    { /* MS: Hier zou je een typecheck kunnen doen... */
      Transition *p_transition;
      char *text;
      ARG arg = pc[1].arg;
      DATA dat;
      dat.val = posmemo_get_variable (pmptr,arg);
      p_transition = dat.input_transition;
      text = p_transition -> text;
      DB_RTS(abs_message ("PRINT_TERM: text = %s", text));

      if (arts_ifd.transduce_option)
	{ current_parse_add_string(text);
	  if (IS_LASTPART (p_transition))
	    current_parse_add_char (' ');
	}
      else
	{ int prefix = 0;
	  int suffix = 0;

	  switch (get_transition_lex_type (p_transition))
	    { case Infix:	suffix = 1; 
	      case Prefix:	prefix = 1;
	          		break;
	      case Suffix:	suffix = 1;
	      default:	break;
	    };

	  if (!arts_ifd.label_bracket) current_parse_add_space(indent);

	  if (suffix) current_parse_add_string ("\"-");
	  else current_parse_add_char ('\"');
	  current_parse_add_string (text);

	  if (prefix) current_parse_add_string ("-\"");
	  else current_parse_add_char ('\"');

	  if (!arts_ifd.label_bracket) current_parse_add_char ('\n');
	};

      /* Continue with next instruction */
      NEXT;
    }

/*-------------------------------------------------------------
// PRINT_LEX: print the text belonging to a lexicon transition
//-----------------------------------------------------------*/
static vptr exec_print_lex ()
    { /* Pick corresponding transition from saved one in stack frame */
      /* MS: Hier zou je een typecheck kunnen doen... */
      Transition* p_transition = (Transition*)posmemo_get_son(pmptr,pc[1].arg);
      char *text = p_transition -> text;
      DB_RTS(abs_message ("PRINT_LEX: text = %s", text));

      if (arts_ifd.transduce_option)
	{ current_parse_add_string (text);
	  if (IS_LASTPART (p_transition))
	    current_parse_add_char(' ');
	}
      else
	{ int prefix = 0;
	  int suffix = 0;

	  switch (get_transition_lex_type (p_transition))
	    { case Infix:  suffix = 1;
	      case Prefix: prefix = 1; break;
	      case Suffix: suffix = 1; break;
	      default: break;
	    };

	  if (!arts_ifd.label_bracket)
	    current_parse_add_space(1);

	  if (suffix)
	    current_parse_add_string("\"-");
	  else current_parse_add_char('\"');
	  current_parse_add_string(text);

	  if (prefix)
	    current_parse_add_string("-\"");
	  else current_parse_add_char('\"');

	  if (!arts_ifd.label_bracket)
	    current_parse_add_char('\n');
	};

      /* Continue with next instruction */
      NEXT;
    }

/*-------------------------------------------------------------
// PRINT_RE: print the text belonging to a regular expression
// PRINT_OTHER: print the text not matched by anything
// For the time being we use the same code.
//-----------------------------------------------------------*/
static vptr exec_print_re ()
    { /* Pick up argument */
      ARG var = pc[1].arg;
      Transition *p_transition;
      char *text;

      /* Pick transition from saved location in frame */
      if (var < pmptr -> nr_formals)
	p_transition = pass2_fp[PASS2_FRAME_HEAD_LEN + var + 1].input_transition;
      else
	{ DATA dat;
	  dat.val = posmemo_get_variable (pmptr,var);
	  p_transition = dat.input_transition;
	};

      text = p_transition -> text;
      DB_RTS(abs_message ("PRINT_RE(or OTHER): text = %s", text));

      if (arts_ifd.transduce_option)
	{ current_parse_add_string (text);
	  if (IS_LASTPART (p_transition))
	    current_parse_add_char (' ');
	}
      else
	{ if (!arts_ifd.label_bracket)
	    current_parse_add_space (indent);

	  current_parse_add_char ('\"');
	  current_parse_add_string (text);

	  if (!arts_ifd.label_bracket)
	    current_parse_add_string ("\"\n");
	  else current_parse_add_char ('\"');
	};

      /* Continue with next instruction */
      NEXT;
    }

/*---------------------------------------------------------------
// PRINT_POS: print a position
//-------------------------------------------------------------*/
static vptr exec_print_pos ()
    { /* Pick up argument */
      ARG local = pc[1].arg;

      /* If transducing, print the saved position */
      if (arts_ifd.transduce_option)
	{ /* Pick up text from frame or posmemo and print it */
	  Value val = posmemo_get_variable (pmptr, local);
	  if (val.text_par == NULL)
	    current_parse_add_string ("chaos awaits the pointer in posmemo");
	  else current_parse_add_string (val.text_par);
	}
      else
	{ current_parse_add_space (indent);
	  current_parse_add_string ("$POS\n");
	};

      /* Continue with next instruction */
      NEXT;
    }

/*---------------------------------------------------------------
// PRINT_PEN: print a penalty
//-------------------------------------------------------------*/
static vptr exec_print_pen ()
    { /* Pick up penalty */
      int pen = pc[1].arg;
      DB_RTS(abs_message ("PRINT_PEN: pen = %d", pen));

      if (!arts_ifd.label_bracket && !arts_ifd.transduce_option)
	{ current_parse_add_space(indent);
	  if (pen == 1) current_parse_add_string("$PENALTY");
	  else
	    { current_parse_add_string("$PENALTY(");
	      current_parse_add_int(pen);
	      current_parse_add_char(')');
	    };
	  current_parse_add_char('\n');
	};

      /* Continue with next instruction */
      NEXT;
    }

/*---------------------------------------------------------------
// PRINT_N: print nonterminal name
//-------------------------------------------------------------*/
static vptr exec_print_n ()
    { /* Pick up argument */
      char* text = pc[1].val.text_par;
      DB_RTS(abs_message ("PRINT_N: nont = %s", text));

      if (!arts_ifd.label_bracket)
	current_parse_add_space (indent);
      current_parse_add_string (text);

      /* Continue with next instruction */
      NEXT;
    }

static vptr exec_print_if ()
    { /* Pick up argument */
      ARG formal = pc[1].arg;
      Value val = pass2_fp[PASS2_FRAME_HEAD_LEN + formal + 1].val;

      DB_RTS(abs_message ("PRINT_IF: int formal = %d", val.int_par));
      print_integer_affix (val.int_par, 1);

      /* Continue with next instruction */
      NEXT;
    }

static vptr exec_print_il ()
    { /* Pick up argument */
      ARG local = pc[1].arg;
      Value val = posmemo_get_local(pmptr,local);

      DB_RTS(abs_message ("PRINT_IL: int local = %d", val.int_par));
      print_integer_affix (val.int_par, 1);

      /* Continue with next instruction */
      NEXT;
    }

static vptr exec_print_sf ()
    { /* Pick up arguments */
      ARG formal = pc[1].arg;
      ARG domain = pc[2].arg;
      Value val = pass2_fp[PASS2_FRAME_HEAD_LEN + formal + 1].val;

      DB_RTS(abs_message ("PRINT_SF: set formal = %08x", val.set_par));
      print_set_affix(val.set_par, domain, 1);

      /* Continue with next instruction */
      NEXT;
    }

static vptr exec_print_sl ()
    { /* Pick up arguments */
      ARG local = pc[1].arg;
      ARG domain = pc[2].arg;
      Value val = posmemo_get_local(pmptr,local);

      DB_RTS(abs_message ("PRINT_SL: set local = %08x", val.set_par));
      print_set_affix(val.set_par, domain, 1);

      /* Continue with next instruction */
      NEXT;
    }

static vptr exec_print_tf ()
    { ARG formal = pc[1].arg;
      Value val = pass2_fp[PASS2_FRAME_HEAD_LEN + formal + 1].val;

      DB_RTS(abs_message ("PRINT_TF: text formal = %s", val.text_par));
      print_text_affix(val.text_par, 1);

      /* Continue with next instruction */
      NEXT;
    }

static vptr exec_print_tl ()
    { ARG local = pc[1].arg;
      Value val = posmemo_get_local(pmptr,local);

      DB_RTS(abs_message ("PRINT_TL: text local = %s", val.text_par));
      print_text_affix(val.text_par, 1);

      /* Continue with next instruction */
      NEXT;
    }

static vptr exec_print_vf ()
    { exec_illegal ();
      NEXT;
    }

static vptr exec_print_vl ()
    { exec_illegal ();
      NEXT;
    }


/*------------------------------------------------------------------------------
// Generateve grammar instructions:
//----------------------------------------------------------------------------*/
static vptr exec_choice ()
#ifndef GEN_RTS
    { exec_illegal ();
      NEXT;
    }
#else /* GEN_RTS */
#define NR_TRIES_PER_ALTGROUP 1
    { ARG choice_nr = pc[1].arg;
      ARG nr_alts = pc[2].arg;

      logexec(choice);
      opc_count(choice);
      ALT_DEPTH_CHECK;

      if (get_nr_parses())
	{ /* a sentence is generated, we can stop */
	  choice_admin_remove (alt_depth, choice_nr);
	  chosen_freq = -2;
	}
      else if (choice_admin_get_nr_tries (alt_depth, choice_nr) >=
      	 (NR_TRIES_PER_ALTGROUP * nr_alts))
	{ /* nr_alts == nr of tries, which all failed, so fail */
	  choice_admin_remove(alt_depth, choice_nr);
	  chosen_freq = -3;
	}
      else
	{ /* failed to generate a sentence, but not everything is tried */
	  PUSH_ADDR(pc);
	  chosen_freq = rand() % nr_alts;
//        chosen_freq = choice_admin_get_nr_tries(alt_depth, choice_nr);
	  choice_admin_incr_nr_tries(alt_depth, choice_nr);
	};
    }
#endif /* GEN_RTS */

static vptr exec_uchoice ()
#ifndef GEN_RTS
    { exec_illegal ();
      NEXT;
    }
#else /* GEN_RTS */
    { ARG choice_nr = pc[1].arg;
      if (((chosen_freq == -2) || (chosen_freq == -3)) &&
	   !choice_admin_get_nr_tries(alt_depth, choice_nr))
	{ /* Step to next instruction */
	  NEXT;
	}
      else
	{ CODE* choice_addr;
	  POP_ADDR(choice_addr);
	  assert(choice_addr);
	  GOTO(choice_addr);
	};
    }
#endif /* GEN_RTS */

static vptr exec_alt_marker ()
#ifndef GEN_RTS
    { exec_illegal ();
      NEXT;
    }
#else /* GEN_RTS */
    { ARG my_freq = 1;
      CODE* next_alt = pc[1].code;

      ALT_DEPTH_CHECK;
      if ((chosen_freq < my_freq) && (chosen_freq >= 0))
	{ /* we're gonna generate for this alternative */
	  chosen_freq = -1;
	  ALT_DEPTH_INCREASE;
	}
      else if (chosen_freq >= 0)
	{ /* this alternative is not chosen, goto next one */
	  chosen_freq -= my_freq;
	}

      /* else we're done, so skip this alternative */
      NEXT;
    }
#endif /* GEN_RTS */

static vptr exec_ualt_marker ()
#ifndef GEN_RTS
    { exec_illegal ();
      NEXT;
    }
#else /* GEN_RTS */
    { ALT_DEPTH_CHECK;
      ALT_DEPTH_DECREASE;
      NEXT;
    }
#endif /* GEN_RTS */

/*-------------------------------------------------------------------------
// Instruction: ROOT_CREATE (nr formals, nr locals, nr sons)
// Description: Setup an initial stack frame 
// Note that the frame head of this stack frame is never to be used
//-----------------------------------------------------------------------*/
static vptr exec_root_create ()
    { ARG nr_formals = pc[1].arg;
      ARG nr_locals = pc[2].arg;
      ARG nr_sons = pc[3].arg;
      int i;

      sp[0].data = fp;	        /* store old fp */
      fp = sp;

      sp -= nr_formals + nr_locals + nr_sons + VAR_OFFSET;
      NR_FORMALS = nr_formals;
      NR_LOCALS = nr_locals;
      NR_SONS = nr_sons;
      DB_RTS (NONT_NR = 0);	// for in FAIL.

      START_INPUT_STATE = i_state;
      fp[NR_SUCCS_OFF].arg = 0;
      fp[NR_SONS_DONE].arg = 0;
      fp[PENORIG].penalty = curr_penalty;

      /*
	 Initialise the locals and sons to 0, since otherwise equal_posmemo()
	 may look at uninitialised values sometimes (and fail to detect
	 identity). This happens because not every alternative uses (sets) the
	 full set of locals and sons so some may remain uninitialised.
      */
      for (i = 0; i < nr_locals + nr_sons; i++)
	VARIABLE(nr_formals + i).arg = 0;

      DB_RTS(abs_message ("ROOT_CREATE: start_input_state = %p, sp = %p, stack bottom = %p",
			  i_state, sp, stack));
      fp[PMPRODPTR].pmprod = NULL;

      /* Continue with next instruction */
      NEXT;
    }

static vptr exec_root_call ()
    { ARG rule_nr = pc[1].arg;
      CODE* jmp_point = pc[2].code;
      CODE* fail_addr = pc[3].code;

      DB_RTS(abs_message ("ROOT_CALL: current time is %.3fs.", get_parse_time()));
      if (posmemo_is_unknown (START_INPUT_STATE, rule_nr)) /* first try */
	posmemo_set_blocked(START_INPUT_STATE, rule_nr);

      /* Set up precall stack frame */
      sp -= FRAME_HEAD_LEN;
      sp[FAIL_ADDR_OFF].code = fail_addr;
      sp[SUCC_ADDR_OFF].code = next_pc;
      push_parse_call (rule_nr);

      SON_FP(fp[NR_SONS_DONE].arg) = sp;
      fp[NR_SONS_DONE].arg++;
      sp[NONT_NR_OFF].arg = rule_nr;
      DB_RTS(abs_message ("Root call: Starting parse, penalty = %d", curr_penalty));

      /* Jump to next instruction */
      GOTO (jmp_point);
    }

/*-------------------------------------------------------------------------
// Instruction: (U)MARK_LREC 
// Description: (Un)mark that calls for the current nonterminal and input
//		state should fail because they are (possibly hidden) left
//		recursive calls
//-----------------------------------------------------------------------*/
static vptr exec_mark_lrec ()
    { /* Pick up input state ptr */
      StateIndicator input_state = START_INPUT_STATE;
      DB_RTS (abs_message("MARK_LREC for nont %ld at pos %u", NONT_NR, STATE_POS (input_state)));

      /* Check if we are unmarked; panic if so */
#ifdef DEBUG_RTS
      if (input_state -> lrec_markers[NONT_NR])
	abs_abort ("MARK_LREC", "Rule %ld already blocked for left recursion", NONT_NR);
#endif /* DEBUG_RTS */
      input_state -> lrec_markers[NONT_NR] = 1;

      /* Continue with next instruction */
      NEXT;
    }

static vptr exec_umark_lrec ()
    { /* Pick up input state ptr */
      StateIndicator input_state = START_INPUT_STATE;
      CODE* redo_addr = pc[1].code;

      DB_RTS (abs_message ("UMARK_LREC for rule %ld \"%s\" at start pos %u", NONT_NR,
			   nonterm_names[NONT_NR], STATE_POS (input_state)));

      /* Check if we are marked; if so, undo the marking and jump to our redo address */
      if (input_state -> lrec_markers[NONT_NR])
	{ /* Undo the marking and redo all alternatives */
	  input_state -> lrec_markers[NONT_NR] = 0;
	  DB_RTS (abs_message ("UMARK_LREC unmarked rule %ld and jumps back", NONT_NR));
	  GOTO (redo_addr);
	};

      /* Continue with next instruction */
      NEXT;
    }

static vptr exec_trace ()
    { /* Continue with next instruction */
      StateIndicator input_state = START_INPUT_STATE;
      char *trace_text = pc[1].str;
      abs_message ("Trace: '%s' at start pos %u", trace_text, STATE_POS (input_state));
      NEXT;
    }

static vptr exec_position ()
    { ARG position = pc[1].arg;

      arts_ifd.alternatives_profile_table[position]++;
      /*
	 abs_message ("Alternative  %d [%ld]:", (int) position,
		      arts_ifd.alternatives_profile_table[position]);
      */

      /* Continue with next instruction */
      NEXT;
    }

/*------------------------------------------------------------------------------
// Illegal instructions
//----------------------------------------------------------------------------*/
static void exec_illegal ()
    { abs_bug ("agfl_interpret", "rts: illegal instruction");
    };

/* Write parse results in the .prd file */
void write_profile_table()
{ int position;

  for (position = 0; position < arts_ifd.nr_positions; position++)
    profile_printf ("%d:%ld\n", position, arts_ifd.alternatives_profile_table[position]);
}


/*------------------------------------------------------------------------------
// Parsing interface
//----------------------------------------------------------------------------*/
static void reset_parser (Trellis* trellis)
{
    reset_parse_result ();
    parse_results_destroy ();
    parse_results_init ();
#ifdef GEN_RTS
    parse_results_start_new ();
#endif /* GEN_RTS */
}

StateIndicator parse_trellis (Trellis* trellis, StateIndicator start_node)
{ reset_parser (trellis);

  if (have_time_limit ()) {
#ifdef DEBUG_TIME_OUT
	abs_message ("setting timeout to %ld secs", arts_ifd.max_parsetime);
#endif /* DEBUG_TIME_OUT */

	set_time_limit();
    }

#ifdef DEBUG_NEGMEMO
  if (neg_memo_option || directors_option)
    print_negmemo_table(trellis, 0);
#endif /* DEBUG_NEGMEMO */

  return (agfl_interpret (trellis, start_node));
}

/*------------------------------------------------------------------------------
// Prelude and postlude of module
//----------------------------------------------------------------------------*/
void arts_init_parser ()
{ init_ambi_stack ();
  posmemo_init ();
  /* agfl_interpret (Translate, NULL, NULL); */

  if (arts_ifd.counters_option)
    initialize_counters ();
}

void arts_end_parser ()
{ posmemo_done ();
  finish_ambi_stack ();
  /* write_profile_table (); */
}
