/*
   File: lxcn_vocabulary.h
   Stores a vocabulary (by which we mean the datastructure through which
   words can be entered with their appropriate information) for later
   retrieval. Since the number of words in the vocabulary may easily be
   of the order of one million, words are stored in a trie structure.

   This module defines the absolute basics for entering, lookup and
   basic (binary) IO. Andre Steenveld will complement the search
   engine with fuzzy matching of words (lxcn_fuzzymatch.{c,h} ?).

   CVS ID: "$Id: lxcn_vocabulary.h,v 1.4 2006/07/26 10:35:36 marcs Exp $"
*/
#ifndef IncLxcnVocabulary
#define IncLxcnVocabulary

#ifdef __cplusplus
extern "C" {
#endif

/* libabase includes */
#include <abase_fileutil.h>

/*
   Define the structure of a vocabulary
*/
typedef struct voc_index_tree_rec *VocIndexTree;
typedef struct voc_trie_rec *VocTrie;
typedef VocTrie Vocabulary;
typedef void *VocInfo;

struct voc_trie_rec
	{ VocIndexTree tails;		/* Balanced tree of subtries */
	  VocInfo info;			/* Associated information */
	  char *search_key;		/* Orig search key */
	  int rem_offset;		/* Tail index in search key */
	};

struct voc_index_tree_rec
	{ VocIndexTree left;		/* Left subtree */
	  VocIndexTree right;		/* Right subtree */
	  VocTrie sub_trie;		/* Associated subtrie */
	  unsigned char key;		/* Search key in index tree */
	  signed char balfac;		/* Balance factor */
	};

/* Introduce nil pointers */
#define voc_index_tree_nil ((VocIndexTree) NULL)
#define voc_trie_nil ((VocTrie) NULL)
#define voc_info_nil ((VocInfo) NULL)
#define vocabulary_nil ((Vocabulary) NULL)

/*
   Add procedure types to process the info nodes as we do not know
   the structure of the information associated with the vocabulary entries

   Note that this also allows multiple information nodes with one entry
   in the vocabulary, for instance the various lexicon nonterminals with
   their affixes belonging to one single word.
*/
typedef void (*AddVocInfoProc)(VocInfo *info_ptr, VocInfo info);
typedef void (*SaveVocInfoProc)(BinFile bf, VocInfo info);
typedef void (*LoadVocInfoProc)(BinFile bf, VocInfo *info);

/* exported routines */
void lxcn_enter_into_vocabulary (Vocabulary *voc, char *pattern,
				 VocInfo info, AddVocInfoProc add_info);
int lxcn_lookup_in_vocabulary (Vocabulary voc, char *pattern, VocInfo *info);
void lxcn_dump_vocabulary (Vocabulary voc);
void lxcn_bin_save_vocabulary (char *path, SaveVocInfoProc sv, Vocabulary voc);
void lxcn_bin_load_vocabulary (char *path, LoadVocInfoProc lv, Vocabulary *voc);

#ifdef __cplusplus
}
#endif
#endif /* IncLxcnVocabulary */
