/*
 * Copyright 2000 KUN.
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Library General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

/* $Id: lexicon_reading.c,v 1.8 2005/03/30 12:33:29 andres Exp $ */

/* standard includes */
#include <assert.h>
#include <stdio.h>
#include <stdlib.h>
#include <ctype.h>
#include <sys/types.h>
#include <setjmp.h>
#include <string.h>

/* local includes */
#include "lexicon.h"

/* libabase includes */
#include <abase_memalloc.h>
#include <abase_dstring.h>

/*------------------------------------------------------------------------------
// IEIEIE! Global alert!
// This is needed for error handling and should be removed as soon as possible
//----------------------------------------------------------------------------*/
jmp_buf lex_file_error_env;

typedef long INT_AFFIX;
typedef char* TEXT_AFFIX;

/*-------------------------------------------------------------------------
// Structure for storing the lexicon affixes
//
// name: affix name
// bitset: the bitset associated with this affix
// is_nont: if false, then this affix is a terminal, otherwise it's a
//          nonterminal
// lhs[]: the indexes of the left-hand-sides (affix nonterminals) that this
//        affix belongs to.
//-----------------------------------------------------------------------*/
typedef struct {
    char *name;
    LXCN_SET bitset;
    int is_nont;
    size_t nr_lhsses;
    off_t *lhs;
} SET_AFFIX;

/* Structure for storing the nonterminal parameters */
enum LexParamTypes {
    LexParamLattice = 1,
    LexParamText = 2,
    LexParamInt = 3
};

typedef struct {
    char type;
    off_t index;
} PARAMETER;

/* Structure for storing the lexicon rules (nonterminals) */
typedef struct {
    char *name;
    int arity;
    PARAMETER *pars;
} NONTERMINAL;

/* Structure for storing the entries */
typedef struct {
    unsigned nontnr;
    LXCN_PENALTY penalty;
    unsigned *pars;
} ENTRY;

/* Generic structure for storing pairs */
typedef struct {
    unsigned index;
    int pointer;
} PAIR;


/*------------------------------------------------------------------------------
// Lexicon structure
//----------------------------------------------------------------------------*/
struct LEXICON {
    lxcn_Trie* trie_buf;
    size_t trie_size;

    size_t nr_entrylist_pairs;
    PAIR *entrylist_pair;
    
    size_t nr_nonterminals;
    NONTERMINAL **nonterminal;
    
    size_t nr_entries;
    ENTRY **entry;
    
    size_t nr_int_affixes;
    INT_AFFIX *int_affix;
    
    size_t nr_text_affixes;
    TEXT_AFFIX *text_affix;
    
    size_t nr_set_affixes;
    SET_AFFIX **set_affix;
};


/*------------------------------------------------------------------------------
// Lexicon file access routines
// Note we will replace this code later with code that does not use longjmp
//----------------------------------------------------------------------------*/
static char 
read_byte (FILE* lex_file)
{   int c = fgetc(lex_file);

    if (c == EOF) longjmp(lex_file_error_env, 666);
    return ((char) c);
}

static int 
read_bool (FILE* lex_file)
{
    char c = read_byte (lex_file);
    return (c == 'T');
}

static char* 
read_string (FILE* lex_file)
{   dstring tmp = abs_init_dstring (10);
    char c;

    while ((c = read_byte(lex_file)) != 0)
       abs_append_dstring_c (tmp, c);

    /* Note finish deallocates */
    return (abs_finish_dstring (tmp));
}

/* This code is only used for reading a small number: to be replaced */
static int 
read_word (FILE* lex_file)
{
    int i = read_byte(lex_file);
    int j = read_byte(lex_file);

    return (i << 8) | j;
}

static LXCN_SET 
read_set(FILE* lex_file)
{
    LXCN_SET set;

    if (fread(&set, sizeof(set), 1, lex_file) != 1) longjmp(lex_file_error_env, 666);
    return set;
}

/* Note: code to be changed in typing */
static off_t 
read_index(FILE* lex_file)
{
    off_t res;

    if (fread(&res, sizeof(res), 1, lex_file) != 1) longjmp(lex_file_error_env, 666);
    return res;
}

static size_t 
read_size(FILE* lex_file)
{
    size_t res;

    if (fread(&res, sizeof(res), 1, lex_file) != 1) longjmp(lex_file_error_env, 666);
    return res;
}

static LXCN_PENALTY 
read_penalty(FILE* lex_file)
{
    LXCN_PENALTY res;

    if (fread(&res, sizeof(res), 1, lex_file) != 1) longjmp(lex_file_error_env, 666);
    return res;
}


/*------------------------------------------------------------------------------
// INT affixes
//----------------------------------------------------------------------------*/
static void 
read_int_affixes(FILE* lex_file, LEXICON* lex)
{
    off_t i;
    size_t nr = read_size (lex_file);

    lex->nr_int_affixes = nr;

    DB_LEX(abs_printf("reading %d INT affixes...\n", nr));

    if (nr) {
        lex->int_affix = (INT_AFFIX *) abs_calloc (nr, sizeof(INT_AFFIX),
						   "read_int_affixes");
  
        for (i = 0; i < nr; i++) {
            lex->int_affix[i] = read_set(lex_file);
        }
    } else lex->int_affix = NULL;
}

static void 
free_int_affixes (LEXICON* lex)
{   /* Note abs_free does a nullity check */
    abs_free (lex->int_affix, "free_int_affixes");
}


/*------------------------------------------------------------------------------
// TEXT affixes
//----------------------------------------------------------------------------*/
static void 
read_text_affixes(FILE* lex_file, LEXICON* lex)
{
    int i;
    TEXT_AFFIX input;
    size_t nr = read_size(lex_file);

    lex->nr_text_affixes = nr;

    DB_LEX(abs_printf("reading %d TEXT affixes...\n", nr));

    if (nr) {
        lex->text_affix = (TEXT_AFFIX*) abs_calloc (nr, sizeof(TEXT_AFFIX), "lex->text_affix");

        for (i = 0; i < nr; i++) {
            input = read_string(lex_file);
            lex->text_affix[i] = input;
            DB_LEX(abs_printf("\tread affix: \"%s\"\n", input));
        }
    } else lex->text_affix = NULL;
}

static void 
free_text_affixes(LEXICON* lex)
{
    off_t i;

    if ((lex->text_affix != NULL) && (lex->nr_text_affixes > 0)) {
        for (i = 0; i < lex->nr_text_affixes; i++) {
            abs_free (lex->text_affix[i], "free_text_affixes");
        }

        abs_free (lex->text_affix, "free_text_affixes: vector");
    }
}

/*------------------------------------------------------------------------------
// Affix names
//----------------------------------------------------------------------------*/
static void 
read_set_affixes (FILE* lex_file, LEXICON* lex)
{
    int i;
    SET_AFFIX* set_affix;

    size_t nr = read_size(lex_file);
    lex->nr_set_affixes = nr;

    DB_LEX(abs_printf("reading %d SET affixes...\n", nr));

    if (nr) {
        lex->set_affix = (SET_AFFIX **) abs_calloc (nr, sizeof(SET_AFFIX*), "lex->set_affixes");
        for(i = 0; i < nr; i++) {
            int j;
            off_t nr_lhsses;

            set_affix = (SET_AFFIX*) abs_malloc (sizeof(SET_AFFIX), "SET_AFFIX");
            set_affix->name = read_string(lex_file);
            DB_LEX(abs_printf("\tread affix %d: \"%s\"\n", i, set_affix->name));
            set_affix->bitset = read_set(lex_file);
            set_affix->is_nont = read_bool(lex_file);

            nr_lhsses = read_size(lex_file);
            set_affix->nr_lhsses = nr_lhsses;
            if (nr_lhsses) {
                set_affix->lhs = (off_t*) abs_calloc (nr_lhsses, sizeof(off_t), "SET_AFFIX->lhs");
                for (j = 0; j < nr_lhsses; j++) {
                    set_affix->lhs[j] = read_index(lex_file);
                    DB_LEX(abs_printf("\t\twith LHS: %d\n", (int)(set_affix->lhs[j])));
                }
            } else {
                set_affix->lhs = NULL;
            }

            lex->set_affix[i] = set_affix;
        }
    } else lex->set_affix = NULL;
}

static void 
free_set_affixes (LEXICON* lex)
{
    off_t i;
    size_t nr = lex->nr_set_affixes;

    if ((lex->set_affix != NULL) && (nr > 0)) {
        for (i = 0; i < nr; i++) {
            abs_free (lex->set_affix[i], "SET_AFFIX");
        }

        abs_free (lex->set_affix, "lex->set_affix");
    }
}


/*------------------------------------------------------------------------------
// Nonterminal names
//----------------------------------------------------------------------------*/
static void 
read_nonterminals(FILE* lex_file, LEXICON* lex)
{
    NONTERMINAL* nonterminal;
    off_t i, j;
    char* nont_name;

    size_t nr = read_size(lex_file);
    lex->nr_nonterminals = nr;

    DB_LEX(abs_printf("reading %d nonterminals\n", nr));

    if (nr) {
        lex->nonterminal = (NONTERMINAL**) abs_calloc (nr, sizeof(NONTERMINAL*), "lex->nonterminal");
        memset(lex->nonterminal, 0, sizeof(NONTERMINAL*) * nr);

        /* TODO: AS, 30/4/05: not all is freed, not clear how or why */
        for (i = 0; i < nr; i++) {
            nont_name = read_string(lex_file);

            if (nont_name[0] == '\0') {
                /* empty nonterminal entry -- this means this nonterminal is not a lexicon nonterminal */
                DB_LEX(abs_printf("reading nonterminal %d: syntax nonterminal\n", (int)i));
                abs_free(nont_name, "read_nonterminals, nont_name");
                nonterminal = NULL;
            } else {
                nonterminal = (NONTERMINAL*) abs_malloc (sizeof(NONTERMINAL), "read_nonterminals, nonterminal");
                nonterminal->name = nont_name;
                DB_LEX(abs_printf("reading nonterminal %d: \"%s/\n", (int)i, nonterminal->name));
                nonterminal->arity = read_word(lex_file);
                DB_LEX(abs_printf("%d\"\n", nonterminal->arity));
    
                if (nonterminal->arity) {
                    nonterminal->pars = (PARAMETER*)
				abs_calloc (nonterminal->arity, sizeof(PARAMETER), "read_nonterminals, parts");
    
                    for (j = 0; j < nonterminal->arity; j++) {
                        char type = read_byte(lex_file);
                        (nonterminal->pars[j]).type = type;
                        switch (type) {
                            case LexParamLattice:
                                (nonterminal->pars[j]).index =
                                    read_index(lex_file);
                                break;
                            case LexParamText:
                            case LexParamInt:
                                break;
                            default:
                                DB_LEX(abs_printf("read_nonterminals: Unknown parameter type 0x%x\n", type));
                                longjmp(lex_file_error_env, 666);
                        }
                    }
                } else {
                    nonterminal->pars = NULL;
                }
            }

            lex->nonterminal[i] = nonterminal;
        }
    } else {
        lex->nonterminal = NULL;
    }
}

static void 
free_nonterminals (LEXICON* lex)
{
    off_t i;
    size_t nr = lex->nr_nonterminals;

    /* TODO: AS, 30/4/05: not all is freed, not clear how or why */
    if ((lex->nonterminal != NULL) && (nr > 0)) {
        for (i = 0; i < nr; i++) {
            if (lex->nonterminal[i] != NULL) {
                abs_free ((lex->nonterminal[i])->name, "free_nonterminals: NONT");
                abs_free ((lex->nonterminal[i])->pars, "free_nonterminals: PARAMS");
            }
        }

        abs_free (lex->nonterminal, "free_nonterminals");
    }
}

/*------------------------------------------------------------------------------
// Trie alloc/read/free routines:
//----------------------------------------------------------------------------*/
#define TS_LENGTH sizeof(long)
static void 
read_trie_size(FILE* lex_file, LEXICON* lex)
{
    unsigned long ptr;

    if (fread((char *)&ptr, TS_LENGTH, 1, lex_file) != 1) {
        longjmp(lex_file_error_env, 666);
    }

    DB_LEX(abs_printf("read_trie_size: size = %lu bytes\n", ptr));
    lex->trie_size = ptr;
}

static void 
read_trie(FILE* lex_file, LEXICON* lex)
{
    read_trie_size(lex_file, lex);
    lex->trie_buf = abs_malloc (lex->trie_size, "read_trie");
  
    if (fread(lex->trie_buf, 1, lex->trie_size, lex_file) != lex->trie_size) {
        longjmp(lex_file_error_env, 666);
    }

    DB_LEX(abs_printf("read_trie: %d bytes\n", lex->trie_size));
}

static void 
free_trie(LEXICON* lex)
{
  if (lex->trie_buf != NULL) abs_free (lex->trie_buf, "trie");
}

lxcn_Trie* 
lxcn_get_lexicon_trie(const LEXICON* lex)
{
    assert ((lex != NULL) && "lxcn_get_lexicon_trie: No lexicon.");
    return (lex -> trie_buf);
}


/*------------------------------------------------------------------------------
// Lexicon entries handling
//----------------------------------------------------------------------------*/
static void 
read_entries(FILE* lex_file, LEXICON* lex)
{
    ENTRY* entry;
    int arity;
    int i, j;
    size_t nr = read_size(lex_file);
   
    lex->nr_entries = nr;
    lex->entry = (ENTRY**) abs_calloc (nr, sizeof(ENTRY*), "lex->entry");

    DB_LEX(abs_printf("-- There are %d entries\n", nr));

    for (i = 0; i < nr; i++) {
        entry = (ENTRY *) abs_malloc (sizeof(ENTRY), "ENTRY");

        entry->nontnr = read_index(lex_file);
        entry->penalty = read_penalty(lex_file);

        arity = lex->nonterminal[entry->nontnr]->arity;
        DB_LEX(abs_printf("Entry %d: %s", i, lex->nonterminal[entry->nontnr]->name));
        if (arity) {
            entry->pars = abs_calloc (arity, sizeof(off_t), "entry->pars");

            DB_LEX(abs_printf("("));
            for (j = 0; j < arity; j++) {
                entry->pars[j] = read_index(lex_file);
#ifdef DEBUG_LEX
                abs_printf("\"%s\"", lex->set_affix[entry->pars[j]]->name);
                if ((j + 1) != arity) {
                    abs_printf(", ");
                }
#endif /* DEBUG_LEX */
            }
            DB_LEX(abs_printf(")"));
        } else {
            entry->pars = NULL;
        }
        DB_LEX(abs_printf("\n"));

        lex->entry[i] = entry;
    }
}

static void 
free_entries(LEXICON* lex)
{
    off_t i;
    size_t nr = lex->nr_entries;

    if ((lex->entry != NULL) && (nr > 0)) {
        for (i = 0; i < nr; i++) {
            abs_free (lex->entry[i], "ENTRY");
        }

        abs_free (lex->entry, "lex->entries");
    }
}


/*------------------------------------------------------------------------------
// Entry list pairs
//----------------------------------------------------------------------------*/
static void 
read_entry_lists_pairs(FILE* lex_file, LEXICON* lex)
{
    off_t i;
    size_t nr = read_size(lex_file);

    lex->nr_entrylist_pairs = nr;
    lex->entrylist_pair = (PAIR*) abs_calloc (nr, sizeof(PAIR), "lex->entrylist_pair");

    for (i = 0; i < nr; i++) {
        lex->entrylist_pair[i].index = read_index(lex_file);
        lex->entrylist_pair[i].pointer = read_index(lex_file);
    }
}

static void 
free_entry_list_pairs(LEXICON* lex)
{
    if (lex->entrylist_pair != NULL) {
        abs_free (lex->entrylist_pair, "lex->entrylist_pair");
    }
}


/*--------------------------------------------------------------------------------
// Function:
//	int lexicon_affix_belongs_to_lhs(LEXICON* lex, off_t aff_nr, off_t lhs_nr)
//
// Description:
//	Checks if affix aff_nr belongs to nontemrinal lhs_nr.
//
// Return value:
//	true if check succeeds, false otherwise.
//------------------------------------------------------------------------------*/
int 
lexicon_affix_belongs_to_lhs (LEXICON* lex, off_t aff_nr, off_t lhs_nr)
{
    SET_AFFIX* aff_entry;
    unsigned i;

    assert(lhs_nr <= lex->nr_set_affixes);
    assert(aff_nr <= lex->nr_set_affixes);

    aff_entry = lex->set_affix[aff_nr];
    for (i = 0; i < aff_entry->nr_lhsses; ++i) {
        if ((aff_entry->lhs[i]) == lhs_nr) {
            return (1);
        }
    }

    return (0);
}

/*------------------------------------------------------------------------------
// Function:
//	int lxcn_try_advance_to_next_entry_in_list(const LEXICON* lex, off_t* lst)
//
// Description:
//	replaces the list index by the next entry's list index.
//
// Return value:
//	0 (false) if there are no more entries;
//	true otherwise.
//----------------------------------------------------------------------------*/
int 
lxcn_try_advance_to_next_entry_in_list (const LEXICON* lex, off_t* lst)
{
    assert(lst);
    *lst = lex->entrylist_pair[*lst].pointer;
    return *lst != -1;
}

void 
lxcn_get_params_from_entry_in_list (const LEXICON* lex, off_t entry_list, off_t *nontnr,
				    unsigned *arity, LXCN_PENALTY *penalty, const LXCN_PARAM **params)
{
    char type;
    off_t i, par_idx;
    ENTRY* cur_entry = lex->entry[lex->entrylist_pair[entry_list].index];
    NONTERMINAL* nonterminal;
    LXCN_PARAM* cur_par;
    *nontnr = cur_entry->nontnr;
    nonterminal = lex->nonterminal[*nontnr];
    *arity = nonterminal->arity;
    *penalty = cur_entry->penalty;

    if (!(*arity)) {
        *params = NULL;
        return;
    }

    cur_par = (LXCN_PARAM*) abs_calloc (*arity, sizeof(LXCN_PARAM), "state");
    *params = cur_par;

    for (i = 0; i < (*arity); i++) {
        type = (nonterminal->pars[i]).type;
        par_idx = cur_entry->pars[i];
        switch (type) {
            case LexParamInt:
                cur_par->kind = lxcn_IntKind;
                cur_par->value.int_par = lex->int_affix[par_idx];
                break;
            case LexParamText:
                cur_par->kind = lxcn_TextKind;
                cur_par->value.text_par = lex->text_affix[par_idx];
                break;
            case LexParamLattice:
                cur_par->kind = lxcn_SetKind;
                cur_par->value.set_par = lex->set_affix[par_idx]->bitset;
                break;
            default:
                DB_LEX(abs_printf("get_params_from_entry_in_list: Error: unknown parameter type\n"));
                longjmp(lex_file_error_env, 666);
        }
        cur_par++;
    }
}


/*------------------------------------------------------------------------------
// Lexicon acces routines
//----------------------------------------------------------------------------*/
size_t 
lexicon_get_nr_nonterminals(LEXICON* lex)
{
    assert(lex);
    return (lex -> nr_nonterminals);
}

char* 
lexicon_get_nont_name(LEXICON* lex, off_t nont_nr)
{
    assert(lex);
    assert((nont_nr >= 0) && (nont_nr < lex->nr_nonterminals));

    return lex->nonterminal[nont_nr]->name;
}

int 
lexicon_nont_exists(LEXICON* lex, off_t nont_nr)
{
    assert(lex);

    if ((nont_nr < 0) || (nont_nr > lex->nr_nonterminals)) {
        return (0);
    }

    return (lex->nonterminal[nont_nr] != NULL);
}

size_t 
lexicon_get_nont_arity(LEXICON* lex, off_t nont_nr)
{
    assert(lex);
    assert((nont_nr >= 0) && (nont_nr < lex->nr_nonterminals));

    return lex->nonterminal[nont_nr]->arity;
}

off_t 
lxcn_find_lex_nonterminal_with_arity(LEXICON* lex, char const *nname, int arity)
/* Simple but expensive search, for now.
** Returns nontnr, or -1 if not found.
*/
{
    size_t nr_l_n = lexicon_get_nr_nonterminals(lex);
    off_t retval;
    for (retval = 0; retval < nr_l_n; retval++) {
        if (lexicon_nont_exists(lex, retval)
	    && ((lex->nonterminal[retval])->arity == arity)
            && !strcmp((lex->nonterminal[retval])->name, nname)) {
                return retval;
        }
    }
    return -1;  /* not found */
}

char* 
lexicon_get_nont_param_name(LEXICON* lex, off_t nont_nr, off_t param_nr)
{
    off_t idx;
    char type;

    assert(lex);
    assert((nont_nr >= 0) && (nont_nr < lex->nr_nonterminals));
    assert((param_nr >= 0) && (param_nr < lex->nonterminal[nont_nr]->arity));

    idx = lex->nonterminal[nont_nr]->pars[param_nr].index;
    type = lex->nonterminal[nont_nr]->pars[param_nr].type;

    switch (type) {
        case LexParamText:
            return "TEXT";
        case LexParamInt:
            return "INT";
        case LexParamLattice:
            return lex->set_affix[idx]->name;
    }

    /* Should be unreachable: */
    return "Unknown type";
}

off_t 
lexicon_get_nont_param_nr(LEXICON* lex, off_t nont_nr, off_t param_nr)
{
    assert(lex);
    assert((nont_nr >= 0) && (nont_nr < lex->nr_nonterminals));
    assert((param_nr >= 0) && (param_nr < lex->nonterminal[nont_nr]->arity));

    return lex->nonterminal[nont_nr]->pars[param_nr].index;
}

size_t 
lexicon_get_nr_setaffixes(LEXICON* lex)
{
    assert(lex);
    return lex->nr_set_affixes;
}

char* 
lexicon_get_setaffix_name(LEXICON* lex, off_t aff_nr)
{
    assert(lex);
    assert(aff_nr >= 0);
    assert(aff_nr < lex->nr_set_affixes);

    return lex->set_affix[aff_nr]->name;
}

LXCN_SET 
lexicon_get_setaffix_bitset(LEXICON* lex, off_t aff_nr)
{
    assert (lex);
    assert(aff_nr >= 0);
    assert(aff_nr < lex->nr_set_affixes);

    return lex->set_affix[aff_nr]->bitset;
}

int 
lexicon_get_setaffix_nont_flag(LEXICON* lex, off_t aff_nr)
{
    assert (lex);
    assert(aff_nr >= 0);
    assert(aff_nr < lex->nr_set_affixes);

    return lex->set_affix[aff_nr]->is_nont;
}

size_t 
lexicon_get_setaffix_nr_lhsses(LEXICON* lex, off_t aff_nr)
{
    assert (lex);
    assert(aff_nr >= 0);
    assert(aff_nr < lex->nr_set_affixes);

    return lex->set_affix[aff_nr]->nr_lhsses;
}

char* 
lexicon_get_setaffix_lhs_name(LEXICON* lex, off_t aff_nr, off_t lhs_nr)
{
    off_t affix_idx;

    assert (lex);
    assert(aff_nr >= 0);
    assert(aff_nr < lex->nr_set_affixes);
    assert(lhs_nr >= 0);
    assert(lhs_nr < lex->set_affix[aff_nr]->nr_lhsses);

    affix_idx = lex->set_affix[aff_nr]->lhs[lhs_nr];

    return lexicon_get_setaffix_name(lex, affix_idx);
}


/*------------------------------------------------------------------------------
// Frees all lexicon tables
//----------------------------------------------------------------------------*/
void 
lexicon_free (LEXICON* lex)
{
    if (lex != NULL)
       { free_trie(lex);
         free_set_affixes(lex);
         free_text_affixes(lex);
         free_int_affixes(lex);
         free_nonterminals(lex);
         free_entries(lex);
         free_entry_list_pairs(lex);

         abs_free (lex, "lexicon_free");
       };
}


/*------------------------------------------------------------------------------
// lexicon initialization
//----------------------------------------------------------------------------*/
LEXICON* 
lexicon_new (FILE* lex_file)
{
    LEXICON* res = (LEXICON*) abs_malloc (sizeof (LEXICON), "lexicon_new");
    /* set all to known value's (and don't trust the machine to do it) */
    memset (res, '\0', sizeof(LEXICON));

    /* but NULL != '\0', for that we need knowledge about the struct! */
    res->trie_buf = NULL;
    res->entrylist_pair = NULL;
    res->nonterminal = NULL;
    res->entry = NULL;
    res->int_affix = NULL;
    res->text_affix = NULL;
    res->set_affix = NULL;

    /* MS: We need to get rid of this setjmp asap */
    if (!setjmp(lex_file_error_env)) {
        /* now try to fill it */
        read_trie(lex_file, res);
        read_int_affixes(lex_file, res);
        read_text_affixes(lex_file, res);
        read_set_affixes(lex_file, res);
        read_nonterminals(lex_file, res);
        read_entries(lex_file, res);
        read_entry_lists_pairs(lex_file, res);

        return (res);
    } else {
        /*
        ** If a read error occurs, we return NULL, 
        ** but free any memory allocated.
        ** (lexicon_free checks for NULL pointers)
        */
        lexicon_free (res);
        return (NULL);
    }
}

