/*
   File: abase_input.c
   Preparing lexer input

   Copyright 2005 Radboud University of Nijmegen
 
   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.
 
   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU Library General Public License for more details.
 
   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.

   CVS ID: "$Id: abase_lex_input.c,v 1.3 2005/05/04 13:18:53 marcs Exp $
*/
#ifdef HAVE_CONFIG_H
#include <config.h>
#endif /* HAVE_CONFIG_H */

/* standard includes */
#include <stdio.h>
#include <ctype.h>
#include <string.h>

/* abase includes */
// #include "abase_string.h"
#include "abase_lex_input.h"

static void set_table (unsigned char* table, const unsigned char* str);
static void set_trans_table (unsigned char* table, const unsigned char* src, const unsigned char* dst);
static void set_list_delimiters(const unsigned char* delims);

/*------------------------------------------------------------------------------
// Global data
//----------------------------------------------------------------------------*/
static unsigned char        blank_table[256];
static unsigned char        terminator_table[256];
static unsigned char        invisible_table[256];
static unsigned char        translate_table[256];
static unsigned char        delimiter_table[3];

/*------------------------------------------------------------------------------
// Token types and markers
//----------------------------------------------------------------------------*/
unsigned char
abs_get_lex_mark(abs_LexemeType lex_type)
{ unsigned char lex_mark;
  switch (lex_type) {
    case Prefix:
      lex_mark = abs_PrefixMark;
      break;
    case Infix:
      lex_mark = abs_InfixMark;
      break;
    case Suffix:
      lex_mark = abs_SuffixMark;
      break;
    case MultiToken:
      lex_mark = abs_MultiTokenMark;
      break;
    default:
      lex_mark = abs_EmptyMark;
      break;
  }
  return (lex_mark);
}

int 
(abs_is_eos)(unsigned char c)
{ return (c == abs_EosMark);
}

int 
(abs_is_space)(unsigned char c)
{ return (c == abs_SpaceMark);
}

int 
abs_is_blank(unsigned char c)
{ return (blank_table[(int)c] != '\0');
}

int 
abs_is_terminator(unsigned char c)
{ return (terminator_table[(int)c] != '\0');
}

int 
abs_is_invisible(unsigned char c)
{ return (invisible_table[(int)c] != '\0');
}

int 
abs_translate(unsigned char c)
{ return (translate_table[(int)c]);
}

void 
abs_init_char_tables(const char *blanks, const char *terminators, const char *invisibles, 
	const char *translate_src, const char *translate_dst, const char *delimiters)
{ set_table (blank_table, blanks);
  set_table (terminator_table, terminators);
  set_table (invisible_table, invisibles);
  set_trans_table (translate_table, translate_src, translate_dst);
  terminator_table[abs_EosMark] = 1;
  set_list_delimiters (delimiters);
}

#if USING_MATCH_LIST
int 
abs_is_delimiter (unsigned char c, Delimiter delim)
{ unsigned char d = delimiter_table[delim];
  return ((c == d) && (d != '\0'));
}
#endif

/*------------------------------------------------------------------------------
// Table functions
//----------------------------------------------------------------------------*/
static void 
set_table (unsigned char* table, const unsigned char* str)
{ unsigned char c;
  unsigned i;
  if (str == NULL) abs_abort ("set_table", "called with NULL string");
  for (i = 0; i < 256; i++) table[i] = '\0';
  while ((c = *str++)) table[c] = c;
}

static void 
set_trans_table (unsigned char* table, const unsigned char* src, const unsigned char* dst)
{ unsigned char c;
  unsigned i;
  if (src == NULL || dst == NULL) {
	abs_abort ("set_trans_table", "called with NULL string");
  }
  for (i = 0; i < 256; i++) table[i] = i;
  while ((c = *src++)) table[c] = *dst++;
}

typedef enum
{ Opener = 0,
  Middler = 1,
  Closer = 2
} Delimiter;

static void 
set_list_delimiters(const unsigned char* delims)
{ if (delims == NULL)
  { delimiter_table[0] = abs_EosMark;
    delimiter_table[1] = abs_EosMark;
    delimiter_table[2] = abs_EosMark;
  }
  else
  { delimiter_table[Opener] = delims[0];
    delimiter_table[Middler] = delims[1];
    delimiter_table[Closer] = delims[2];
  }
}

/*------------------------------------------------------------------------------
// Function:
//        static int
//        check_terminator(abs_SeparatorType sep_type, unsigned char c)
//
// Description:
//        Check whether c is allowed in the context of sep_type.
//        If sep_type is SepRequired, then c must be a terminator,
//        //else if sep_type is SepNoBlank then c must not be a blank,
//        else, don't care.
//
// Return value:
//        True if c allowed with sep_type, or false else.
//----------------------------------------------------------------------------*/
int 
abs_check_terminator (abs_SeparatorType sep_type, unsigned char c)
{ switch (sep_type)
    { case SepDontCare:	return (1);
      case SepRequired:	return (abs_is_terminator (c));
      /* case SepNoBlank: */
      /* return !abs_is_blank(c); */
      /* break; */
      default: abs_abort ("check_terminator", "bad sep_type %d", (int) sep_type);
  }
  return (0);
}

