#define AGFL
/* Implementation of regular expression matching interface.
 *
 * Copyright 2001, KUN.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

/* $Id: pattern.c,v 1.8 2001/10/17 09:30:15 ejv Exp $ */

#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <sys/types.h>
#include <regex.h>
#include "pattern.h"
#ifdef RTS
#include "rtsutil.h"
#endif

#if !defined(AGFL) && !defined(RTS)
#define	STAND_ALONE
#endif

/*
//------------------------------------------------------------------------------
// Module interface implementation
//------------------------------------------------------------------------------
*/

#if defined(STAND_ALONE) || defined(AGFL)

static void*
reg_get_mem(size_t size)
{
  void* mem = malloc(size);
  if (mem == NULL)
  {
    fprintf(stderr, "Out of memory\n");
    exit(1);
  }
  return mem;
}

static void
reg_free_mem(void* mem)
{
  free(mem);
}

#else /* !(STAND_ALONE || AGFL) */
#ifdef RTS

static const char module_name[]	= "pattern.c";

#define reg_get_mem(sz)		(GetMem((sz), module_name))
#define reg_free_mem(mem)	(FreeMem((mem), module_name))

#endif /* RTS */
#endif /* (STAND_ALONE || AGFL) */


/*
**------------------------------------------------------------------------------
** Regexp wrapper functions.
**------------------------------------------------------------------------------
*/

#ifndef RTS
static char* word_terminator_chars = "";
#endif /* RTS */

#include "regex.h"
#include "regex.c"


RegExp*
new_regexp(const char* regexp_txt, int* error_p)
{
  regex_t* buf = reg_get_mem(sizeof(regex_t));
  *error_p = regcomp(buf, regexp_txt, REG_EXTENDED);
  return (RegExp*)buf;
}

void
delete_regexp(RegExp* regexp)
{
  regfree((regex_t*)regexp);
  reg_free_mem(regexp);
}

#define NMATCH	1

const char*
match_regexp(const char* str, RegExp* regexp)
{
  regmatch_t pmatch[NMATCH];
  int ret = regexec((regex_t*)regexp, str, NMATCH, pmatch, 0);
  if (ret == 0 && pmatch[0].rm_so == 0)
    return (const char*)str + pmatch[0].rm_eo;
  return NULL;
}

size_t
regexp_error(int error, RegExp* regexp, char* buf, size_t sz)
{
  return regerror(error, (regex_t*)regexp, buf, sz);
}

/*
//------------------------------------------------------------------------------
// Define AGFL for CDL3 interface for Agfl compiler.
//
// EXTERNAL TEST validate regexp (>TEXT, INT>):400
//
// Description:
//	Check validity of regular expression TEXT. If valid, assign
//	the number of fields in regexp to INT, and succeed.
//	Note: number of fields not implemented.
//
// Side effects:
//	If the expression TEXT is erroneous, print an error message
//	on standard error.
//------------------------------------------------------------------------------
*/

#ifdef AGFL

#include <cdl3rts.h>

void my_abort();	/* TODO: include util.h */

int E400_validate_regexp_TEXT_INT(value v_TEXT, value* v_INT)
{
  int error;
  RegExp* comp = new_regexp(Text(v_TEXT), &error);
  if (error == 0)
  {
    *v_INT = C_INT(1L);		/* number of fields: not implemented */
    delete_regexp(comp);
    return 1;
  }
  if (error < 0)
    my_abort("regexp not supported, please contact agfl@cs.kun.nl!");
  else
  {
    enum { BufSz = 256 };
    char buf[BufSz];
    regexp_error(error, comp, buf, BufSz);
    fprintf(stderr, "agfl: regexp error: %s\n", buf);
  }
  delete_regexp(comp);
  return 0;
}

#endif /* AGFL */

/*
//------------------------------------------------------------------------------
// Define STAND_ALONE for stand-alone executable with simple
// user interface for testing regexps.
//------------------------------------------------------------------------------
*/

#ifdef STAND_ALONE

static char* chop(char* p)
{
  p[strlen(p) - 1] = '\0';
  return p;
}

static char* prompt(char* buf)
{
  enum { MaxIn = 1024 };
  fprintf(stderr, "pattern> ");
  if (fgets(buf, MaxIn, stdin) && buf[0] != '\n')
    return buf;
  else
    return NULL;
}

int main(int argc, char *argv[])
{
  enum { BufSz = 1024 };

  char		buf[BufSz];
  RegExp*	regexp;
  int		error;

  if (argc != 2)
  {
    fprintf(stderr, "Usage: %s <regexp>\n", argv[0]);
    exit(1);
  };

  regexp = new_regexp(argv[1], &error);
  if (error < 0)
  {
    fprintf(stderr, "regex functions not available... update your OS!\n");
    exit(1);
  }
  if (error)
  {
    regexp_error(error, regexp, buf, BufSz);
    fprintf(stderr, "Error %d: %s\n", error, buf);
    exit(1);
  };
  while(prompt(buf))
  {
    char* next;

    chop(buf);
    next = (char*)match_regexp(buf, regexp);
    if (next == NULL)
      printf("No match\n");
    else if (*next == '\0')
      printf("Matched: `%s\'\n", buf);
    else
    {
      *next = '\0';
      printf("Matched prefix: `%s\'\n", buf);
    }
  }
  delete_regexp(regexp);
  return 0;
}

#endif /* STAND_ALONE */
