//
// Copyright (C) 1999, 2000, Marco Kesseler
//

implementation module matrix

import StdList, StdFunc, StdMisc
import basic

//
//	Some standard matrix and vector operations
//

:: Vector a :== [a]
:: Matrix a :== Vector (Vector a)

matMap ::(a -> b) (Matrix a) -> (Matrix b)
matMap f mat = map (map f) mat

matMap2 :: (a b -> c) (Matrix a) (Matrix b) -> (Matrix c)
matMap2 f a b = map2 (map2 f) a b

matMap3 :: (a b c -> d) (Matrix a) (Matrix b) (Matrix c) -> (Matrix d)
matMap3 f a b c = map3 (map3 f) a b c

matMap4 :: (a b c d -> e) (Matrix a) (Matrix b) (Matrix c) (Matrix d) -> (Matrix e)
matMap4 f a b c d = map4 (map4 f) a b c d

//
// The following function collapses a matrix of matrices into
// one big matrix. It first transposes each row of submatrices into
// a small column in which each element is a list of submatix-rows.
// These lists of rows are then flattened into one big row. And finally the
// set of columns is flattened.

matFlatten :: (Matrix (Matrix a)) -> Matrix a
matFlatten mat = (flatten o map ((map flatten) o transpose)) mat

//
// The following function transforms a list of matrices into a matrix
// of lists. First the list of matrices is transposed into a column
// in which each element is a list of rows. Each of these elements
// is then transposed into a row of lists.
//

matZip :: [Matrix a] -> Matrix [a]
matZip list = map transpose (transpose list)

matZip2 :: [Matrix a] -> Matrix (a, a)
matZip2 [a, b] = map2 zip2 a b

matZip3 :: [Matrix a] -> Matrix (a, a, a)
matZip3 [a, b, c] = map3 zip3 a b c

matZip4 :: [Matrix a] -> Matrix (a, a, a, a)
matZip4 [a, b, c, d] = map4 zip4 a b c d

// The following function transforms a matrix of lists into a list of
// matrices. First each row of lists is transposed into a list of rows.
// The resulting column of lists is then transposed into a list of collumns.

matUnzip :: (Matrix [a]) -> [Matrix a]
matUnzip mat = transpose (map transpose mat)

matrixVectorProduct :: (Matrix Real) (Vector Real) -> Vector Real
matrixVectorProduct matrix vector = map (inproduct vector) matrix
where
	inproduct :: [Real] [Real] -> Real
	inproduct x y = inprod x y 0.0
	where
		inprod [x : xs] [y : ys] acc
			= inprod xs ys (acc + x * y)
		inprod [] [] acc
			= acc
		inprod _ _ _ = abort "inproduct: vectors have different size"

// Be aware that transpose[] will deliver an infinite list...

transpose :: (Matrix a) -> Matrix a
transpose [row : rows]
	= appendCollumns row (transpose rows)
	where
		appendCollumns [e : es] [col : cols]
			= [[e : col] : appendCollumns es cols]
		appendCollumns _ _ = []
		
transpose [] = repeat []

