//
// Copyright (C) 1999, 2000, Marco Kesseler
//

implementation module colourTransform

import StdReal, StdArray, StdEnum, StdList
import basic

// The following will be needed for 16 bit samples
//SCALEBITS :== 14	/* avoid overflow */

SCALEBITS	:== 16	/* speedier right-shift on some machines */
CONST_SCALE :== toReal (1 << SCALEBITS)

/* Convert a positive real constant to an integer scaled by CONST_SCALE. */
FIX x :== round (x * CONST_SCALE)

/**************************************************************************************
 *
 * The following code and comments are based on jdmaster.c, by Thomas G. Lane.,
 * part of the Idependent JPEG Group's software, Release 6b.
 *
 * We need to range-limit values to the range
 * 0..MAXSAMPLE; the input value may fall somewhat outside this range
 * due to noise introduced by quantization, roundoff error, etc.
 * In addition, just after the IDCT, a wildly out-of-range value is 
 * possible if the input data is corrupt.  To avoid any chance of indexing
 * off the end of memory and getting a bad-pointer trap, we perform the
 * post-IDCT limiting thus:
 *		x = range_limit[x & MASK];
 * where MASK is 2 bits wider than legal sample data, ie 10 bits for 8-bit
 * samples.  Under normal circumstances this is more than enough range and
 * a correct output will be generated; with bogus input data the mask will
 * cause wraparound, and we will safely generate a bogus-but-in-range output.
 * For the post-IDCT step, we want to convert the data from signed to unsigned
 * representation by adding CENTERJSAMPLE at the same time that we limit it.
 * So the post-IDCT limiting table ends up looking like this:
 *   CENTERJSAMPLE,CENTERJSAMPLE+1,...,MAXJSAMPLE,
 *   MAXJSAMPLE (repeat 2*(MAXJSAMPLE+1)-CENTERJSAMPLE times),
 *   0          (repeat 2*(MAXJSAMPLE+1)-CENTERJSAMPLE times),
 *   0,1,...,CENTERJSAMPLE-1
 * Negative inputs select values from the upper half of the table after
 * masking.
 *
 ****************************************************************************/

/**************** YCbCr -> RGB conversion: most common case **************/
/*
 * YCbCr is defined per CCIR 601-1, except that Cb and Cr are
 * normalized to the range 0..MAXJSAMPLE rather than -0.5 .. 0.5.
 * The conversion equations to be implemented are therefore
 *	R = Y                + 1.40200 * Cr
 *	G = Y - 0.34414 * Cb - 0.71414 * Cr
 *	B = Y + 1.77200 * Cb
 * where Cb and Cr represent the incoming values less MAXJSAMPLE/2.
 * (These numbers are derived from TIFF 6.0 section 21, dated 3-June-92.)
 *
 * To avoid floating-point arithmetic, we represent the fractional constants
 * as integers scaled up by 2^16 (about 4 digits precision); we have to divide
 * the products by 2^16, with appropriate rounding, to get the correct answer.
 * Notice that Y, being an integral input, does not contribute any fraction
 * so it need not participate in the rounding.
 *
 * For even more speed, we avoid doing any multiplications in the inner loop
 * by precalculating the constants times Cb and Cr for all possible values.
 * For 8-bit JSAMPLEs this is very reasonable (only 256 entries per table);
 * for 12-bit samples it is still acceptable.  It's not very reasonable for
 * 16-bit samples, but if you want lossless storage you shouldn't be changing
 * colorspace anyway.
 * The Cr=>R and Cb=>B values can be rounded to integers in advance; the
 * values for the G calculation are left scaled up, since we must add them
 * together before rounding.
 */

:: RangeTable :== {#Int}

:: YccTables =
	{
		cr2R :: {#Int},
		cr2G :: {#Int},
		cb2B :: {#Int},
		cb2G :: {#Int}
	}

:: Colour1Transformer :== !Int -> Int
:: Colour2Transformer :== (!Int, !Int) -> (!Int, !Int)
:: Colour3Transformer :== (!Int, !Int, !Int) -> (!Int, !Int, !Int)
:: Colour4Transformer :== (!Int, !Int, !Int, !Int) -> (!Int, !Int, !Int, Int)
:: ColourNTransformer :== [Int] -> [Int]

:: ColourTransformRecord =
	{
		signedRangeLimit	:: Colour1Transformer,
		unsignedRangeLimit	:: Colour1Transformer,
		ycc2RGB 			:: Colour3Transformer,
		yccK2CMYK			:: Colour4Transformer
	}

//
// Actually, only precision 8 and 12 will be used for JPEG decoding. We don't
// care. The other entries will never be evaluated.

SignedRangeLimitFunction :: !Int -> Colour1Transformer
SignedRangeLimitFunction precision = ColourTransformFunctions.[precision].signedRangeLimit

YccK2CMYKFunction :: !Int -> Colour4Transformer
YccK2CMYKFunction precision = ColourTransformFunctions.[precision].yccK2CMYK

Ycc2RGBFunction :: !Int -> Colour3Transformer
Ycc2RGBFunction precision = ColourTransformFunctions.[precision].ycc2RGB

ColourTransformFunctions :: {ColourTransformRecord}
ColourTransformFunctions =: {functions precision \\ precision <- [0..12]}
where
	functions precision =
		{
			signedRangeLimit	= SignedRangeLimitFunction RangeTable rangeMask centerSample,
			unsignedRangeLimit	= UnsignedRangeLimitFunction RangeTable rangeMask,
			ycc2RGB				= Ycc2RGB RangeTable YccTables rangeMask,
			yccK2CMYK			= YccK2CMYK RangeTable YccTables rangeMask maxSample
		}
	where
	
		positive_valid	=> [0..maxSample]
		maxSample		= (1 << precision) - 1
		centerSample	= 1 << (precision - 1)
		extraEntries	= 2 * (maxSample + 1) - centerSample
		rangeMask 		= (1 << (precision + 2)) - 1

		SignedRangeLimitFunction :: RangeTable Int Int Int -> Int
		SignedRangeLimitFunction table rangeMask centerSample i
			= table.[(i bitand rangeMask) + centerSample]

		UnsignedRangeLimitFunction :: RangeTable Int Int -> Int
		UnsignedRangeLimitFunction table rangeMask i
			= table.[i bitand rangeMask]

		RangeTable = {s \\ s <- (positive_valid ++ positive_invalid ++ negative_invalid ++ negative_valid)}
		where
			positive_invalid	= repeatn extraEntries maxSample
			negative_invalid 	= repeatn extraEntries 0
			negative_valid		= [0..(centerSample - 1)]

		Ycc2RGB :: RangeTable YccTables Int (Int, Int, Int) -> (Int, Int, Int)
		Ycc2RGB range ycc rangeMask (y, cb, cr)
		= let!
			/* Note: if the inputs were computed directly from RGB values,
    		 * range-limiting would be unnecessary here; but due to possible
   			 * noise in the DCT/IDCT phase, we do need to apply range limits.
    		 */
  			red		= range.[(y + ycc.cr2R.[cr]) bitand rangeMask]
   			green 	= range.[(y + ((ycc.cb2G.[cb] + ycc.cr2G.[cr]) >> SCALEBITS)) bitand rangeMask]
   		 	blue	= range.[(y + ycc.cb2B.[cb]) bitand rangeMask]
      	
			in (red, green, blue)
		
		YccK2CMYK :: RangeTable YccTables Int Int (Int, Int, Int, Int) -> (Int, Int, Int, Int)
		YccK2CMYK range ycc rangeMask maxSample (y, cb, cr, k)
		= let!
			/* Note: if the inputs were computed directly from RGB values,
			 * range-limiting would be unnecessary here; but due to possible
			 * noise in the DCT/IDCT phase, we do need to apply range limits.
			 */
			cyan 	= maxSample - range.[(y + ycc.cr2R.[cr]) bitand rangeMask]
			magenta = maxSample - range.[(y + ((ycc.cb2G.[cb] + ycc.cr2G.[cr]) >> SCALEBITS)) bitand rangeMask]
			yellow	= maxSample - range.[(y + ycc.cb2B.[cb]) bitand rangeMask]

			in (cyan, magenta, yellow, k)
		
		YccTables :: YccTables
		YccTables =
			{
				cr2R = {red x		\\ x <- xs},
				cr2G = {green_r x	\\ x <- xs},
				cb2B = {blue x		\\ x <- xs},
				cb2G = {green_b x	\\ x <- xs}
			}
		where	
			red x		= ((FIX (1.40200/2.0)) * x + half) >> SCALEBITS
			green_r x	= ~(FIX (0.71413636/2.0)) * x
    		blue x		= ((FIX (1.77200/2.0)) * x + half) >> SCALEBITS
			green_b x	= ~(FIX (0.3441363/2.0)) * x + half
			half		= 1 << (SCALEBITS-1)

			/* i is the actual input pixel value, in the range 0..MAXJSAMPLE */
			/* The Cb or Cr value we are thinking of is x = i - MAXJSAsMPLE/2 */

			xs			= [2*i - maxSample \\ i <- positive_valid]


