implementation module ioTypes


//////////////////////////////////////////////////
//                                               /
// Common types for the event I/O system.        /
//                                               /
// / / / / / / / / / / / / / / / / / / / / / / / /


import StdEnv
import picture
import intrface


  //-------------------------------------------------//
 //  The IO System                                  //
//-------------------------------------------------//


:: IOSystem *s *io :== [ DeviceSystem s io ]

:: DeviceSystem *s *io =  MenuSystem   [MenuDef   s io]
				 		| TimerSystem  [TimerDef  s io]
						| WindowSystem [WindowDef s io]
						| DialogSystem [DialogDef s io]

//
//	Definition of IOState: the environment parameter on which all
//									I/O functions operate.
//	Definition of EVENTS : the environment parameter to start I/O.
//
//	The operations on the IOState to create interactions.
//
:: *EVENTS :== *OS
:: *IOState *s :== !(!IOadmin s, !OS)


//---------------------------------------------------


:: IOadmin *s = { io_quit            :: !Bool,
			 	  io_menuState       :: !Perhaps (MenuWinAdmin s),
				  io_windowState     :: ![(WindowAdmin s)], 
				  io_timers          :: ![TimerAdmin s],
				  io_dialogs         :: ![DialogAdmin s],
				  io_noticeresult    :: !Perhaps NoticeButtonId,
				  io_ddehandler      :: !String ->(s -> *((IOState s) -> (s,IOState s))),
				  io_delayedmessages :: ![ CrossCallInfo ]
			    }


//	OpenEvents retrieves the event stream from the world.
OpenEvents :: !*World -> (!*EVENTS, !*World)
OpenEvents w
	| open_ok    =  (os, w)
    | otherwise  =  abort "This world doesn't contain events."
where
	(open_ok, os) = WinInitOs

 /*	CloseEvents replaces the event stream in the world. Event streams that
 	have been retrieved from the world, must be closed before they can be
 	opened again, otherwise a run-time error will occur. */
CloseEvents :: !*EVENTS !*World -> *World
CloseEvents os w 
  |  WinCloseOs os =  w
                   =  w 
 
PackIOState   :: !(IOadmin s) !OS -> IOState s
PackIOState admin os = (admin,os)

UnpackIOState :: !(IOState s) -> (!IOadmin s, !OS)
UnpackIOState io  =  io

UnpackIOStateWithCheck :: !(IOState s) -> (!IOadmin s, !OS)
UnpackIOStateWithCheck io=:(admin,os) 
  | admin.io_quit  =  abort "Operations on empty IOState not allowed.\n"
                   =  io

EventsToOs :: !EVENTS -> OS
EventsToOs ev = ev

OsToEvents :: !OS -> EVENTS
OsToEvents os = os

ChangeOs :: !(OS -> OS) !(IOState s) -> (IOState s) 
ChangeOs change (adm, os) = (adm, change os)


EmptyIOadmin :: IOadmin s
EmptyIOadmin = { io_quit             =  False,
			     io_menuState        =  Nope,
			 	 io_windowState      =  [],
				 io_timers           =  [],
				 io_dialogs          =  [],
				 io_noticeresult     =  Nope,
				 io_ddehandler       =  \c s io -> (s,io),
				 io_delayedmessages  =  []  
			   }




  //-------------------------------------------------//
 //  Common Defs and simple functions               //
//-------------------------------------------------//




:: ItemTitle     :== String

:: SelectState =  Able | Unable
:: MarkState   =  Mark | NoMark

:: KeyboardState :== (!KeyCode, !KeyState, !Modifiers)
:: KeyCode       :== Char
:: KeyState      =  KeyUp | KeyDown | KeyStillDown

:: MouseState    :== (!MousePosition, !ButtonState, !Modifiers)
:: MousePosition :== (!Int, !Int)
:: ButtonState   =  ButtonUp | ButtonDown | ButtonDoubleDown |
                    ButtonTripleDown | ButtonStillDown

/* Modifiers indicates the meta keys that have been pressed (TRUE)
   or not (FALSE): (Shift, Alternate/Meta, Control, Control)
*/

:: Modifiers     :== (!Bool,!Bool,!Bool,!Bool)

:: PictureDomain :== Rectangle


  //----------------------------------------------//
 //  The Defs for the menu device.               //
//----------------------------------------------//


    
:: MenuDef *s *io
       =  PullDownMenu MenuId MenuTitle SelectState [MenuElement s io]

:: MenuElement *s *io
       =  MenuItem MenuItemId ItemTitle KeyShortcut SelectState (MenuFunction s io)
         |  CheckMenuItem MenuItemId ItemTitle KeyShortcut SelectState MarkState
                          (MenuFunction s io)
         |  SubMenuItem MenuId ItemTitle SelectState [MenuElement s io]
         |  MenuItemGroup MenuItemGroupId [MenuElement s io]
         |  MenuRadioItems MenuItemId [RadioElement s io]
         |  MenuSeparator

:: RadioElement *s *io
       =  MenuRadioItem MenuItemId ItemTitle KeyShortcut SelectState
                        (MenuFunction s io)


:: MenuTitle             :== String
:: MenuFunction *s *io   :== !s -> *( !io -> (s, io)) 
:: MenuId                :== Int
:: MenuItemId            :== Int
:: MenuItemGroupId       :== Int
:: KeyShortcut           =  Key KeyCode | NoKey

//-----------------------------------------------------------------------

:: MenuKind  =  MenuBarK
			  | PullDownK
              | SubMenuK
              | ItemGroupK
              | RadioGroupK

:: MenuItemKind  =  RadioItemK
                  | CheckItemK
                  | ItemK

:: MenuAdmin *s = {  mkind     :: !MenuKind,
				     mid       :: !Int,
				     mtitle    :: !String,
				     melements :: ![MAdmin s],
				     menabled  :: !Bool,
				     mhandle   :: !HMENU,
					 mparent   :: !HMENU,
					 mpos      :: !Int
			       }

:: ItemAdmin *s = {  ikind     :: !MenuItemKind, 
				     iid       :: !Int,
					 ihandle   :: !HITEM,
				     ititle    :: !String,
				     ikey      :: !KeyShortcut,
				     ienabled  :: !Bool,
				     imarked   :: !Bool,
				     ifunction :: !MenuFunction s (IOState s),
				 	 iparent   :: !HMENU,
					 ipos      :: !Int
			       }

:: MenuWinAdmin *s = {  mbwinhandle :: !HWND,
				 	    mbmenu      :: !MAdmin s
				      }

:: MAdmin *s =  Menu !(MenuAdmin s)
              | Item !(ItemAdmin s)  
			  | Separator






  //------------------------------------------------//
 //  The Defs for the window device.               //
//------------------------------------------------//


:: WindowDef * s * io
   =  ScrollWindow WindowId WindowPos WindowTitle
                   ScrollBarDef ScrollBarDef
                   PictureDomain MinimumWindowSize InitialWindowSize
                   (UpdateFunction s) [WindowAttribute s io]
   |  FixedWindow  WindowId WindowPos WindowTitle
                   PictureDomain
                   (UpdateFunction s)  [WindowAttribute s io]
 
:: WindowId          :== Int
:: WindowPos         :== (!Int, !Int)
:: WindowTitle       :== String

:: ScrollBarDef      =  ScrollBar ThumbValue ScrollValue
:: ThumbValue        =  Thumb  Int
:: ScrollValue       =  Scroll Int
:: MinimumWindowSize :== (!Int, !Int)
:: InitialWindowSize :== (!Int, !Int)

:: UpdateFunction * s      :== UpdateArea -> s -> (s,[DrawFunction]) 
:: GoAwayFunction * s * io :== s ->  *(io -> (s, io))
 
:: WindowAttribute * s * io =
      Activate   (WindowFunction s io)
   |  Deactivate (WindowFunction s io)
   |  GoAway     (WindowFunction s io)
   |  Mouse      SelectState (MouseFunction    s io)
   |  Keyboard   SelectState (KeyboardFunction s io)
   |  Cursor     CursorShape
   |  StandByWindow 

:: WindowFunction   * s * io :== s ->  *(io -> (s, io) )
:: KeyboardFunction * s * io :== KeyboardState ->  s ->  *(io -> (s, io))  
:: MouseFunction    * s * io :== MouseState ->     s ->  *(io -> (s, io))  
 

:: CursorShape =  StandardCursor | BusyCursor     | IBeamCursor |
                  CrossCursor    | FatCrossCursor | ArrowCursor | HiddenCursor


//------------------------------------------------------------------


:: WindowKind =  ScrollK
			   | FixedK


:: WindowAdmin *s = {  wkind       :: !WindowKind,
					   wid         :: !Int,
					   whandle     :: !HWND,
			           wtitle      :: !String,
					   wminwidth   :: !Int,
					   wminheight  :: !Int,
					   wupdatef    :: !UpdateFunction s,
				 	   wdomain     :: !PictureDomain,
                       wpicstate   :: !PictureState,
					   whscroll    :: !Int,
					   wvscroll    :: !Int,  
					   whthumb     :: !Int,
					   wvthumb     :: !Int,  
					   wgoaway     :: !WindowFunction s (IOState s),
					   wactivate   :: !WindowFunction s (IOState s),
					   wdeactivate :: !WindowFunction s (IOState s),
					   wkeyboardf  :: !KeyboardFunction s (IOState s),
					   wkeybable   :: !Bool,
					   wmousef     :: !MouseFunction s (IOState s),
					   wmouseable  :: !Bool,
					   wcursor     :: !CursorShape
                    }				
				      



  //----------------------------------------------//
 //  The Defs for the timer device.              //
//----------------------------------------------//


:: TimerDef * s * io		=  Timer TimerId SelectState TimerInterval (TimerFunction s io)
:: TimerId      			:== Int
:: TimerInterval			:== Int
:: TimerFunction * s * io	:== TimerState ->  s -> *( io -> (s,io) ) 
:: TimerState    			:== Int


//-----------------------------------------------------------

:: TimerAdmin *s = { tid       :: !Int,
                     thandle   :: !HITEM,
                     tinterval :: !Int,
					 tlasttime :: !Int,
                     tfunction :: !TimerFunction s (IOState s),
				     table     :: !Bool
			  	   }


  //------------------------------------------------//
 //  The Defs for the dialog device.               //
//------------------------------------------------//


:: * DialogState * s * io :== (!DialogAdmin s, !OS)

:: DialogInfo :== [ (!DialogItemId, !ItemInfo) ]

:: ItemInfo = EditInfo !String
            | RadioGroupInfo !DialogItemId
			| CheckGroupInfo ![ (!DialogItemId, !Bool) ]
			| ControlStateInfo !ControlState 


:: DialogDef * s * io
   =  PropertyDialog DialogId DialogTitle [DialogAttribute] (SetFunction s io)
                     (ResetFunction s io) [DialogItem s io]
   |  CommandDialog  DialogId DialogTitle [DialogAttribute]
                     DialogItemId [DialogItem s io]
   |  AboutDialog    ApplicationName PictureDomain [DrawFunction]
                     (AboutHelpDef s io)

:: ApplicationName :== String
:: AboutHelpDef * s * io = NoHelp
                         | AboutHelp ItemTitle (AboutHelpFunction s io)
:: AboutHelpFunction * s * io :== s ->  *(io -> (s,io)) 

:: DialogAttribute
   =  DialogPos    Measure Measure
   |  DialogSize   Measure Measure
   |  DialogMargin Measure Measure
   |  ItemSpace    Measure Measure
   |  StandByDialog

:: DialogId    :== Int
:: DialogTitle :== String
:: Measure     =   MM Real | Inch Real | Pixel Int
 
:: DialogItem * s * io
   =  DialogButton DialogItemId ItemPos ItemTitle SelectState (ButtonFunction s io)
   |  DialogIconButton DialogItemId ItemPos PictureDomain IconLook SelectState (ButtonFunction s io)
   |  StaticText   DialogItemId ItemPos String
   |  DynamicText  DialogItemId ItemPos TextWidth String
   |  EditText     DialogItemId ItemPos TextWidth NrEditLines String
   |  DialogPopUp  DialogItemId ItemPos SelectState DialogItemId [RadioItemDef s io]
   |  RadioButtons DialogItemId ItemPos RowsOrColumns DialogItemId [RadioItemDef s io]
   |  CheckBoxes   DialogItemId ItemPos RowsOrColumns [CheckBoxDef s io]
   |  Control      DialogItemId ItemPos PictureDomain SelectState ControlState
                       ControlLook ControlFeel (DialogFunction s io)
 
:: DialogItemId  :== Int
:: RowsOrColumns =  Rows Int | Columns Int
:: ItemPos       =  Left | Center | Right        |
                    RightTo DialogItemId         |
                    Below DialogItemId           |
                    XOffset DialogItemId Measure |
                    YOffset DialogItemId Measure |
                    XY Measure Measure           |
                    ItemBox Int Int Int Int
:: IconLook      :== SelectState -> [DrawFunction]
:: TextWidth     :== Measure
:: NrEditLines   :== Int
:: RadioItemDef * s * io =  RadioItem DialogItemId ItemTitle SelectState (DialogFunction s io)
:: CheckBoxDef * s * io  =  CheckBox DialogItemId ItemTitle SelectState MarkState (DialogFunction s io)
 
:: ControlState =  IntCS    Int                       |
                   BoolCS   Bool                      |
                   RealCS   Real                      | 
                   StringCS String                    |
                   PairCS   ControlState ControlState |
                   ListCS   [ControlState]
:: ControlLook :== SelectState ->  ControlState -> [DrawFunction] 
:: ControlFeel :== MouseState ->   ControlState -> (ControlState,[DrawFunction]) 
 
:: SetFunction    * s * io :== ButtonFunction s io
:: ResetFunction  * s * io :== ButtonFunction s io
:: DialogFunction * s * io :== DialogInfo -> (DialogState s io) -> DialogState s io
:: ButtonFunction *s *io :== DialogInfo -> ( s -> * (io -> (s,io)))

 
:: NoticeDef       =  Notice [String] NoticeButtonDef [NoticeButtonDef]
:: NoticeButtonDef =  NoticeButton NoticeButtonId ItemTitle
:: NoticeButtonId  :== Int


//---------------------------------------------------------------------

PackDialogInfo :: ![ !(!DialogItemId, !ItemInfo) ] -> DialogInfo
PackDialogInfo info = info

UnpackDialogInfo :: !DialogInfo -> [ (!DialogItemId, !ItemInfo) ] 
UnpackDialogInfo info = info

PackDialogState :: !(DialogAdmin s) !OS -> DialogState s (IOState s)
PackDialogState a os = (a,os)

UnpackDialogState :: !(DialogState s io) -> ( !DialogAdmin s, !OS)
UnpackDialogState s = s

:: DialogAdmin *s = { dId     :: !Int,
                      dHandle :: !HWND,
					  dKind   :: !DialogKind,
					  dItems  :: ![ControlAdmin s]
					}

:: DialogKind = ModalK | ModelessK

:: ControlAdmin *s = { cId      :: !Int,
                       cHandle  :: !HWND,
					   cKind    :: !ControlKind s,
					   cEnabled :: !Bool,
					   cItems   :: ![ CheckRadioAdmin s ]
				     }

:: ControlKind *s = ButtonK       !(ButtonFunction s (IOState s))
                  | IconButtonK   !PictureState !PictureDomain !IconLook !(ButtonFunction s (IOState s))
			      | StaticTextK
			      | DynamicTextK  
			      | EditTextK     !NrEditLines  !String
			      | PopupK        !DialogItemId 
			      | RadioButtonsK !DialogItemId 
			      | CheckBoxesK   
			      | CustomK       !PictureState !PictureDomain !ControlState !ControlLook !ControlFeel !(DialogFunction s (IOState s))



:: CheckRadioAdmin *s = { caId      :: !DialogItemId,
                          caHandle  :: !HWND,
                          caKind    :: !CheckRadioKind,
						  caEnabled :: !Bool,
						  caFunct   :: !DialogFunction s (IOState s)
						}

:: CheckRadioKind = RadioK | CheckK !Bool




  //------------------------------------------------//
 //      Helper functions                          //
//------------------------------------------------//


CursorShape2Code :: !CursorShape -> Int
CursorShape2Code StandardCursor =  CURSARROW 
CursorShape2Code BusyCursor     =  CURSBUSY
CursorShape2Code IBeamCursor    =  CURSIBEAM
CursorShape2Code CrossCursor    =  CURSCROSS
CursorShape2Code FatCrossCursor =  CURSFATCROSS
CursorShape2Code ArrowCursor    =  CURSARROW
CursorShape2Code HiddenCursor   =  CURSHIDDEN



SetBetween :: !Int !Int !Int -> Int
SetBetween x low up 
  | x < low   = low
  | x > up    = up
              = x

MakeMouseState :: !Int !Int !Int !Int -> MouseState
MakeMouseState state x y mods = ( (x,y), toState state, toMods mods )
where
	toState BUTTONUP         = ButtonUp 
	toState BUTTONDOWN       = ButtonDown 
	toState BUTTONDOUBLEDOWN = ButtonDoubleDown
	toState BUTTONTRIPLEDOWN = ButtonTripleDown 
	toState buttonstilldown  = ButtonStillDown

MakeKeyboardState :: !Int !Int !Int -> KeyboardState
MakeKeyboardState chcode state mods = ( toChar chcode, toState state, toMods mods )
where
  toState KEYDOWN   = KeyDown
  toState KEYUP     = KeyUp
  toState keyrepeat = KeyStillDown

toMods :: !Int -> Modifiers
toMods i = ( shifton, alton, ctrlon, ctrlon )
where
	shifton  =  i bitand SHIFTBIT <> 0
	alton    =  i bitand ALTBIT   <> 0
	ctrlon   =  i bitand CTRLBIT  <> 0


SelectStateEqual :: !SelectState !SelectState -> Bool
SelectStateEqual Able   Able   = True
SelectStateEqual Unable Unable = True
SelectStateEqual _      _      = False

MarkEqual  :: !MarkState !MarkState -> Bool
MarkEqual  Mark   Mark   = True
MarkEqual  NoMark NoMark = True
MarkEqual  _      _      = False

Enabled  :: !SelectState -> Bool
Enabled   Able  = True
Enabled Unable  = False

Marked  :: !MarkState -> Bool
Marked  Mark    =  True
Marked  NoMark  =  False


:: Id   :== Int
:: Size :== (Int,Int)


:: Perhaps a  =  OK !a
			   | Nope

RectangleDimensions :: !Rectangle -> (Int,Int)
RectangleDimensions ((l,t),(r,b))  = (w,h)
where
    w = abs (r-l)
	h = abs (b-t)




StateMap2 :: ( x s -> s) ![x] !s -> s
StateMap2 f [x:xs] s = StateMap2 f xs (f x s)
StateMap2 _ _ s = s

Select :: (x -> Bool) x ![x] -> (Bool, x)
Select cond n [x:xs]
  | cond x  =  (True, x)
            =  Select cond n xs
Select _ n _ = (False, n)


// smap :: (a s -> (b,s)) [a] s  -> ([b],s)
smap f as s :== smap_ as s
where
	smap_ []       s  =  ([],s)
	smap_ [a:rest] s  =  ([b:restb],finals)
	where
		(b,s2)         = f a s 
		(restb,finals) = smap_ rest s2



Error :: !String !String !String -> .x
Error rule modulename error = abort ( "Error in rule " +++ rule +++ " [" +++ modulename +++ "]: " +++ error +++ ".\n")

