module Calculator;

/*
	A simple desk calculator.

	Run the program using the "No Console" option (Application options).

	To generate an application for this program the memory of the Clean
	0.8 application should be set to at least 1600K.
	The linker needs an additional 500K of free memory inside or outside
	the Clean 0.8 application.
	To launch the generated application 550K of free memory is needed
	outside the Clean 0.8 application.
*/

import StdEnv;

import deltaEventIO, deltaPicture, deltaWindow, deltaTimer, deltaSystem;


    
::	* Number	:== (Which,Mode,Real,Char,Real);
::	Which			=  First | Second;
::	Mode			=  HasPoint Int | NoPoint | Clear;
::	* IO		:==  IOState Number;
::	* NumIO	:== (Number,IO);

     
	FileID		:== 1;
	QuitID		:== 2;
	WndwID		:== 1;
	TimerID		:== 1;
	Corner		:== (130,40);
	PicDomain	:== ((0,0),(5 + ButRight,5 + ButBot));
	ButRight		:== 5 +  NrButsX * 25 ;
	ButBot		:== 35 +  NrButsY * 20 ;
	NrButsX		:== 4;
	NrButsY		:== 6;
	Buttons		:== "sctA\261\303^C789/456*123-0.=+";
	PlusMin		:== '\261';
	Sqrt_new			:== '\303';
	ClearKey		:== '\033';
	Infinity		:== exp 3000.0;
	Blink			:== TicksPerSecond / 20;

    

Start	:: * World -> * World;
Start world =  CloseEvents events` world`;
	where {
	(state,events`)=: StartIO [menu,window,timer] (First,NoPoint,0.0,' ',0.0) [] events;
	(events,world`)=: OpenEvents world;
	menu		=: MenuSystem [PullDownMenu FileID "File" Able [
				    MenuItem QuitID "Quit" (Key 'Q') Able Quit]];
	window	=: WindowSystem [
				    FixedWindow WndwID Corner "Clcltr" PicDomain UpdateWindow
				      [GoAway Quit, Keyboard Able HandleKey, Mouse Able HandleClick]];
	timer		=: TimerSystem [Timer TimerID Unable Blink (UnHilite ((0,0),(0,0)))];
	};

//	The menu function for the Quit command

Quit	:: Number IO -> NumIO;
Quit nr io =  (nr, QuitIO io);

//	The Update function for the window

UpdateWindow	:: UpdateArea Number -> (Number, [DrawFunction]);
UpdateWindow area nr=:(First ,mode,num,op,arg2) =  (nr, [DrawCalculator num]);
UpdateWindow area nr=:(Second,mode,arg1,op,num) =  (nr, [DrawCalculator num]);

//	The keyboard function of the window

HandleKey	:: KeyboardState Number IO -> NumIO;
HandleKey (key,KeyUp,mod_new) nr io =  (nr,io);
HandleKey (key,KeyStillDown,mod_new) nr io =  (nr,io);
HandleKey ('v',down,mod_new) nr io =  HandleButton Sqrt_new nr io;
HandleKey ('/',down,mod_new) nr io =  HandleButton '/' nr io;
HandleKey (key,down,mod_new) nr io
	| key == ClearKey =  HandleButton 'C' nr io;
	| key == DelKey || key == BackSpKey =  HandleButton 'A' nr io;
	| key == EnterKey || key == ReturnKey =  HandleButton '=' nr io;
	| key == 'p' || key == 'm' =  HandleButton PlusMin nr io;
HandleKey (key,down,mod_new) nr io
	=  HandleButton key nr io;

//	The mouse function of the window

HandleClick	:: MouseState Number IO -> NumIO;
HandleClick (pos,ButtonUp,mod_new) nr io =  (nr,io);
HandleClick (pos,ButtonStillDown,mod_new) nr io =  (nr,io);
HandleClick ((mx,my),buttondown,mod_new) nr io
	| in_button =  HandleButton (Buttons !! butnr) nr io;
	=  (nr,io);
	where {
	(in_button,butnr)=: CalcButNr mx my;
	};

CalcButNr	:: Int Int -> (Bool,Int);
CalcButNr x y
	| x < 5 || x > ButRight || y < 35 || y > ButBot =  (False,3);
	=  (True,  NrButsX * ((y - 35) / 20)  +  (x - 5) / 25 );

//	Handle the press of a (virtual) calculator button

HandleButton	:: Char Number IO -> NumIO;

HandleButton '=' nr=:(First,mode,arg1,op,arg2) io
	=  (nr, HiliteButton '=' io);
HandleButton '=' (Second,mode,arg1,op,arg2) io
	=  ((First,Clear,num,' ',0.0), ShowEffect '=' num io);
	where {
	num=:ApplyOp op arg1 arg2;
		
	};
	
HandleButton PlusMin (First,mode,arg1,op,arg2) io
	=  ((First,mode,num,op,arg2), ShowEffect PlusMin num io);
	where {
	num=:0.0 - arg1;
		
	};
HandleButton PlusMin (Second,mode,arg1,op,arg2) io
	=  ((Second,mode,arg1,op,num), ShowEffect PlusMin num io);
	where {
	num=:0.0 - arg2;
		
	};
	
HandleButton mop (First,mode,arg1,op,arg2) io
	| mop == '^' || mop == Sqrt_new || mop == 's' || mop == 'c'  || mop == 't'
		=  ((First,Clear,num,op,arg2), ShowEffect mop num io);
	where {
	num=:ApplyMonOp mop arg1;
		
	};
HandleButton mop (Second,mode,arg1,op,arg2) io
	| mop == '^' || mop == Sqrt_new || mop == 's' || mop == 'c'  || mop == 't'
		=  ((Second,Clear,arg1,op,num), ShowEffect mop num io);
	where {
	num=:ApplyMonOp mop arg2;
		
	};

HandleButton 'C' (First,mode,arg1,op,arg2) io
	=  ((First,Clear,0.0,' ',0.0), ShowEffect 'C' 0.0 io);
HandleButton 'C' (Second,mode,arg1,op,arg2) io
	=  ((Second,Clear,arg1,op,0.0), ShowEffect 'C' 0.0 io);

HandleButton 'A' (which,mode,arg1,op,arg2) io
	=  ((First,Clear,0.0,' ',0.0), ShowEffect 'A' 0.0 io);

HandleButton '.' (First,Clear,arg1,op,arg2) io
	=  ((First,HasPoint 1,0.0,' ',0.0), ShowEffect '.' 0.0 io);
HandleButton '.' (Second,Clear,arg1,op,arg2) io
	=  ((Second,HasPoint 1,arg1,op,0.0), ShowEffect '.' 0.0 io);
HandleButton '.' (which,NoPoint,arg1,op,arg2) io
	=  ((which,HasPoint 1,arg1,op,arg2), HiliteButton '.' io);
HandleButton '.' nr=:(which,HasPoint i,arg1,op,arg2) io
	=  (nr, HiliteButton '.' io);

HandleButton opt (First,mode,arg1,op,arg2) io
	| opt == '/' || opt == '*' || opt == '+' || opt == '-'
			=  ((Second,Clear,arg1,opt,0.0), HiliteButton opt io);
HandleButton opt (Second,mode,arg1,op,arg2) io
	| opt == '/' || opt == '*' || opt == '+' || opt == '-'
			=  ((Second,Clear,num,opt,0.0), ShowEffect opt num io);
	where {
	num=:ApplyOp op arg1 arg2;
		
	};

HandleButton dig (First,Clear,arg1,op,arg2) io
	| dig >= '0' && dig <= '9'
			=  ((First,NoPoint,num,op,arg2), ShowEffect dig num io);
	where {
	num=:DTOR dig;
		
	};
HandleButton dig (First,NoPoint,arg1,op,arg2) io
	| dig >= '0' && dig <= '9'
			=  ((First,NoPoint,num,op,arg2), ShowEffect dig num io);
	where {
	num=:AddDigit dig arg1;
		
	};
HandleButton dig (First,HasPoint i,arg1,op,arg2) io
	| dig >= '0' && dig <= '9' =  ((First,HasPoint (inc i),num,op,arg2),
	        ShowEffect dig num io);
	where {
	num=:AddFract i dig arg1;
		
	};
HandleButton dig (Second,Clear,arg1,op,arg2) io
	| dig >= '0' && dig <= '9'
			=  ((Second,NoPoint,arg1,op,num), ShowEffect dig num io);
	where {
	num=:DTOR dig;
		
	};
HandleButton dig (Second,NoPoint,arg1,op,arg2) io
	| dig >= '0' && dig <= '9'
			=  ((Second,NoPoint,arg1,op,num), ShowEffect dig num io);
	where {
	num=:AddDigit dig arg2;
		
	};
HandleButton dig (Second,HasPoint i,arg1,op,arg2) io
	| dig >= '0' && dig <= '9' =  ((Second,HasPoint (inc i),arg1,op,num),
	        ShowEffect dig num io);
	where {
	num=:AddFract i dig arg2;
		
	};

HandleButton but nr io =  (nr,io);


//	Apply an operator to its operand(s)

ApplyOp	:: Char Real Real -> Real;
ApplyOp '+' arg1 arg2 =  arg1 + arg2;
ApplyOp '-' arg1 arg2 =  arg1 - arg2;
ApplyOp '*' arg1 arg2 =  arg1 * arg2;
ApplyOp '/' arg1 0.0  =  Infinity;
ApplyOp '/' arg1 arg2 =  arg1 / arg2;
ApplyOp chr arg1 arg2 =  abort ("ApplyOp: Unknown operator: " +++  toString chr );

ApplyMonOp	:: Char Real -> Real;
ApplyMonOp Sqrt_new arg | arg >= 0.0 =  sqrt arg;
	                    =  Infinity;
ApplyMonOp '^'  arg =  arg * arg;
ApplyMonOp 's'  arg =  sin arg;
ApplyMonOp 'c'  arg =  cos arg;
ApplyMonOp 't'  arg =  tan arg;
ApplyMonOp chr  arg =  abort ("ApplyMonOp: Unknown operator: " +++  toString chr );
	

//	Draw the calculator

DrawCalculator	:: Real Picture -> Picture;
DrawCalculator num pict
	=  DrawNumber num (DrawButtons_and_Venster (
			SetFontName "Courier" (SetFontSize 10 pict)));

DrawButtons_and_Venster	:: Picture -> Picture;
DrawButtons_and_Venster pict
	=  DrawButtonLines 0 NrButsY 35 20 (DrawVenster frame);
	where {
	frame=: FillRectangle PicDomain (SetPenPattern LtGreyPattern pict);
	};

DrawVenster	:: Picture -> Picture;
DrawVenster pict
	=  DrawRectangle ((5,5),(ButRight,25)) (SetPenPattern BlackPattern erase);
	where {
	erase=: EraseVenster pict;
	};

EraseVenster	:: Picture -> Picture;
EraseVenster pict =  EraseRectangle ((6,6),(dec ButRight,24)) pict;

DrawButtonLines	:: Int Int Int Int Picture -> Picture;
DrawButtonLines butnr 0 y ofs pict =  pict;
DrawButtonLines butnr nr y ofs pict
	=  DrawButtonLines (butnr + NrButsX) (dec nr) (y + ofs) ofs
			(DrawButtons butnr NrButsX 5 25 (y + 2) pict);

DrawButtons	:: Int Int Int Int Int Picture -> Picture;
DrawButtons butnr 0 x ofs y pict =  pict;
DrawButtons butnr nr x ofs y pict
	=  DrawButtons (inc butnr) (dec nr) (x + ofs) ofs y
			(DrawButton butnr (x + 2) y pict);

DrawButton	:: Int Int Int Picture -> Picture;
DrawButton butnr x y pict
	=  DrawChar (Buttons !! butnr) (MovePenTo (x + 7,y + 11) button);
	where {
	button	=: DrawRectangle butrect (EraseRectangle butrect shade);
	shade		=: FillRectangle ((x + 2,y + 2),(x + 21,y + 16)) black;
	black		=: SetPenPattern BlackPattern pict;
	butrect	=: ((x,y),(x + 19,y + 14));
	};

DrawNumber	:: Real Picture -> Picture;
DrawNumber num pict =  DrawString (HandleError (toString num)) (EraseVenster pos);
	where {
	pos=: MovePenTo (10,18) pict;
	};

HandleError	:: String -> String;
HandleError str | c == '-' || (c >= '0' && c <= '9') =  str;
	                =  "Error";
	where {
	c=: str !! 0;
		
	};

//	Show the effect of a (virtual) key-press of the calculator

ShowEffect	:: Char Real IO -> IO;
ShowEffect but num io
	=  DrawInActiveWindow [DrawNumber num] (HiliteButton but io);

HiliteButton	:: Char IO -> IO;
HiliteButton but io
	=  EnableTimer TimerID (ChangeTimerFunction TimerID (UnHilite butrect) disable);
	where {
	disable=: DisableActiveKeyboard (DisableActiveMouse hilite);
	hilite =: DrawInActiveWindow [DrawHilite butrect] io;
	butrect=: GetButRect but;
	};
	
DrawHilite	:: Rectangle Picture -> Picture;
DrawHilite but pict
	=  SetPenMode CopyMode (FillRectangle but (SetPenMode XorMode pict));

GetButRect	:: Char -> Rectangle;
GetButRect but =  ((x,y),(x + 19,y + 14));
	where {
	x=: 7 +  25 * (nr mod NrButsX) ;
	y=: 37 +  20 * (nr / NrButsX) ;
	nr=: GetButNr 0 but Buttons;
	};

GetButNr	:: Int Char String -> Int;
GetButNr i but buttons
	| i >=  # buttons  =  3;
	| but ==  buttons !! i  =  i;
	=  GetButNr (inc i) but buttons;

//	Device function for the timer

UnHilite	:: Rectangle TimerState Number IO -> NumIO;
UnHilite but ts nr io =  (nr, DisableTimer TimerID unhilite);
	where {
	unhilite=: DrawInActiveWindow [DrawHilite but] enable;
	enable=: EnableActiveKeyboard (EnableActiveMouse io);
	};

//	Adding digits to real numbers:

AddDigit	:: Char Real -> Real;
AddDigit dig num =   DTOR dig  +  10.0 * num ;

AddFract	:: Int Char Real -> Real;
AddFract nrfracs dig num
	=    DTOR dig  /  10.0 ^  toReal nrfracs + num;

DTOR	:: Char -> Real;
DTOR c =  toReal ( toInt c  - 48);
