implementation module EdText;

/*	Operations on the Text: A list of lists of lines
	Operations on Lines: lists of strings
	
	To prevent space-leaks strategic strictness annotations are necessary
*/

import	StdClass;
import StdInt, StdString, StdChar, StdBool;
from StdMisc import abort;

import EdConstants, EdTypes;

     

EmptyText	:== [[EmptyLine]];
EmptyLine	:== [NewlStr];

LineNr  id	:== id mod LinesPerBlock;
BlockNr id	:== id / LinesPerBlock;


    

/*	Retrieve a line from the text */

Text_GetLine	:: Int Text -> TLine;
Text_GetLine id text =  Select (Select text (BlockNr id)) (LineNr id);

/*	Retrieve a number of lines from the text: arg1:1st linenr, arg2:nr of lines */

Text_GetLines	:: Int Int Text -> [TLine];
Text_GetLines id 0 text =  [];
Text_GetLines id nr text =  GetLines (LineNr id) (BlockNr id) nr text;

GetLines	:: !Int !Int !Int !Text -> [TLine];
GetLines lid 0 nr [block : blocks] =  GetLinesFromBlock lid nr block blocks;
GetLines lid bid nr [block : blocks] =  GetLines lid (dec bid) nr blocks;
GetLines lid bid nr [] =  abort "GetLines: wrong line number";

GetLinesFromBlock	:: !Int !Int !Block !Text -> [TLine];
GetLinesFromBlock 0 nr block blocks =  GetNrLines nr block blocks;
GetLinesFromBlock lid nr [line:lines] blocks
	=  GetLinesFromBlock (dec lid) nr lines blocks;
GetLinesFromBlock lid nr [] blocks
	=  abort "GetLinesFromBlock: wrong line number";

GetNrLines	:: !Int !Block !Text -> [TLine];
GetNrLines 0 block blocks =  [];
GetNrLines nr [line:lines] blocks =  let! {
		strict1;
		} in
		[line : strict1];
	where {
	strict1=GetNrLines (dec nr) lines blocks;
		
	};
GetNrLines nr [] [block:blocks] =  GetNrLines nr block blocks;
GetNrLines nr block blocks =  abort "GetNrLines: wrong number of lines";

/*	Replace a line in the text */

Text_SetLine	:: Int TLine Text -> Text;
Text_SetLine id line text =  Update2 text (BlockNr id) (LineNr id) line;

/*	Insert a line in the text */

Text_InsertLine	:: Int TLine Text -> Text;
Text_InsertLine id line text
	=  InsertLine line (LineNr id) (BlockNr id) text;

InsertLine	:: TLine Int Int Text -> Text;
InsertLine line 0 0 [] =  [[line]];
InsertLine line lid 0 [block:blocks]
	| save = 	[block` : blocks];
	= 	let! {
		strict1;
		} in
		[block` : strict1];
	where {
	(save,block`,carry)=: InsertInBlock line lid (dec LinesPerBlock) block;
	strict1=InsertLine carry 0 0 blocks;
		};
InsertLine line lid bid [block:blocks]
	= 	let! {
		strict1;
		} in
		[block : strict1];
	where {
	strict1=InsertLine line lid (dec bid) blocks;
		
	};
InsertLine line lid bid []
	=  abort "InsertLine: wrong line number";

InsertInBlock	:: TLine Int Int Block -> (!Bool,!Block,!TLine);
InsertInBlock line 0 max block =  (save, [line:block`], carry);
	where {
	(save,block`,carry)=: IsBlockTooLong max block;
	};
InsertInBlock line lid max [this:lines] =  (save, [this:block], carry);
	where {
	(save,block,carry)=: InsertInBlock line (dec lid) (dec max) lines;
	};
InsertInBlock line lid max []
	=  abort "InsertInBlock: wrong line number";

IsBlockTooLong	:: Int Block -> (!Bool,!Block,!TLine);
IsBlockTooLong n [] =  (True, [], EmptyLine);
IsBlockTooLong 0 [line] =  (False, [], line);
IsBlockTooLong n [line:lines] =  (save, [line:block], carry);
	where {
	(save,block,carry)=: IsBlockTooLong (dec n) lines;
	};

/*	Remove a line from a text */

Text_RemoveLine	:: Int Text -> Text;
Text_RemoveLine 0 text=:[[line]] =  EmptyText;
Text_RemoveLine id text =  RemoveLine (LineNr id) (BlockNr id) text;

RemoveLine	:: Int Int Text -> Text;
RemoveLine 0   0 [[line]] =  [];
RemoveLine lid 0 [block]  =  let! {
		strict1;
		} in
		[strict1];
	where {
	strict1=RemoveFromBlock lid block;
		
	};
RemoveLine lid 0 [ block : blocks=: [[line:lines] : rest] ]
	=  let! {
		strict1;
		strict2;
		} in
		[strict1 : strict2];
	where {
	strict1=RemoveFromBlockAndAppend lid line block;
		strict2=RemoveLine 0 0 blocks;
		
	};
RemoveLine lid bid [block:blocks] =  let! {
		strict1;
		} in
		[block : strict1];
	where {
	strict1=RemoveLine lid (dec bid) blocks;
		
	};
RemoveLine lid bid [] =  abort "RemoveLine: wrong line number";

RemoveFromBlock	:: Int Block -> Block;
RemoveFromBlock 0 [line:lines] =  lines;
RemoveFromBlock lid [line:lines] =  let! {
		strict1;
		} in
		[line : strict1];
	where {
	strict1=RemoveFromBlock (dec lid) lines;
		
	};
RemoveFromBlock lid [] =  abort "RemoveFromBlock: wrong line number";

RemoveFromBlockAndAppend	:: Int TLine Block -> Block;
RemoveFromBlockAndAppend 0 line [this:lines] =  Append lines line;
RemoveFromBlockAndAppend lid line [this:lines]
	=  let! {
		strict1;
		} in
		[this : strict1];
	where {
	strict1=RemoveFromBlockAndAppend (dec lid) line lines;
		
	};
RemoveFromBlockAndAppend lid line []
	=  abort "RemoveFromBlockAndAppend: wrong line number";

/*	Cut a Selection from the Text to the Clipboard */

Text_CutSelection	:: Selection Text -> (Clipboard,Text);
Text_CutSelection ((ln1,cn1,ln2,cn2),spix) text
	| ln1 == ln2 =  ([],text);	// Clearing in one line only affects current line
	=  CutSelection (LineNr ln1) (BlockNr ln1)
		               (LineNr ln2) (BlockNr ln2) cn1 cn2 text;

CutSelection	:: !Int !Int !Int !Int !Int !Int !Text -> (!Clipboard,!Text);
CutSelection li1 0 li2 bi2 cn1 cn2 [block:blocks]
	=  (clip, LinesToText lines);
	where {
	(clip,lines)=: CutSelFromBlock li1 li2 bi2 cn1 cn2 block blocks;
	};
CutSelection li1 bi1 li2 bi2 cn1 cn2 [block:blocks]
	=  (clip, [block : rest]);
	where {
	(clip,rest)=: CutSelection li1 (dec bi1) li2 (dec bi2) cn1 cn2 blocks;
	};

CutSelFromBlock	:: !Int !Int !Int !Int !Int !Block !Text -> (!Clipboard,![TLine]);
CutSelFromBlock 0 li2 bi2 cn1 cn2 [line:lines] blocks
	=  let! {
		last;
		strict1;
		strict3;
		strict4;
		strict5;
		} in
		([strict1 : clip], [strict3 : rest]);
	where {
	(clip,second,rest)=: strict4;
	(first,last)=: strict5;
	li2`=: If (bi2 == 0) (dec li2) li2;
	strict1=Line_LineToString last;
		strict3=Line_GlueLine first second;
		strict4=CutRestSelection li2` bi2 cn2 lines blocks;
		strict5=Line_SplitLine cn1 line;
		};
CutSelFromBlock li1 li2 bi2 cn1 cn2 [line:lines] blocks
	=  (clip, [line : rest]);
	where {
	(clip,rest)=: CutSelFromBlock (dec li1) li2` bi2 cn1 cn2 lines blocks;
	li2`=: If (bi2 == 0) (dec li2) li2;
	};

CutRestSelection	:: !Int !Int !Int !Block !Text -> (!Clipboard,!TLine,![TLine]);
CutRestSelection 0 0 cn2 [line:lines] blocks
	=  let! {
		str;
		strict1;
		} in
		([str], last, Lines_and_BlocksToLines lines blocks);
	where {
	str=: Line_LineToString (strict1);
	(first,last)=: Line_SplitLine cn2 line;
	strict1=Reverse first [];
		};
CutRestSelection li2 0 cn2 [line:lines] blocks
	=  let! {
		strict1;
		} in
		([strict1 : clip], last, rest);
	where {
	(clip,last,rest)=: CutRestSelection (dec li2) 0 cn2 lines blocks;
	strict1=Line_LineToString line;
		};
CutRestSelection li2 bi2 cn2 lines blocks
	=  CutBlock li2 bi2 cn2 lines blocks;

CutBlock	:: !Int !Int !Int !Block !Text -> (!Clipboard,!TLine,![TLine]);
CutBlock li2 bi2 cn2 [] [block:blocks]
	=  CutRestSelection li2 (dec bi2) cn2 block blocks;
CutBlock li2 bi2 cn2 [line:lines] blocks
	=  let! {
		strict1;
		} in
		([strict1 : clip], last, rest);
	where {
	(clip,last,rest)=: CutBlock li2 bi2 cn2 lines blocks;
	strict1=Line_LineToString line;
		};
CutBlock li2 bi2 cn2 lines blocks
	=  abort "CutBlock: wrong selection";

/*	Copy a Selection to the Clipboard */

Text_CopySelection	:: Selection Text -> Clipboard;
Text_CopySelection ((ln1,cn1,ln2,cn2),spix) text
	| ln1 == ln2 =  let! {
		strict1;
		} in
		[strict1];
	=  PreciseClipboard cn1 cn2 lines;
	where {
	line =: Line_LineToString (Text_GetLine ln1 text);
	lines=: Text_GetLines ln1 (inc (ln2 - ln1)) text;
	strict1=line% ( cn1, dec cn2);
		};

PreciseClipboard	:: !Int !Int ![TLine] -> Clipboard;
PreciseClipboard cn1 cn2 [line:lines]
	=  let! {
		strict1;
		strict2;
		} in
		[strict1 : strict2];
	where {
	str=: Line_LineToString line;
	strict1=str % (cn1, dec (# str));
		strict2=TransformToClip cn2 lines;
		};
PreciseClipboard cn1 cn2 []
	=  abort "PreciseClipboard: wrong selection";

TransformToClip	:: !Int ![TLine] -> Clipboard;
TransformToClip cn2 [line] =  let! {
		strict1;
		} in
		[strict1];
	where {
	strict1=(Line_LineToString line) % (0, dec cn2);
		
	};
TransformToClip cn2 [line:lines]
	=  let! {
		strict1;
		strict2;
		} in
		[strict1 : strict2];
	where {
	strict1=Line_LineToString line;
		strict2=TransformToClip cn2 lines;
		
	};
TransformToClip cn2 []
	=  abort "TransformToClip: wrong selection";

/*	Paste the Clipboard into the Text */

Text_PasteClipboard	:: Clipboard Int Int Text -> Text;
Text_PasteClipboard [] lnr cnr text
	=  abort "Text_PasteClipboard: empty clipboard";
Text_PasteClipboard [str] lnr cnr text
	=  text;				// Pasting of one string only affects current line
Text_PasteClipboard clip lnr cnr text
	=  InsertClipboard clip (LineNr lnr) (BlockNr lnr) cnr text;

InsertClipboard	:: !Clipboard !Int !Int !Int !Text -> Text;
InsertClipboard clip lid 0 cnr [block:blocks]
	=  LinesToText (InsertClipInBlock clip lid cnr block blocks);
InsertClipboard clip lid bid cnr [block:blocks]
	=  let! {
		strict1;
		} in
		[block : strict1];
	where {
	strict1=InsertClipboard clip lid (dec bid) cnr blocks;
		
	};
InsertClipboard clip lid bid cnr []
	=  abort "InsertClipboard: wrong line number";

InsertClipInBlock	:: !Clipboard !Int !Int !Block !Text -> [TLine];
InsertClipInBlock [str:rest] 0 cnr [line:lines] blocks
	=  let! {
		strict1;
		strict2;
		} in
		[strict1 : strict2];
	where {
	(part1,part2)=: ChopLine cnr line;
	strict1=Line_MakeLine (part1 +++ str);
		strict2=InsertClipInLine rest part2 lines blocks;
		};
InsertClipInBlock clip lid cnr [line:lines] blocks
	=  let! {
		strict1;
		} in
		[line : strict1];
	where {
	strict1=InsertClipInBlock clip (dec lid) cnr lines blocks;
		
	};
InsertClipInBlock clip lid cnr [] blocks
	=  abort "InsertClipInBlock: wrong line number";

InsertClipInLine	:: !Clipboard !String !Block !Text -> [TLine];
InsertClipInLine [str] last lines blocks
	=  let! {
		strict1;
		strict2;
		} in
		[strict1 : strict2];
	where {
	strict1=Line_MakeLine (str +++ last);
		strict2=Lines_and_BlocksToLines lines blocks;
		
	};
InsertClipInLine [str:rest] last lines blocks
	=  let! {
		strict1;
		strict2;
		} in
		[strict1 : strict2];
	where {
	strict1=Line_MakeLine str;
		strict2=InsertClipInLine rest last lines blocks;
		
	};
InsertClipInLine [] last lines blocks
	=  abort "InsertClipInLine: wrong clipboard";

Lines_and_BlocksToLines	:: !Block !Text -> [TLine];
Lines_and_BlocksToLines [] blocks =  BlocksToLines blocks;
Lines_and_BlocksToLines [line:lines] blocks
	=  let! {
		strict1;
		} in
		[line : strict1];
	where {
	strict1=Lines_and_BlocksToLines lines blocks;
		
	};
	
BlocksToLines	:: !Text -> [TLine];
BlocksToLines [] =  [];
BlocksToLines [block:blocks] =  Lines_and_BlocksToLines block blocks;

ChopLine	:: !Int !TLine -> (!String,!String);
ChopLine cnr line
	| cnr == 0 =  ("",str);
	=  (str % (0, dec cnr), str % (cnr,  dec (# str)));
	where {
	str=: Line_LineToString line;
	};

/*	Clear a Selection from the Text

::	Text_ClearSelection Selection Text -> Text;
	Text_ClearSelection ((ln1,cn1,ln2,cn2),spix) text
	-> text, IF = ln1 ln2	== Clearing in one line only affects current line
	-> ClearSelection (LineNr ln1) (BlockNr ln1)
		               (LineNr ln2) (BlockNr ln2) cn1 cn2 text;

::	ClearSelection !INT !INT !INT !INT !INT !INT !Text -> Text;
	ClearSelection li1 0 li2 bi2 cn1 cn2 [block:blocks]
	-> LinesToText (ClearSelFromBlock li1 li2 bi2 cn1 cn2 block blocks);
	ClearSelection li1 bi1 li2 bi2 cn1 cn2 [block:blocks]
	-> [block : !ClearSelection li1 (-- bi1) li2 (-- bi2) cn1 cn2 blocks];

::	ClearSelFromBlock !INT !INT !INT !INT !INT !Block !Text -> [TLine];
	ClearSelFromBlock 0 li2 bi2 cn1 cn2 [line:lines] blocks
	-> [!Line_GlueLine first last : !rest],
	(last,rest): !ClearRestSelection li2' bi2 cn2 lines blocks,
	(first,dum): !Line_SplitLine cn1 line,
	li2': If (= bi2 0) (-- li2) li2;
	ClearSelFromBlock li1 li2 bi2 cn1 cn2 [line:lines] blocks
	-> [line : !ClearSelFromBlock (-- li1) li2' bi2 cn1 cn2 lines blocks],
	li2': If (= bi2 0) (-- li2) li2;

::	ClearRestSelection !INT !INT !INT !Block !Text -> (!TLine,![TLine]);
	ClearRestSelection 0 0 cn2 [line:lines] blocks
	-> (last, Lines&BlocksToLines lines blocks),
	(dum,last): Line_SplitLine cn2 line;
	ClearRestSelection li2 0 cn2 [line:lines] blocks
	-> ClearRestSelection (-- li2) 0 cn2 lines blocks;
	ClearRestSelection li2 bi2 cn2 lines [block:blocks]
	-> ClearRestSelection li2 (-- bi2) cn2 block blocks;
	ClearRestSelection li2 bi2 cn2 lines blocks
	-> ABORT "ClearRestSelection: wrong selection";
*/

/*	Transform a list of STRINGs into a Text */

Text_StringsToText	:: [String] -> Text;
Text_StringsToText [] =  EmptyText;
Text_StringsToText strings =  StringsToText strings;

StringsToText	:: [String] -> Text;
StringsToText [] =  [];
StringsToText strings =  let! {
		strict1;
		} in
		[block : strict1];
	where {
	(block,rest)=: SplitTake_and_Transform LinesPerBlock strings;
	strict1=StringsToText rest;
		};

SplitTake_and_Transform	:: Int [String] -> (![TLine],![String]);
SplitTake_and_Transform n [] =  ([],[]);
SplitTake_and_Transform 0 list =  ([],list);
SplitTake_and_Transform n [str:r] =  ([line:more], rest);
	where {
	(more,rest)=: SplitTake_and_Transform (dec n) r;
	line=: MakeLine 0 (# str) str;
	};

/*	Transform a list of Lines into a Text */

LinesToText	:: [TLine] -> Text;
LinesToText [] =  [];
LinesToText lines =  let! {
		strict1;
		} in
		[block : strict1];
	where {
	(block,rest)=: SplitTake LinesPerBlock lines;
	strict1=LinesToText rest;
		};

SplitTake	:: Int [TLine] -> (![TLine],![TLine]);
SplitTake n [] =  ([],[]);
SplitTake 0 list =  ([],list);
SplitTake n [line:r] =  ([line:more], rest);
	where {
	(more,rest)=: SplitTake (dec n) r;
	};

/*	Transform a STRING into a TLine */

Line_MakeLine	:: String -> TLine;
Line_MakeLine str =  MakeLine 0 (# str) str;

MakeLine	:: Int Int String -> TLine;
MakeLine i len string
	| i >= len =  [];
	=  let! {
		strict1;
		} in
		[str : strict1];
	where {
	(str,newi)=: UptoTab i len string;
	strict1=MakeLine newi len string;
		};

UptoTab	:: Int Int String -> (!String,!Int);
UptoTab i len string
	|  string !! i  == TabChar =  (TabStr, inc i);
	=  UptoNextTab i len string;

UptoNextTab	:: Int Int String -> (!String,!Int);
UptoNextTab i len string
	| i >= len || char == TabChar =  ("",i);
	=  ( toString char  +++ str, newi);
	where {
	(str,newi)=: UptoNextTab (inc i) len string;
	char=: string !! i;
	};

/*	Transform a TLine into a STRING */

Line_LineToString	:: TLine -> String;
Line_LineToString [] =  "";
Line_LineToString [str:rest] =  str +++  Line_LineToString rest ;

/*	Split a line into a part before (reversed) and a part after the cursor */

Line_SplitLine	:: Int TLine -> (TLine,TLine);
Line_SplitLine cursor line
	| common == "" =  (before,after);
	=  ([befcom : before], [aftcom : after]);
	where {
	(before,after,common,sofar)=: PrimitiveSplit 0 cursor line [];
	befcom=: common % (0, dec comcur);
	aftcom=: common % (comcur, dec (# common));
	comcur=: cursor - sofar;
	};

PrimitiveSplit	:: Int Int TLine TLine -> (!TLine,!TLine,!String,!Int);
PrimitiveSplit sofar cursor [] before
	| sofar == cursor =  (before,[],"",sofar);
	=  abort "PrimitiveSplit: cursor not in line";
PrimitiveSplit sofar cursor after=:[str:rest] before
	| sofar == cursor =  (before,after,"",sofar);
	| newsof > cursor =  (before,rest,str,sofar);
	=  PrimitiveSplit newsof cursor rest [str:before];
	where {
	newsof=: sofar +  # str ;
	};

/*	Glue a a part before (reversed) and a part after the cursor into one line */

Line_GlueLine	:: TLine TLine -> TLine;
Line_GlueLine bef aft =  GlueLine (Reverse bef []) aft;

GlueLine	:: TLine TLine -> TLine;
GlueLine [] aft =  aft;
GlueLine before=:[bstr] [] =  before;
GlueLine [bstr] after=:[astr:rest]
	| bstr == TabStr || astr == TabStr =  [bstr : after];
	=  [bstr +++ astr : rest];
GlueLine [bstr:rest] aft =  let! {
		strict1;
		} in
		[bstr : strict1];
	where {
	strict1=GlueLine rest aft;
		
	};

/*	Glue a part before and a part after the cursor into one reversed line */

Line_GlueAfter	:: TLine TLine -> TLine;
Line_GlueAfter bef aft =  Reverse_and_RemoveNl (Line_GlueLine bef aft) [];

Reverse_and_RemoveNl	:: TLine TLine -> TLine;
Reverse_and_RemoveNl [str] rev
	| str == NewlStr =  rev;
	=  [str % (0,  # str  - 2) : rev];
Reverse_and_RemoveNl [str:rest] rev =  Reverse_and_RemoveNl rest [str:rev];

/*	Calculate the nr of chars in a line */

Line_NrChars	:: TLine -> Int;
Line_NrChars [] =  0;
Line_NrChars [str]
	|  str !! last  == NewlChar =  last;
	=  inc last;
	where {
	last=:(dec (# str));
		
	};
Line_NrChars [str:rest] =   # str  +  Line_NrChars rest ;


/*	Functions to change the current line */

LineToBefore	:: TLine TLine -> TLine;
LineToBefore [] before =  before;
LineToBefore [str] before
	| last < 1 =  before;
	|  str !! last  <> NewlChar =  [str : before];
	=  [str % (0, dec last) : before];
	where {
	last=: dec (# str);
	};
LineToBefore [str:rest] before =  LineToBefore rest [str:before];

BeforeToLine	:: TLine -> TLine;
BeforeToLine [] =  [NewlStr];
BeforeToLine before=:[TabStr:rest] =  Reverse [NewlStr : before] [];
BeforeToLine [str:rest] =  Reverse [str +++ NewlStr : rest] [];

/*	Cut a small Selection from the current line to the Clipboard */

CutFromCurLine	:: Selection CurLine -> (Clipboard,TLine,TLine);
CutFromCurLine ((ln1,cn1,ln2,cn2),psel) (bef,aft,lnr,cnr)
	| cn2 == cnr    =  ([clip1],before,aft);
	=  ([clip2],bef, after); // IF = cn1 cnr,
	where {
	(clip1,before)=: CutFromBefore (cn2 - cn1) bef "";
	(clip2, after)=: CutFromAfter  (cn2 - cn1) aft "";
	};

CutFromBefore	:: Int TLine String -> (String,TLine);
CutFromBefore 0 before clip =  (clip,before);
CutFromBefore n [TabStr:rest] clip
	=  CutFromBefore (dec n) rest (TabStr +++ clip);
CutFromBefore n [str:rest] clip
	| n > len =  CutFromBefore (n - len) rest (str +++ clip);
	| n == len =  (str +++ clip, rest);
	=  (str % (i, dec len)  +++ clip, [str % (0, dec i) : rest]);
	where {
	i=: len - n;
	len=:(# str);
		};

CutFromAfter	:: Int TLine String -> (String,TLine);
CutFromAfter 0 before clip =  (clip,before);
CutFromAfter n [TabStr:rest] clip
	=  CutFromAfter (dec n) rest (clip +++ TabStr);
CutFromAfter n [str:rest] clip
	| n > len =  CutFromAfter (n - len) rest (clip +++ str);
	| n == len =  (clip +++ str, rest);
	=  (clip +++  str % (0, dec n) , [str % (n, dec len) : rest]);
	where {
	len=:(# str);
		
	};


/*	Miscellaneous functions */

If	:: Bool t t -> t;
If True t e =  t;
If fals t e =  e;

Append	:: [t] t -> [t];
Append [] x =  [x];
Append [f:r] x =  [f : Append r x];

Reverse :: [t] [t] -> [t];
Reverse [] res =  res;
Reverse [f:r] res =  Reverse r [f:res];

Select	:: [t] !Int -> t;
Select [f:r] 0 =  f;
Select [f:r] i =  Select r (dec i);
Select list  i =  abort "MODULE EdText, Select: index out of range";

Update2	:: [[t]] !Int !Int t -> [[t]];
Update2 [f:r] 0 j x =  [Update_new f j x : r];
Update2 [f:r] i j x =  let! {
		strict1;
		} in
		[f : strict1];
	where {
	strict1=Update2 r (dec i) j x;
		
	};
Update2 list  i j x =  abort "MODULE EdText, Update2: index out of range";
	
Update_new	:: [t] !Int t -> [t];
Update_new [f:r] 0 x =  [x : r];
Update_new [f:r] i x =  let! {
		strict1;
		} in
		[f : strict1];
	where {
	strict1=Update_new r (dec i) x;
		
	};
Update_new list  i x =  abort "MODULE EdText, Update: index out of range";
