module LifeGameExample;

/*	This is the version of the LifeGame program that was used in chapter
	5 of the Concurrent Clean version 0.8.0 language manual to explain the
	I/O system of Concurrent Clean. A more sophisticated version (with
	multiple windows) can be found in LifeGame.icl.

	Run the program using the "No Console" option (Application options).

	To generate an application for this program the memory of the Clean
	0.8 application should be set to at least 1800K.
	The linker needs an additional 500K of free memory inside or outside
	the Clean 0.8 application.
	To launch the generated application 600K of free memory is needed
	outside the Clean 0.8 application.
*/

import StdEnv, deltaEventIO, deltaMenu;
import deltaWindow, deltaPicture, deltaTimer;
import Life;

    
::	* State	:== (Generation, Size);
::	Size			:== Int;
::	* IO		:==  IOState State;

    
Start	:: * World -> * World;
Start world =  CloseEvents events` world`;
		where {
		(state,events`)=: StartIO [window, null, menus] start_state [] events;
		(events,world`)=: OpenEvents world;
		start_state    =: ([], StartCellSize);
		window=: WindowSystem
					[	ScrollWindow WindowID Corner "Life" ScrollBarH_and_V
						ScrollBarH_and_V PicDomain MinSize InitSize UpdateWindow
						[GoAway Quit, Mouse Able Track]
					];
		null=:   TimerSystem [Timer TimerID Unable NullInterval TimerStep];
		menus=:  MenuSystem [file, optns, cmnds];
		file=:   PullDownMenu FileMenuID "File" Able
					[ MenuItem QuitID "Quit" (Key 'Q') Able Quit
					];
		optns=:  PullDownMenu OptionsMenuID "Options" Able
					[	MenuItem EraseID "Erase All Cells" (Key 'E') Able Erase,
				  		SubMenuItem CellSizeID "Cell Size" Able [sizes]
					];
		sizes=:  MenuRadioItems Size8ID
					[	MenuRadioItem Size1ID  "1 * 1" NoKey Able (ChangeSize 1),
						MenuRadioItem Size2ID  "2 * 2" NoKey Able (ChangeSize 2),
						MenuRadioItem Size4ID  "4 * 4" NoKey Able (ChangeSize 4),
						MenuRadioItem Size8ID  "8 * 8" NoKey Able (ChangeSize 8),
						MenuRadioItem Size16ID "16*16" NoKey Able (ChangeSize 16)
					];
		cmnds=:  PullDownMenu CommandsMenuID "Commands" Able
					[	MenuItem PlayID "Play" (Key 'P') Able Play,
						MenuItem HaltID "Halt" (Key 'H') Unable Halt,
						MenuItem StepID "Step" (Key 'S') Able Step
					];
		};

Quit	:: State IO -> (State, IO);
Quit state io =  (state, QuitIO io);

Play	:: State IO -> (State, IO);
Play state io =  (state, ChangeIOState [mouse,disable,enable,null] io);
		where {
		mouse  =: DisableActiveMouse;
		disable=: DisableMenuItems [PlayID, StepID, EraseID];
		enable =: EnableMenuItems [HaltID];
		null   =: EnableTimer TimerID;
		};

Halt	:: State IO -> (State, IO);
Halt state io =  (state, ChangeIOState [null,mouse,disable,enable] io);
		where {
		null   =: DisableTimer TimerID;
		mouse  =: EnableActiveMouse;
		disable=: DisableMenuItems [HaltID];
		enable =: EnableMenuItems [PlayID, StepID, EraseID];
		};

TimerStep	:: TimerState State IO -> (State, IO);
TimerStep tstate state io =  Step state io;

Step	:: State IO -> (State, IO);
Step (g,s) io =  ((next,s), ChangeIOState [draw_died,draw_new] io);
		where {
		draw_died		=: DrawInActiveWindow (Map (EraseCell s) died);
		draw_new			=: DrawInActiveWindow (Map (DrawCell  s) new);
		(next,new,died)=: LifeGame g;
		};

Erase	:: State IO -> (State, IO);
Erase (gen, size) io =  (([], size), DrawInActiveWindow [erase] io);
		where {
		erase=: EraseRectangle PicDomain;
		};

ChangeSize	:: Int State IO -> (State, IO);
ChangeSize ns (g,s) io =  ((g,ns), DrawInActiveWindow [erase:cells] io);
		where {
		erase=: EraseRectangle PicDomain;
		cells=: Map (DrawCell ns) g;
		};

UpdateWindow	:: UpdateArea State -> (State,[DrawFunction]);
UpdateWindow update_area state=:(g,s) =  (state, Map (DrawCell s) g);

Track	:: MouseState State IO -> (State, IO);
Track (pos, ButtonUp, modifiers) state io =  (state, io);
Track ((x,y), buttondown, (shift,option,command,control)) (g,s) io
	| command =  ((RemoveCell cell g, s), erase);
	=  ((InsertCell cell g, s), draw );
		where {
		erase=: DrawInActiveWindow [EraseCell s cell] io;
		draw =: DrawInActiveWindow [DrawCell  s cell] io;
		cell =: (nx / s, ny / s);  nx=: x -  x mod s ;  ny=: y -  y mod s ;
		};

DrawCell	:: Size LifeCell -> DrawFunction;
DrawCell size (x, y)
	= 	FillRectangle ((px, py), (px + size, py + size));
		where {
		px=: x * size;  py=: y * size;
		};

EraseCell	:: Size LifeCell -> DrawFunction;
EraseCell size (x, y)
	= 	EraseRectangle ((px, py), (px + size, py + size));
		where {
		px=: x * size;  py=: y * size;
		};

     
	FileMenuID			:== 1;
		QuitID 			:== 11;
	OptionsMenuID 		:== 2;
		EraseID			:== 21;
		CellSizeID 		:== 22;
			Size1ID		:== 221;
			Size2ID		:== 222;
			Size4ID		:== 223;
			Size8ID		:== 224;
			Size16ID		:== 225;
	CommandsMenuID 	:== 3;
		PlayID			:== 31;
		HaltID			:== 32;
		StepID			:== 33;

	WindowID				:== 1;
		Corner			:== (30, 50);
		PicDomain		:== ((0, 0), (1000, 1000));
		MinSize			:== (100, 100);
		InitSize			:== (512, 320);
		ScrollBarH_and_V	:== ScrollBar (Thumb 0) (Scroll StartCellSize);

	TimerID				:== 1;
		NullInterval	:== 0;

	StartCellSize			:== 8;
