implementation module EdTextFind;

import	StdClass;
import StdInt,StdChar,StdString,StdBool;

import EdText,EdConstants,EdTypes;

     
	IsOpenBracket  c :==  c == '('  || ( c == '{'  ||  c == '[' );
	IsCloseBracket c :==  c == ')'  || ( c == '}'  ||  c == ']' );
	Corresponds a b :==   a == '('  &&  b == ')'   || (  a == '['  &&  b == ']'   ||
	                        a == '{'  &&  b == '}'  );

    

//
//	Text_Balance returns the Selection of the smallest piece of balanced text surrounding
//	the cursor or the current selection.
//

Text_Balance	:: Int Int Int Int Text -> (Bool,PartSel);
Text_Balance bln bcn eln ecn text
	| not down || not up || not (Corresponds upb downb) =  (False, (0,0,0,0));
	=  (True, (lu,cu,eln + ld,inc cd));
	where {
	(down,ld,cd,downb)=: BalanceDownText eln ecn text;
	(up  ,lu,cu,upb  )=: BalanceUpText   bln bcn text;
	};

BalanceDownText	:: !Int !Int !Text -> (!Bool,!Int,!Int,!Char);
BalanceDownText ln cn [block:blocks]
	| ln < LinesPerBlock =  BalanceDownBlock ln cn block blocks;
	=  BalanceDownText (ln - LinesPerBlock) cn blocks;

BalanceDownBlock	:: !Int !Int !Block !Text -> (!Bool,!Int,!Int,!Char);
BalanceDownBlock 0 cn [line:lines] blocks
	=  BalanceDown 0 0 cn line` lines blocks;
	where {
	line`=: RemoveChars cn line;
	};
BalanceDownBlock ln cn [line:lines] blocks
	=  BalanceDownBlock (dec ln) cn lines blocks;

BalanceDown	:: !Int !Int !Int !TLine !Block !Text -> (!Bool,!Int,!Int,!Char);
BalanceDown ps ln cn [str:strs] lines blocks
	| fnd =  (True,ln,cn + ci,brak);
	=  BalanceDown ps` ln (cn + len) strs lines blocks;
	where {
	(fnd,ps`,ci,brak)	=: BalanceDownString ps 0 len str;
	len					=: # str;
	};
BalanceDown ps ln cn [] [line:lines] blocks
	=  BalanceDown ps (inc ln) 0 line lines blocks;
BalanceDown ps ln cn [] [] [[line:lines]:blocks]
	=  BalanceDown ps (inc ln) 0 line lines blocks;
BalanceDown ps ln cn [] [] []
	=  (False,0,0,' ');

BalanceDownString	:: !Int !Int !Int !String -> (!Bool,!Int,!Int,!Char);
BalanceDownString ps ci len str
	| ci >= len =  (False,ps,len,' ');
	| close && ps == 0 =  (True,0,ci,char);
	| close =  BalanceDownString (dec ps) (inc ci) len str;
	| IsOpenBracket char =  BalanceDownString (inc ps) (inc ci) len str;
	=  BalanceDownString ps      (inc ci) len str;
	where {
	close=: IsCloseBracket char;
	char =: str !! ci;
	};

BalanceUpText	:: !Int !Int !Text -> (!Bool,!Int,!Int,!Char);
BalanceUpText ln cn text
	| fnd =  (True,ln,dec (cn - ci),brak);
	=  BalanceUp 0 (dec ln) text;
	where {
	(fnd,ps,ci,brak)=: BalanceUpLine 0 0 (TakeChars cn (Text_GetLine ln text) []);
	};

BalanceUp	:: !Int !Int !Text -> (!Bool,!Int,!Int,!Char);
BalanceUp ps ln text
	| ln < 0 =  (False,0,0,' ');
	| fnd =  (True,ln,cn,brak);
	=  BalanceUp ps` (dec ln) text;
	where {
	(fnd,ps`,ci,brak)	=: BalanceUpLine ps 0 (Reverse line []);
	line					=: Text_GetLine ln text;
	cn						=:  Line_NrChars line  - ci;
	};

BalanceUpLine	:: !Int !Int !TLine -> (!Bool,!Int,!Int,!Char);
BalanceUpLine ps ci [] =  (False,ps,ci,' ');
BalanceUpLine ps ci [str:strs]
	| fnd =  (True,0,dec (ci + (len - ci`)),brak);
	=  BalanceUpLine ps` (ci + len) strs;
	where {
	(fnd,ps`,ci`,brak)=: BalanceUpString ps (dec len) str;
	len					=: # str;
	};

BalanceUpString	:: !Int !Int !String -> (!Bool,!Int,!Int,!Char);
BalanceUpString ps ci str
	| ci < 0 =  (False,ps,0,' ');
	| open && ps == 0 =  (True,0,ci,char);
	| open =  BalanceUpString (dec ps) (dec ci) str;
	| IsCloseBracket char =  BalanceUpString (inc ps) (dec ci) str;
	=  BalanceUpString ps      (dec ci) str;
	where {
	open=: IsOpenBracket char;
	char=: str !! ci;
	};

RemoveChars	:: !Int !TLine -> TLine;
RemoveChars 0 line =  line;
RemoveChars i [str:strs]
	| i >= len =  RemoveChars (i - len) strs;
	=  let! {
		strict1;
		} in
		[strict1 : strs];
	where {
	len=: # str;
	strict1= str % (i, dec len);
		};

TakeChars	:: !Int !TLine !TLine -> TLine;   // Returns the TLine reversed!!!
TakeChars 0 line res =  res;
TakeChars i [str:strs] res
	| i >= len =  TakeChars (i - len) strs [str:res];
	=  let! {
		strict1;
		} in
		[strict1 : res];
	where {
	len=: # str;
	strict1=str % (0, dec i);
		};

//
//	Text_ReplaceAll replaces all ocurrences of 'find' by 'replace'
// It returns the text-selection of the lastly replaced string and the new text
//

Text_ReplaceAll :: String String Bool Bool Text -> (Bool,PartSel,Text);
Text_ReplaceAll find replace ignore match text
	| find == "" || NoGoodFind find || NoGoodFind replace =  (False,(0,0,0,0),text);
	=  (found,(ln,cn,ln,cn +  # replace ),text`);
	where {
	(found,ln,cn,text`)=: ReplaceAll False 0 0 0 find replace text ignore match;
	};

ReplaceAll	:: !Bool !Int !Int !Int !String !String !Text !Bool !Bool
	           -> (!Bool,!Int,!Int,!Text);
ReplaceAll found li lnr cnr find replace [] i m =  (found,lnr,cnr,[]);
ReplaceAll found li lnr cnr find replace [block:blocks] i m
	=  let! {
		block`;
		rest;
		strict3;
		} in
		(found`,lnr`,cnr`,[block`:rest]);
	where {
	(found`,lnr`,cnr`,rest)	 =: ReplaceAll found1 li` lnr1 cnr1 find replace blocks i m;
	(found1,lnr1,cnr1,block`)=: strict3;
	li`							 =: li + LinesPerBlock;
	strict3=ReplaceAllInBlock found li lnr cnr find replace block i m;
		};

ReplaceAllInBlock	:: !Bool !Int !Int !Int !String !String !Block !Bool !Bool
	                  -> (!Bool,!Int,!Int,!Block);
ReplaceAllInBlock found li lnr cnr find replace [] i m =  (found,lnr,cnr,[]);
ReplaceAllInBlock found li lnr cnr find replace [line:lines] i m
	=  let! {
		line`;
		rest;
		strict3;
		} in
		(found`,lnr`,cnr`,[line`:rest]);
	where {
	(found`,lnr`,cnr`,rest)=: ReplaceAllInBlock found2 (inc li) lnr1 cnr1 find replace lines i m;
	(found1,cnr1,line`)=: strict3;
	lnr1=: If found1 li lnr;
	found2=: found || found1;
	strict3=ReplaceAllInLine False 0 cnr find replace line i m;
		};

ReplaceAllInLine	:: !Bool !Int !Int !String !String !TLine !Bool !Bool -> (!Bool,!Int,!TLine);
ReplaceAllInLine found ci cnr find replace [] i m =  (found,cnr,[]);
ReplaceAllInLine found ci cnr find replace [TabStr:strs] i m
	=  (found`,cnr`,[TabStr:rest]);
	where {
	(found`,cnr`,rest)=: ReplaceAllInLine found (inc ci) cnr find replace strs i m;
	};
ReplaceAllInLine found ci cnr find replace [str:strs] i m
	=  let! {
		str`;
		rest;
		strict3;
		} in
		(found`,cnr`,[str`:rest]);
	where {
	(found`,cnr`,rest)=: ReplaceAllInLine found2 (ci + len`) cnr2 find replace strs i m;
	(found1,cnr1,str`)=: strict3;
	cnr2					=: If found1 (ci + cnr1) cnr;
	found2				=: found || found1;
	len					=: # str;
	len`					=: # str`;
	strict3=ReplaceAllInString False 0 len 0 find replace str i m;
		};

ReplaceAllInString	:: !Bool !Int !Int !Int !String !String !String !Bool !Bool
	                   -> (!Bool,!Int,!String);
ReplaceAllInString found ci len cnr find replace str i m
	| len < flen =  (found,cnr,str);
	| StringMatch ci flen find str i m =  ReplaceAllInString True ci` len` ci find replace str` i m;
	=  ReplaceAllInString found (inc ci) (dec len) cnr find replace str i m;
	where {
	ci`	=: ci +  # replace ;
	len`	=: len - flen;
	str`	=:  MySlice str 0 (dec ci)  +++  replace +++  MySlice str (ci + flen) (dec slen)  ;
	flen	=: # find;
	slen	=: # str;
	};

	
//
//	Text_Find finds the next or previous occurrence of 'find'
// It returns the text-selection of the found string
//

Text_Find	:: Int Int Int String Bool Bool Bool Bool Text -> (Bool,PartSel);
Text_Find nrlines lnr cnr find ignore backw wrap match text
	| find == "" || NoGoodFind find =  (False,(0,0,0,0));
	| backw =  (found1,(l1,c1,l1,c1 + flen));
	=  (found2,(l2,c2,l2,c2 + flen));
	where {
	(found1,l1,c1)=: FindPrevious nrlines lnr cnr find ignore wrap match text;
	(found2,l2,c2)=: FindNext lnr lnr cnr find ignore wrap match text text;
	flen			  =: # find;
	};

FindPrevious	:: !Int !Int !Int !String !Bool !Bool !Bool !Text -> (!Bool,!Int,!Int);
FindPrevious nrlines lnr cnr find i w m text
	| fnd =  (True,lnr,dec (cnr - ci));
	=  FindUp nrlines (dec lnr) flen find i w m text;
	where {
	(fnd,ci)	=: FindUpLine 0 flen find i m (TakeChars cnr (Text_GetLine lnr text) []);
	flen		=: # find;
	};

FindUp	:: !Int !Int !Int !String !Bool !Bool !Bool !Text -> (!Bool,!Int,!Int);
FindUp nrlines ln flen find i wrap m text
	| ln < 0 && wrap =  FindUp nrlines (dec nrlines) flen find i False m text;
	| ln < 0 =  (False,0,0);
	| fnd =  (True,ln,cn);
	=  FindUp nrlines (dec ln) flen find i wrap m text;
	where {
	(fnd,ci)	=: FindUpLine 0 flen find i m (Reverse line []);
	line		=: Text_GetLine ln text;
	cn			=:  Line_NrChars line  - ci;
	};

FindUpLine	:: !Int !Int !String !Bool !Bool !TLine -> (!Bool,!Int);
FindUpLine ci flen find i m [] =  (False,ci);
FindUpLine ci flen find i m [str:strs]
	| flen > len || not fnd =  FindUpLine (ci + len) flen find i m strs;
	=  (True,dec (ci + (len - ci`)));
	where {
	(fnd,ci`)=: FindUpString (len - flen) flen find i m str;
	len		=: # str;
	};

FindUpString	:: !Int !Int !String !Bool !Bool !String -> (!Bool,!Int);
FindUpString ci flen find i m str
	| ci < 0 =  (False,ci);
	| StringMatch ci flen find str i m =  (True,ci);
	=  FindUpString (dec ci) flen find i m str;

FindNext	:: !Int !Int !Int !String !Bool !Bool !Bool !Text !Text -> (!Bool,!Int,!Int);
FindNext li lnr cnr find i w m [block:blocks] text
	| lnr < LinesPerBlock =  FindInBlock li lnr cnr find i w m block blocks text;
	=  FindNext li (lnr - LinesPerBlock) cnr find i w m blocks text;

FindInBlock :: !Int !Int !Int !String !Bool !Bool !Bool !Block !Text !Text
	            -> (!Bool,!Int,!Int);
FindInBlock li 0 cnr find i w m [line:lines] blocks text
	=  FindInLine li cnr cnr find i w m line lines blocks text;
FindInBlock li lnr cnr find i w m [line:lines] blocks text
	=  FindInBlock li (dec lnr) cnr find i w m lines blocks text;

FindInLine	:: !Int !Int !Int !String !Bool !Bool !Bool !TLine !Block !Text !Text
	           -> (!Bool,!Int,!Int);
FindInLine li ci cnr find i w m [str:strs] lines blocks text
	| cnr < len =  let! {
		strict1;
		} in
		Find li ci find i w m [strict1:strs] lines blocks text;
	=  FindInLine li ci (cnr - len) find i w m strs lines blocks text;
	where {
	len=: # str;
	strict1=str % (cnr, dec len);
		};

Find	:: !Int !Int !String !Bool !Bool !Bool !TLine !Block !Text !Text -> (!Bool,!Int,!Int);
Find li ci find i w m [str:strs] lines blocks text
	| found =  (True,li,ci + cnr);
	=  Find li (ci + len) find i w m strs lines blocks text;
	where {
	(found,cnr)=: FindInString 0 (# find) len find i m str;
	len		  =: # str;
	};
Find li ci find i w m [] [line:lines] blocks text
	=  Find (inc li) 0 find i w m line lines blocks text;
Find li ci find i w m [] [] [[line:lines]:blocks] text
	=  Find (inc li) 0 find i w m line lines blocks text;	
Find li ci find i wrap m [] [] [] [[line:lines]:blocks]
	| wrap =  Find 0 0 find i False m line lines blocks EmptyText;
	=  (False,0,0);

FindInString	:: !Int !Int !Int !String !Bool !Bool !String -> (!Bool,!Int);
FindInString ci flen len find i m str
	| len < flen =  (False,0);
	| StringMatch ci flen find str i m =  (True,ci);
	=  FindInString (inc ci) flen (dec len) find i m str;


//
//	Text_Replace replaces the string found on the given pos with 'replace'
//

Text_Replace	:: Int Int String String Text -> (Bool,Text);
Text_Replace l1 c1 find replace text
	| NoGoodFind replace =  (False,text);
	=  let! {
		strict1;
		} in
		(True,strict1);
	where {
	line`=: ReplaceInLine c1 (# find) replace (Text_GetLine l1 text);
	strict1=Text_SetLine l1 line` text;
		};

ReplaceInLine	:: !Int !Int !String !TLine -> TLine;
ReplaceInLine ci flen replace [str:rest]
	| ci < len =  [str` : rest];
	=  let! {
		strict1;
		str`;
		} in
		[str : strict1];
	where {
	str`=:  MySlice str 0 (dec ci)  +++  replace +++  MySlice str (ci + flen) (dec len)  ;
	len=: # str;
	strict1=ReplaceInLine (ci - len) flen replace rest;
		};

//
//	StringMatch checks if the string 'find' occurs in 'string' at position 'si'
//

StringMatch	:: !Int !Int !String !String !Bool !Bool -> Bool;
StringMatch si flen find string ignore match
	| match && (NonOutlineChar (dec si) string || NonOutlineChar (si + flen) string) =  False;
	| ignore =  IgnoreCaseStringMatch 0 si flen find string;
	=  NormalStringMatch 0 si flen find string;

IgnoreCaseStringMatch	:: !Int !Int !Int !String !String -> Bool;
IgnoreCaseStringMatch fi si flen find string
	| fi >= flen =  True;
	| IgnoreCase_sm__gr_C (find !! fi) (string !! si) =  False;
	=  IgnoreCaseStringMatch (inc fi) (inc si) flen find string;

NormalStringMatch	:: !Int !Int !Int !String !String -> Bool;
NormalStringMatch fi si flen find string
	| fi >= flen =  True;
	|  find !! fi  <>  string !! si  =  False;
	=  NormalStringMatch (inc fi) (inc si) flen find string;

IgnoreCase_sm__gr_C	:: !Char !Char -> Bool;
IgnoreCase_sm__gr_C c1 c2
	| c1 == c2 =  False;
	| c1 >= FirstSmall && c1 <= LastSmall =    toInt c1  - CaseOffset  <>  toInt c2 ;
	| c2 >= FirstSmall && c2 <= LastSmall =   toInt c1  <>   toInt c2  - CaseOffset ;
	=  True;

//
//	Functions to change the current line
//

AddCharBefore	:: Char TLine -> TLine;
AddCharBefore char [] =  [toString char];
AddCharBefore char bef=:[str:rest]
	| str == TabStr =  [charst : bef];
	=  [str +++ charst : rest];
	where {
	charst=: toString char;
	};

AddStringBefore	:: String TLine -> TLine;
AddStringBefore str [] =  Reverse (Line_MakeLine str) [];
AddStringBefore str before =  AddLineBefore (Line_MakeLine str) before;

AddLineBefore	:: TLine TLine -> TLine;
AddLineBefore [str] before=:[first:rest]
	| str == TabStr || first == TabStr =  [str:before];
	=  [first +++ str:rest];
AddLineBefore [str:rest] before =  AddLineBefore rest [str:before];

AddWordBefore	:: String TLine -> TLine;
AddWordBefore word [] =  [word];
AddWordBefore word before=:[TabStr:rest] =  [word:before];
AddWordBefore TabStr before =  [TabStr:before];
AddWordBefore word [str:rest] =  let! {
		strict1;
		} in
		[strict1 : rest];
	where {
	strict1=str +++ word;
		
	};

RemoveCharBefore	:: TLine -> (Char,TLine);
RemoveCharBefore [str:rest]
	| last == 0 =  (char, rest);
	=  let! {
		strict1;
		char;
		} in
		(char, [strict1 : rest]);
	where {
	char=: str !! last;
	last=: dec (# str);
	strict1=str % (0, dec last);
		};

RemoveWordBefore	:: TLine -> (String,TLine);
RemoveWordBefore [str:rest]
	| last == 0 =  (str,rest);
	| not (NonOutlineChar last str) =  let! {
		strict1;
		strict2;
		} in
		(strict1, [strict2 : rest]);
	| more == "" =  let! {
		word;
		} in
		(word,rest);
	=  let! {
		word;
		} in
		(word,[more:rest]);
	where {
	(word,more)=: RemoveWordBack str (dec last);
	last=: dec (# str);
	strict1=str % (last, last);
		strict2= str % (0, dec last);
		};

RemoveWordBack	:: String Int -> (!String,!String);
RemoveWordBack str i
	| i < 0 =  (str,"");
	| NonOutlineChar i str =  RemoveWordBack str (dec i);
	=  (str % (inc i, dec (# str)), str % (0, i));

AddCharAfter	:: Char TLine -> TLine;
AddCharAfter char [] =  [toString char];
AddCharAfter char bef=:[str:rest]
	| str == TabStr =  [charst : bef];
	=  [charst +++ str : rest];
	where {
	charst=: toString char;
	};

AddWordAfter	:: String TLine -> TLine;
AddWordAfter word [] =  [word];
AddWordAfter word after=:[TabStr:rest] =  [word : after];
AddWordAfter TabStr after =  [TabStr:after];
AddWordAfter word [str:rest] =  let! {
		strict1;
		} in
		[strict1 : rest];
	where {
	strict1=word +++ str;
		
	};

RemoveCharAfter	:: TLine -> (Char,TLine);
RemoveCharAfter [str:rest]
	| last == 0 =  (char, rest);
	=  (char, [str % (1, last) : rest]);
	where {
	char=: str !! 0;
	last=: dec (# str);
	};

RemoveWordAfter	:: TLine -> (String,TLine);
RemoveWordAfter [str:rest]
	| len == 1 =  (str,rest);
	| not (NonOutlineChar 0 str) =  let! {
		strict1;
		strict2;
		} in
		(strict1,[strict2 : rest]);
	| more == "" =  (word,rest);
	=  let! {
		strict3;
		} in
		(word,[more:rest]);
	where {
	(word,more)=: strict3;
	len=:(# str);
		strict3=RemoveWordFront str 1 len;
		strict1= str % (0, 0);
		strict2= str % (1, dec len);
		};

RemoveWordFront	:: String Int Int -> (!String,!String);
RemoveWordFront str i len
	| i >= len =  (str,"");
	| NonOutlineChar i str =  RemoveWordFront str (inc i) len;
	=  (str % (0, dec i), str % (i, dec len));


/*	NoGoodFind checks if the string passed to it contains non-printable characters */

NoGoodFind	:: String -> Bool;
NoGoodFind ""  =  False;
NoGoodFind str =  NonPrintablesInString (dec (# str)) str;

NonPrintablesInString	:: Int String -> Bool;
NonPrintablesInString 0 str
	=   char < FirstPrintable  ||  char > LastPrintable ;
	where {
	char=: str !! 0;
	};
NonPrintablesInString id str
	| char < FirstPrintable || char > LastPrintable =  True;
	=  NonPrintablesInString (dec id) str;
	where {
	char=: str !! id;
	};

/* NonOutlineChar checks whether the string contains an outline char on the indicated pos */

NonOutlineChar	:: !Int !String -> Bool;
NonOutlineChar i str
	| i < 0 || i >=  # str  ||
                c == ' ' || c == '"' || c == '\'' || c == '(' || c == ')' || c == '!' ||
		          c == ',' || c == '.' || c == ':'  || c == ';' || c == '[' || c == '?' ||
		          c == ']' || c == '{' || c == '|'  || c == '}' || c == NewlChar =  False;
	=  True;
	where {
	c=: str !! i;
	};

/* Misc. function(s) */

MySlice	:: String Int Int -> String;
MySlice str b e | b > e =  "";
	                =  str % (b, e);

Reverse :: [t] [t] -> [t];
Reverse [] res =  res;
Reverse [f:r] res =  Reverse r [f:res];
