#ifdef ARDUINO

#include <Arduino.h>
#include <stdbool.h>
#include <stdint.h>

#include "arduino.h"
#include "interface.h"
#include "symbols.h"
#include "mem.h"
#include "client.h"

#if defined(ARDUINO_ARCH_AVR)
	#define COMMUNICATOR Serial
	uint8_t apins[] = {A0, A1, A2, A3, A4, A5, A6, A7};
	uint8_t dpins[] = {0, 1, 2, 3, 4, 5, 6, 7, 8, 9, 10, 11, 12, 13};
//NodeMCU stuff
#elif defined(ARDUINO_ARCH_ESP8266)
	#include <ESP8266WiFi.h>
	#include "mtaskwifi.h"
	#define PORT 8123
	WiFiServer server(PORT);
	WiFiClient client;
	#define COMMUNICATOR client
	uint8_t apins[] = {A0};
	uint8_t dpins[] = {D0, D1, D2, D3, D4, D5, D6, D7};
//Unknown arduino device
#elif defined(ARDUINO_ARCH_ESP32)
	#include <WiFi.h>
	#include "mtaskwifi.h"
	#define PORT 8123
	WiFiServer server(PORT);
	WiFiClient client;
	#define COMMUNICATOR client
	uint8_t apins[] = {A0};
	uint8_t dpins[] = {0};
//Unknown arduino device
#else
	#error Unknown arduino device
#endif

// Additional sensors that require Wire should be added using
// ||
#if defined(HAVE_LIGHTSENSOR) || defined(HAVE_AIRQUALITYSENSOR)
	#include <Wire.h>
#endif

#ifdef HAVE_OLEDSHIELD
	#include <SPI.h>
	#include <Adafruit_GFX.h>
	#include <Adafruit_SSD1306.h>

	#define OLED_RESET 1
	Adafruit_SSD1306 display(OLED_RESET);
#endif

uint64_t getmillis(void)
{
	return millis();
}

void msdelay(unsigned long ms)
{
	delay(ms);
}

bool input_available(void)
{
#ifdef ARDUINO_ARCH_ESP8266
	if (!client.connected()) {
#ifdef HAVE_OLEDSHIELD
		display.print("Server disconnected");
		display.display();
		delay(1000);
#endif
		reset();
	}
#endif
	return COMMUNICATOR.available();
}

uint8_t read_byte(void)
{
	while (!input_available())
		msdelay(5);
	return COMMUNICATOR.read();
}

void write_byte(uint8_t b)
{
	COMMUNICATOR.write(b);
}

uint8_t read_apin(uint16_t t)
{
	return analogRead(t);
}

void write_apin(uint16_t p, uint8_t v)
{
#ifdef ARDUINO_ARCH_ESP32
	//TODO not implemented in the Arduino libraries for esp32 yet (https://github.com/espressif/arduino-esp32/issues/4)
#else
	analogWrite(p, v);
#endif
}

bool read_dpin(uint16_t t)
{
	return digitalRead(t);
}

void write_dpin(uint16_t p, bool v)
{
	digitalWrite(p, v ? HIGH : LOW);
}

void set_pinmode(uint16_t p, uint8_t mode)
{
	pinMode(p, mode == PMINPUT ? INPUT :
		mode == PMOUTPUT ? OUTPUT : INPUT_PULLUP);
}

uint16_t translate_pin(uint16_t i)
{
	//Digital pin
	if ((i & 1) > 0) {
		i >>= 1;
		return i >= sizeof(dpins) ? 0 : dpins[i];
	//Analog pin
	} else {
		i >>= 1;
		return i >= sizeof(apins) ? 0 : apins[i];
	}
}

#if defined(HAVE_DHT)

#if defined(ARDUINO_AVR_UNO)
#include <dht.h>
dht DHT;
#elif defined(ARDUINO_ESP8266_WEMOS_D1MINI)
#include <WEMOS_SHT3X.h>
SHT3X sht30(0x45);
#endif

#define dht_real_read(pin, type)\
	switch (type) {\
	case DHT11:\
		DHT.read11(pin);\
		break;\
	case DHT21:\
		DHT.read21(pin);\
		break;\
	case DHT22:\
		DHT.read22(pin);\
		break;\
	}\

bool dht_init(uint16_t pin, uint8_t type)
{
	return true;
	(void)pin;
	(void)type;
}

float get_dht_temp(uint16_t pin, uint8_t type)
{
#if defined(ARDUINO_AVR_UNO)
	dht_real_read(pin, type);
	//msg_log(SC("dht temp: %u\n"), (uint16_t)(DHT.temperature*10.0));
	return DHT.temperature;
#elif defined(ARDUINO_ESP8266_WEMOS_D1MINI)
	sht30.get();
	//msg_log(SC("dht temp: %u\n"), (uint16_t)(sht30.cTemp*10.0));
	return sht30.cTemp;
#endif
}

float get_dht_humidity(uint16_t pin, uint8_t type)
{
#if defined(ARDUINO_AVR_UNO)
	dht_real_read(pin, type);
	//msg_log(SC("dht humid: %u\n"), (uint16_t)(DHT.humidity*10.0));
	return DHT.humidity*10.0;
#elif defined(ARDUINO_ESP8266_WEMOS_D1MINI)
	sht30.get();
	//msg_log(SC("dht humid: %u\n"), (uint16_t)(sht30.humidity*10.0));
	return sht30.humidity*10.0;
#endif
}
#endif

#ifdef HAVE_I2CBUTTON
#include <LOLIN_I2C_BUTTON.h>
I2C_BUTTON button(0);
bool i2c_init(uint8_t addr)
{
	button.~I2C_BUTTON();
	new (&button) I2C_BUTTON(addr);
	return true;
}

uint8_t i2c_abutton(uint8_t addr)
{
	if (button.get() == 0) {
		return button.BUTTON_A;
	} else {
		die("Button error");
	}
	(void)addr;
}

uint8_t i2c_bbutton(uint8_t addr)
{
	(void)addr;
	if (button.get() == 0) {
		return button.BUTTON_B;
	} else {
		die("Button error");
	}
	(void)addr;
}
#endif


#ifdef HAVE_LIGHTSENSOR
#include <BH1750.h>
BH1750 lightSensor(0x23);
bool lightsensor_init(uint8_t addr) {
	lightSensor.~BH1750();
	new (&lightSensor) BH1750(addr);
	return lightSensor.begin(BH1750::ONE_TIME_HIGH_RES_MODE);
}

float get_light(uint8_t addr)
{
	return lightSensor.readLightLevel();
	(void)addr;
}
#endif

#ifdef HAVE_AIRQUALITYSENSOR
#include <SparkFunCCS811.h>
CCS811 airqualitySensor(0x5b);
bool airqualitysensor_init(uint8_t addr) {
	airqualitySensor.~CCS811();
	new (&airqualitySensor) CCS811(addr);
	CCS811Core::CCS811_Status_e stat = airqualitySensor.beginWithStatus();
	msg_log(airqualitySensor.statusString(stat));
	return stat;
}

void set_environmental_data(float humid, float temp, uint8_t addr) {
	airqualitySensor.setEnvironmentalData(humid, temp);
	(void)addr;
}

uint16_t get_tvoc(uint8_t addr) {
	ledmatrix_clear();
	airqualitySensor.readAlgorithmResults();
	return airqualitySensor.getTVOC();
	(void)addr;
}

uint16_t get_co2(uint8_t addr) {
	airqualitySensor.readAlgorithmResults();
	return airqualitySensor.getCO2();
	(void)addr;
}
#endif

#ifdef HAVE_LEDMATRIX
#include <WEMOS_Matrix_LED.h>
MLED mled(5);

bool ledmatrix_init(uint16_t dataPin, uint16_t clockPin)
{
	//mled.~MLED();
	//new (&mled) MLED(5, dataPin, clockPin);
	return true;
}
void ledmatrix_dot(uint8_t x, uint8_t y, bool state)
{
	mled.dot(x, y, state);
}
void ledmatrix_intensity(uint8_t intensity)
{
	mled.intensity = intensity;
}
void ledmatrix_clear(void)
{
	mled.clear();
}
void ledmatrix_display(void)
{
	mled.display();
}
#endif

void real_setup(void)
{
#if defined(ARDUINO_AVR_UNO)
	Serial.begin(9600);
#elif defined(ARDUINO_ARCH_ESP8266) || defined(ARDUINO_ARCH_ESP32)
#ifdef HAVE_OLEDSHIELD
	display.begin(SSD1306_SWITCHCAPVCC, 0x3C);
	display.clearDisplay();
	display.setTextSize(1);
	display.setCursor(0, 0);
	display.setTextColor(WHITE);
#endif

#if defined(HAVE_LIGHTSENSOR) || defined(HAVE_AIRQUALITYSENSOR)
	Wire.begin();
#endif

	const char *ssids[] = SSIDS;
	const char *wpas[] = WPAS;

	Serial.begin(115200);
	Serial.println("Peripherals:");

#ifdef HAVE_OLEDSHIELD
	Serial.println("Have OLED");
#endif
#ifdef HAVE_LEDMATRIX
	Serial.println("Have LEDMatrix");
	mled.clear();
	mled.display();
#endif
#ifdef HAVE_DHT
	Serial.println("Have DHT");
#endif
#ifdef HAVE_AIRQUALITYSENSOR
	Serial.println("Have AQS");
#endif
#ifdef HAVE_LIGHTSENSOR
	Serial.println("Have Lightsensor");
#endif
	int ssid = -1;
	do {
		Serial.println("Scanning...");
#ifdef HAVE_OLEDSHIELD
		display.println("Scanning..");
		display.display();
#endif
		int n = WiFi.scanNetworks();
		Serial.print("Found ");
		Serial.print(n);
		Serial.println(" networks");
#ifdef HAVE_OLEDSHIELD
		display.print(n);
		display.println(" found");
		display.display();
#endif
		if (n == 0) {
#ifdef HAVE_OLEDSHIELD
			display.println("none found");
			display.display();
#endif
			Serial.println("No networks found");
			delay(1000);
			continue;
		}
		for (unsigned j = 0; j<sizeof(ssids)/sizeof(char *); j++) {
			for (int i = 0; i < n; i++) {
				if (strcmp(WiFi.SSID(i).c_str(),
						ssids[j]) == 0) {
#ifdef HAVE_OLEDSHIELD
					display.print("Try ");
					display.println(WiFi.SSID(i));
					display.display();
#endif
					Serial.print("Connect to: ");
					Serial.println(WiFi.SSID(i));
					ssid = j;
				}
			}
		}
	} while (ssid == -1);
	WiFi.scanDelete();
	WiFi.begin(ssids[ssid], wpas[ssid]);
	while (WiFi.status() != WL_CONNECTED) {
		delay(1000);
		Serial.print(".");
#ifdef HAVE_OLEDSHIELD
		display.print(".");
		display.display();
#endif
	}
	Serial.print("\n");
#ifdef HAVE_OLEDSHIELD
	display.clearDisplay();
	display.setCursor(0, 0);
	display.print("Connected to: ");
	display.print(ssids[ssid]);
	display.print(" ");
	display.print(WiFi.localIP());
	display.print(":");
	display.println(PORT);
	display.display();
#endif
	Serial.println("WiFi connected");
	Serial.println("IP address: ");
	Serial.println(WiFi.localIP());

	server.begin();
	server.setNoDelay(true);
	Serial.print("Server started on port: ");
	Serial.println(PORT);

	Serial.println("Waiting for a client to connect.");
	while (!(client = server.available())) {
		delay(10);
	}

#ifdef HAVE_OLEDSHIELD
	display.println("Client:");
	display.print(client.remoteIP());
	display.display();
#endif
	Serial.println("Client connected\n");
#endif
	pinMode(LED_BUILTIN, OUTPUT);
	digitalWrite(LED_BUILTIN, LOW);
}

long lastping = 0;
void real_yield(void)
{
#ifdef ARDUINO_ARCH_ESP8266
	yield();
	if (millis()-lastping > 300) {
		write_byte(MTFPING);
		write_byte('\n');
		lastping = millis();
	}
#endif
}

#if LOGLEVEL > 0
void msg_log(const char *fmt, ...)
{
#if defined(ARDUINO_AVR_UNO)
	char buf[128];
	va_list args;
	va_start(args, fmt);
	vsnprintf_P(buf,128,fmt,args);
	va_end(args);
	write_byte(MTFDEBUG);
	for (uint8_t i = 0; i<128; i++) {
		if (buf[i] == '\0') {
			write_byte(i);
			break;
		}
	}
	for (uint8_t i = 0; i<128; i++) {
		if (buf[i] == '\0')
			break;
		write_byte(buf[i]);
	}
#else
	char buf[128];
	va_list args;
	va_start(args, fmt);
	vsnprintf_P(buf,128,fmt,args);
	va_end(args);
	Serial.write(buf);
	Serial.write('\r');
#endif
}
#endif

void die(const char *fmt, ...)
{
#ifdef HAVE_OLEDSHIELD
	display.clearDisplay();
	display.print("Error: ");
	display.print(fmt);
	display.display();
#endif
	Serial.println(fmt);
	while (1) {
		msdelay(1000);
		Serial.print("die");
	}
}

void pdie(char *s)
{
	die(s);
}

void reset(void)
{
#if defined(ARDUINO_AVR_UNO)
	Serial.end();
#elif defined(ARDUINO_ARCH_ESP8266)
#ifdef HAVE_LEDMATRIX
	mled.clear();
	mled.display();
#endif
	client.stop();
#endif
	mem_reset();
	real_setup();
}

void loop()
{
	real_loop();
}

void setup()
{
	real_main();
}
#endif
